#include "processor.h"
#include "name_resolver.h"
#include "exception.h"

#include <algorithm>
#include <akaxiso2/util/scoped_ptr.h>
#include <akaxiso2/util/string_funcs.h>


using namespace osx;


bool processor::is_form_qualified(const std::string &val) {
  if (aka::trim(val) == "unqualified")
    return false;
  if (aka::trim(val) == "qualified")
    return true;

  throw fatal_error(val + " does not specify form(qualified/unqualified).", __FILE__, __LINE__);
  return false;
}

bool processor::is_use_required(const std::string &val) {
  return aka::trim(val) == "required";
}

bool processor::is_use_prohibited(const std::string &val) {
  return aka::trim(val) == "prohibited";
}

aka::qname processor::create_attribute_name(const aka::qname &name, 
					    const std::string &form) {

  bool qualified = false;
  if (!form.empty())
    qualified = is_form_qualified(form);
  else if (attribute_qualified_)
    qualified = true;

  if (qualified)
    return name;

  if (name.get_namespace_id() == target_ns_id_)
    return aka::qname(name.local());
  else
    return name;
}


aka::qname processor::create_element_name(const aka::qname &name, 
					  const std::string &form) {
  bool qualified = false;
  if (!form.empty())
    qualified = is_form_qualified(form);
  else if (element_qualified_)
    qualified = true;

  if (qualified)
    return name;

  if (name.get_namespace_id() == target_ns_id_)
    return aka::qname(name.local());
  else
    return name;
}


aka::qname processor::create_anonymous_name(const aka::qname &name, const std::string &postfix,
					    size_t index) {
  std::ostringstream ostm;
  if (!name.local().empty())
    ostm << name.local() << '_';
  ostm << postfix << index;
  return aka::qname(name.get_namespace_id(), ostm.rdbuf()->str());
}


bool processor::is_attrDeclsEmpty(const xs::attrDecls &attrDecls) {
  return attrDecls.c0_.empty() && (attrDecls.anyAttribute_.get() == 0);
}



void processor::scan(const unit_props &unit) {

  current_unit_ = &registry_[unit];
  target_ns_id_ = unit.target_ns_id_;
  const xs::schema &schema = *unit.schema_;

  attribute_qualified_ = is_form_qualified(schema.attributeFormDefault_);
  element_qualified_ = is_form_qualified(schema.elementFormDefault_);

  const xs::schema_sequence_s0_holder &tops = schema.array0_;
  for (xs::schema_sequence_s0_holder::const_iterator it = tops.begin(); it != tops.end(); ++it) {
    const xs::schemaTop &top = it->schemaTop_;

    for (xs::schemaTop::const_iterator tit = top.begin(); tit != top.end(); ++tit) {
      const aka::qname &name = tit->get_name();
      if (name == aka::qname("&redefinable")) {
	const xs::redefinable &redefinable = aka::item_cast<xs::redefinable>(*tit);
	process_redefinable(redefinable);
      }
      else if (name == aka::qname("xs:element")) {
	const xs::topLevelElement &el = 
	  aka::item_cast<const xs::topLevelElement>(*tit);
	aka::qname name(target_ns_id_, el.name_);
	topLevelElements_.add(name, &el);
      }
      else if (name == aka::qname("xs:attribute")) {
	const xs::topLevelAttribute &attr = aka::item_cast<const xs::topLevelAttribute>(*tit);
	aka::qname name(target_ns_id_, attr.name_);
	topLevelAttributes_.add(name, &attr);
      }
      else if (name == aka::qname("xs:notation")) {
	// Nothing to do here.
      }
      else {
	assert(!"Must not reach here.");
      }
    }
  }
}


void processor::process(const unit_props &unit) {

  current_unit_ = &registry_[unit];
  target_ns_id_ = unit.target_ns_id_;
  const xs::schema &schema = *unit.schema_;

  attribute_qualified_ = is_form_qualified(schema.attributeFormDefault_);
  element_qualified_ = is_form_qualified(schema.elementFormDefault_);

  {
    for (topLevelSimpleTypes::iterator it = topLevelSimpleTypes_.begin();
	 it != topLevelSimpleTypes_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_) {
	process_topLevelSimpleType(it->second.name_, it->second.t_);
	current_unit_->toplevels_.insert(it->second.name_);
      }
    }
  }

  {
    for (topLevelAttributes::iterator it = topLevelAttributes_.begin();
	 it != topLevelAttributes_.end(); ++it)
      if (it->first.get_namespace_id() == target_ns_id_)
	process_topLevelAttribute(it->second.name_, *it->second.t_);
  }
  
  {
    for (topLevelAttributeGroups::iterator it = topLevelAttributeGroups_.begin();
	 it != topLevelAttributeGroups_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_)
	process_topLevelAttributeGroup(it->second.name_, it->second.t_);
    }
  }

  {
    for (topLevelElements::iterator it = topLevelElements_.begin();
	 it != topLevelElements_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_)
	process_topLevelElement(it->second.name_, it->second.t_); 
    }
  }
  
  {
    for (topLevelComplexTypes::iterator it = topLevelComplexTypes_.begin();
	 it != topLevelComplexTypes_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_) {
	process_topLevelComplexType(it->second.name_, it->second.t_);
	current_unit_->toplevels_.insert(it->second.name_);
      }
    }
  }

  {
    for (namedGroups::iterator it = namedGroups_.begin();
	 it != namedGroups_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_) {
	process_namedGroup(it->second.name_, it->second.t_);
	current_unit_->toplevels_.insert(it->second.name_);
      }
    }
  }
  
  {
    for (topLevelElements::iterator it = topLevelElements_.begin();
	 it != topLevelElements_.end(); ++it)
      if (it->first.get_namespace_id() == target_ns_id_)
	check_reference(*it->second.t_); 
  }
}

aka::qname processor::process_topLevelElement(const aka::qname &name, 
					      const xs::topLevelElement *el) {



  if (current_unit_->elements_.exists(name))
    return current_unit_->elements_.get(name).type_;

  if (verbose_)
    ostm_ << "processing <xs:element name=\"" << name << "\"/>" << std::endl;

  el = topLevelElements_.get(name);

  if (!el->substitutionGroup_.empty() || !el->block_.empty())
    throw not_supported(__FILE__, __LINE__);

  element_def edef;

  if (el->abstract_) {
    throw not_supported(__FILE__, __LINE__); // not supported.
    return aka::qname();
  }

  if (!el->type_.empty()) {
    if (!el->c0_.empty())
      throw fatal_error();
    edef.type_ = el->type_;
  }
  else if (!el->c0_.empty()) {
    assert(el->type_.empty());// Must satisfy.
    edef.type_ = process_typeChoice(name, el->c0_);
    current_unit_->toplevels_.insert(edef.type_);
  }
  else {
    /** empty <xs:element name="foo"/> */
    edef.type_ = aka::qname(registry_.get_nill_type());
  }
  assert(!edef.type_.empty());

  if (!el->fixed_.empty()) {
    edef.is_fixed_ = true;
    edef.default_ = el->fixed_;
  }
  else if (!el->default_.empty()) {
    edef.is_fixed_ = false;
    edef.default_ = el->default_;
  }

  current_unit_->elements_.insert(name, edef);

  return edef.type_;
}

void processor::check_reference(const xs::topLevelElement &el) {
  if (el.type_.empty())
    return;
  if (registry_.is_predefined(el.type_))
    return;
  if (registry_.is_xs_any(el.type_))
    return;
  if (current_unit_->simpleTypes_.exists(el.type_))
    return;
  topLevelComplexTypes_.get(el.type_);
}

void processor::process_redefinable(const xs::redefinable &redefinable) {

  for (xs::redefinable::const_iterator rit = redefinable.begin();
       rit != redefinable.end(); ++rit) {
    const aka::qname &redefname = rit->get_name();
    if (redefname == aka::qname("xs:simpleType")) {
      const xs::topLevelSimpleType &st = 
	aka::item_cast<const xs::topLevelSimpleType>(*rit);
      aka::qname stname(target_ns_id_, st.name_);
      topLevelSimpleTypes_.add(stname, &st);
    }
    else if (redefname == aka::qname("xs:complexType")) {
      const xs::topLevelComplexType &ct = 
	aka::item_cast<const xs::topLevelComplexType>(*rit);
      aka::qname ctname(target_ns_id_, ct.name_);
      topLevelComplexTypes_.add(ctname, &ct);
    }
    else if (redefname == aka::qname("xs:group")) {
      const xs::namedGroup &gr = aka::item_cast<const xs::namedGroup>(*rit);
      aka::qname name(target_ns_id_, gr.name_);
      namedGroups_.add(name, &gr);
    }
    else if (redefname == aka::qname("xs:attributeGroup")) {
      const xs::namedAttributeGroup &ag = 
	aka::item_cast<const xs::namedAttributeGroup>(*rit);
      aka::qname name(target_ns_id_, ag.name_);
      topLevelAttributeGroups_.add(name, &ag);
    }
    else {
      assert(!"Must not reach here.");
    }
  }
}


void processor::process_topLevelAttribute(const aka::qname &name, 
					  const xs::topLevelAttribute &tat) {

  if (current_unit_->gattributes_.exists(name))
    return;

  if (verbose_)
    ostm_ << "processing <xs:attribute name=\"" << name << "\"/>" << std::endl;

  gattribute_def attrdef;

  attrdef.name_ = name;
  if (!tat.type_.empty()) {
    if (tat.simpleType_.get() != 0)
      throw fatal_error();
    attrdef.type_ = tat.type_;
  }
  else if (tat.simpleType_.get() != 0) {
    assert(tat.type_.empty());
    attrdef.type_ = process_localSimpleType(name, *tat.simpleType_.get());
    current_unit_->toplevels_.insert(attrdef.type_);
  }
  else {
    // ur-type.
    attrdef.type_ = aka::qname("xs:anySimpleType");
  }

  if (!tat.default_.empty()) {
    attrdef.default_ = tat.default_;
  }
  else if (!tat.fixed_.empty()) {
    attrdef.default_ = tat.fixed_;
    attrdef.is_fixed_ = true;
  }

  current_unit_->gattributes_.insert(attrdef.name_, attrdef);
}

void processor::process_topLevelSimpleType(const aka::qname &name, 
					   const xs::topLevelSimpleType *st) {
  if (registry_.is_predefined(name))
    return;
  if (registry_.type_exists(name))
    return;

  if (verbose_)
    ostm_ << "processing <xs:simpleType name=\"" << name << "\"/>" << std::endl;

  // Check restriction/extension/union.... !!!!!!!!

  st = topLevelSimpleTypes_.get(name);
  aka::qname type = process_simpleDerivation(st->simpleDerivation_);

  class_def *def = new simpletype_def(name);
  def->set_value_type(type);
  current_unit_->classes_.insert(name, def);
  current_unit_->simpleTypes_.add(name); 
}

const class_def &processor::process_topLevelComplexType(const aka::qname &name, 
							const xs::topLevelComplexType *ct) {

  if (complexType_processing_.find(name) != complexType_processing_.end())
    throw fatal_error(); // cyclic processing.

  if (registry_.type_exists(name))
    return registry_.classdef_get(name);

  complexType_processing_.insert(name);

  if (ct == 0)
    ct = topLevelComplexTypes_.get(name);
  if (verbose_)
    ostm_ << "processing <xs:complexType name=\"" << name << "\"/>" << std::endl;

  const class_def &def = process_complexTypeModel(name, ct->complexTypeModel_);

  complexType_processing_.erase(name);

  return def;
}

void processor::process_topLevelAttributeGroup(const aka::qname &name, 
					       const xs::namedAttributeGroup *ag) {
  if (attrG_processing_.find(name) != attrG_processing_.end())
    throw fatal_error();

  if (registry_.attributeGroup_exists(name))
    return;

  attrG_processing_.insert(name);

  if (ag == 0)
    ag = topLevelAttributeGroups_.get(name);

  if (verbose_)
    ostm_ << "processing <xs:attributeGroup name=\"" << name << "\"/>" << std::endl;

  attributeGroup_def def;
  def.name_ = name;
  process_attrDecls(name, def, ag->attrDecls_);
  current_unit_->attributeGroups_.insert(def.name_, def);
  attrG_processing_.erase(name);
}


void processor::process_attrDecls(const aka::qname &hint,
				  attribute_defs &attrdefs, const xs::attrDecls &decls) {
  
  for (xs::attrDecls_c0::const_iterator ait = decls.c0_.begin();
       ait != decls.c0_.end(); ++ait) {
    if (ait->get_name() == aka::qname("xs:attribute")) {
      const xs::attribute &attr = aka::item_cast<xs::attribute>(*ait);
      process_attribute(hint, attrdefs, attr);
    }
    else if (ait->get_name() == aka::qname("xs:attributeGroup")) {
      const xs::attributeGroupRef &attrGroupRef =
	aka::item_cast<const xs::attributeGroupRef>(*ait);
      process_topLevelAttributeGroup(attrGroupRef.ref_);
      const attribute_defs &grdefs = registry_.attributeGroup_get(attrGroupRef.ref_);
      attrdefs.attributes_.insert(attrdefs.attributes_.end(), 
				  grdefs.attributes_.begin(), grdefs.attributes_.end());
      if (grdefs.has_anyAttribute_) {
	attrdefs.has_anyAttribute_ = grdefs.has_anyAttribute_;
	attrdefs.namespace_list_ = grdefs.namespace_list_;
      }
    }
    else {
      assert(!"Must not reach here.");
    }
  }

  xs::wildcard *anyAttribute = decls.anyAttribute_.get();

  if (attrdefs.has_anyAttribute_ && (anyAttribute != 0)) {
    std::string message = "xs:anyAttribute is declared in xs:attributeGroup and ";
    message += "\"" + hint.qualified() + "\".";
    throw fatal_error(message, __FILE__, __LINE__);
  }

  if (anyAttribute != 0) {
    attrdefs.has_anyAttribute_ = true;
    attrdefs.namespace_list_ = anyAttribute->namespace_;
  }
}

attribute_type processor::get_attribute_type(const aka::qname &hint, const xs::attribute &attr) {
  
  attribute_type attrtype;
  if (!attr.name_.empty())
    attrtype.name_ = create_attribute_name(aka::qname(attr.name_), attr.form_);
  else {
    // referenced attribute has global name.
    attrtype.name_ = attr.ref_;
  }
  
  /** Get type */
  if (!attr.ref_.empty()) {
    if (!attr.type_.empty() || (attr.simpleType_.get() != 0))
      throw fatal_error();
    const gattribute_def &gattr = registry_.get_globalAttribute(attr.ref_);
    attrtype.type_ = gattr.type_;
    attrtype.default_ = gattr.default_;
    attrtype.is_fixed_ = gattr.is_fixed_;
  }
  else if (!attr.type_.empty()) {
    if (attr.simpleType_.get() != 0)
      throw fatal_error();
    attrtype.type_ = attr.type_;
  }
  else if (attr.simpleType_.get() != 0) {
    aka::qname name(hint);
    name.local() += "_" + attr.name_;
    attrtype.type_ = process_localSimpleType(name, *attr.simpleType_.get());
  }
  else {
    // ur-type.
    attrtype.type_ = aka::qname("xs:anySimpleType");
  }

  if (!attr.default_.empty()) {
    if (is_use_required(attr.use_))
      throw fatal_error();
    attrtype.default_ = attr.default_;
  }
  else if (!attr.fixed_.empty()) {
    attrtype.default_ = attr.fixed_;
    attrtype.is_fixed_ = true;
  }
  else if (is_use_required(attr.use_)) {
    attrtype.required_ = true;
  }

  return attrtype;
}

void processor::process_attribute(const aka::qname &hint,
				  attribute_defs &attrdefs, const xs::attribute &attr) {
  attribute_type attrtype = get_attribute_type(hint, attr);
  attrdefs.attributes_.push_back(attrtype);
}

void processor::restrict_attrDecls(const aka::qname &hint,
				   attribute_defs &attrdefs, const xs::attrDecls &attrDecls) {

  for (xs::attrDecls_c0::const_iterator ait = attrDecls.c0_.begin();
       ait != attrDecls.c0_.end(); ++ait) {
    if (ait->get_name() == aka::qname("xs:attribute")) {
      
      const xs::attribute &attr = aka::item_cast<xs::attribute>(*ait);

      if (is_use_prohibited(attr.use_)) {
	if (attr.simpleType_.get() != 0)
	  throw fatal_error(); // !!!!! OK?

	// name, ref both must not be empty. or both must not be non-empty.
	if (attr.name_.empty() == attr.ref_.empty())
	  throw fatal_error();
	if (!attr.name_.empty())
	  prohibit_attribute(attrdefs, aka::qname(attr.name_));
	else if (!attr.ref_.empty()) {
	  prohibit_attribute(attrdefs, attr.ref_);
	}
      }
      else {
	attribute_type atype = get_attribute_type(hint, attr);
	restrict_attribute(attrdefs, atype);
      }
    }
    else if (ait->get_name() == aka::qname("xs:attributeGroup")) {
      const xs::attributeGroupRef &ref = 
	aka::item_cast<const xs::attributeGroupRef>(*ait);
      const attributeGroup_def &def = registry_.attributeGroup_get(ref.ref_);
      for (attribute_types::const_iterator it = def.attributes_.begin();
	   it != def.attributes_.end(); ++it) {
	const attribute_type &atype = *it;
	restrict_attribute(attrdefs, atype);
	// Don't have to consider restriction, because attributeGroup does not have @use.
      }
    }
    else {
      assert(!"Must not reach here.");
    }
  }

  xs::wildcard *ana = attrDecls.anyAttribute_.get();
  if (ana != 0) {
    if (attrdefs.has_anyAttribute_)
      attrdefs.namespace_list_ = ana->namespace_;
    else {
      throw fatal_error("Try to extend anyAttribute in xs:restriction.", __FILE__, __LINE__);
    }
  }
}

void processor::restrict_attribute(attribute_defs &attrdefs, 
				   const attribute_type &atype) {
  
  attribute_type &to_restrict = attrdefs.get(atype.name_);
  to_restrict.required_ = atype.required_;
  to_restrict.default_ = atype.default_;
  to_restrict.is_fixed_ = atype.is_fixed_;
  // restriction any_attribute.... !!!!!!!
}

void processor::prohibit_attribute(attribute_defs &attrdefs,
				   const aka::qname &tagname) {
  attrdefs.erase(tagname);
}

void processor::extend_attrDecls(const aka::qname &name,
				 attribute_defs &attrdefs, const xs::attrDecls &attrDecls) {

  for (xs::attrDecls_c0::const_iterator ait = attrDecls.c0_.begin();
       ait != attrDecls.c0_.end(); ++ait) {
    if (ait->get_name() == aka::qname("xs:attribute")) {
      const xs::attribute &attr = aka::item_cast<xs::attribute>(*ait);
      extend_attribute(name, attrdefs, attr);
    }
    else if (ait->get_name() == aka::qname("xs:attributeGroup")) {
      const xs::attributeGroupRef &agref = aka::item_cast<xs::attributeGroupRef>(*ait);
      const xs::namedAttributeGroup *ag = topLevelAttributeGroups_.get(agref.ref_);
      extend_attrDecls(name, attrdefs, ag->attrDecls_);
    }
    else {
      assert(!"Must not reach here.");
    }
  }
}


void processor::extend_attribute(const aka::qname &name,
				 attribute_defs &attrdefs, const xs::attribute &attr) {

  if (!attr.name_.empty() && ! attr.ref_.empty())
    throw fatal_error("@name and @ref should not be specified at the same time.", 
		      __FILE__, __LINE__);

  if (!attr.name_.empty()) {
    if (!attrdefs.exists(aka::qname(attr.name_)))
      process_attribute(name, attrdefs, attr);
    else {
      assert(!"Not implemented.");
    }
  }
  else if (!attr.ref_.empty()) {
    if (!attrdefs.exists(attr.ref_))
      process_attribute(name, attrdefs, attr);
    else {
      assert(!"Not implemented.");
    }
  }
  
  // anyAttribute? !!!!!!!!!!!!!!!!!!!!!!!!!!
  // extension??
}


const class_def &processor::process_namedGroup(const aka::qname &name, const xs::namedGroup *gr) {
  if (registry_.type_exists(name))
    return registry_.classdef_get(name);

  if (gr == 0)
    gr = namedGroups_.get(name);

  assert(gr->c0_.size() == 1);
  const aka::item &item = gr->c0_.front();
  if (item.get_name() == aka::qname("xs:sequence")) {
    const xs::simpleExplicitGroup &seg = aka::item_cast<xs::simpleExplicitGroup>(item);
    return process_sequence(name, aka::occurrence(), seg.nestedParticle_);
  }
  else if (item.get_name() == aka::qname("xs:choice")) {
    const xs::simpleExplicitGroup &seg = aka::item_cast<xs::simpleExplicitGroup>(item);
    return process_choice(name, aka::occurrence(), seg.nestedParticle_);
  }
  else if (item.get_name() == aka::qname("xs:all")) {
    const xs::namedGroup_sequence_c0_all &all = 
      aka::item_cast<xs::namedGroup_sequence_c0_all>(item);
    return process_all(name, aka::occurrence(), all.allModel_.element_);
  }
  assert(!"Must not reach here.");
  return *static_cast<class_def*>(0);
}

aka::qname processor::process_typeChoice(const aka::qname &name, const xschoice &choice) {
  if (choice.size() != 1)
    throw fatal_error();

  const aka::item &item = *choice.begin();
  if (item.get_name() == aka::qname("xs:simpleType"))
    return process_localSimpleType(name, aka::item_cast<const xs::localSimpleType>(item));
  else if (item.get_name() == aka::qname("xs:complexType"))
    return process_localComplexType(name, aka::item_cast<const xs::localComplexType>(item));
  assert(!"Must not reach here.");
  return aka::qname();
}

aka::qname processor::process_localSimpleType(const aka::qname &name, 
					      const xs::localSimpleType &lst) {
  
  aka::qname type = process_simpleDerivation(lst.simpleDerivation_); 
  class_def *def = new simpletype_def(name);
  def->set_value_type(type);
  current_unit_->classes_.insert(name, def);
  current_unit_->simpleTypes_.add(name); /////!!!!!!!!
  return name;
}

aka::qname processor::process_localComplexType(const aka::qname &name,
					       const xs::localComplexType &lct) {
  process_complexTypeModel(name, lct.complexTypeModel_);
  return name;
}


const class_def &processor::process_complexTypeModel(const aka::qname &name, 
						     const xs::complexTypeModel &model) {
  
  if (model.size() == 0) { // <xs:complexType name="hoge"/>
    class_def *def = new sequence_def(name, aka::occurrence());
    current_unit_->classes_.insert(name, def);
    return *def;
  }

  const class_def *def;

  assert(model.size() == 1);
  const aka::item &item = model.front();
  if (item.get_name() == aka::qname("xs:simpleContent")) {
    def = &process_simpleContent(name, aka::item_cast<xs::simpleContent>(item));
  }
  else if (item.get_name() == aka::qname("xs:complexContent")) {
    def = &process_complexContent(name, aka::item_cast<xs::complexContent>(item));
  }
  else if (item.get_name() == aka::qname("&s0")) {
    const xs::complexTypeModel_s0 &cdef = aka::item_cast<xs::complexTypeModel_s0>(item);
    class_def * seqdef = new sequence_def(name, aka::occurrence());

    element_type etype = process_complexTypeParticle(name, cdef.typeDefParticle_);
    if (!etype.empty())
      seqdef->etypes_.push_back(etype);

    process_attrDecls(name, seqdef->adefs_, cdef.attrDecls_);
    current_unit_->classes_.insert(name, seqdef);
    def = seqdef;
  }

  assert(def != 0);
  def->adefs_.check_dup_names();
  return *def;
}

const class_def &processor::process_simpleContent(const aka::qname &name,
						  const xs::simpleContent &sc) {
  const xs::simpleContent_choice &choice = sc.c0_;
  assert(choice.size() == 1);

  const aka::item &item = choice.front();
  if (item.get_name() == aka::qname("xs:extension"))
    return process_simpleExtensionType(name, aka::item_cast<xs::simpleExtensionType>(item));
  else if (item.get_name() == aka::qname("xs:restriction"))
    return process_simpleRestrictionType(name, aka::item_cast<xs::simpleRestrictionType>(item));
  
  assert(!"Must not reach here.");
  return *static_cast<const class_def*>(0);
}


const class_def &processor::process_simpleExtensionType(const aka::qname &name, 
							const xs::simpleExtensionType &se) {
  if (registry_.classdef_exists(se.base_)) {
    const class_def &super_def = registry_.classdef_get(se.base_);
    if (super_def.id_ != simplecontent_id)
      throw fatal_error();
    class_def *def = super_def.derive(name);
    process_attrDecls(name, def->adefs_, se.attrDecls_);
    current_unit_->classes_.insert(name, def);
    return *def;
  }
  else if (registry_.simpleType_exists(se.base_)) {
    simplecontent_def *def = new simplecontent_def(name);
    def->set_value_type(se.base_);
    process_attrDecls(name, def->adefs_, se.attrDecls_);
    current_unit_->classes_.insert(name, def);
    return *def;
  }

  throw fatal_error("Failed to find type, " 
		    + aka::quote(se.base_.qualified()) + ".", __FILE__, __LINE__);
  return *static_cast<const class_def*>(0);
}

const class_def &processor::process_simpleRestrictionType(const aka::qname &name, 
							  const xs::simpleRestrictionType &sr) {
  if (current_unit_->classes_.exists(sr.base_)) {
    const class_def *super_def = current_unit_->classes_.get(sr.base_);
    if (super_def->id_ != simpletype_id)
      throw fatal_error();
    class_def *def = super_def->derive(name);
    //process_attrDecls(*def->adefs_, se.attrDecls_);  !!!!!!!!!!!!!!!!
    // add code for prohibit !!!!!!!!!!!!!!!!!
    current_unit_->classes_.insert(name, def);
    return *def;
  }
  else if (current_unit_->simpleTypes_.exists(sr.base_)) {
    class_def *def = new simplecontent_def(name);
    def->set_value_type(sr.base_);
    current_unit_->classes_.insert(name, def);
    return *def;
  }

  throw fatal_error();
  return *static_cast<const class_def*>(0);
}


element_type processor::process_complexTypeParticle(const aka::qname &name,
						    const xs::typeDefParticle &particle) {
  
  if (particle.empty()) {
    return element_type();
  }

  assert(particle.size() == 1); // Must contain one paticle.
  const aka::item &item = particle.front();
  if (item.get_name() == aka::qname("xs:group")) {
    const xs::groupRef &gr = aka::item_cast<xs::groupRef>(item);
    process_namedGroup(gr.ref_); // Make sure referred group processed.
    element_type etype;
    etype.name_ = aka::qname(gr.ref_.local());
    etype.type_ = gr.ref_;
    etype.occ_ = occurrence(gr);
    etype.is_group_ = true;
    return etype;
  }

  element_type el;
  const class_def *particle_def;

  if (item.get_name() == aka::qname("xs:all")) {
    const xs::all &all = aka::item_cast<xs::all>(item);
    aka::qname allname = name;
    allname.local() += "_all";
    particle_def = &process_all(allname, occurrence(all), all.allModel_.element_);
    el.occ_ = occurrence(all);
    el.name_ = aka::qname("*");
  }
  else if (item.get_name() == aka::qname("xs:choice")) {
    const xs::explicitGroup &gr = aka::item_cast<xs::explicitGroup>(item);
    aka::qname choicename(name);
    choicename.local() += "_choice";
    particle_def = &process_choice(choicename, occurrence(gr), gr.nestedParticle_);
    el.occ_ = occurrence(gr);
    el.name_ = aka::qname("*");
  }
  else if (item.get_name() == aka::qname("xs:sequence")) {
    const xs::explicitGroup &gr = aka::item_cast<const xs::explicitGroup>(item);
    aka::qname seqname = name;
    seqname.local() += "_sequence";
    particle_def = &process_sequence(seqname, occurrence(gr), gr.nestedParticle_);
    el.occ_ = occurrence(gr);
    el.name_ = aka::qname("*");
  }
  else {
    throw fatal_error();
  }
  
  el.type_ = particle_def->get_name();
  el.is_group_ = true;
  return el;
}


const class_def &processor::process_sequence(const aka::qname &name, 
					     const aka::occurrence &occ,
					     const xs::nestedParticle &np) {

  class_def *def = new sequence_def(name, occ);
  current_unit_->classes_.insert(name, def);
  process_nestedParticle(def->etypes_, name, np);
  return *def;
}

const class_def &processor::process_choice(const aka::qname &name, 
					   const aka::occurrence &occ,
					   const xs::nestedParticle &np) {
  choice_def *def = new choice_def(name, occ);
  current_unit_->classes_.insert(name, def);
  process_nestedParticle(def->etypes_, name, np);
  return *def;
}

const class_def &processor::process_all(const aka::qname &name, 
					const aka::occurrence &occ,
					const xs::narrowMaxMin_array &ams) {
  all_def *def = new all_def(name, occ);
  current_unit_->classes_.insert(name, def);
  process_all_members(def->etypes_, name, ams);
  return *def;
}

void processor::process_groupRef(const aka::qname &name, const xs::groupRef &gr) {
  const class_def *refdef = current_unit_->classes_.get(gr.ref_);
  class_def *def = refdef->derive(name);
  current_unit_->classes_.insert(name, def);
}

aka::qname processor::process_simpleDerivation(const xs::simpleDerivation &der) {
  assert(der.size() == 1);
  const aka::item &item = der.front();
  if (item.get_name() == aka::qname("xs:restriction")) {
    const xs::restriction &re = aka::item_cast<xs::restriction>(item);
    if (re.base_.empty())
      throw fatal_error("simpleType in xs:restriction is not supported.", __FILE__, __LINE__);

    process_topLevelSimpleType(re.base_);
    return re.base_;
  }
  return aka::qname("xs:anySimpleType");
}

void processor::process_all_members(element_types &etypes, 
				    const aka::qname &hint, const xs::narrowMaxMin_array &ams) {

  for (xs::narrowMaxMin_array::const_iterator it = ams.begin();
       it != ams.end(); ++it) {
    element_type membertype = 
      process_allElementMember(*it, hint);
    etypes.push_back(membertype);
  }
  etypes.check_dup_names();
}


element_type processor::process_localElement(const xs::localElement &el, const aka::qname &hint) {
  // block, form, nillable is ignored.

  aka::qname type;
  aka::qname name;

  if (!el.ref_.empty()) {
    if (!el.type_.empty() || !el.c0_.empty())
      throw fatal_error();
    type = process_topLevelElement(el.ref_);
    name = el.ref_;
  }
  else if (!el.type_.empty()) {
    if (!el.c0_.empty())
      throw fatal_error();
    name = aka::qname(target_ns_id_, el.name_);
    type = el.type_;
  }
  else if (!el.c0_.empty()) {
    name = aka::qname(target_ns_id_, el.name_);
    type = aka::qname(target_ns_id_, hint.local() + '_' + name.local());
    type = process_typeChoice(type, el.c0_);
  }
  else {
    /* @ref, @type is not spcified.  content is ur-type. */
    name = aka::qname(target_ns_id_, el.name_);
    type = aka::qname("xs:anyType");
  }

  assert(!name.empty());
  assert(!type.empty());

  element_type et;

  et.name_ = create_element_name(name, el.form_);
  et.type_ = type;
  et.occ_ = aka::occurrence(el.minOccurs_, el.maxOccurs_);

  if (!el.fixed_.empty()) {
    if (!el.default_.empty())
      throw fatal_error();
    et.default_ = el.fixed_;
    et.is_fixed_ = true;
  }
  else if (!el.default_.empty()) {
    et.default_ = el.default_;
  }


  
  return et;
}


element_type processor::process_allElementMember(const xs::narrowMaxMin &ame, 
						 const aka::qname &hint) {
  xs::localElement lel;

  lel.attributes_ = ame.attributes_;
  lel.id_ = ame.id_;
  lel.annotation_ = ame.annotation_;
  lel.ref_ = ame.ref_;
  lel.name_ = ame.name_;
  lel.c0_ = ame.c0_;
  lel.identityConstraint_ = ame.identityConstraint_;
  lel.type_ = ame.type_;
  lel.default_ = ame.default_;
  lel.fixed_ = ame.fixed_;
  lel.nillable_ = ame.nillable_;
  lel.block_ = ame.block_;
  lel.form_ = ame.form_;

  return process_localElement(lel, hint);
}


element_type processor::process_member_groupRef(const xs::groupRef &ref) {
  element_type et;
  et.name_ = aka::qname(ref.ref_.local());
  et.type_ = ref.ref_;
  et.occ_ = aka::occurrence(ref.minOccurs_, ref.maxOccurs_);
  et.is_group_ = true;
  return et;
}

element_type processor::process_any(const xs::any &any) {
  element_type et;
  et.occ_ = aka::occurrence(any.minOccurs_, any.maxOccurs_);
  et.name_ = aka::qname("any");
  et.type_ = aka::qname("xs:anyType");
  et.is_group_ = true;
  et.namespace_list_ = any.namespace_;
  return et;
}

const class_def &processor::process_complexContent(const aka::qname &name, 
						   const xs::complexContent &cc) {
  const xs::complexContent_choice &contents = cc.c0_;
  assert(contents.size() == 1);
  const aka::item &item = contents.front();
  if (item.get_name() == aka::qname("xs:restriction"))
    return process_complexRestrictionType(name, 
					  aka::item_cast<xs::complexRestrictionType>(item));
  else if (item.get_name() == aka::qname("xs:extension"))
    return process_extensionType(name, aka::item_cast<xs::extensionType>(item));

  assert(!"Must not reach here.");
  return *static_cast<const class_def*>(0);
}

const class_def &processor::process_complexRestrictionType(const aka::qname &name, 
							   const xs::complexRestrictionType &cr) {
  const class_def &superdef = process_topLevelComplexType(cr.base_);
  // lack of !Restriction check!.!!!!!
  class_def *def = new sequence_def(name, aka::occurrence());
  element_type etype;
  etype = process_complexTypeParticle(name, cr.typeDefParticle_);

  if (!etype.empty())
    def->etypes_.push_back(etype);
  def->adefs_ = superdef.adefs_;
  
  current_unit_->classes_.insert(name, def);

  restrict_attrDecls(name, def->adefs_, cr.attrDecls_);
  return *def;
}

const class_def &processor::process_extensionType(const aka::qname &name, 
						  const xs::extensionType &et) {

  const class_def *superdef = &process_topLevelComplexType(et.base_);
  class_def *def = superdef->derive(name);

  if (def->etypes_.empty()) {
    element_type etype = 
      process_complexTypeParticle(name, et.typeDefParticle_);
    if (!etype.empty())
      def->etypes_.push_back(etype);
  }
  else {
    element_type el = process_complexTypeParticle(name, et.typeDefParticle_);
    if (!el.empty()) {
      aka::qname particlename(name);
      particlename.local() += "_particle";
      class_def *particle_def = new sequence_def(particlename, aka::occurrence());
      particle_def->etypes_ = def->etypes_;
      current_unit_->classes_.insert(particlename, particle_def);
      def->etypes_.clear();
//       el.name_ = aka::qname("@");
      particle_def->etypes_.push_back(el);

      el.name_ = aka::qname(name.local()); // extended member name.
      el.type_ = particlename;
      el.is_group_ = true;
      def->etypes_.push_back(el);
    }
  }
  extend_attrDecls(name, def->adefs_, et.attrDecls_);
  current_unit_->classes_.insert(name, def);
  return *def;

  // !!!! lack of @block check.

}


void processor::process_nestedParticle(element_types &etypes, const aka::qname &hint, 
				       const xs::nestedParticle &np) {
  
//   size_t all_index = 0;
  size_t choice_index = 0;
  size_t sequence_index = 0;

  for (xschoice::const_iterator it = np.begin(); 
       it != np.end(); ++it) {
    const aka::item &item = *it;
    element_type el;

    if (item.get_name() == aka::qname("xs:element")) {
      el = process_localElement(aka::item_cast<const xs::localElement>(item), hint);
    }
    else if (item.get_name() == aka::qname("xs:group")) {
      el = process_member_groupRef(aka::item_cast<const xs::groupRef>(item));
    }
//     else if (item.get_name() == aka::qname("xs:all")) {
//       aka::qname allname = create_anonymous_name(hint, "a", all_index);
//       el = process_localAll(allname, aka::item_cast<const xs::all>(item));
//       el.name_ = aka::qname("*");
//       ++all_index;
//     }
    else if (item.get_name() == aka::qname("xs:choice")) {
      aka::qname choicename = create_anonymous_name(hint, "c", choice_index);
      el = process_localChoice(choicename, aka::item_cast<const xs::explicitGroup>(item));
      el.name_ = aka::qname("*");
      ++choice_index;
    }
    else if (item.get_name() == aka::qname("xs:sequence")) {
      aka::qname sequencename = create_anonymous_name(hint, "s", sequence_index);
      el = process_localSequence(sequencename, aka::item_cast<const xs::explicitGroup>(item));
      el.name_ = aka::qname("*");
      ++sequence_index;
    }
    else if (item.get_name() == aka::qname("xs:any")) {
      el = process_any(aka::item_cast<const xs::any>(item));
      el.is_group_ = true;
    }
    else {
      assert(!"Must not reach here.");
    }
    etypes.push_back(el);
  }
//   etypes.check_dup_names();
}


element_type processor::process_localAll(const aka::qname &name, const xs::all &all) {
  process_all(name, occurrence(all), all.allModel_.element_);
  element_type et;
  et.name_ = aka::qname(name.local());
  et.occ_ = occurrence(all);
  et.type_ = name;
  et.is_group_ = true;
  return et;
}

element_type processor::process_localChoice(const aka::qname &name, const xs::explicitGroup &cho) {
  process_choice(name, occurrence(cho), cho.nestedParticle_);
  element_type et;
  et.name_ = aka::qname(name.local());
  et.occ_ = occurrence(cho);
  et.type_ = name;
  et.is_group_ = true;
  return et;
}

element_type processor::process_localSequence(const aka::qname &name, 
					      const xs::explicitGroup &seq) {
  process_sequence(name, occurrence(seq), seq.nestedParticle_);
  element_type et;
  et.name_ = aka::qname(name.local());
  et.occ_ = occurrence(seq);
  et.type_ = name;
  et.is_group_ = true;
  return et;
}

