/* -*- c++ -*- */
#ifndef OSIXAKA2_CLASSDEFS_H__
#define OSIXAKA2_CLASSDEFS_H__

#include <akaxiso2/configuration.h>
#include <akaxiso2/util/string_funcs.h>
#include <osixaka2/exception.h>
#include <osixaka2/platform.h>

#include <set>

namespace osx {

  template<class T>
  struct simple_map {
    typedef std::map<aka::qname, T, aka::qname_less> map_type;
    typedef TYPENAME map_type::iterator iterator;
    typedef TYPENAME map_type::const_iterator const_iterator;
  
    map_type map_;
  
    void insert(const aka::qname &name, const T& t);
    void erase(const aka::qname &name);
    void erase(iterator to_erase) { map_.erase(to_erase); }
    const T& get(const aka::qname &name) const;
    T& get(const aka::qname &name);
    bool exists(const aka::qname &name) const;
    void clear();
    bool empty() const { return map_.empty(); }
  
    iterator begin() { return map_.begin(); }
    iterator end() { return map_.end(); }
    const_iterator begin() const { return map_.begin(); }
    const_iterator end() const { return map_.end(); }
  
  };

  template<class T>
  void simple_map<T>::insert(const aka::qname &name, const T& t) {
    //assert(aka::qname("xs:anyType") != name);
    std::pair<TYPENAME map_type::iterator, bool> res =
      map_.insert(OSX_TYPENAME map_type::value_type(name, t));
    if (!res.second) {
      throw fatal_error(name.qualified() + " is registered more than once.",
			__FILE__, __LINE__);
    }
  }

  template<class T>
  void simple_map<T>::erase(const aka::qname &name) {
    TYPENAME map_type::iterator it = map_.find(name);
    if (it == map_.end())
      throw fatal_error();
    map_.erase(it);
  }

  template<class T>
  const T& simple_map<T>::get(const aka::qname &name) const {
    TYPENAME map_type::const_iterator it = map_.find(name);
    if (it == map_.end())
      throw fatal_error();
    return it->second;
  }

  template<class T>
  T& simple_map<T>::get(const aka::qname &name) {
    TYPENAME map_type::iterator it = map_.find(name);
    if (it == map_.end()) {
      throw fatal_error("specified type "
			+ aka::quote(name.qualified()) 
			+ " not found.",
			__FILE__, __LINE__);
    }
    return it->second;
  }

  template<class T>
  bool simple_map<T>::exists(const aka::qname &name) const {
    TYPENAME map_type::const_iterator it = map_.find(name);
    return it != map_.end();
  }

  template<class T>
  void simple_map<T>::clear() {
    map_.clear();
  }


  typedef std::set<aka::qname, aka::qname_less> qname_set;

  class class_def;
  typedef simple_map<class_def*> class_defs;

  /* element */
  struct element_type {
    element_type() 
      : is_fixed_(false), is_group_(false), is_ptrmember_(false), occ_required_(false) {}
    aka::qname name_;
    aka::qname member_name_;
    aka::qname type_;
    aka::occurrence occ_;
    std::string default_;
    std::string namespace_list_;

    bool is_fixed_;
    bool is_group_;
    bool is_ptrmember_;
    bool occ_required_;
    bool empty() const { return name_.empty() && type_.empty(); }
  };

  /* attribute */
  struct attribute_type {
    attribute_type() : required_(false), is_fixed_(false) {}
    aka::qname name_;
    aka::qname type_;
    bool required_;
    bool is_fixed_;
    std::string default_;
  };

  struct element_types : std::list<element_type> {
    void check_dup_names() const;
  };

  typedef std::list<attribute_type> attribute_types;


  struct attribute_defs {
    attribute_defs() : has_anyAttribute_(false) {}

    attribute_types attributes_;
    bool has_anyAttribute_;
    std::string namespace_list_;

    bool exists(const aka::qname &name) const;
    void erase(const aka::qname &name);
    attribute_type &get(const aka::qname &name);

    bool has_attributes() const {
      return !attributes_.empty() || has_anyAttribute_;
    }

    void check_dup_names() const;
  };

  class attributeGroup_def : public attribute_defs {
  public:
    aka::qname name_;
  };
  typedef simple_map<attributeGroup_def> attrGroup_defs;


  /* attribute */
  struct gattribute_def {
    gattribute_def() : is_fixed_(false) {}
    aka::qname name_;
    aka::qname type_;
    std::string default_;
    bool is_fixed_;
  };
  typedef simple_map<gattribute_def> gattribute_defs;


  enum def_type {
    simpletype_id = 1,
    simplecontent_id,
    array_id,
    sequence_id,
    choice_id,
    all_id,
    //   enclose_id,
    //   disclose_id,
    //   any_id,
    //   any_array_id,
    //   any_attribute_id,
    //   fixed_id
  };

  class class_def {
    aka::qname value_type_;
    aka::qname name_;
  public:
    class_def(const aka::qname &name, const aka::occurrence &occ, def_type id)
      : name_(name), occ_(occ), id_(id) {}

    void set_value_type(const aka::qname &value_type) { 
      value_type_ = value_type; 
      assert(!value_type_.empty());
    }
    const aka::qname &get_value_type() const { 
      assert(!value_type_.empty());
      return value_type_; 
    }
    const aka::qname &get_name() const { 
      assert(!name_.empty());
      return name_; 
    }

    aka::occurrence occ_;
    def_type id_;
    class_def *derive(const aka::qname &name) const;
    element_types etypes_;
    attribute_defs adefs_;
    bool accept_attributes() const { return (id_ == sequence_id) || (id_ == all_id); }
  };


  class sequence_def : public class_def {
  public:
    sequence_def(const aka::qname &name, const aka::occurrence &occ) 
      : class_def(name, occ, sequence_id) {}
  };

  class all_def : public class_def {
  public:
    all_def(const aka::qname &name, const aka::occurrence &occ) 
      : class_def(name, occ, all_id) {}
  };

  class choice_def : public class_def {
  public:
    choice_def(const aka::qname &name, const aka::occurrence &occ)
      : class_def(name, occ, choice_id) {}
  };

  class simplecontent_def : public class_def {
  public:
    simplecontent_def(const aka::qname &name)
      : class_def(name, aka::occurrence(), simplecontent_id) {}
  };

  class simpletype_def : public class_def {
  public:
    simpletype_def(const aka::qname &name)
      : class_def(name, aka::occurrence(), simpletype_id) {}
  };


  class array_def : public class_def {
  public:
    array_def(const aka::qname &name)
      : class_def(name, aka::occurrence(), array_id){}
  };


  /* element */
  struct element_def {
    element_def() 
      : is_fixed_(false) {}
    aka::qname name_;
    aka::qname type_;
    std::string default_;
    bool is_fixed_;
  };

  typedef simple_map<element_def> element_defs;


}

#endif /* CLASSDEFS_H__ */
