#include "akaxisonizer.h"
#include <iostream>
#include <ctype.h>

using namespace osx;


bool akaxisonizer::array_required(const class_def &def) const {

  switch (def.id_) {
  case sequence_id:
  case all_id:
  case simpletype_id:
  case simplecontent_id:
    return (def.occ_.minOccurs_ != 1) || (def.occ_.maxOccurs_ != 1);
  case choice_id:
  case array_id:
    break;
  }
  return false;
}


bool akaxisonizer::array_required(const element_type &etype) const {

  if (registry_.is_predefined(etype.type_))
    return (etype.occ_.minOccurs_ != 1) || (etype.occ_.maxOccurs_ != 1);

  const class_def &def = registry_.classdef_get(etype.type_);

  switch (def.id_) {
  case sequence_id:
  case all_id:
  case simpletype_id:
  case simplecontent_id:
    return (etype.occ_.minOccurs_ != 1) || (etype.occ_.maxOccurs_ != 1);
  case choice_id:
  case array_id:
    break;
  }
  return false;
}

bool akaxisonizer::is_array(const element_type &etype) const {
  return (etype.occ_.minOccurs_ != 1) || (etype.occ_.maxOccurs_ != 1);
}

bool akaxisonizer::occ_required(const element_type &etype) const {
  return (etype.occ_.minOccurs_ != 1) || (etype.occ_.maxOccurs_ != 1);
}


void akaxisonizer::add_missings() {
  for (class_defs::iterator it = regunit_.classes_.begin();
       it != regunit_.classes_.end(); ++it) {
    class_def *def = it->second;
    add_missings_for_class(def);
  }
}

// Add array definitions, and copy def->occ_ to element_type::occ_.
void akaxisonizer::add_missings_for_class(class_def *def) {

  if (def->id_ == array_id)
    return;

  // Confirm occurrence, and add array decls if required.
  for (element_types::iterator it = def->etypes_.begin();
       it != def->etypes_.end(); ++it) {

    if (registry_.is_xs_any(it->type_)) 
      continue;  // array for anyType will be inserted in replace_any().

    if (registry_.is_predefined(it->type_))
      continue;

    class_def &childdef = registry_.classdef_get(it->type_);
    if (childdef.id_ == array_id)
      continue;

    if (array_required(*it) && !array_required(childdef)) {
      it->type_ = define_array(childdef.get_name());
      assert(!it->type_.empty());
    }
    else if (!array_required(*it) && array_required(childdef)) {
      it->type_ = define_array(childdef.get_name());
      it->occ_ = childdef.occ_;
      assert(!it->type_.empty());
    }
    else if (array_required(*it) && array_required(childdef)) {
      aka::qname arrayname = define_array(childdef.get_name());
      aka::qname holdername = childdef.get_name();
      holdername.local() += "_holder";
      it->type_ = holdername;
      
      if (!regunit_.classes_.exists(holdername)) {
	class_def *holder = new array_def(holdername);
	holder->set_value_type(childdef.get_name());
	element_type el;
	el.name_ = childdef.get_name();
	el.type_ = arrayname;
	el.occ_ = childdef.occ_;
	holder->etypes_.push_back(el);
	regunit_.classes_.insert(holdername, holder);
	assert(!el.type_.empty());
      }
    }
    if (!registry_.is_toplevel(childdef.get_name()))
      add_missings_for_class(&childdef);
  }

}



void akaxisonizer::optimize() {
  if (verbose_)
    std::cout << "Optimization" << std::flush;

  bool optimizing = true;
  while (optimizing) {
    if (verbose_)
      std::cout << "." << std::flush;
    optimizing = false;
    for (class_defs::iterator it = regunit_.classes_.begin();
	 it != regunit_.classes_.end(); ++it) {
      class_def *def = it->second;
      optimizing |= simplify_class(def);
    }
  }

  if (verbose_)
    std::cout << "done." << std::endl;
}


bool akaxisonizer::simplify_class(class_def *def) {

  if (def->etypes_.empty()) // Nothing to optimize.
    return false;

  bool optimized = simplify_children(def);

  if (def->etypes_.size() == 1)
    optimized |= simplify_one_element_type(def);
  
  return optimized;
}

bool akaxisonizer::simplify_children(class_def *def) {
  
  bool optimized = false;

  for (element_types::iterator it = def->etypes_.begin();
       it != def->etypes_.end(); ++it) {

    if (!regunit_.type_exists(it->type_)) 
      continue; // Optimization over namespaces are not performed.
    
    // Don't optimize toplevel types.
    if (regunit_.is_toplevel(it->type_))
      continue;
    
    // If predefined type, no optimization.
    // Because element_type reference in the upper level required to be updated.
    if (!regunit_.classes_.exists(it->type_))
      continue;


    class_def *child = regunit_.classes_.get(it->type_);
    
    if ((def->id_ == sequence_id) &&
	is_array(*it)) {
      // If occurrence of element is (0, 1), and child is array.
      // Convert array to ptrmember.    
      if ((child->id_ == array_id)
	  && (it->occ_.minOccurs_ == 0) 
	  && (it->occ_.maxOccurs_ == 1)) {
	it->type_ = child->get_value_type();
	optimized = true;
	continue;
      }
    }

    // If child is array, and child array item is sequence,
    // and sequence has just one child element without attributes,
    // Then replace type to array of child element.
    // @test : XMLSchema.xsd and test6.xsd.
    if ((def->id_ == sequence_id)
	&& it->is_group_
	&& is_array(*it)
	&& (child->id_ == array_id)) {
      const class_def &gcdef = registry_.classdef_get(child->get_value_type());
      if ((def->id_ == sequence_id)
	  && (gcdef.etypes_.size() == 1)
	  && !gcdef.adefs_.has_attributes()
	  && !is_array(gcdef.etypes_.front())) {
	const element_type &gc_etype = gcdef.etypes_.front();
	it->type_ = define_array(gc_etype.type_);
	it->name_ = gc_etype.name_;
	it->is_group_ = gc_etype.is_group_;
	it->namespace_list_ = gc_etype.namespace_list_;
	optimized = true;
	continue;
      }
    }

    // one item choice optimization.
    if ((def->id_ == sequence_id)
	&& it->is_group_
	&& !child->adefs_.has_attributes()
	&& (child->id_ == choice_id)
	&& (child->etypes_.size() == 1)) {
      element_type &etype = child->etypes_.front();
      it->type_ = define_array(etype.type_);
      it->name_ = etype.name_;
      it->is_group_ = etype.is_group_;
      optimized = true;
    }
    
    // If element is group, and child is a sequence, and does not have any attributes,
    // move child elements to parent.
    if ((def->id_ == sequence_id)
	&& it->is_group_
	&& !is_array(*it)
	&& (child->id_ == sequence_id)
	&& !child->adefs_.has_attributes()) {
      def->etypes_.insert(it, child->etypes_.begin(), child->etypes_.end());
      it = def->etypes_.erase(it);
      optimized = true;
      continue;
    }
  }
  return optimized;
}  

bool akaxisonizer::simplify_one_element_type(class_def *def) {

  /** One child element optimization. */

  element_type &etype = def->etypes_.front();

  // Don't optimize toplevel types.
  if (regunit_.is_toplevel(etype.type_))
    return false;
  
  // If predefined type, no optimization.
  // Because element_type reference in the upper level required to be updated.
  if (!regunit_.classes_.exists(etype.type_))
    return false;
  
  // If class has attributes, it's complexType.
  // Could not be optimized.
  class_def *child = regunit_.classes_.get(etype.type_);

  if (!def->adefs_.has_attributes() && etype.is_group_) {

    // parent does not have attributes.
    // Child is one particle.
    // Therefore expand child elements to parent to erase anonymous types.

    switch (child->id_) {
    case sequence_id:
    case all_id:
    case choice_id:
      def->etypes_ = child->etypes_;
      def->adefs_ = child->adefs_;
      def->id_ = child->id_;
      return true;
    case array_id:
      def->set_value_type(child->get_value_type());
      def->etypes_.clear();
      def->id_ = child->id_;
      return true;
    case simpletype_id:
    case simplecontent_id:
      break; // Nothing to optimize.
    }
  }
  
  // child does not have attributes.
  // Move elements to parent. 
  bool child_expandable = (child->id_ != choice_id) && (child->id_ != array_id);
  if (!child->adefs_.has_attributes() && child_expandable && etype.is_group_) { 
    def->etypes_ = child->etypes_;
    def->id_ = child->id_;
    return true;;
  }
  return false;
}

aka::qname akaxisonizer::define_array(const aka::qname &valuename) {
  
  aka::qname arrayname = valuename;
  arrayname.local() = registry_.get_array_prefix() 
    + arrayname.local() 
    + registry_.get_array_suffix();
  
  if (regunit_.classes_.exists(arrayname))
    return arrayname;
  
  class_def *ardef = new array_def(arrayname);
  ardef->set_value_type(valuename);
  regunit_.classes_.insert(arrayname, ardef);
  
  assert(!arrayname.empty());
  return arrayname;
}





bool akaxisonizer::is_ptrmember(const element_type &etype) const {
  if (registry_.is_predefined(etype.type_))
    return (etype.occ_.minOccurs_ == 0) && (etype.occ_.maxOccurs_ == 1);

  const class_def &def = registry_.classdef_get(etype.type_);

  switch (def.id_) {
  case sequence_id:
  case all_id:
  case simpletype_id:
  case simplecontent_id:
    return (etype.occ_.minOccurs_ == 0) && (etype.occ_.maxOccurs_ == 1);
  case array_id:
  case choice_id:
    return false;
//     if (!etype.occ_.in_range(0))      // Should be in optimization.!!!!!!
//       return true;
  }
  return false;
}



bool akaxisonizer::replace_any(const aka::qname &type, element_type &etype) {

  if (type == aka::qname("xs:anyType")) {
    if ((etype.occ_.minOccurs_ == 0) && (etype.occ_.maxOccurs_ == 1)) {
      etype.type_ = aka::qname("aka:any");
      etype.is_ptrmember_ = true;
    }
    else if ((etype.occ_.minOccurs_ != 1) || (etype.occ_.maxOccurs_ != 1)) {
      etype.type_ = aka::qname("aka:any_array");
      etype.occ_required_ = true;
    }
    else {
      etype.type_ = aka::qname("aka:any");
    }
    return true;
  }
  else if (type == aka::qname("xs:anySimpleType")) {
    if ((etype.occ_.minOccurs_ == 0) && (etype.occ_.maxOccurs_ == 1)) {
      etype.type_ = aka::qname("xs:string");
      etype.is_ptrmember_ = true;
    }
    else if ((etype.occ_.minOccurs_ != 1) || (etype.occ_.maxOccurs_ != 1)) {
      etype.type_ = aka::qname("xs:string_array");
      etype.occ_required_ = true;
    }
    else {
      etype.type_ = aka::qname("xs:string");
      etype.is_ptrmember_ = true;
    }
    return true;
  }
  return false;
}

bool akaxisonizer::replace_any(element_type &etype) {

  if (replace_any(etype.type_, etype))
    return true;

  if (registry_.classdef_exists(etype.type_)) {
    const class_def &def = registry_.classdef_get(etype.type_);
    if (def.id_ == array_id) {
      if (replace_any(def.get_value_type(), etype))
	return true;
    }
  }
  return false;
}

bool akaxisonizer::add_member_number(aka::qname &name, const std::string &prefix, int index) {

  std::string::size_type pos = name.local().find_first_of('*');
  if (pos == std::string::npos)
    return false;

  name.local().resize(name.local().size() - 1);

  std::ostringstream ostm;
  ostm << index;
  name.local() += prefix + ostm.rdbuf()->str();
  return true;
}

void akaxisonizer::fix_member_name(class_def *def) {
  int seqindex = 0;
  int choindex = 0;
  int allindex = 0;
  int array_index = 0;

  for (element_types::iterator it = def->etypes_.begin();
       it != def->etypes_.end(); ++it) {

    if (registry_.is_predefined(it->type_) 
	|| registry_.is_xs_any(it->type_)) {
      assert(it->name_.local().find_first_of('*') == std::string::npos);
      continue;
    }

    class_def &child = registry_.classdef_get(it->type_);
    if (child.id_ == sequence_id) {
      if (add_member_number(it->name_, "s", seqindex))
	++seqindex;
    }
    if (child.id_ == choice_id) {
      if (add_member_number(it->name_, "c", choindex))
	++choindex;
    }
    if (child.id_ == all_id) {
      if (add_member_number(it->name_, "a", allindex))
	++allindex;
    }
    if (child.id_ == array_id) {
      if (add_member_number(it->name_, "array", array_index))
	++array_index;
    }
    assert(it->name_.local().find_first_of('*') == std::string::npos);
  }
}

void akaxisonizer::rename_duplicate_member(class_def *def) {

  if (def->id_ != sequence_id) {
    for (element_types::iterator it = def->etypes_.begin();
	 it != def->etypes_.end(); ++it) {
      it->member_name_ = it->name_;
    }
  }    
  else {
    std::map<aka::qname, long> members;
    std::map<aka::qname, long> member_count;
    element_types::iterator it;
    for (it = def->etypes_.begin(); it != def->etypes_.end(); ++it) {
      members[it->name_] = 0;
      member_count[it->name_] = 0;
    }
    for (it = def->etypes_.begin(); it != def->etypes_.end(); ++it)
      ++ members[it->name_];
    
    for (it = def->etypes_.begin(); it != def->etypes_.end(); ++it) {
      if (members[it->name_] > 1) {
	it->member_name_ = it->name_;
	const std::string &memname = it->member_name_.local();
	if (isdigit(memname[memname.size() - 1]))
	  it->member_name_.local() += "_*";
	else
	  it->member_name_.local() += '*';
	add_member_number(it->member_name_, "", member_count[it->name_]);
	++member_count[it->name_];
      }
      else
	it->member_name_ = it->name_;
    }
  }
}

bool akaxisonizer::replace_builtin(element_type &etype) {

  if (!registry_.is_builtin(etype.type_))
    return false;

  if (is_ptrmember(etype))
    etype.is_ptrmember_ = true;
  else if (occ_required(etype)) {
    etype.type_ = registry_.get_predefined_array(etype.type_);
    etype.occ_required_ = true;
  }
  return true;
}


void akaxisonizer::set_array_properties(const class_def *def, element_type &etype) {

  switch (def->id_) {
  case sequence_id:
  case all_id:
    if (is_ptrmember(etype))
      etype.is_ptrmember_ = true;
    else if (occ_required(etype))
      etype.occ_required_ = true;
    break;
  case choice_id:
    if (occ_required(etype))
      etype.occ_required_ = true;
    break;
  case array_id:
    break; // Don't have to set array properties for array.
  case simpletype_id:
  case simplecontent_id:
    // does not have children.  No array available.
    break;
  }
}



void akaxisonizer::akaxisonize() {

  for (class_defs::iterator it = regunit_.classes_.begin();
       it != regunit_.classes_.end(); ++it) {
    class_def *def = it->second;

    fix_member_name(def);
    rename_duplicate_member(def);

    for (element_types::iterator it = def->etypes_.begin();
	 it != def->etypes_.end(); ++it) {

      // Must call replace_any() before repalce_builtin().
      if (replace_any(*it)) 
	continue;
      else if (replace_builtin(*it))
	continue;
      else 
	set_array_properties(def, *it);
    }      

    for (attribute_types::iterator ait = def->adefs_.attributes_.begin();
	 ait != def->adefs_.attributes_.end(); ++ait) {
      if (ait->type_ == aka::qname("xs:anySimpleType"))
	ait->type_ = aka::qname("xs:string");
    }
  }

  for (element_defs::iterator eit = regunit_.elements_.begin();
       eit != regunit_.elements_.end(); ++eit) {
    if (eit->second.type_ == aka::qname("xs:anyType"))
      eit->second.type_ = aka::qname("aka:wildcard");
    else if (eit->second.type_ == aka::qname("xs:anySimpleType"))
      eit->second.type_ = aka::qname("xs:string");
  }
}

