/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_SIMPLETYPE_H__
#define AKAXISO2_FRAMEWORK_SIMPLETYPE_H__

/**
 * @file akaxiso2/framework/simpletype.h
 * @brief @ref simpletypes -related classes/templates 
 */

#include <akaxiso2/framework/type_registry.h>
#include <akaxiso2/framework/member.h>
#include <akaxiso2/framework/operators.h>
#include <akaxiso2/framework/xmltype.h>
#include <akaxiso2/util/sstream.h>

namespace aka2 {

  template<class V, class VL>
  struct default_value : public default_op {
    typedef V value_type;

    default_value() : has_default_(false) {}
    virtual ~default_value(){}

    virtual void set_default(const std::string &defval) {
      aka2::isstream istm(defval);
      VL::read_text(&default_, istm, system_preconditions());
      has_default_ = true;
    }

    virtual bool has_default() const {
      return has_default_;
    }

    virtual const void *value() const {
      assert(has_default_);
      return &default_;
    }

  private:
    V default_;
    bool has_default_;
  };


  template<class L>
  struct simpletype_op_dispatcher : public simpletype_op {
  public:
    virtual schematype_id get_schematype() const { return simpletype_id; }
    virtual std::string get_typename() const { return L::get_xmltype(); }
    virtual const attribute_types *get_attribute_types() const { return 0; }
    virtual const attribute_type *get_anyattr_type() const { return 0; }

    virtual void write_text(const void* elm,
                            std::ostream &ostm,
                            const preconditions &pcd) const {
      L::write_text(elm, ostm, pcd);
    }

    virtual void read_text(void* elm, aka2::isstream &istm,
                           const preconditions &pcd) const {
      L::read_text(elm, istm, pcd);
    }

    virtual void construct(void *e) const {  L::construct(e); }
    virtual void copy_construct(void *e, const void *src) const { L::copy_construct(e, src); }
    virtual void destruct(void *e) const { L::destruct(e); }
    virtual size_t class_size() const { return L::class_size(); }
    virtual bool equals(const void *lhs, const void *rhs) const { return L::equals(lhs, rhs); }
  };


  template<class L>
  struct simpletype_statics {
    static simpletype_op_dispatcher<L> dispatcher_;
    static occurrence occ_;
  };

  template<class L>
  simpletype_op_dispatcher<L> simpletype_statics<L>::dispatcher_;

  template<class L>
  occurrence simpletype_statics<L>::occ_;


  /**
   * @brief template to define @ref simpletypes leaf class.
   *
   * Leaf classes for @ref simpletypes are derived from this template class.\n
   * @see @ref simpletypes
   * @param T value class type
   * @param L leaf class type
   */
  template<class V, class L=xiso::leaf<V> >
  class simpletype : public simpletype_statics<L>, 
		     public xmltype_statics<L> {
  public:
    virtual ~simpletype(){}
    typedef V value_type;

    static void initialize() {
      system_type_registry().add(L());
    }
    static void uninitialize() {}

    /**
     * @brief write element value to std::ostream
     *
     * Cast elm to const V&, and users get variable to write to std::ostream.
     * preconditionals will provide conditionals.
     * If user need custom-writing logic, override this method.
     * @param elm pointer to value
     * @param ostm std::ostream to output.
     * @param pfs preconditions to write value.
     * @exception aka2::error thrown when error occurs.
     */
    static void write_text(const void *elm, std::ostream &ostm,
                           const preconditions &pfs) {
      const V &value = *reinterpret_cast<const V*>(elm);
      ostm << value;
    }

    /**
     * @brief read element value from aka2::isstream.
     *
     * Cast elm to V&, and users get variable to set read value.
     * preconditionals will provide conditionals.
     * If user need custom-writing logic, override this method.
     * @param elm pointer to value
     * @param istm aka2::isstream to get text.
     * @param pfs preconditions to parse value.
     * @exception aka2::error thrown when error occurs.
     */
    static void read_text(void *elm, aka2::isstream &istm,
                          const preconditions &pfs) {
      V &value = *reinterpret_cast<V*>(elm);
      istm >> value;
      if (istm.fail())
        throw error(std::string("Failed to parse simpletype value, \"")
		    + istm.rdbuf()->str() + "\".",
		    __FILE__, __LINE__);
    }


    static void construct(void *e) { new (e) V(); }
    static void copy_construct(void *e, const void *src) {
      new (e) V(*static_cast<const V*>(src));
    }
    static size_t class_size() { return sizeof(V); }
    static void destruct(void *elm) { static_cast<V*>(elm)->~V(); }

    static bool equals(const void *lhs, const void *rhs) {
      return *static_cast<const V*>(lhs) == *static_cast<const V*>(rhs);
    }

    static default_op* create_default_op() { return new default_value<V, L>; }
    static simpletype_op_dispatcher<L> *get_attribute_dispatcher() {
      return &L::dispatcher_;
    }
  };

} // namespace aka2

#endif
