/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_PRECONDITIONS_H__
#define AKAXISO2_FRAMEWORK_PRECONDITIONS_H__

/**
 * @file akaxiso2/framework/preconditions.h
 * @brief preconditions for read/write simpleType values.
 */

#include <akaxiso2/framework/types.h>
#include <akaxiso2/util/shared_ptr.h>
#include <map>
#include <set>

namespace aka2 {
  class prefix_map;

  void reset_for_new_document();

  struct property_operator {
    virtual ~property_operator() { }
    virtual void reset_for_new_document(void *e) const = 0;
  };

  template <class T>
  struct T_property_operator : public property_operator {
    virtual void reset_for_new_document(void *e) const {
      T* t = static_cast<T*>(e);
      t->clear();
    }
  };

  struct precondition {
    precondition(void *property, property_operator *prop_op)
      : property_(property), prop_op_(prop_op) { }
    void *property_;
    shared_ptr<property_operator> prop_op_;

    void reset_for_new_document() const {
      prop_op_->reset_for_new_document(property_);
    }
  };


  typedef std::set<std::string> id_constraint;

  /**
   * @brief preconditions to read/write simpletype values.
   *
   * preconditions class are created by deserializer.\n
   * Created instances are passed to aka2::simpletype<> methods to give preconditions 
   * to serialize/deserialize simpletype values during deserialization.\n
   * \n
   * This class will be exteded when we need other preconditions for simpletypes.\n
   * Currently namespace prefix/URI association map is obtained.\n
   * @see @ref simpletypes
   */
  class preconditions {
    shared_ptr<prefix_map> prefixes_;
    typedef std::map<std::string, precondition> properties;
    properties properties_;
    preconditions(const preconditions &); // hidden copy-ctor.
    mutable id_constraint id_constraint_;
    void clear_properties();
  public:
    /**
     * @brief const version to get prefix and namespace URI association map.
     * @return get reference to current prefix_map.
     */
    prefix_map &get_prefixes() const { return *prefixes_; }

    /**
     * @brief add property.
     * @param name property name.
     * @param t instance of property.
     */
    template<class T>
    void add_property(const std::string &name, T &t) {
      add_property(name, &t, new T_property_operator<T>());
    }

    /**
     * @brief add property.
     * @param name property name.
     * @param t instance of property.
     */
    template<class T>
    void add_property(const std::string &name, T &t, property_operator *op) {
      add_property_ptr(name, &t, op);
    }

    template<class T>
    T& get_property(const std::string &name) const {
      return *static_cast<T*>(get_property_ptr(name));
    }

    template<class T>
    const T& const_get_property(const std::string &name) const {
      return *static_cast<T*>(const_get_property_ptr(name));
    }

    void add_property_ptr(const std::string &name, void *p, property_operator *prop_op);
    void *get_property_ptr(const std::string &name) const;
    const void *const_get_property_ptr(const std::string &name) const;

    bool check_id(const std::string &value) const;

    void clear_all();
    void reset_for_new_document() const;

    shared_ptr<preconditions> clone(bool copy_prefixes) const;
    explicit preconditions();
    ~preconditions();
  };
}

#endif
