/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_CHOICE_H__
#define AKAXISO2_FRAMEWORK_CHOICE_H__

/** 
 * @file akaxiso2/framework/choice.h 
 * @brief @ref aka_choice -related classes/templates
 */

#include <akaxiso2/framework/qname.h>
#include <akaxiso2/framework/item.h>
#include <akaxiso2/framework/model_check.h>
#include <akaxiso2/framework/xmltype.h>
#include <vector>

namespace aka2 {

  template<class C>
  class typed_item_iterator : public item_iterator {
  public:
    typed_item_iterator(const C& container) :
      container_(container), current_(container.begin()) {}
    virtual ~typed_item_iterator(){}

    const C &container_;
    TYPENAME C::const_iterator current_;

    virtual bool has_next() const { return current_ != container_.end(); }
    virtual const item *next() {
      const item &v = *current_;
      ++current_;
      return &v;
    }
  };


  template<class L>
  class choice_op_dispatcher : public choice_op {
  public:
    virtual schematype_id get_schematype() const { return choice_id; }
    virtual std::string get_typename() const { return L::get_xmltype(); }
    virtual const attribute_types* get_attribute_types() const { return 0; }
    virtual const attribute_type *get_anyattr_type() const { return 0; }
    virtual void construct(void *e) const {  L::construct(e); }
    virtual void copy_construct(void *e, const void *src) const { L::copy_construct(e, src); }
    virtual void destruct(void *e) const { L::destruct(e); }
    virtual size_t class_size() const { return L::class_size(); }

    virtual bool equals(const void *lhs, const void *rhs) const {
      return L::equals(lhs, rhs);
    }

    virtual size_t size(const void *container) const { return L::size(container); }

    /** attribute_info getter. */
    virtual const item_types &get_item_types() const { return L::item_types_; }

    virtual void push(void *container, item &i) const {
      L::push_function(container, i);
    }
    virtual item_iterator* get_iterator(const void *e) const {
      return L::get_iterator(e);
    }
  };


  template<class L>
  struct choice_statics {
    static item_types item_types_;
    static choice_op_dispatcher<L> dispatcher_;
    static occurrence occ_;
  };

  template<class L>
  item_types choice_statics<L>::item_types_;

  template<class L>
  occurrence choice_statics<L>::occ_;

  template<class L>
  choice_op_dispatcher<L> choice_statics<L>::dispatcher_;


  /**
   * @brief template to define @ref aka_choice leaf class.
   *
   * Leaf classes for @ref aka_choice are derived from this template class.\n
   * Users don't have to use this class directly.\n
   * Please use aka2::sequential_choice<> template for sequence containers, 
   * or aka2::associative_choice<> template for choice containers.
   * 
   * @param T value class type
   * @param L leaf class type
   */
  template<class T, class L>
  class choice : public choice_statics<L>,
		 public xmltype_statics<L> {
  public:
    typedef T value_type;
    virtual ~choice(){}

    static void initialize() {
      if (!system_type_registry().add(L()))
	return;
      L::item_types_.clear();
      L l; l.model();
      check_emptiable(L::item_types_);
    }
    static void uninitialize() {
      L::item_types_.clear();
    }

    static typed_item_iterator<T>* get_iterator(const void *container) {
      return new typed_item_iterator<T>(*static_cast<const T*>(container));
    }


    static void construct(void *e) {
      new (e) T();
    }
    static void copy_construct(void *e, const void *src) {
      new (e) T(*static_cast<const T*>(src));
    }
    static size_t class_size() { return sizeof(T); }
    static void destruct(void *elm) { static_cast<T*>(elm)->~T(); }

    static bool equals(const void *lhs, const void *rhs) {
      return choice_equals(lhs, rhs, L::dispatcher_);
    }

    static size_t size(const void *container) { return static_cast<const T*>(container)->size(); }

    static void push_function(void *container, aka2::item &i) {
      L().methods.push(*static_cast<T*>(container), i);
    }

    /**
     * @brief define occurrence of @ref aka_choice itself.
     * @param minOccurs minOccurs
     * @param maxOccurs maxOccurs
     */
    static void occurrence(int minOccurs, int maxOccurs) {
      L::occ_ = aka2::occurrence(minOccurs, maxOccurs);
    }

    /**
     * @brief define item as an child element.
     * @param tagname tag name
     * @param il leaf class of child element.
     * @param minOccurs minOccurs
     * @param maxOccurs maxOccurs
     */
    template<class IL>
    void item(const std::string &tagname, const IL& il, int minOccurs = 1, int maxOccurs = 1) {
      IL::initialize();
      element_op &op = IL::dispatcher_;
      qname qn(tagname);

      itemtype itype(op);
      default_op *defop = IL::create_default_op();
      if (defop != 0)
	itype.set_default_op(defop);
      itype.set_name(qn);

      schematype_id id = itype.get_schematype();
      if ((id == array_id) || (id == choice_id))
	itype.set_occurrence(aka2::occurrence(minOccurs, maxOccurs));
      else {
	if ((minOccurs != 1) && (maxOccurs != 1)) {
	  throw tagged_error("element", tagname, "has wrong occurrence.", __FILE__, __LINE__);
	}
      }
      std::pair<item_types::iterator, bool> res = 
	L::item_types_.insert(item_types::value_type(qn, itype));
      if (!res.second) {
	throw tagged_error("element", tagname, "was declared twice.", __FILE__, __LINE__);
      }
    }

    /**
     * @brief define wildcard (aka2::any, xs:anyType in XML Schema) child element.
     * @param tagname tag name
     * @param ns_list namespace list. 
     * available values are "##any", "##other:target_namespace_uri", 
     * "##local" or namespace uri list.
     */
    void any(const std::string &tagname, const std::string &ns_list = "##any") {
      aka2::itemtype itype(aka2::any_op::dispatcher_);
      qname qn(tagname);
      itype.set_name(qn);
      itype.set_ns_list(ns_list);
      std::pair<item_types::iterator, bool> res = 
	L::item_types_.insert(item_types::value_type(qn, itype));
      if (!res.second)
	throw tagged_error("choice", L::get_xmltype(), "any child element is already declared.",
			   __FILE__, __LINE__);
    }

    /**
     * @brief define array of wildcard (aka2::any_array, xs:anyType[] in XML Schema) 
     * child elements.
     *
     * @param tagname tag name
     * @param minOccurs minOccurs
     * @param maxOccurs maxOccurs
     * @param ns_list namespace list. 
     * available values are "##any", "##other:target_namespace_uri", 
     * "##local" or namespace uri list.
     */
    void any(const std::string &tagname, int minOccurs, int maxOccurs, 
	     const std::string &ns_list = "##any") {
      aka2::itemtype itype(aka2::any_array_op::dispatcher_);
      itype.set_name(qname(tagname));
      itype.set_occurrence(minOccurs, maxOccurs);
      itype.set_ns_list(ns_list);
      std::pair<item_types::iterator, bool> res = 
	L::item_types_.insert(item_types::value_type(qname(tagname), itype));
      if (!res.second)
	throw tagged_error("choice", L::get_xmltype(), "any child element is already declared.",
			   __FILE__, __LINE__);
    }


    /**
     * @brief define element having fixed value.
     * @param tagname tag name
     * @param fixed_value fixed value
     * @param il leaf class for fixed value class.
     */
    template<class IL>
    void fixed_item(const std::string &tagname, const std::string &fixed_value, const IL& il) {
      if (IL::dispatcher_.get_schematype() != simpletype_id)
	throw tagged_error("element", tagname, "fixed value should be simpleType.",
			   __FILE__, __LINE__);
      IL::initialize();
      element_op &op = aka2::fixed<IL>::dispatcher_;
      itemtype itype(op); 
      itype.set_default_op(IL::create_default_op());
      itype.setup_default_value(fixed_value);
      itype.set_name(qname(tagname));
      std::pair<item_types::iterator, bool> res = 
	L::item_types_.insert(item_types::value_type(qname(tagname), itype));
      if (!res.second)
	throw tagged_error("element", tagname, "already declared.", __FILE__, __LINE__);
    }


    /**
     * @brief define array of elements having fixed value.
     * @param tagname tag name
     * @param fixed_value fixed value
     * @param il leaf class for fixed value class.
     * @param minOccurs minOccurs
     * @param maxOccurs maxOccurs
     */
    template<class IL>
    void fixed_array(const std::string &tagname, const std::string &fixed_value, const IL& il, 
		     int minOccurs, int maxOccurs) {
      if (IL::dispatcher_.get_schematype() != array_id)
	throw tagged_error("element", tagname, " should be array.", __FILE__, __LINE__);

      qname qn(tagname);
      aka::occurrence occ(minOccurs, maxOccurs);

      IL::initialize();      
      element_op &op = IL::dispatcher_;
      itemtype itype(op);

      typedef TYPENAME IL::item_leaf_type item_leaf_type;
      default_op *defop = item_leaf_type::create_default_op();
      assert(defop != 0);
      itype.set_default_op(defop);
      itype.setup_default_value(fixed_value);
      itype.set_name(qn);
      itype.set_occurrence(occ);

      std::pair<item_types::iterator, bool> res = 
	L::item_types_.insert(item_types::value_type(qn, itype));
      if (!res.second)
	throw tagged_error("element", tagname, "already declared.", __FILE__, __LINE__);
    }

    static default_op* create_default_op() { return 0; }

    static const item_types::value_type* find_item_type(const qname &name) {
      item_types::const_iterator it = L::item_types_.find(name);
      if (it == L::item_types_.end())
      	return 0;
      return &*it;
    }

    static const item_types::value_type* find_item_type(const element_op &eo) {
      for (item_types::const_iterator it = L::item_types_.begin(); it != L::item_types_.end(); ++it)
    	if (&it->second.op() == &eo)
	  return &*it;
      return 0;
    }
  };


  /**
   * @brief Helper class to operate aka2::choice for sequence container.
   * @param C container type
   * @param L leaf class
   */
  template<class C, class L>
  struct sequential_choice_moc {
  private:
    C& c_;
  public:
    typedef TYPENAME C::value_type value_type;
    typedef TYPENAME C::iterator iterator;
    typedef TYPENAME C::const_iterator const_iterator;

    /**
     * @brief
     */
    sequential_choice_moc(C &container) : c_(container) {}

    iterator begin() { return c_.begin(); }
    iterator end() { return c_.end(); }
    const_iterator begin() const { return c_.begin(); }
    const_iterator end() const { return c_.end(); }

    /**
     * @brief insert child element.
     * @param v child element instance.
     * @param tagname tag name
     * @exception aka2::error thrown when tag name is not defined, or tagname is not for class I.
     */
    template<class I>
    void push_back(const I &v, const std::string &tagname) {
      const item_types::value_type *vt = L::find_item_type(qname(tagname));
      if (vt == 0)
	unknown_tagname_error(tagname);
      const itemtype &itype = vt->second;
      void *newv = itype.op().replicate(&v);
      if (itype.get_schematype() == fixed_id)
	itype.set_item_to_default(newv);
      c_.push_back(item(newv, itype));
    }

    /**
     * @brief Find child elements by tag name.
     * @param r container(std::vector<>, std::list<>) to get results.
     * @param tagname tag name
     * @return number of elements found.
     * @exception aka2::error thrown when tagname is not defined. 
     */
    template<class R>
    int find_elements(R &r, const std::string &tagname) const {
      qname qn(tagname);
      const item_types::value_type *itype = L::find_item_type(qn);
      if (itype == 0)
	unknown_tagname_error(tagname);
      
      int count = 0;
      for (const_iterator it = c_.begin(); it != c_.end(); ++it) {
	if (it->get_name() == qn) {
          r.push_back(item_cast<TYPENAME R::value_type>(*it));
	  ++count;
	}
      }
      return count;
    }

  };

  /**
   * @brief aka2::choice<> template speialized for 
   * <a href="http://www.sgi.com/tech/stl/Sequence.html" target="_blank">sequence container</a>.
   *
   * Use this template to define @ref aka_choice leaf class 
   * when you are using std::vector<>, std::list<> as @ref aka_choice containers.
   * @param T container type
   * @param L leaf class
   */
  template<class T, class L=xiso::leaf<T> >
  class sequential_choice : public choice<T, L> {
  public:
    virtual ~sequential_choice(){}

    /**
     * @brief Helper class to operate choice container.
     */
    typedef  sequential_choice_moc<T, L> moc;

    /**
     * @brief Helper class to operate const choice container.
     */
    typedef  sequential_choice_moc<const T, L> const_moc;
    sequential<T> methods;
  };


  /**
   * @brief Helper class to operate @ref aka_choice for associative container.
   * @param C container type.
   * @param L leaf class of @ref aka_choice
   */
  template<class C, class L>
  struct associative_choice_moc {
    /** @brief value_type */
    typedef TYPENAME C::value_type value_type;
    /** @brief iterator as C::iterator. */
    typedef TYPENAME C::iterator iterator;
    /** @brief const_iterator as C::const_iterator. */
    typedef TYPENAME C::const_iterator const_iterator;
    /**
     * @brief constructor
     * @param container @ref aka_choice container.
     */
    associative_choice_moc(C &container) : c_(container) {}

    /**
     * @brief returns the first iterator of a container.
     * @return the first iterator of a container.
     */
    iterator begin() { return c_.begin(); }

    /**
     * @brief returns the end iterator of a container.
     * @return the end iterator of a container.
     */

    iterator end() { return c_.end(); }

    /** 
     * @brief const version of begin() 
     * @return the first const_iterator of a container
     */
    const_iterator begin() const { return c_.begin(); }

    /** 
     * @brief const version of begin() 
     * @return the end const_iterator of a container
     */
    const_iterator end() const { return c_.end(); }

    /**
     * @brief Find child element by tag name.
     * @param r container(std::vector<>, std::list<>) to get results.
     * @param tagname tag name
     * @return number of elements found.
     * @exception aka2::error thrown when tag name is not defined. 
     */
    template<class R>
    int find_elements(R &r, const std::string &tagname) const {
      qname qn(tagname);
      const item_types::value_type *itype = L::find_item_type(qn);
      if (itype == 0)
	unknown_tagname_error(tagname);
      
      int count = 0;
      for (TYPENAME C::const_iterator it = c_.begin(); it != c_.end(); ++it) {
      	if (it->get_name() == qn) {
          r.push_back(item_cast<TYPENAME R::value_type>(*it));
	  ++count;
	}
      }
      return count;
    }

  private:
    C& c_;
  };


  /**
   * @brief aka2::choice<> template speialized for 
   * <a href="http://www.sgi.com/tech/stl/UniqueAssociativeContainer.html">
   * unique associative container</a>.
   *
   * Use this template to define @ref aka_choice leaf class 
   * when you are using std::map<>, std::set<> as @ref aka_choice containers.
   * @param T value class (container)
   * @param L leaf class
   */
  template<class T, class L=xiso::leaf<T> >
  class associative_choice : public choice<T, L> {
  public:
    virtual ~associative_choice(){}

    /** @brief Helper class to operate containers. */
    typedef  associative_choice_moc<T, L> moc;
    /** @brief Helper class to operate const containers. */
    typedef  associative_choice_moc<const T, L> const_moc;
    associative<T> methods;
  };

} // namespace aka2

#endif
