/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_ANY_H__
#define AKAXISO2_FRAMEWORK_ANY_H__

/**
 * @file akaxiso2/framework/any.h
 * @brief ur-type classes
 */

#include <akaxiso2/framework/qname.h>
#include <akaxiso2/framework/document.h>
#include <akaxiso2/util/shared_ptr.h>
#include <vector>

namespace aka2 {

  /** 
   * @brief akaxiso builtin type to represent xs:anyAttribute contents.
   *
   * akaxiso uses aka2::any_attribute to store xs:anyAttribute contents.
   * The member, value_, is for xs:anySimpleType.
   */
  struct wc_attribute {
    wc_attribute(){}
    wc_attribute(const qname &name, const std::string &value) 
      : name_(name), value_(value) {}
    /** tag name of attribute */
    qname name_;
    /** value of attribute (xs:anySimpleType) */
    std::string value_;
  };

  /**
   * @struct any_attributes
   * @brief array of any attributes.
   */
  typedef std::vector<wc_attribute> wc_attributes;

  /**
   * @brief akaxiso builtin type to represent xs:anyType in XML Schema.
   *
   * akaxiso uses aka2::any class to store wildcard (xs:anyType) values.\n
   * Text nodes in mixed content will be ignored.\n
   * If wildcard element has child elements, children are put to the children_ member. \n
   * If wildcard element has text entity like <element>TEST</element>, 
   * text entity is put to the value_ member.\n
   * Both children_ and value_ member should not have their contents at the same time.
   */
  struct wildcard {
    /** @brief tag name */
    qname name_;
    /** @brief any attributes */
    wc_attributes attributes_;
    /** @brief value of this element if this wildcard element has value of xs:anySimpleType. */
    std::string value_;
    /** @brief array of any element if this wildcard element has child elements. */
    std::vector<wildcard> children_; 
    bool empty() const { return value_.empty() && attributes_.empty() && children_.empty(); }
  };

  struct any {
    wildcard wc_;
    document document_;
    bool empty() const { return (document_.empty() == 0) && wc_.empty(); }
    bool has_document() const { return !document_.empty(); }
  };

  /**
   * @struct any_array
   * @brief array of aka2::any.
   */
  typedef std::vector<wildcard> wc_array; 
  typedef std::vector<any> any_array; 
}

bool operator==(const aka2::wildcard &lhs, const aka2::wildcard &rhs);

#endif
