/* -*- c++ -*- */
#include "formatter_base.h"
#include "../../exception.h"

using namespace aka2;

namespace {
  struct escape_entry {
    uchar_t uchar_;
    uchar_t escaped_[16];
  };

  const escape_entry escape_table[] = {
    {'"', {'&', 'q', 'u', 'o', 't', ';', 0}},
    {'&', {'&', 'a', 'm', 'p', ';',      0}},
    {'<', {'&', 'l', 't', ';',           0}},
    {0,   {                              0}}
  };

#ifdef __LITTLE_ENDIAN_COMPUTER__
  static const bom utf16_bom = {{ '\xff', '\xfe' }, 2};
  //static const bom utf32_bom = {{ 0xff, 0xfe, 0, 0 }, 4};
#endif

#ifdef __BIG_ENDIAN_COMPUTER__
  static const bom utf16_bom = {{ '\xfe', '\xff' }, 2};
  //static const bom utf32_bom = {{ 0, 0, 0xfe, 0xff }, 4};
#endif
  
} // namespace


#ifdef _WIN32

#include <stdio.h>
#include <io.h>
#include <fcntl.h> 
#include <iostream>

#ifdef __BORLANDC__
# define _setmode setmode
#endif

void formatter_base::set_binary_mode_for_stdio() {
  if (ostm_ == static_cast<std::ostream*>(&std::cout))
    fd_ = _fileno(stdout);
  else if (ostm_ == static_cast<std::ostream*>(&std::cerr))
    fd_= _fileno(stderr);
  else if (ostm_ == static_cast<std::ostream*>(&std::clog))
    fd_ = _fileno(stderr);
  if (fd_ != -1) {
    *ostm_ << std::flush;
    saved_stream_mode_ = _setmode(fd_, _O_BINARY);
  }
}

void formatter_base::revert_stream_mode() {
  if (ostm_ == 0)
    return;

  if (fd_ != -1) {
    *ostm_ << std::flush;
    _setmode(fd_, saved_stream_mode_);
  }
}

#else

void formatter_base::set_binary_mode_for_stdio() {}
void formatter_base::revert_stream_mode() {}

#endif


void formatter_base::prepare(std::ostream &ostm) {
  ostm_ = &ostm;
  if (is_binary_encoding(encoding_))
    set_binary_mode_for_stdio();
  bom_ = check_bom(encoding_);
}

void formatter_base::finish() {
  revert_stream_mode();
  ostm_ = 0;
}


void formatter_base::write(const std::string &value) {
  ustring unistr = lcp_to_ucs2(value);
  write(unistr);
}

void formatter_base::write_attribute_entity(const std::string &entity) {
  write_entity(entity, 0);
}

void formatter_base::write_text_entity(const std::string &entity) {
  write_entity(entity, 1);
}

void formatter_base::write_entity(const std::string &value, const int escape_index) {
  ustring unistr = lcp_to_ucs2(value);
  ustring escaped;
  for (ustring::const_iterator it = unistr.begin(); it != unistr.end(); ++it) {
    const uchar_t uchar = *it;
    const escape_entry *entry = &escape_table[escape_index];
    while (entry->uchar_ != 0) {
      if (entry->uchar_ == uchar) {
        escaped += entry->escaped_;
        break;
      }
      ++entry;
    }
    if (entry->uchar_ == 0)
      escaped += uchar;
  }
  write(escaped);
}


const bom* formatter_base::check_bom(const std::string &encoding) const {
  if ((strcasecmp("utf-16", encoding.c_str()) == 0) ||
      (strcasecmp("utf16", encoding.c_str()) == 0)) {
    return &utf16_bom;
  }
  else
    return 0;
}


