/* -*- c++ -*- */
#include "ptrmember_handler.h"

#include "sequence_handler.h"
#include "choice_handler.h"
#include "all_handler.h"
#include "simplecontent_handler.h"
#include "simpletype_handler.h"
#include "array_handler.h"

#include "../../framework/operators.h"
#include "../../framework/membertype.h"

#include <cassert>

using namespace aka2;

ptrmember_handler::ptrmember_handler(bool emptiable, void *e, int depth,
				     const qname &tagname, const element_props &props, 
				     parser_context &context) 
  : handler(emptiable, depth, props, context), ptrmember_(e), e_(0), 
    pop_(static_cast<const ptrmember_op&>(props.op())), tagname_(tagname),
    appeared_(false) {
  is_element_ = props_.is_element();
}

ptrmember_handler::~ptrmember_handler() {
}


validation_result ptrmember_handler::query_element(const qname &tagname, const attribute_values &attrs) {
  if (appeared_)
    return skip;
  
  if (is_element_)
    return parse_element(tagname, attrs);
  else {
    validation_result res = find_particle(tagname, attrs, &e_, emptiable_);
    if (res == ok)
      pop_.set(ptrmember_, e_);
    return res;
  }
}


validation_result ptrmember_handler::query_next(const qname &tagname, const attribute_values &attrs) {
  
  if (is_element_) {
    if (appeared_)
      return skip; 
    if (!context_.compare_tagname(tagname, tagname_))
      return can_skip() ? skip : invalid; 
    return ok;
  }
  else {
    void *e = 0;
    bool emptiable = emptiable_;
    validation_result res = find_particle(tagname, attrs, &e, emptiable);
    pop_.get_value_op().destroy(e);
    e_ = 0;
    return res;
  }
}

validation_result ptrmember_handler::parse_element(const qname &tagname, const attribute_values &attrs) {
  if (appeared_)
    return skip;

  if (!context_.compare_tagname(tagname, tagname_)) {
    if (can_skip())
      return skip;
    context_.report_wrong_occurrence(props_.get_name(), 
				     props_.get_occurrence(), 0, __FILE__, __LINE__);
    return invalid;
  }

  emptiable_ = false; // Element found.

  const element_op &op = pop_.get_value_op();
  e_ = pop_.create_member(ptrmember_);

  switch (op.get_schematype()) {
  case sequence_id:
  case choice_id:
  case all_id: {
    create_particle_handler(false, e_, op, depth_ + 1, tagname, props_); // item element.
    return ok;
  }
  case simplecontent_id:
  case simpletype_id: {
    create_simple_handler(e_, op, depth_ + 1, tagname, props_); // item element.
    return ok;
  }
  case any_id: {
    handler *h = create_any_handler(*static_cast<any*>(e_), depth_ + 1,
				    tagname, props_);
    
    validation_result res = h->query_element(tagname, attrs);
    assert(res == ok);
    return ok;
  }
  case array_id: // array/ptrmember should not be under the ptrmember.
  case ptrmember_id:
  case enclose_id: // The use of enclose/disclose is restricted for <xs:sequence>.
  case disclose_id:
  case any_array_id: // any/any_array should not be under ptrmember.
  case fixed_id: // fixed_id should not be under the ptrmember. ( Because it is an array. )
  case wildcard_attribute_id: // attribute is not handled here.
  case wildcard_id:  // any child should not be under ptrmember.
    assert(!"Must not reach here.");
  }
  return invalid;
}

validation_result ptrmember_handler::find_particle(const qname &tagname, const attribute_values &attrs, void **ep, bool &emptiable) {

  bool child_emptiable = emptiable || props_.get_occurrence().in_range(0);


  const element_op &vop = pop_.get_value_op();
  schematype_id id = vop.get_schematype();
  void *e = vop.create();
  
  if ((id == any_id) && context_.ignore_any())
    return invalid;

  switch (vop.get_schematype()) {
  case sequence_id:
  case choice_id: 
  case all_id: {
    handler *h = create_particle_handler(child_emptiable, e, vop, depth_, tagname, props_);
    validation_result res = h->query_element(tagname, attrs);
    if (res == ok) {
      *ep = e;
      emptiable = false;
      return ok;
    }    
    context_.rollup_to(h);
    break;
  }
  case array_id: {
    handler *h = create_array_handler(child_emptiable, e, 
				      depth_, 
				      tagname, props_);
    validation_result res = h->query_element(tagname, attrs);
    if (res == ok) {
      *ep = e;
      emptiable = false;
      return ok;
    }    

    context_.rollup_to(h);
    break;
  }
  case any_id: { // <xs:any>
    bool next_found = seek_forward(tagname, attrs);
    if (!next_found) {
      handler *h = create_any_handler(*static_cast<any*>(e), depth_ + 1,
				      tagname, props_);
      validation_result res = h->query_element(tagname, attrs);
      assert(res == ok);
      emptiable = false;
      return ok;
    }
    break;
  }
  case simplecontent_id: // simpletype, simplecontent could not be particle.
  case simpletype_id:
  case any_array_id:
  case ptrmember_id: // nested ptrmember has no meaning.
  case enclose_id: // The use of enclose/disclose is restricted for ptrmember.
  case disclose_id:
  case fixed_id: // fixed should not be particle.
  case wildcard_attribute_id: // attribute is not handled here.
  case wildcard_id:
    assert(!"Must not reach here.");
  }

  if (can_skip()) {
    vop.destroy(e);
    *ep = e;
    return skip;
  }
  return invalid;
}

validation_result ptrmember_handler::end_element(const qname &tagname) {
  if (appeared_)
    return ok;
  if (can_skip())
    return ok;
  context_.report_wrong_occurrence(tagname, 
				   props_.get_occurrence(), 0, __FILE__, __LINE__);
  return invalid;
}


bool ptrmember_handler::parse_entity(const std::string &entity) {
  return true;
}

node ptrmember_handler::get_node() {
  return node(ptrmember_, pop_);
}

void ptrmember_handler::receive_child(const node &child) {
  assert(&pop_.get_value_op() == &child.op());
  assert(e_ == child.ptr());
  appeared_ = true;
  e_ = 0;
}

bool ptrmember_handler::can_skip() {
  if (emptiable_)
    return true;
  return props_.get_occurrence().in_range(0);
}

void ptrmember_handler::abort() {
  e_ = 0;
}
