#include "choice_handler.h"

#include "sequence_handler.h"
#include "all_handler.h"
#include "simplecontent_handler.h"
#include "simpletype_handler.h"
#include "array_handler.h"
#include "any_handler.h"
#include "../../framework/item.h"
#include "../../framework/construct.h"

#include <cassert>

using namespace aka2;
  
choice_handler::choice_handler(bool emptiable, 
			       void *cho, const choice_op &cop,
			       int depth,
			       const element_props &props,
			       parser_context &context) 
  : handler(emptiable, depth, props, context), 
    cho_(cho), e_(0), cop_(cop), 
    occurrence_(props.get_occurrence()), 
    count_(0),
    itypes_(cop_.get_item_types()) {
  current_ = itypes_.end();
  searching_ = itypes_.end();
}

choice_handler::~choice_handler() {
  if (e_ != 0)
    abort();
}


validation_result choice_handler::query_element(const qname &tagname, const attribute_values &attrs) {

  assert(current_ == itypes_.end());

  if (occurrence_.maxOccurs_ == count_)
    return skip;

  item_types::const_iterator it = find_item(tagname);
  if (it != itypes_.end()) {
    current_ = it;
    emptiable_ = false;
    return parse_element(attrs);
  }

  bool child_emptiable = emptiable_ || props_.is_emptiable();
  validation_result res = find_particle(tagname, attrs, current_, &e_, child_emptiable);
  if (res == ok)
    return ok;

  // <xs:any>
  if (!context_.ignore_any()) {

    it = find_item(qname("&any"));
    if (it != itypes_.end()) {
      const itemtype &itype = it->second;
      schematype_id id = itype.get_schematype();

      if (id == any_array_id) {
	e_ = itype.op().create();
	handler *h = 
	  create_any_array_handler(child_emptiable, 
				   *static_cast<any_array*>(e_),
				   depth_, itype);
	validation_result res = h->query_element(tagname, attrs);
	if (res == ok) {
	  current_ = it;
	  emptiable_ = false;
	}
	else {
	  context_.rollup_to(this);
	  itype.op().destroy(e_);
	  e_ = 0;
	}
	return res;
      }

      assert(id == any_id);
      bool next_found = seek_forward(tagname, attrs);
      if (!next_found) {
	current_ = it;
	e_ = itype.op().create();
	handler * h = create_any_handler(*static_cast<any*>(e_), 
					 depth_ + 1, 
					 tagname, current_->second);
	validation_result res = h->query_element(tagname, attrs);
	assert(res == ok);
	emptiable_ = false;
	return ok;
      }
    }
  }

  if (can_skip())
    return skip;

  //context_.report_no_element(tagname, __FILE__, __LINE__);
  return invalid;
}


validation_result choice_handler::query_next(const qname &tagname, const attribute_values &attrs) {
  

  item_types::const_iterator it = find_item(tagname);
  if (current_ == it)
    return skip;
  if (it != itypes_.end())
    return ok;

  //it = itypes_.end()
  bool emptiable = emptiable_;
  validation_result res = find_particle(tagname, attrs, it, 0, emptiable);
    
  
  if (res == ok)
    return ok;

  return can_skip() ? skip : invalid;
}


validation_result choice_handler::parse_element(const attribute_values &attrs) {

  const element_op &iop = current_->second.op();
  const qname &tagname = current_->first;
  const itemtype &itype = current_->second;

  e_ = iop.create();

  switch (iop.get_schematype()) {
  case sequence_id:
  case choice_id:
  case all_id: {
    create_particle_handler(false, e_, iop, depth_ + 1, tagname, itype);
    break;
  }
  case simplecontent_id:
  case simpletype_id:
  case fixed_id: {
    create_simple_handler(e_, iop, depth_ + 1, tagname, itype);
    break;
  }
  case array_id: {
    handler *h = create_array_handler(false, e_, 
				      depth_, 
				      tagname, itype);
    validation_result res = h->query_element(tagname, attrs);
    assert(res == ok);
    break;
  } 
  case any_id: {
    handler *h = create_any_handler(*static_cast<any*>(e_), 
				    depth_ + 1,
				    tagname, current_->second);
    validation_result res = h->query_element(tagname, attrs);
    assert(res == ok);
    break;
  }
  case any_array_id: {
    handler *h = 
      create_any_array_handler(false, *static_cast<any_array*>(e_),
			       depth_, 
			       itype);
    validation_result res = h->query_element(tagname, attrs);
    assert(res == ok);
    break;
  }
  case enclose_id: // enclose/disclose are for <xs:sequence>.
  case disclose_id:
  case ptrmember_id: // ptrmember should not be under choice.
  case wildcard_attribute_id: // attribute is not handled here.
  case wildcard_id:
    assert(!"Must not reach here.");
  }
  return ok;
}


validation_result choice_handler::find_particle(const qname &tagname, 
						const attribute_values &attrs,
						item_types::const_iterator &found,
						void **ep,
						bool &emptiable) {

  if (searching_ != itypes_.end()) {
    return invalid; //currently searching particle.  It could be called by find_next().
                  // To avoid searchnig xs:any's in items, return invalid to stop search.
  }

  handler *current = 0;
  void *e = 0;

  for (item_types::const_iterator it = itypes_.begin(); it != itypes_.end(); ++it) {

    const itemtype &itype = it->second;
    if (itype.is_element())
      continue;

    searching_ = it;

    schematype_id id = itype.get_schematype();
    switch (id) {
    case sequence_id:
    case choice_id: 
    case all_id: {
      e = itype.op().create();
      current = create_particle_handler(emptiable, e, itype.op(), 
					depth_, 
					tagname, itype);
      break;
    }
    case array_id: {
      e = itype.op().create();
      current = create_array_handler(emptiable, e, 
				     depth_, 
				     tagname, itype);
      break;
    }
    case any_id: // xs:any is handled later.
    case any_array_id: 
      continue;
    case ptrmember_id:
    case simpletype_id:    // simpletype is not a particle.
    case simplecontent_id: // simplecontent is not a particle.
    case enclose_id:       // enclose is not a particle.
    case disclose_id:      // disclose is not a particle.
    case fixed_id:         // fixed is always element.(XML-Schema spec. restriction of akaxiso.)
    case wildcard_attribute_id: // attribute is not handled here.
    case wildcard_id:
      assert("!Must not reach here.");
    }

    validation_result res = current->query_element(tagname, attrs);

    if (res == ok) {
      found = it;
      searching_ = itypes_.end();
      if (ep != 0)
	*ep = e;
      else
	itype.op().destroy(e);
      emptiable = false;
      return ok;  // Element given by tagname found.
    }

    // Rollup.
    context_.rollup_to(current);
    // Remove current.
    assert(context_.top() == current);
    context_.pop();
    if (e != 0)
      itype.op().destroy(e);

    assert((res == skip) || (res == invalid));
  }
  searching_ = itypes_.end();
  return skip;
}


validation_result choice_handler::end_element(const qname &tagname) {
  if (occurrence_.in_range(count_))
    return ok;
  context_.report_wrong_occurrence(tagname, occurrence_, count_, __FILE__, __LINE__);
  return invalid;
}


bool choice_handler::parse_entity(const std::string &entity) {
  return true;
}

node choice_handler::get_node() {
  return node(cho_, cop_);
}

void choice_handler::receive_child(const node &child) {
  assert(current_ != itypes_.end());
  assert(&child.op() == &current_->second.op());
  item i(child.ptr(), current_->second);
  cop_.push(cho_, i);
  ++count_; // occurrence increased.
  e_ = 0; // Item is passed to choice object.  Clear the pointer.
  current_ = itypes_.end();
}

bool choice_handler::can_skip() {
  if (emptiable_)
    return true;
  if (occurrence_.in_range(count_))
    return true;

  if ((count_ == 0) && props_.is_emptiable())
    return true;

  return false;
}

void choice_handler::abort() {
  if ((current_ != itypes_.end()) && (e_ != 0)) {
    const element_op &iop = current_->second.op();
    iop.destroy(e_);
  }
  e_ = 0;
}

item_types::const_iterator choice_handler::find_item(const qname &tagname) const {
  item_types::const_iterator it = itypes_.find(tagname);
  if (it != itypes_.end())
    return it;
  for (it = itypes_.begin(); it != itypes_.end(); ++it) {
    if (context_.compare_tagname(tagname, it->first))
      return it;
  }
  return itypes_.end();
}
