//CUPPA:include=+
//CUPPA:include=-
#include <cppunit/extensions/HelperMacros.h>
#include <cppunit/TestAssert.h>

#include <akaxiso/akaxiso.h>
#include <akaxiso/content_model.h>

#include <iostream>

//CUPPA:namespace=+
//CUPPA:namespace=-


namespace {

  struct SeqSample {
    std::string string_;
    std::string default_attr_;
    std::string required_attr_;
  };


  struct SeqSampleLeaf : public aka::sequence<SeqSample, SeqSampleLeaf> {
    void model() {
      member("string", &SeqSample::string_);
      attribute("default_attr", &SeqSample::default_attr_).set_default("default_value");
      attribute("required_attr", &SeqSample::required_attr_).required(true);
      fixed_attribute<std::string>("fixed_optional", "fixed_optional_value").required(false);
      fixed_attribute<std::string>("fixed_required", "fixed_required_value", 
				   xiso::leaf<std::string>()).required(true);
    }
  };

}


class attributeTest : public CppUnit::TestFixture {
  CPPUNIT_TEST_SUITE(attributeTest);
//CUPPA:suite=+
  CPPUNIT_TEST(test_default);
  CPPUNIT_TEST(test_equality0);
  CPPUNIT_TEST(test_equality1);
  CPPUNIT_TEST(test_copy);
  CPPUNIT_TEST(test_replicate);
  CPPUNIT_TEST(test_serialize);
  CPPUNIT_TEST(test_parse);
  CPPUNIT_TEST(test_wrong_fixed_attribute);
  CPPUNIT_TEST(test_lack_of_required_attribute);
  CPPUNIT_TEST(test_optional_attribute);
  CPPUNIT_TEST(test_optional_fixed_attribute);
//CUPPA:suite=-
  CPPUNIT_TEST_SUITE_END();
private:

//    // your stuff...

public:

  virtual void setUp() { 
//     XMLPlatformUtils::Initialize();
    aka::initialize();
    aka::doctype("mydoc", SeqSampleLeaf());
  }

  virtual void tearDown(){
    aka::uninitialize();
//     XMLPlatformUtils::Terminate();
  }


  static void setValues(SeqSample &seq){
    seq.string_ = "Test String for a member in Sequence.";
    seq.default_attr_ = "test_default";
    seq.required_attr_ = "test_reqruied";
  }

//CUPPA:decl=+
  void test_default() {
    SeqSample seq;
    aka::construct_element(seq, SeqSampleLeaf());
    CPPUNIT_ASSERT_MESSAGE("Default value not set.", seq.default_attr_ == "default_value");
  }

  void test_equality0(){
    SeqSample seq;
    SeqSample seq1;

    setValues(seq);
    setValues(seq1);
    bool isEqual = aka::equals(seq, seq1, SeqSampleLeaf());

    CPPUNIT_ASSERT_MESSAGE("Equality test failed.", isEqual);
  }

  void test_equality1(){
    SeqSample seq;
    SeqSample seq1;

    setValues(seq);
    setValues(seq1);
    seq.required_attr_ = "";
    
    bool isEqual = aka::equals(seq, seq1, SeqSampleLeaf());
    
    CPPUNIT_ASSERT_MESSAGE("Equality test failed.", !isEqual);
  }

  void test_copy() {
    SeqSample seq;
    SeqSample seq1;

    setValues(seq);
    
    aka::copy_element(seq1, seq, SeqSampleLeaf());
    bool isEqual = aka::equals(seq, seq1, SeqSampleLeaf());
    
    CPPUNIT_ASSERT_MESSAGE("Sequence::CopyTo failed.", isEqual);
  }

  void test_replicate(){
    SeqSample seq;

    setValues(seq);
    
    SeqSample *seq1 = aka::replicate_element(seq, SeqSampleLeaf());
    bool isEqual = aka::equals(seq, *seq1, SeqSampleLeaf());
    
    delete seq1;
    CPPUNIT_ASSERT_MESSAGE("Sequence::CopyTo failed.", isEqual);
  }

  void test_serialize() {
    std::ostringstream ostm;
    SeqSample seq;
    setValues(seq);

    aka::xml_serializer ser;
    ser.serialize(seq, "mydoc", ostm);

//     std::cout << ostm.rdbuf()->str() << std::endl;
  }

  void test_parse() {
    std::ostringstream ostm;
    aka::xml_serializer ser;
    
    SeqSample seq;
    setValues(seq);
    ser.serialize(seq, "mydoc", ostm);
    
    aka::expat_deserializer deserializer;
    aka::document deserialized = deserializer.deserialize(ostm.rdbuf()->str());

    CPPUNIT_ASSERT_MESSAGE("Wrong Document name", aka::document_of(deserialized, "mydoc"));

    bool is_equal = aka::equals(*aka::root_cast<SeqSample>(deserialized), 
				seq, SeqSampleLeaf());
    if (!is_equal) {
      ser.serialize(deserialized, ostm);
      std::cout << ostm.str();
    }

    CPPUNIT_ASSERT_MESSAGE("Deserialized document is not the same as original.", is_equal);
  }


  void test_wrong_fixed_attribute() {
    bool deserialize_error = false;



    try {
      aka::expat_deserializer deserializer;
      deserializer.deserialize("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
		   "<mydoc "
		   " default_attr=\"default_value\" "
		   " required_attr=\"required_value\" "
		   //" fixed_optional=\"fixed_optional_value\" "
		   " fixed_optional=\"changed_fixed_optional_value\" "
		   " fixed_required=\"fixed_required_value\" "
		   ">"
		   "<string>Test String for a member in Sequence.</string>\n"
		   "</mydoc>\n");
    }
    catch (const std::exception &) {
      deserialize_error = true;
    }
    CPPUNIT_ASSERT_MESSAGE("Fixed value accepts non-fixed value.", deserialize_error);

    deserialize_error = false;
    try {
      aka::expat_deserializer deserializer;
      deserializer.deserialize("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
		   "<mydoc "
		   " default_attr=\"default_value\" "
		   " required_attr=\"required_value\" "
		   " fixed_optional=\"fixed_optional_value\" "
		   " fixed_required=\"changed_fixed_required_value\" "
		   ">"
		   "<string>Test String for a member in Sequence.</string>\n"
		   "</mydoc>\n");
    }
    catch (const std::exception &) {
      deserialize_error = true;
    }
    CPPUNIT_ASSERT_MESSAGE("Fixed value accepts non-fixed value.", deserialize_error);
  }


  void test_lack_of_required_attribute() {
    bool deserialize_error = false;
    try {
      aka::expat_deserializer deserializer;
      deserializer.deserialize("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
		   "<mydoc "
		   " default_attr=\"default_value\" "
		   //" required_attr=\"required_value\" "
		   " fixed_optional=\"fixed_optional_value\" "
		   " fixed_required=\"fixed_required_value\" "
		   ">"
		   "<string>Test String for a member in Sequence.</string>\n"
		   "</mydoc>\n");
    }
    catch (const std::exception &) {
      deserialize_error = true;
    }
    CPPUNIT_ASSERT_MESSAGE("The lack of required attribute not handled.", deserialize_error);
    
    deserialize_error = false;
    try {
      aka::expat_deserializer deserializer;
      deserializer.deserialize("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
		   "<mydoc "
		   " default_attr=\"default_value\" "
		   " required_attr=\"required_value\" "
		   " fixed_optional=\"fixed_optional_value\" "
		   //" fixed_required=\"fixed_required_value\" "
		   ">"
		   "<string>Test String for a member in Sequence.</string>\n"
		   "</mydoc>\n");
    }
    catch (const std::exception &) {
      deserialize_error = true;
    }
    CPPUNIT_ASSERT_MESSAGE("Deserialize error for fixed attribute not caught.", deserialize_error);
  }

  void test_optional_attribute() {
    aka::expat_deserializer deserializer;
    aka::document 
      deserialized = deserializer.deserialize("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
			    "<mydoc "
			    //" default_attr=\"default_value\" "
			    " required_attr=\"required_value\" "
			    " fixed_optional=\"fixed_optional_value\" "
			    " fixed_required=\"fixed_required_value\" "
			    ">"
			    "<string>Test String for a member in Sequence.</string>\n"
			    "</mydoc>\n");
    SeqSample *root = aka::root_cast<SeqSample>(deserialized);
    CPPUNIT_ASSERT_MESSAGE("Optional attribute does not have default value.", 
			   root->default_attr_ == "default_value");
  }


  void test_optional_fixed_attribute() {
    bool deserialize_error = false;
    aka::expat_deserializer deserializer;
    try {
      deserializer.deserialize("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
		   "<mydoc "
		   " default_attr=\"default_value\" "
		   " required_attr=\"required_value\" "
		   //			    " fixed_optional=\"fixed_optional_value\" "
		   " fixed_required=\"fixed_required_value\" "
		   ">"
		   "<string>Test String for a member in Sequence.</string>\n"
		   "</mydoc>\n");
    }
    catch (const std::exception &) {
      deserialize_error = true;
    }
    CPPUNIT_ASSERT_MESSAGE("failed to handle lack of optional fixed attribute.", !deserialize_error);
  }
//CUPPA:decl=-
};

//CUPPA:impl=+
//CUPPA:impl=-

CPPUNIT_TEST_SUITE_REGISTRATION(attributeTest);
