#include "dtd_processor.h"
#include "processor.h"
#include <akaxiso/imported/expat_parser.h>
#include <akaxiso/schema/schema_xiso.h>

#include <iostream>
#include <fstream>
#include "classdefs.h"


/**
 * xs:complexType -> complexTypeModel -> contentDef -> typeDefParticle -> particle ->
 * explicitGroup -> 
 * particle  -> localElement, explicitGroup
 */

namespace {


  /* attribute */
  struct attdecl {
    attdecl() : required_(false) {}
    std::string elmname_;
    std::string name_;
    std::string type_;
    bool required_;
    std::string default_;
  };


  typedef std::multimap<std::string, attdecl> attdecl_map;
  typedef std::map<std::string, xs::topLevelComplexType> complexTypes;

  struct handler : aka::expat_dtd_handler {
    handler(const registry &reg)  : registry_(reg) { 
      aka::construct_element(topLevelElement_, xs::topLevelElementLeaf());
    }
    virtual ~handler() { }
    virtual void elementDeclHandler(const XML_Char *name, XML_Content *model);
    virtual void attlistDeclHandler(const XML_Char *elname,
				    const XML_Char *attname,
				    const XML_Char *att_type,
				    const XML_Char *dflt,
				    int isrequired);
    virtual void startDoctypeDeclHandler(const XML_Char *doctypeName,
					 const XML_Char *sysid,
					 const XML_Char *pubid,
					 int has_internal_subset);
    virtual void endDoctypeDeclHandler();

    xs::explicitGroup process_explicitGroup(XML_Content &model);
    void process_contentDef(xs::contentDef &cdef, XML_Content &model);

    const registry &registry_;
    xs::topLevelElement topLevelElement_;
    attdecl_map attdecls_;
    complexTypes complexTypes_;
  };

  void set_occurrence(long &minOccurs, long &maxOccurs, XML_Content_Quant quant) {
    switch (quant) {
    case XML_CQUANT_NONE:
      break;
    case XML_CQUANT_OPT:
      minOccurs = 0;
      maxOccurs = 1;
      break;
    case XML_CQUANT_REP:
      minOccurs = 0;
      maxOccurs = aka::unbounded;
      break;
    case XML_CQUANT_PLUS:
      minOccurs = 1;
      maxOccurs = aka::unbounded;
      break;
    }
  }


  xs::explicitGroup handler::process_explicitGroup(XML_Content &model) {

    xs::explicitGroup eg;
    xs::particleLeaf::moc moc(eg.particle_);

    aka::construct_element(eg, xs::explicitGroupLeaf());
    set_occurrence(eg.minOccurs_, eg.maxOccurs_, model.quant);
    
    for (size_t index = 0; index < model.numchildren; ++index) {
      XML_Content &child = model.children[index];

      switch (child.type) {
      case XML_CTYPE_EMPTY: {
	xs::localElement le;
	aka2::construct_element(le, xs::localElementLeaf());
	le.name_ = model.name;
	le.type_ = aka::qname(registry_.get_nill_type());
	moc.push_back(le, "xs:element");
	set_occurrence(le.minOccurs_, le.maxOccurs_, child.quant);
	break;
      }
      case XML_CTYPE_ANY: {
	xs::any any;
	aka2::construct_element(any, xs::anyLeaf());
	any.minOccurs_ = 0;
	any.maxOccurs_ = aka::unbounded;
	moc.push_back(any, "xs:any");
	break;
      }
      case XML_CTYPE_MIXED: {
	assert(!"Must not reach here.");
	break;
      }
      case XML_CTYPE_NAME: {
	xs::localElement le;
	aka::construct_element(le, xs::localElementLeaf());
	le.name_ = child.name;
	le.type_ = aka::qname(child.name);
	set_occurrence(le.minOccurs_, le.maxOccurs_, child.quant);
	moc.push_back(le, "xs:element");
	break;
      }
      case XML_CTYPE_CHOICE: {
	xs::explicitGroup choice = process_explicitGroup(child);
	set_occurrence(choice.minOccurs_, choice.maxOccurs_, child.quant);
	moc.push_back(choice, "xs:choice");
	break;
      }
      case XML_CTYPE_SEQ: {
	xs::explicitGroup sequence = process_explicitGroup(child);
	set_occurrence(sequence.minOccurs_, sequence.maxOccurs_, child.quant);
	moc.push_back(sequence, "xs:sequence");
	break;
      }
      }
    }
    return eg;
  }


  void handler::process_contentDef(xs::contentDef &cdef, XML_Content &model) {
    xs::typeDefParticleLeaf::moc moc(cdef.particle_);

    switch (model.type) {
    case XML_CTYPE_EMPTY: {
      // Empty complexType.  Nothing to define.
      break;
    }
    case XML_CTYPE_ANY: {
      xs::explicitGroup eg;
      aka::construct_element(eg, xs::explicitGroupLeaf());
      xs::particleLeaf::moc moc(eg.particle_);

      xs::any any;
      aka2::construct_element(any, xs::anyLeaf());
      any.minOccurs_ = 0;
      any.maxOccurs_ = aka::unbounded;
      moc.push_back(any, "xs:any");
      break;
    }
    case XML_CTYPE_MIXED:
    case XML_CTYPE_NAME: {
      assert(!"Must not reach here.");
      break;
    }
    case XML_CTYPE_CHOICE: {
      xs::explicitGroup choice = process_explicitGroup(model);
      moc.push_back(choice, "xs:choice");
      break;
    }
    case XML_CTYPE_SEQ: {
      xs::explicitGroup sequence = process_explicitGroup(model);
      moc.push_back(sequence, "xs:sequence");
      break;
    }
    }
  }

  void handler::elementDeclHandler(const XML_Char *name, XML_Content *model) {
    xs::topLevelComplexType tlc;
    aka::construct_element(tlc, xs::topLevelComplexTypeLeaf());
    tlc.name_ = name;
    if (model->type == XML_CTYPE_MIXED)
      tlc.mixed_ = true;

    xs::contentDef cdef;
    process_contentDef(cdef, *model);

    xs::complexTypeModelLeaf::moc moc(tlc.model_);
    moc.push_back(cdef, "&contentDef");
    complexTypes_.insert(complexTypes::value_type(name, tlc));
  }
  
  void handler::attlistDeclHandler(const XML_Char *elname,
				   const XML_Char *attname,
				   const XML_Char *att_type,
				   const XML_Char *dflt,
				   int isrequired) {
    attdecl adecl;
    adecl.elmname_ = elname;
    adecl.name_ = attname;
    adecl.type_ = att_type;
    adecl.required_ = isrequired != 0;
    if (dflt != 0)
      adecl.default_ = dflt;
    attdecls_.insert(attdecl_map::value_type(adecl.elmname_, adecl));
  }


  void handler::startDoctypeDeclHandler(const XML_Char *doctypeName,
					const XML_Char *sysid,
					const XML_Char *pubid,
					int has_internal_subset) {
    aka::construct_element(topLevelElement_, xs::topLevelElementLeaf());
    topLevelElement_.name_ = doctypeName;
    topLevelElement_.type_ = aka::qname(doctypeName);
  }
  
  void handler::endDoctypeDeclHandler() {

  }

}

dtd_processor::dtd_processor(registry &reg, std::ostream &ostm, bool verbose, bool do_dump)
  : registry_(reg), ostm_(&ostm), verbose_(verbose), do_dump_(do_dump), schema_(0) { }

int dtd_processor::process(const std::string &base, const std::string &filename) { 
  if (schema_ != 0)
    delete schema_;

  aka::expat_parser parser;
  std::ifstream file(filename.c_str());
  handler h(registry_);

  parser.set_dtd_handler(h);
  parser.set_base(base);

  try {
    parser.parse_istream(file, filename);
  }
  catch (const std::exception &e) {
    std::cerr << e.what() << std::endl
	      << "failed to process document." << std::endl;
  }

  /** Insert declaration elements to created root object. */

  /** Construct xs::schema document. */
  schema_ = new xs::schema;
  aka::construct_element(*schema_, xs::schemaLeaf());

  /** Insert top level element */
  xs::annotatedSchemaTop top;
  xs::schemaTopLeaf::moc act_moc(top.schemaTop_);
  act_moc.push_back(h.topLevelElement_, "xs:element");
  schema_->annotatedSchemaTops_.push_back(top);

  /** Insert complexTypes */
  for (complexTypes::iterator it = h.complexTypes_.begin(); 
       it != h.complexTypes_.end(); ++it) {

    xs::redefinable redefinable;
    xs::redefinableLeaf::moc rd_moc(redefinable);

    // get contentDef to store xs::complexType.
    xs::topLevelComplexType &tlc = it->second;
    assert(tlc.model_.size() == 1);

    aka::item item = tlc.model_.front();
    xs::contentDef &cdef = aka::item_cast<xs::contentDef>(item);

    // Insert attribute declarations to attrDecls.
    // construct attrDecls.
    xs::attributeChoiceLeaf::moc at_moc(cdef.attrDecls_.choice_);

    // get attribute associated with element named it->first.
    attdecl_map::iterator attrbegin = h.attdecls_.lower_bound(it->first);
    attdecl_map::iterator attrend = h.attdecls_.upper_bound(it->first);
    for (attdecl_map::iterator ait = attrbegin; ait != attrend; ++ait) {
      attdecl &adecl = ait->second;

      xs::attribute attr;
      aka::construct_element(attr, xs::attributeLeaf());
      attr.name_ = adecl.name_;
      if (adecl.required_) {
	if (!adecl.default_.empty())
	  attr.use_ = "fixed";
	else
	  attr.use_ = "required";
      }
      attr.type_ = aka::qname(ait->second.type_);

      /** insert attrDecls -> contentdef; */
      at_moc.push_back(attr, "xs:attribute");
    }
    rd_moc.push_back(it->second, "xs:complexType");
    xs::annotatedSchemaTop top;
    xs::schemaTopLeaf::moc act_moc(top.schemaTop_);
    act_moc.push_back(redefinable, "&redefinable");
    schema_->annotatedSchemaTops_.push_back(top);
  }

  if (do_dump_) {
    aka::xml_serializer ser;
    try {
      ser.serialize(*schema_, "xs:schema", std::cout);
    }
    catch (const std::exception &e) {
      *ostm_ << "Serialization failed." << std::endl
	     << e.what() << std::endl;
    }
  }
  return 0;
}

xs::schema* dtd_processor::get_schema() {
  return schema_;
}

