/*
 * AJD4JP
 * Copyright (c) 2011  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に
 * 同意できる場合にのみ利用可能です。
 */
package ajd4jp;


class EraWork {
	static AJD[][] st_ed;
	static {
		st_ed = new AJD[4][];
		try {
			st_ed[0] = new AJD[] {
				new AJD( 1868, 9, 8 ), new AJD( 1912, 7, 30 )
			};
			st_ed[1] = new AJD[] {
				new AJD( 1912, 7, 30 ), new AJD( 1926, 12, 25 )
			};
			st_ed[2] = new AJD[] {
				new AJD( 1926, 12, 25 ), new AJD( 1989, 1, 7 )
			};
			st_ed[3] = new AJD[] {
				new AJD( 1989, 1, 8 ), null
			};
		}
		catch( AJDException e ) {}
	}
}

/**
 * 和暦の年号。明治以降の年号を持ちます。<br>
 * 明治6年より過去は太陽暦ではないため、正しい対応とはなりません。<br>
 * 昭和～平成以外では、前年号の最終日と次年号の開始日が同一日です。このクラスで該当日を判定する場合は、未来の年号(大正と昭和なら昭和)を採用します。
 */
public enum Era {
	/** 明治 */
	MEIJI( "明治", EraWork.st_ed[0][0], EraWork.st_ed[0][1] ),
	/** 大正 */
	TAISHO( "大正", EraWork.st_ed[1][0], EraWork.st_ed[1][1] ),
	/** 昭和 */
	SHOWA( "昭和", EraWork.st_ed[2][0], EraWork.st_ed[2][1] ),
	/** 平成 */
	HEISEI( "平成", EraWork.st_ed[3][0], EraWork.st_ed[3][1] );

	private String name;
	private AJD	start, end = null;
	private Era( String n, AJD s, AJD e ) {
		name = n;
		start = s;
		end = e;
	}

	static Era get( Day ajd ) {
		final Era[]	era = values();
		for ( int i = era.length - 1; i >= 0; i-- ) {
			if ( era[i].start.compareTo( ajd ) <= 0 )	return era[i];
		}
		return null;
	}

	/**
	 * 和暦年の取得。
	 * @param yyyy 西暦年。
	 * @param mm 月。
	 * @param dd 日。
	 */
	public static Era.Year getEra( int yyyy, int mm, int dd ) throws AJDException {
		return new Era.Year( new AJD( yyyy, mm, dd ) );
	}

	/**
	 * 年号名の取得。
	 * @return 明治～平成。
	 */
	public String getName() { return name; }
	/**
	 * 開始日の取得。
	 * @return 開始日。
	 */
	public AJD getStart() { return start; }
	/**
	 * 終了日の取得。
	 * @return 終了日。平成はnullを返します。
	 */
	public AJD getEnd() { return end; }

	/**
	 * 日付範囲判定。指定日付が年号の期間中のものか判定します。
	 * @param date 判定日付。
	 * @return true:その年号範囲、false:その年号の範囲外。
	 */
	public boolean isValid( AJD date ) {
		if ( date == null )	return false;
		if ( start.compareTo( date ) > 0 )	return false;
		if ( end == null )	return true;
		date = date.trim();
		if ( end.compareTo( date ) < 0 )	return false;
		return true;
	}

	/**
	 * 西暦での年の取得。
	 * @param yy 年号での年。
	 * @return yyに対応する西暦年。年号の範囲を超過する年でも値を返します。
	 */
	public int getAD( int yy ) {
		return start.getYear() + yy - 1;
	}

	private int getYear( int yyyy ) {
		return yyyy - start.getYear() + 1;
	}

	/**
	 * 和暦の年を表します。
	 */
	public static class Year implements java.io.Serializable {
		private static final long serialVersionUID = 1;
		private Era era;
		private int yy;

		private Year() {}
		/**
		 * コンストラクタ。
		 */
		public Year( Day ajd ) {
			era = Era.get( ajd );
			int y4 = ajd.getYear();
			yy = ( era == null )?	y4:	era.getYear( y4 );
		}
		/**
		 * 年号の取得。
		 * @return 年号。明治より前はnullです。
		 */
		public Era getEra() { return era; }
		/**
		 * 年の取得。
		 * @return 年。明治より前は西暦で返します。
		 */
		public int getYear() { return yy; }
		/**
		 * 文字列取得。
		 * @return 年号yy年のフォーマット。
		 */
		public String toString() {
			return String.format( "%s%d年", ( era == null )? "": era.getName(), yy );
		}
	}
}

