/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.jetspeed.tutorial.modules.actions.portlets;


// Java 
import java.util.List;
import java.sql.Connection;


// Turbine 
import org.apache.turbine.util.Log;
import org.apache.turbine.util.RunData;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.DynamicURI;

// Velocity 
import org.apache.velocity.context.Context;

//Torque 
import org.apache.torque.util.Criteria;
import org.apache.torque.Torque;

// Jetspeed 
import org.apache.jetspeed.portal.portlets.VelocityPortlet;
import org.apache.jetspeed.modules.actions.portlets.VelocityPortletAction;
import org.apache.jetspeed.util.PortletConfigState;
import org.apache.jetspeed.util.StringUtils;
import org.apache.jetspeed.util.template.JetspeedLink;
import org.apache.jetspeed.util.template.JetspeedLinkFactory;


// Tutorial
import org.apache.jetspeed.tutorial.om.Cafe;
import org.apache.jetspeed.tutorial.om.CafePeer;

/**
 * Tutorial 9 
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: TutorialCafeAction.java,v 1.1 2004/04/08 17:03:54 taylor Exp $
 */
public class TutorialCafeAction extends VelocityPortletAction
{
    // tutorial reader: you may consider putting these in a constants file
    // im leaving them here for easy reading of the tutorial
    public static final String UI_MODE = "js_mode";
    public static final String UI_ROW_ID = "js_rowid";

    public static final String UI_EDIT = "Edit";
    public static final String UI_UPDATE = "Update";
    public static final String UI_ADD = "Add";
    public static final String UI_DELETE = "Delete";
    public static final String UI_REFRESH = "Refresh";
    public static final String UI_VALIDATE = "Validate";

    private static final String SESSION_CAFE = "cafe.instance";
    private static final String SESSION_UPDATE_MODE = "cafe.mode";
    private static final String UI_CAFE = "cafe";

    private static final String NO_CURRENT_REC = "Cafe Tutorial: no current record";

    /**
     * Build the normal state content for this portlet.
     *
     * @param portlet The velocity-based portlet that is being built.
     * @param context The velocity context for this request.
     * @param rundata The turbine rundata context for this request.
     */

    protected void buildNormalContext(VelocityPortlet portlet,
                                       Context context,
                                       RunData rundata)
    {
        String mode = null;
        Cafe cafe = null;

        try
        {
            mode = this.getQueryParameter(rundata, UI_MODE, UI_REFRESH);
            cafe = (Cafe)rundata.getUser().getTemp(SESSION_CAFE);

            // refresh mode - simply keep state of fields from session, 
            // the request is for another portlet or simply a refresh
            //
            if (mode.equalsIgnoreCase(UI_REFRESH))
            {
                if (cafe != null)
                {
                    rundata.getParameters().setProperties(cafe);
                }                
            }
            else if (mode.equalsIgnoreCase(UI_EDIT) ||
                     mode.equalsIgnoreCase(UI_DELETE))
            {
                int rowid = Integer.parseInt(this.getQueryParameter(rundata, UI_ROW_ID));

                context.put(UI_ROW_ID, String.valueOf(rowid));

                // get the primary key and put the object in the context
                Criteria criteria = new Criteria();
                criteria.add( CafePeer.CAFE_ID, rowid);
                List cafes = CafePeer.doSelect(criteria);
                if (cafes != null && cafes.size() > 0)
                {
                    cafe = (Cafe)cafes.get(0);
                }
                else
                {
                    throw new Exception("Cafe for id="+rowid+" not found.");
                }

                rundata.getUser().setTemp(SESSION_CAFE, cafe);

                rundata.getUser().setTemp(SESSION_UPDATE_MODE, mode);

            }
            else if (mode.equalsIgnoreCase(UI_ADD))
            {
                cafe = new Cafe(); 
                cafe.setCafeName("");
                rundata.getUser().setTemp(SESSION_CAFE, cafe);
                rundata.getUser().setTemp(SESSION_UPDATE_MODE, mode);
            }

            context.put(UI_CAFE, cafe);
            context.put(UI_MODE, mode);


        }
        catch (Exception e)
        {
            Log.error(e);
            rundata.setMessage(e.toString());
        }
    }



    /**
     * Update the portlet state from the form.
     *
     * @param rundata The turbine rundata context for this request.
     * @param context The velocity context for this request.
     */
    public void doUpdate(RunData rundata, Context context) throws Exception
    {
        Cafe cafe = null;
        Connection con = null;

        try
        {
            con = Torque.getConnection();

            cafe = (Cafe)rundata.getUser().getTemp(SESSION_CAFE);

            if(cafe == null)
            {
                Log.error(NO_CURRENT_REC);
                rundata.setMessage(NO_CURRENT_REC);
                return;
            }
        
            rundata.getParameters().setProperties(cafe);

            validate(cafe);

            cafe.save(con);

            con.commit();

            returnToBrowser(rundata, true);
            
        }
        catch (Exception e)
        {
            Log.error("error in saving cafe: " + e);
            rundata.setMessage(e.toString());
            if (con != null)
            {
                con.rollback();
            }

        }
        finally
        {
            try
            {
                Torque.closeConnection(con);
            }
            catch (Exception e)
            {}
        }

    }

    /**
     * Delete current record
     *
     * @param rundata The turbine rundata context for this request.
     * @param context The velocity context for this request.
     */
    public void doDelete(RunData rundata, Context context) throws Exception
    {
        Cafe cafe = null;
        Connection con = null;

        try
        {
            con = Torque.getConnection();

            cafe = (Cafe)rundata.getUser().getTemp(SESSION_CAFE);

            if(cafe == null)
            {
                Log.error(NO_CURRENT_REC);
                rundata.setMessage(NO_CURRENT_REC);
                return;
            }
        

            CafePeer.doDelete(cafe, con);

            con.commit();

            returnToBrowser(rundata, true);
            
        }
        catch (Exception e)
        {
            Log.error("error in deleting cafe: " + e);
            rundata.setMessage(e.toString());

            if (con != null)
            {
                con.rollback();
            }

        }
        finally
        {
            try
            {
                Torque.closeConnection(con);
            }
            catch (Exception e)
            {}
        }

    }

    /**
     * Cancel editing or deletion
     *
     * @param rundata The turbine rundata context for this request.
     * @param context The velocity context for this request.
     */
    public void doCancel(RunData rundata, Context context) throws Exception
    {
        returnToBrowser(rundata, false);
    }

    /** 
     * Helper function to get query param from request
     *
     */
    public static String getQueryParameter(RunData rundata, String attrName)
    {
        String str = rundata.getParameters().getString(attrName);
        if(str != null && str.length() > 0 && str.trim().length() > 0 )
            return str;
        return null;
    }

    /** 
     * Helper function to get query param from request with a default value
     *
     */
    public static String getQueryParameter(RunData rundata, String attrName, String defaultValue)
    {
        String str = getQueryParameter(rundata, attrName);
        if(str == null)
            return defaultValue;
        else
            return str;
    }


    /**
     *   validate that its a valid record
     *
     */
    private void validate(Cafe cafe) throws Exception
    {

        if(isEmpty(cafe.getCafeName()))
        {
            throw new Exception("Cafe Name must be entered.");
        }
    }

    /**
     * Check for string bein empty or null
     * this should be put in some utility package
     *
     */
    public static boolean isEmpty(String str)
    {
        if(!(str != null && str.trim().length() > 0))
        {
            return true;
        }
        return false;
    }

    /** 
     * redirect back to browser
     *
     */
    private void returnToBrowser(RunData rundata, boolean refresh)
    {
        try
        {
            JetspeedLink link = JetspeedLinkFactory.getInstance(rundata);
            DynamicURI duri = link.getPaneByName("TutorialCafeBrowser");
            if (refresh)
            {
                duri.addQueryData(CafeBrowserAction.BROWSER_COMMAND, CafeBrowserAction.BROWSER_REFRESH);
            }
            rundata.setRedirectURI(duri.toString());
            JetspeedLinkFactory.putInstance(link);
        }
        catch (Exception e)
        {
            Log.error(e);
            rundata.setMessage(e.toString());
        }
    }

}
