# Copyright (C) 2005 FishGrove Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# $Id: Affelio.pm,v 1.18 2005/07/01 08:19:42 slash5234 Exp $

package Affelio;
{
    use strict;
    use lib("../extlib");
    use lib("../../extlib");
    use DBI;
    use Config::Tiny;
    use Config::IniFiles;
    use Error qw(:try);
    use lib("../lib");
    use Affelio::SNS::FriendManager;
    use Affelio::Managing::ProfileManager;
    use Affelio::Managing::GroupManager;
    use Affelio::Managing::PermissionManager;
    use Affelio::Managing::MessageManager;
    use Affelio::Managing::ApplicationManager;
    use Affelio::Managing::AccessLogManager;
    use Affelio::misc::Debug;
    use Affelio::misc::L10N;
    use Affelio::misc::WebInput;
    use Affelio::exception::TaintedInputException;
    use Affelio::exception::DBException;
    use Affelio::exception::SystemException;

    ######################################################################
    #Constructor
    ######################################################################
    sub new{
	my $class = shift;
	my %param = @_;

	debug_print("Affelio::new: start.");

	###################################
	#Config dir
	###################################
	my $cfg_dir = $param{ConfigDir};
	my $cfg_path = $param{ConfigDir} . "/affelio.cfg";
	debug_print("Affelio::new: cfg_path= [$cfg_path]");
	#cfg_path               Configuration file's path
	#cfg_dir                Configuration file's directory

	###################################
	#Mode
	###################################
	my $mode = $param{Mode};
	if(!defined($mode)){
	    $mode ="";
	}
	#mode= "init"                Setup mode
	#mode= ""                    Normal execution 

	###################################
	#Guest/Owner current status
	###################################
	# How can we know if we are in owner mode? or guest mode?
	my $guest_owner_switch = $param{GuestOwnerSwitch};
	#guest_owner_switch = ""          guest
	#guest_owner_switch = "owner"     owner

	###################################
	#Caller
	###################################
	my $caller = "";
	$caller = $param{Caller};
	if($caller){
	    if($caller =~ /([\w]+)/){
		$caller =$1;
	    }else{
		throw Affelio::exception::TaintedInputException("Affelio caller");
	    }
	}else{
	    $caller="";
	}
	debug_print("Affelio::new: caller= [$caller] ");
	#caller= ""            ... if AffelioCore called me.
	#caller= "install_name"... if application "install_name" called me.

	###################################
	#Managers
	###################################
	my $lh = "";
	my $db = "";
	my $pm = "";
	my $fm = "";
	my $gm = "";
	my $perm = "";
	my $am = "";
	my $alm = "";

	###################################
	#Blessing
	###################################
	my $self = {cfg_path => $cfg_path,
		    cfg_dir => $cfg_dir,
		    lh => $lh,
		    db => $db,
		    pm => $pm,
		    fm => $fm,
		    gm => $gm,
		    perm => $perm,
		    am => $am,
		    alm => $alm,
		    mode => $mode,
		    guest_owner_switch => $guest_owner_switch,
		    caller => $caller
		    };

	bless $self, $class;

	###################################
	#Initialization
	###################################
	#Read site config. and user preferences.
	$self->read_site_config();
	$self->read_user_prefs();

	#Connect to DB
	$self->openDB();

	#Load locale 
	$self->load_Locale();

	#Load profile manager
	$self->load_ProfileManager();

	#Load friend manager
	$self->load_FriendManager();

	#Load group manager
	$self->load_GroupManager();

	#Load permission manager
	$self->load_PermissionManager();

	#Load permission manager
	$self->load_MessageManager();

	#Load permission manager
	$self->load_AccessLogManager();

	if($self->{mode} ne "init"){
	    $self->load_ApplicationManager();
	}

	debug_print("Affelio::new: end.");
	return $self;
    }

    ######################################################################
    #Destructor
    ######################################################################
    sub DESTROY{
	my $self = shift;
	if(defined($self->{db})){
	    try{
		$self->{db}->disconnect;
	    }catch Error with{
	    };
	}
    }

    ######################################################################
    #set_owner_mode:   Flag this Affelio as "owner" mode.
    ######################################################################
    sub set_owner_mode{
	my $self = shift;
	debug_print("Affelio::set_owner_mode  OK");
	$self->{guest_owner_switch} = "owner";
    }

    ######################################################################
    #openDB
    ######################################################################
    sub openDB{
	my $self = shift;

	undef($self->{db});

	#Load username and password
	my $DBConfig = Config::Tiny->new();
	$DBConfig = Config::Tiny->read("$self->{site__user_dir}/db.cfg");
	my $rootproperty = $DBConfig->{_}->{rootproperty};
	my $db_type = $DBConfig->{db}->{type};
	my $db_dbname   = $DBConfig->{db}->{dbname};
	my $db_username = $DBConfig->{db}->{username};
	my $db_password = $DBConfig->{db}->{password};
	my $db_hostname = $DBConfig->{db}->{hostname};
	my $db_port = $DBConfig->{db}->{port};

	if($db_type eq "mysql"){
	    
	    my $dsn = "DBI:mysql:$db_dbname:$db_hostname:$db_port";
	    eval{
		$self->{db} = DBI->connect($dsn, $db_username, $db_password) or die("Cannot connect to MySQL: " . $DBI::errstr);
	    };
	    if($@){
		throw Affelio::exception::DBException("MySQL open error");
	    }

	}elsif ($db_type eq "sqlite"){
	    eval{
		$self->{db} = DBI->connect("dbi:SQLite:dbname=$self->{site__user_dir}/DB.sqlite", "", "") or die("Cannot connect to SQLite: " . $DBI::errstr);
	    };
	    if($@){
		throw Affelio::exception::DBException("SQLite open error");
	    }
	}else{
	    throw Affelio::exception::SystemException("DB type not specified");
	}

	if(!defined($self->{db})){
	    throw Affelio::exception::DBException("DB Handle not defined after open. very weird.");
	}

	eval{
	    $self->{db}->{RaiseError} = 1;
	};
	if($@){
	    throw Affelio::exception::DBException("DB Error Setting");
	}

    }

    ######################################################################
    #openAppDB
    ######################################################################
    sub openAppDB{
	my $self = shift;
	my $app_install_name = shift;

	#Load username and password
	my $DBConfig = Config::Tiny->new();
	$DBConfig = Config::Tiny->read("$self->{site__user_dir}/db.cfg");
	my $rootproperty = $DBConfig->{_}->{rootproperty};
	my $db_type = $DBConfig->{appdb}->{type};
	my $db_dbname   = $DBConfig->{appdb}->{dbname};
	my $db_username = $DBConfig->{appdb}->{username};
	my $db_password = $DBConfig->{appdb}->{password};
	my $db_hostname = $DBConfig->{appdb}->{hostname};
	my $db_port = $DBConfig->{appdb}->{port};

	my $ret="";

	###MySQL
	if($db_type eq "mysql"){
	    my $dsn = "DBI:mysql:$db_dbname:$db_hostname:$db_port";

	    eval{
		$ret = DBI->connect($dsn, $db_username, $db_password);
	    };
	    if($@){
		throw Affelio::exception::DBException("MySQL open error");
	    }
	}
	###SQLite
	if($db_type eq "sqlite"){
	    eval{
		$ret = DBI->connect("dbi:SQLite:dbname=$self->{site__user_dir}/appdata/$app_install_name/DB.sqlite", "", "");
	    };
	    if($@){
		throw Affelio::exception::DBException("SQLite open error");
	    }
	}

	eval{
	    $ret->{RaiseError} = 1;
	};
	if($@){
	    throw Affelio::exception::DBException("DB Error Setting");
	}

	$ret->{RaiseError} = 1;

	return($ret);
    }

    ######################################################################
    #load_Locale
    ######################################################################
    sub load_Locale{
	my $self = shift;
	$self->{lh} = Affelio::misc::L10N->get_handle(($self->{site__locale},"en_us"));
    }

    ######################################################################
    #load_ProfileManager
    ######################################################################
    sub load_ProfileManager{
	my $self = shift;
	try{
	    $self->{pm} = new Affelio::Managing::ProfileManager($self,
								$self->{mode});
	}
	catch Error with {
	    my $ex = shift;
	    if($self->{mode} ne "init"){
		throw Affelio::exception::SystemException("Cannot load profilemanager: $ex");
	    }
	};
    }

    ######################################################################
    #load_FriendManager
    ######################################################################
    sub load_FriendManager{
	my $self = shift;
	try{
	    $self->{fm} = new Affelio::SNS::FriendManager($self);
	}catch Error with {
	    my $ex = shift;
	    if($self->{mode} ne "init"){
		throw Affelio::exception::SystemException("Cannot load FriendManager: $ex");
	    }
	};
    }

    ######################################################################
    #load_GroupManager
    ######################################################################
    sub load_GroupManager{
	my $self = shift;
	try{
	    $self->{gm} = new Affelio::Managing::GroupManager($self);
	}
	catch Error with {
	    my $ex = shift;
	    if($self->{mode} ne "init"){
		throw Affelio::exception::SystemException("Cannot load GroupManager: $ex");
	    }
	};

    }

    ######################################################################
    #load_PermissionManager
    ######################################################################
    sub load_PermissionManager{
	my $self = shift;
	try{
	    $self->{perm} = new Affelio::Managing::PermissionManager($self);
	}
	catch Error with {
	    my $ex = shift;
	    if($self->{mode} ne "init"){
		throw Affelio::exception::SystemException("Cannot load PermissionManager: $ex");
	    }
	};
    }

    ######################################################################
    #load_MessageManager
    ######################################################################
    sub load_MessageManager{
	my $self = shift;
	try{
	    $self->{mesgm} = new Affelio::Managing::MessageManager($self);
	}
	catch Error with {
	    my $ex = shift;
	    if($self->{mode} ne "init"){
		throw Affelio::exception::SystemException("Cannot load MessageManager: $ex");
	    }
	};

    }

    ######################################################################
    #load_AccessLogManager
    ######################################################################
    sub load_AccessLogManager{
	my $self = shift;
	try{
	    $self->{alm} = new Affelio::Managing::AccessLogManager($self);
	}
	catch Error with {
	    my $ex = shift;
	    if($self->{mode} ne "init"){
		throw Affelio::exception::SystemException("Cannot load AccessLogManager: $ex");
	    }
	};

    }

    ######################################################################
    #load_ApplicationManager
    ######################################################################
    sub load_ApplicationManager{
	my $self = shift;
	try{
	    ###########################
	    #Start ApplicationManager 
	    ###########################
	    $self->{am} = new Affelio::Managing::ApplicationManager($self);

	    ###########################
	    #Configure "Top page"
	    ###########################
	    if(($self->{userpref__toppage_app_installname} ne "")
	       && ($self->{userpref__toppage_app_installname} ne "Affelio")){

		   $self->{userpref__toppage_app_path}= 
		       "/apps/" .
		       $self->{userpref__toppage_app_installname} . "/" .
		       $self->{am}->{apps}->{"$self->{userpref__toppage_app_installname}"}->{guest_index};

	       }else{
		   $self->{userpref__toppage_app_installname} = "Affelio";
		   $self->{userpref__toppage_app_path}="index.cgi?mode=index";
	       }

	}catch Error with {
	    my $ex = shift;
	    if($self->{mode} ne "init"){
		throw Affelio::exception::SystemException("Cannot load ApplicationManager: $ex");
	    }
	};

    }


    ######################################################################
    #read_site_config
    #    read Affelio.cfg file
    ######################################################################
    sub read_site_config{
	my $self = shift;
	debug_print("Affelio::read_site_config: start.");

	my $wi = new Affelio::misc::WebInput();

	my $Config = Config::Tiny->new();

	$Config = Config::Tiny->read($self->{cfg_path});
	if($@ || !$Config){
	    throw Affelio::exception::SystemException("affelio.cfg not found");
	}

	#print "$self->{cfg_path}\n";

	my $rootproperty = $Config->{_}->{rootproperty};
	$self->{site__fs_root} = $Config->{site_config}->{fs_root};
	$self->{site__web_root} = $Config->{site_config}->{web_root};

	#Oh well, this config parameter has been "char_set" for a while..
	#For now, locale = char_set 
	$self->{site__locale} = $Config->{site_config}->{char_set};
	if($self->{site__locale} eq ""){
	    $self->{site__locale} = "ja";
	}

	$self->{site__template} = $Config->{site_config}->{template};
	if($self->{site__template} eq ""){
	    $self->{site__template} = "default";
	}

	#$self->{cmd__nkf} = $Config->{command}->{nkf};
	$self->{cmd__sendmail} = $Config->{command}->{sendmail};
	
	#Determine userdata/..../ directory
	my $dir;
	eval{
	    opendir(DIR, "$self->{cfg_dir}/userdata");
	    while (defined($dir = readdir(DIR))) {
		if(($dir ne '.') && ($dir ne '..') 
		   && ($dir ne 'default') && ($dir ne 'CVS')){
		    $self->{site__user_dir} = 
			$wi->PTN_dirname("$self->{cfg_dir}/userdata/$dir");
		}
	    }
	};
	if($@){
	    throw Affelio::exception::SystemException("cannot open userdata directory.");
	}
	closedir(DIR);

	#Load username and password
	my $Config2 = Config::Tiny->new();
	$Config2 = Config::Tiny->read("$self->{site__user_dir}/login.cfg");
	if($@ || !$Config2){
	    throw Affelio::exception::SystemException("login.cfg not found");
	}
	my $rootproperty2 = $Config2->{_}->{rootproperty};
	$self->{site__username} = $wi->PTN_nickname($Config2->{auth}->{username});
	$self->{site__password} = $Config2->{auth}->{password};
	
	#Determine session/..../ directory
	$self->{site__session_dir} = $wi->PTN_dirname("$self->{cfg_dir}/session");
	debug_print("Affelio::read_site_config: session_dir = [$self->{site__session_dir}]");
	debug_print("Affelio::read_site_config: end.");
    }

    ######################################################################
    #write_user_prefs
    ######################################################################
    sub write_user_prefs{
	my $self = shift;
	debug_print("Affelio::write_user_prefs: start.");

	my $Config = Config::Tiny->new();
	$Config = Config::Tiny->read("$self->{site__user_dir}/preference.cfg");
	if($@ || !$Config){
	    throw Affelio::exception::SystemException("preference.cfg not found");
	}

	my $rootproperty = $Config->{_}->{rootproperty};

	#Write preferences

	$Config->{messaging}->{emailflg} = 
	    $self->{userpref__mesging__emailflg};
	$Config->{preference}->{toppage_app_installname} = 
	    $self->{userpref__toppage_app_installname};
	$Config->{preference}->{emailack_friendship_recv} = 
	    $self->{userpref__emailack_fr_recv};
	$Config->{preference}->{preferred_hosting_service} =
	    $self->{userpref__preferred_hosting_service};
	$Config->{preference}->{skin}=
	    $self->{userpref__skin};

	$Config->write("$self->{site__user_dir}/preference.cfg");
	if($@ || !$Config){
	    throw Affelio::exception::SystemException("preference.cfg is not writable.");
	}

	debug_print("Affelio::write_user_prefs: end.");
    }

    ######################################################################
    #read_user_prefs
    ######################################################################
    sub read_user_prefs{
	my $self = shift;
	debug_print("Affelio::read_user_prefs: start.");
	
	my $wi = new Affelio::misc::WebInput;

	my $Config = Config::Tiny->new();
	$Config = Config::Tiny->read("$self->{site__user_dir}/preference.cfg");
	if($@ || !$Config){
	    throw Affelio::exception::SystemException("Cannot open preference.cfg");
	}

	my $rootproperty = $Config->{_}->{rootproperty};
	
	#Read preferences
	$self->{userpref__toppage_app_installname} 
	= $wi->PTN_word($Config->{preference}->{toppage_app_installname});
	$self->{userpref__emailack_fr_recv} 
	= $wi->PTN_word($Config->{preference}->{emailack_friendship_recv});
	$self->{userpref__preferred_hosting_service}
	= $wi->PTN_URL($Config->{preference}->{preferred_hosting_service});
	$self->{userpref__skin}
	= $wi->PTN_word($Config->{preference}->{skin});

	$self->{userpref__mesging__emailflg}
	= $wi->PTN_word($Config->{messaging}->{emailflg});
	if($self->{userpref__mesging__emailflg} eq ""){
	    $self->{userpref__mesging__emailflg} = "yes";
	}

	debug_print("Affelio::read_user_prefs: end.");
    }
    
    ######################################################################
    #get_guest_owner_list
    ######################################################################
    sub get_guest_owner_list{
	my $self = shift;
	my $output_ref = shift;

	my @guest_owner_switch=();
	$output_ref->{'guest_owner_switch'} = \@guest_owner_switch;

	my $owner_mode_selected="";
	if($self->{guest_owner_switch} eq "owner"){
	    $owner_mode_selected="true";
	}else{
	    $owner_mode_selected="";
	}

	my $guest_mode_url="";
	my $owner_mode_url="";
	if($self->{caller}){ 
	    $guest_mode_url
		= $self->{am}->{apps}->{$self->{caller}}->{guest_index};
	    $owner_mode_url
		= $self->{am}->{apps}->{$self->{caller}}->{owner_index};
	}else{
	    $guest_mode_url= $self->{site__web_root} . "/index.cgi?mode=index";
	    $owner_mode_url= $self->{site__web_root} . "/admin.cgi";
	}

	if($owner_mode_selected){
	    push(@guest_owner_switch,
		 {'selected' => $owner_mode_selected,
		  'name' => "Owner",
		  'url' => $guest_mode_url,
		  'image_over' => "$self->{site__web_root}/icons/owner_over.jpg"
		  }
		 );

	}else{
	    push(@guest_owner_switch,
		 {'selected' => $owner_mode_selected,
		  'name' => "Owner",
		  'url' => $owner_mode_url,
		  'image_normal' => "$self->{site__web_root}/icons/owner_normal.jpg",
		  }
		 );
	}#else

    }#method


    ######################################################################
    #get_module_list
    ######################################################################
    sub get_module_list{
	my $self = shift;
	my $output_ref = shift;
	my $visitor_afid =shift;
	my $visitor_type =shift;

	my $selected ="";
	my @modules=();
	$output_ref->{'modules'} = \@modules;

	#########################
	#Affelio core 
	#########################
	if($self->{caller}){
	    #Affelio.pm is called by an application
	    $selected ="";
	}else{
	    $selected ="true";
	}

	my $dest_url;
	if($self->{guest_owner_switch} eq "owner"){
	    ####Owner
	    if($selected eq "true" ){
		$dest_url = $self->{site__web_root} . "/index.cgi?mode=index";
	    }else{
		$dest_url = $self->{site__web_root} . "/admin.cgi";
	    }
	}else{
	    ####Guest
	    if($selected eq "true" ){
		$dest_url = $self->{site__web_root} . "/admin.cgi";
	    }else{
		$dest_url = $self->{site__web_root} . "/index.cgi?mode=index";
	    }
	}
	push(@modules,
	     {'name' => "Affelio",
	      'image_normal' => "$self->{site__web_root}/icons/affelio_normal.jpg",
	      'image_over' => "$self->{site__web_root}/icons/affelio_over.jpg",
	      'url' => $dest_url,
	      'selected' => $selected}
	     );

	#########################
	#Applications
	#########################
	my $tmp_name;
	my $this_app_ref;
	while ( ($tmp_name, $this_app_ref) = each( %{$self->{am}->{apps}} ) ){
	    my %this_app = %$this_app_ref;

	    debug_print("Affelio::get_module_list: $this_app{'install_name'}");

	    #Is this application  permmited to be printed at the tab??
	    $self->{am}->prepare_app_perm_table($this_app{'install_name'});

	    my $perm_to_tab=0;
	    $perm_to_tab
		= $self->{am}->get_summed_app_perm($visitor_afid,
						   $visitor_type,
						   $this_app{'install_name'},
						   "DF_visibility");

	    if($perm_to_tab ==1 ){
		if($this_app{'install_name'} eq $self->{caller}){
		    $selected = "true";
		}else{
		    $selected = "";
		}

		my $dest_url = "$self->{site__web_root}/apps/$this_app{'install_name'}/";
		if($self->{guest_owner_switch} eq "owner"){
		    #Owner
		    if($selected eq "true"){
			$dest_url .= $this_app{'guest_index'};
		    }else{
			$dest_url .= $this_app{'owner_index'};
		    }
		}else{
		    #Guest
		    if($selected eq "true"){
			$dest_url .= $this_app{'owner_index'};
		    }else{
			$dest_url .= $this_app{'guest_index'};
		    }
		}
		
		push(@modules,
		     {'selected' => $selected,
		      'name' => $this_app{'install_title'},
		      'url' => $dest_url,
		      'image_normal' => "$self->{site__web_root}/apps/$this_app{'install_name'}/icons/normal.jpg",
		      'image_over' => "$self->{site__web_root}/apps/$this_app{'install_name'}/icons/over.jpg"
		      }
		     );
	    }
	}#foreach

    }#method


    ######################################################################
    #translate_templateL10N
    ######################################################################
    sub translate_templateL10N{
	my $af=shift;
	my $mesg = shift;
	
	my $tag_body ="";
	my $text_value="";
	my $param_value="";
	
	while( $mesg =~ /<AF_M ([^>]+)>/ ){
	    $tag_body = $1;
	    
	    $tag_body =~ /text(\s*)=(\s*)["']([^"']*)["'](\s*)param(\s*)=(\s*)["']([^"']*)["']/;
	    $text_value=$3;
	    $param_value=$7;
	    if($text_value eq ""){
		$tag_body =~ /text(\s*)=(\s*)["']([^"']*)["']/;
		$text_value=$3;
	    }
	    
	    my $sbst = $af->{lh}->maketext($text_value, $param_value);
	    
	    debug_print("Affelio::translate tag_body = [$tag_body]\n");
	    debug_print("Affelio::translate \t text=[$text_value]\n");
	    debug_print("Affelio::translate \t param=[$param_value]\n");
	    debug_print("Affelio::translate \t sbst=[$sbst]\n");
	    
	    $mesg =~ s/\Q<AF_M $tag_body>\E/$sbst/g;
	}
	
	return($mesg);
    }
    
    
}#package
1;
##########################################################################

