# Copyright (C) 2005 FishGrove Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
package Diary;

use strict;

use lib '../../extlib';
use lib '../../lib';

use DBI;
use Jcode;
use LWP::UserAgent;
use HTTP::Request::Common qw(POST);
use AffelioApp;
use HTML::Template;

##############################################
# Constructor for diary
##############################################

sub new {
	my ($proto, $afap) = @_;
	unless ($afap) { die("Diary::new: Error: missing username\n"); }

	my $self = {};
	$self->{afap}  = $afap;
	$self->{uname} = $afap->{af}->{site__username};
	$self->{datadir} = $afap->get_userdata_dir();
	$self->{dbh}   = $afap->get_userdata_dbh;
	$self->{entry_table} = "diary_$afap->{install_name}_entries";
	$self->{comment_table} = "diary_$afap->{install_name}_comments";
	$self->{tb_table} = "diary_$afap->{install_name}_tb";
	$self->{max_entries} = 365;
	$self->{recent_entries_no} = 10;
	$self->{header_title} = 'Affelio Diary';
	$self->{header_show} = 0;
	
	my $DBConfig = Config::Tiny->new();
	$DBConfig = Config::Tiny->read("$self->{afap}->{af}->{site__user_dir}/db.cfg");
	$self->{dbtype} = $DBConfig->{db}->{type};

	my @rets;
	if ($self->{dbtype} eq 'mysql') {
		@rets = getall($self, "SHOW TABLES like '$self->{entry_table}'");
	}
	else { # SQLite
		@rets = getall($self, "SELECT * FROM sqlite_master WHERE type = 'table' AND name = '$self->{entry_table}'");
	}

	# entries
	my $pkey_modifier = $self->{dbtype} eq 'mysql' ? " AUTO_INCREMENT PRIMARY KEY " : " PRIMARY KEY ";
	create_table($self, $self->{entry_table},
	"CREATE TABLE $self->{entry_table} (
		id		INTEGER $pkey_modifier ,
		title		TEXT,
		contents	TEXT,
		year		INTEGER,
		month		INTEGER,
		day		INTEGER,
		timestamp	INTEGER
	)");

	# comments
	create_table($self, $self->{comment_table},
	"CREATE TABLE $self->{comment_table} (
		id		INTEGER,
		user		TEXT,
		comment		TEXT,
		timestamp	INTEGER
	)");

	# trackbacks
	create_table($self, $self->{tb_table},
	"CREATE TABLE $self->{tb_table} (
		id		INTEGER,
		title		TEXT,
		url		TEXT,
		excerpt		TEXT,
		blog_name	TEXT,
		timestamp	INTEGER
	)");

	bless $self, $proto;
	return $self;
}


##############################################
# Destructor
##############################################

sub DESTROY {
	my $self = shift;
	$self->{dbh}->disconnect;
}


##############################################
# addEntry
##############################################

sub addEntry {
	my $self     = shift;
	my $title    = $self->escape(shift);
	my $contents = $self->escape(shift);
	my $time     = shift;

	unless ($time) { $time = time; }

	my ($sec, $min, $hour, $mday, $mon, $year) = localtime($time);
	$year += 1900; $mon += 1;
	
	# prevent double submit
	my @same = $self->getall("SELECT id FROM $self->{entry_table} WHERE title = '$title' AND contents = '$contents'");
	if($#same >= 0) { return; }

	# log rotation
	if($self->getColumn("SELECT count(*) FROM $self->{entry_table}") >= $self->{max_entries}) {
		my $erase = $self->getColumn("SELECT MIN(timestamp) FROM $self->{entry_table}");
		my $erase_id = $self->getColumn("SELECT id FROM $self->{entry_table} WHERE timestamp = '$erase'");
		$self->removeEntry($erase_id);
	}
	
	$self->{dbh}->do("INSERT INTO $self->{entry_table} VALUES (NULL, '$title', '$contents', $year, $mon, $mday, $time)");
	
	# send trackback ping by using urls in entry
	my $id = $self->getColumn("SELECT MAX(id) FROM $self->{entry_table}");
#	$self->send_trackback_ping($id, $title, $contents);
}


##############################################
# updateEntry
##############################################

sub updateEntry {
	my $self     = shift;
	my $id       = $self->escape(shift, 'int');
	my $title    = $self->escape(shift);
	my $contents = $self->escape(shift);
	$self->{dbh}->do("UPDATE $self->{entry_table} SET title = '$title', contents = '$contents' WHERE id = $id");
}


##############################################
# removeEntry
##############################################

sub removeEntry {
	my $self = shift;
	my $id   = $self->escape(shift, 'int');
	$self->{dbh}->do("DELETE FROM $self->{entry_table} WHERE id = $id");
	$self->{dbh}->do("DELETE FROM $self->{comment_table} WHERE id = $id");
	$self->{dbh}->do("DELETE FROM $self->{tb_table} WHERE id = $id");
	if (-f $self->{datadir}."$id.stor") {
		unlink $self->{datadir}."$id.stor";
	}
	$self->removeUploadedImage($id);
}


##############################################
# getEntry
##############################################

sub getEntry {
	my $self = shift;
	my $id   = $self->escape(shift, 'int');
	my @ret = $self->getall("SELECT * FROM $self->{entry_table} WHERE id = $id");
	return $ret[0];
}

##############################################
# existsEntry
##############################################

sub existsEntry {
	my $self = shift;
	my $id   = $self->escape(shift, 'int');
	return $self->getColumn("SELECT COUNT(*) FROM $self->{entry_table} WHERE id = $id") > 0;
}

##############################################
# getEntries
#############################################

sub getEntries {
	my $self  = shift;
	my $year  = $self->escape(shift, 'int');
	my $month = $self->escape(shift, 'int');
	my $day   = $self->escape(shift, 'int');

	my $query = "SELECT * FROM $self->{entry_table} WHERE year = $year AND month = $month";

	if ($day) {
		$query .= " AND day = $day";
	}

	$query .= " ORDER BY timestamp DESC";

	return $self->getall($query);
}


##############################################
# getNewestEntries
##############################################

sub getNewestEntries {
	my ($self, $num) = @_;
	unless ($num) { $num = 5; }
	return $self->getall("SELECT * FROM $self->{entry_table} ORDER BY timestamp DESC LIMIT $num");
}


##############################################
# addComment
##############################################

sub addComment {
	my $self    = shift;
	my $id      = $self->escape(shift, 'int');
	my $user    = shift;
	my $comment = $self->escape(shift);
	my $time    = time;
	
	my @same = $self->getall("SELECT id FROM $self->{comment_table} WHERE user = '$user' AND comment = '$comment'");
	if($#same >= 0) { return; }
	
	$self->{dbh}->do("INSERT INTO $self->{comment_table} VALUES ($id, '$user', '$comment', $time)");
}


##############################################
# getComments
##############################################

sub getComments {
	my $self = shift;
	my $id   = $self->escape(shift, 'int');
	return $self->getall("SELECT * FROM $self->{comment_table} WHERE id = $id ORDER BY timestamp");
}

##############################################
# getVisitorInfo
##############################################

sub getVisitorInfo {
	my $self = shift;
	my $id   = $self->escape(shift, 'int');

	my $uname = $self->getColumn("SELECT user FROM $self->{comment_table} WHERE id = $id");
	if ($uname) {
		if ($uname =~ /([^<]*)<br \/>(.*)/) {
			return ($1, $2);
		}
		else {
			return ($uname);
		}
	}
	return ("");
}

##############################################
# getCommentsNo
##############################################

sub getCommentsNo {
	my $self = shift;
	my $id   = $self->escape(shift, 'int');
	return $self->getColumn("SELECT count(*) FROM $self->{comment_table} WHERE id = $id");
}

sub getColumn {
	my ($self, $query) = @_;
	my $sth = $self->{dbh}->prepare($query);
	$sth->execute;
	my $num;
	$sth->bind_columns(undef, \$num);
	$sth->fetch;
	$sth->finish;
	if($num) {
		return $num;
	}
	else {
		return 0;
	}
}

##############################################
# get_HTML_header
##############################################

sub get_HTML_header {
	my $self = shift;

	return "" if ($self->{header_show} == 1) ;

	# conetnt type
	my $header = 
		"Content-type: text/html; charset=UTF-8\n".
		"Pragma: no-cache\n\n"; 
	
	# affelio header
	$header .= $self->{afap}->get_HTML_header($self->{header_title});
	
	my $tmpl = HTML::Template->new(filename => "./templates/menu.tmpl");

	# calender
	my $calender;
	if($self->{afap}->{cgi}->param('year') and $self->{afap}->{cgi}->param('month')) {
		$calender = $self->getCalender($self->{afap}->{cgi}->param('year'), $self->{afap}->{cgi}->param('month'));
	}
	elsif($self->{afap}->{cgi}->param('id')) {
		my $id = 
		my @date = $self->getall("SELECT year, month FROM $self->{entry_table} WHERE id = ".$self->{afap}->{cgi}->param('id'));
		$calender = $self->getCalender($date[0]->{year}, $date[0]->{month});
	}
	else {
		$calender = $self->getCalender;
	}

	# archives
	my @archives = $self->getall("SELECT DISTINCT year, month FROM $self->{entry_table} LIMIT 10");
	if ($#archives >= 0) {
		shift @archives unless $archives[0]->{year};
		$tmpl->param(ARCHIVES => \@archives);
	}
	
	# recent entries
	my @entries = $self->getall("SELECT id, title FROM $self->{entry_table} ORDER BY timestamp DESC LIMIT 10");
	if ($#entries >= 0) {
		$tmpl->param(RECENT_ENTRIES => \@entries);
	}

	# recent comments
	my @comments = $self->getall("SELECT $self->{comment_table}.id, title, user FROM $self->{entry_table}, $self->{comment_table} WHERE $self->{entry_table}.id = $self->{comment_table}.id ORDER BY $self->{comment_table}.timestamp DESC LIMIT 10");
	if ($#comments >= 0) {
		$tmpl->param(RECENT_COMMENTS => \@comments);
	}

	# recent trackbacks
	my @trackbacks = $self->getall("SELECT id, blog_name, title FROM $self->{tb_table} ORDER BY timestamp DESC LIMIT 10");
	if ($#trackbacks >= 0) {
		$tmpl->param(RECENT_TRACKBACKS => \@trackbacks);
	}
	
	$tmpl->param(
		CALENDER => $calender, 
		access_control_URL => $self->{afap}->get_URL("access_control"),
	);
	if ($self->{afap}->check_access('write_diary')) {
		$tmpl->param(EDITABLE => 1);
	}
	$header .= $tmpl->output;

	$self->{header_show} = 1;

	return $header;
}


##############################################
# get_HTML_footer
##############################################

sub get_HTML_footer {
	my $self = shift;
	my $tmpl = HTML::Template->new(filename => "./templates/footer.tmpl");
	return $tmpl->output().$self->{afap}->get_HTML_footer;
}

##############################################
# redirection
##############################################

sub getRedirection {
	my ($self, $file) = @_;
	my $webroot = $self->{afap}->get_site_info('web_root');
	return
		"Content-type: text/html; charset=UTF-8\n".
		"Location: $webroot/apps/$self->{afap}->{install_name}/$file"."\n\n";
}

##############################################
# checkAccess
##############################################

sub checkAccess {
	my ($self, $page_name) = @_;
	unless ($self->{afap}->check_access($page_name)) {
		$self->accessErrorExit("このページにアクセスする権限がありません");
	}
}

##############################################
# errorExit
##############################################

sub errorExit {
	my $self = shift;
	my $msg  = $self->escape(shift);
	
	my $tmpl = new HTML::Template(filename => './templates/error.tmpl');
	$tmpl->param(MESSAGE => $msg);

	unless ($self->{header_show}) {
		print "Content-type: text/html; charset=UTF-8\n\n";
		print $self->{afap}->get_HTML_header($self->{header_title});
	}
	print $tmpl->output;
	print $self->{afap}->get_HTML_footer;
	exit;
}

##############################################
# accessErrorExit
##############################################

sub accessErrorExit {
	my $self = shift;
	my $msg  = $self->escape(shift);
  	my $affelio_id = $self->{afap}->get_visitor_info("afid");
 	my $visitor_type=$self->{afap}->get_visitor_info("type");

	$visitor_type="pb" if ($visitor_type eq "");

	my $tmpl = new HTML::Template(filename => "./templates/access_error.tmpl");
	$tmpl->param(
		AFID	=> $affelio_id,
		VIS_TYPE=> $visitor_type,
		MESSAGE	=> $msg,
	);

	unless ($self->{header_show}) {
		print "Content-type: text/html; charset=UTF-8\n\n";
		print $self->{afap}->get_HTML_header($self->{header_title});
	}
	print $tmpl->output;
	print $self->{afap}->get_HTML_footer;
	exit;
}

##############################################
# getCalender
##############################################

sub getCalender {
	my $self = shift;
	my $year = $self->escape(shift, 'int');
	my $mon  = $self->escape(shift, 'int');
	
	unless ($mon) { my $d; ($d, $d, $d, $d, $mon, $year) = localtime(time); $year += 1900; $mon += 1;  }
	my @weeks = $self->weekly_days($year, $mon);
	
	my $tmpl = HTML::Template->new(filename => "./templates/calender.tmpl");

	my $last_mon = $mon - 1;
	my $next_mon = $mon + 1;
	my $lastyear = $year;
	my $nextyear = $year;
	if ($mon == 1) {
		$last_mon = 12;
		$lastyear = $year - 1;
	}
	elsif ($mon == 12) {
		$next_mon = 1;
		$nextyear = $year + 1;
	}
	
	$tmpl->param(
		YEAR => $year, MONTH => $mon,
		LAST_MON => $last_mon, NEXT_MON => $next_mon,
		LASTYEAR => $lastyear, NEXTYEAR => $nextyear,
	);

	my @days = $self->getall("SELECT day FROM $self->{entry_table} WHERE year = $year AND month = $mon");
	my @daytable = (0 .. 31);
	$daytable[0] = '';
	foreach(@days) {
		$daytable[$_->{day}] = "<a href=\"list_diary.cgi?year=$year&month=$mon&day=$_->{day}\"><b>$_->{day}</b></a>";
	}

	my @weeks_param;
	foreach(@weeks) {
		if($_->[0] == '' and $_->[6] == '') { next; }
		push @weeks_param,
		{
			SUN => $daytable[$_->[0]],
			MON => $daytable[$_->[1]],
			TUE => $daytable[$_->[2]],
			WED => $daytable[$_->[3]],
			THU => $daytable[$_->[4]],
			FRI => $daytable[$_->[5]],
			SAT => $daytable[$_->[6]],
		};
	}

	$tmpl->param(WEEKS => \@weeks_param);

	return $tmpl->output;
}


##############################################
# addTrackback
##############################################

sub addTrackback {
	my $self      = shift;
	my $id        = $self->escape(shift, 'int');
	my $title     = $self->escape(shift);
	my $url       = $self->escape(shift);
	my $excerpt   = $self->escape(shift);
	my $blog_name = $self->escape(shift);
	my $timestamp = $self->escape(shift, 'int');
	$self->{dbh}->do("INSERT INTO $self->{tb_table} VALUES($id, '$title', '$url', '$excerpt', '$blog_name', $timestamp)");
}

##############################################
# getTrackbacks
##############################################

sub getTrackbacks {
	my $self = shift;
	my $id   = $self->escape(shift, 'int');
	my @ret  = $self->getall("SELECT * FROM $self->{tb_table} WHERE id = $id");
	
	foreach (@ret) {
		$_->{excerpt} = Jcode::convert($_->{excerpt}, 'utf8');
	}
	reset (@ret);

	return @ret;
}
	
##############################################
# getTrackbacksNo
##############################################

sub getTrackbacksNo {
	my $self = shift;
	my $id   = $self->escape(shift, 'int');
	return $self->getColumn("SELECT COUNT(*) FROM $self->{tb_table} WHERE id = $id");
}

##############################################
# sendTrackbackPing
##############################################

sub sendTrackbackPing {
	my ($self, $url, $title, $contents, $id) = @_;
	
	$id = $self->getColumn("SELECT MAX(id) FROM $self->{entry_table}") unless ($id);

	my %form = (
		title => $title, 
		excerpt => $contents, 
		url => $self->{afap}->get_site_info('web_root')."/apps/$self->{afap}->{install_name}/show_diary.cgi?id=$id",
		blog_name => $self->{afap}->get_owner_info('nickname')."'s Affelio Diary",
	);
	my $req = POST($url, [%form]);
	my $ua = new LWP::UserAgent;
	my $res = $ua->request($req);
	my $str = $res->as_string;
	if ($str =~ /<error>[^1]*1[^<]*<\/error>/) {
		$self->errorExit('Trackback Pingの送信に失敗しました'.$str);
	}
}

##############################################
# setRDFURL
##############################################

sub setRDFURL {
	my ($self, $url) = @_;
	local (*OUT);

	open(OUT, "> $self->{datadir}url");
	print OUT $url;
	close(OUT);
}

##############################################
# getRDFURL
##############################################

sub getRDFURL {
	my $self = shift;
	if (-f "$self->{datadir}url") {
		local (*IN);
		open (IN, "$self->{datadir}url");
		my $rssfile = <IN>;
		$rssfile =~ s/[\r\n]//g;
		close(IN);
		return $rssfile;
	}
	return undef;
}

##############################################
# unsetRDF
##############################################

sub unsetRDFURL {
	my $self = shift;
	unlink("$self->{datadir}url") if (-f "$self->{datadir}url");
}

##############################################
# getRSS
##############################################

sub getRSS {
	my ($self, $count) = @_;
	unless ($count) { $count = 5; }
	
	my $tmpl = new HTML::Template(filename => './templates/rss.tmpl');

	my @entries = $self->getNewestEntries($count);
	my @item_list;
	my @items;
	my $web_root = $self->{afap}->get_site_info('web_root');
	my $uname = $self->{afap}->get_owner_info('nickname');

	foreach (@entries) {
		my $link = "$web_root/apps/$self->{afap}->{install_name}/show_diary.cgi?id=$_->{id}";
		push @item_list, { LINK => $link, };
		my ($sec, $min, $hour, $mday, $mon, $year) = localtime($_->{timestamp});
		push @items, {
			TITLE	=> $_->{title},
			LINK	=> $link,
			DESCRIPTION => $_->{contents},
			DATE	=> sprintf("%4d-%02d-%02dT%02d:%02d+09:00", $year, $mon, $mday, $hour, $min),
			CREATOR	=> $uname,
			TPING	=> $web_root."apps/$self->{afap}->{install_name}/tb.cgi/$_->{id}",
		};
	}
	
	$tmpl->param(
		LINK		=> $web_root,
		NICKNAME	=> $uname,
		ITEM_LIST	=> \@item_list,
		ITEMS		=> \@items,
	);

	return $tmpl->output;
}

##############################################
# getURLDescription
##############################################

sub getURLDescription {
	my $self	= shift;
	my $id		= $self->escape(shift, 'int');
	
	my ($entry) = $self->getall("SELECT * FROM $self->{entry_table} WHERE id = $id");
	my $tmpl = new HTML::Template(filename => "./templates/tpingrdf.tmpl");
	my ($sec, $min, $hour, $mday, $mon, $year) = localtime($entry->{timestamp});
	$year += 1900; $mon += 1;
	
	$tmpl->param(
		TITLE => $entry->{title},
		TURL => "$self->{afap}->{af}->{site__web_root}/apps/$self->{afap}->{install_name}/tb/tb.cgi/$id",
		IDENT => "$self->{afap}->{af}->{site__web_root}/apps/$self->{afap}->{install_name}/show_diary.cgi?id=$id",
		DESCRIPTION => $entry->{contents},
		CREATOR => $self->{afap}->{af}->{user__nickname},
		DATE => sprintf("%4d-%02d-%02dT%02d:%02d+09:00", $year, $mon, $mday, $hour, $min),
	);

	return $tmpl->output;
}

##############################################
# saveUploadedImage
##############################################

sub saveUploadedImage {
	use File::Basename;
	my ($self, $filename, $id) = @_;
	
	$id = $self->getColumn("SELECT MAX(id) FROM $self->{entry_table}") unless ($id);
	
	my $afap = $self->{afap};
       
	my $file;
	my $buf;
	my $filesize = 0;
	while (my $bytesread = read($filename, $buf, 1024)) {
		$file .= $buf;
		$self->errorExit('ファイルのアップロード：ファイルサイズが大きすぎます') if (++$filesize >= 300);
	}

	my $imgdir = "$self->{datadir}img/";
	unless (-d $imgdir) {
		mkdir $imgdir;
	}
	my $basedir = $imgdir."$id/";
	unless (-d $basedir) {
		mkdir $basedir;
	}

	fileparse_set_fstype('MSDOS');
	my $distfile = $basedir.basename($filename);

	unless (basename($filename) =~ /^[a-zA-Z0-9\.\-\_]{1,28}\.(jpg|jpeg|png|gif|bmp)$/i) {
		$self->errorExit('ファイルのアップロード：不正なファイルタイプです');
	}

	local (*OUT);
	open(OUT, "> $distfile") or $self->errorExit('ファイルのアップロード：ファイル書き込みに失敗しました');
	binmode OUT;
	print OUT $file;
	close(OUT);
}

##############################################
# removeUploadedImage
##############################################

sub removeUploadedImage {
	my ($self, $id) = @_;
	
	$id = $self->getColumn("SELECT MAX(id) FROM $self->{entry_table}") unless ($id);
	
	my $imgdir = "$self->{datadir}img/$id/";
	if (-d $imgdir) {
		local (*DIR);
		opendir(DIR, $imgdir);
		while (my $file = readdir(DIR)) {
			unlink ($imgdir.$file) if (-f $imgdir.$file);
		}
		closedir(DIR);
		rmdir $imgdir;
	}
}

##############################################
# getUploadedImages
##############################################

sub getUploadedImages {
	my ($self, $id, $width, $height) = @_;

	$width  = "&w=$width" if ($width);
	$height = "&h=$height" if ($height);
	
	my $imgdir = "$self->{datadir}img/$id/";
	my $ret;
	
	local (*DIR);
	opendir(DIR, $imgdir);
	while (my $file = readdir(DIR)) {
		if (-f $imgdir.$file) {
			$ret .= "<a href=\"show_image.cgi?id=$id&filename=$file\" target=\"_blank\">".
			"<img src=\"show_image.cgi?id=$id&filename=$file$width$height\" border=\"0\" />".
			"</a><br />";
		}
	}
	closedir(DIR);

	return $ret ? "<p>$ret</p>" : "";
}

##############################################
# Internal functions
##############################################

sub create_table {
	my ($self, $table_name, $sql) = @_;
	
	my @rets;
	if ($self->{dbtype} eq 'mysql') {
		@rets = getall($self, "SHOW TABLES like '$table_name'");
	}
	else { # SQLite
		@rets = getall($self, "SELECT * FROM sqlite_master WHERE type = 'table' AND name = '$table_name'");
	}

	if ($#rets < 0) {
		$self->{dbh}->do($sql);
	}
}

sub send_trackback_ping {
	my ($self, $id, $title, $contents) = @_;
	my @urls = $contents =~ /(s?https?:\/\/[-_.!~*'()a-zA-Z0-9;\/?:\@&=+\$,%#]+)/g;

	foreach(@urls) {
		my $url = $self->discover_tb($_);
		if($url) {
			my %form = (
				title => $title, 
				excerpt => $contents, 
				url => "$self->{afap}->{af}->{site__web_root}/apps/$self->{afap}->{install_name}/show_diary.cgi?id=$id",
				blog_name => "$self->{afap}->{af}->{user__nickname}'s affelio diary",
			);
			my $req = POST($url, [%form]);
			my $ua = new LWP::UserAgent;
			my $res = $ua->request($req);
			my $str = $res->as_string;
		}
	}
}

sub escape {
	my ($self, $str, $type) = @_;

	if ($type eq 'int') {
		return int($str);
	}
	else {
		$str =~ s/'/"/g;
		$str =~ s/&/&amp;/g;
		$str =~ s/"/&quot;/g;
		$str =~ s/</&lt;/g;
		$str =~ s/>/&gt;/g;
		$str =~ s/&lt;(\/?)(br|p|i|b|strong|em|u|font)([^&]*)&gt;/<$1$2$3>/gi;
		$str =~ s/\r\n/<br \/>/g;
		$str =~ s/[\r\n]/<br \/>/g;
		$str =~ s/(s?https?:\/\/[-_.!~*'()a-zA-Z0-9;\/?:\@&=+\$,%#]+)/<a href="$1">$1<\/a>/g;
	}

	return $str;
}

sub getall {
	my ($self, $query) = @_;

	my $sth = $self->{dbh}->prepare($query);
	$sth->execute;

	my @ret;
	while(my $row = $sth->fetchrow_hashref) {
		push @ret, $row;
	}
	$sth->finish;

	return @ret;
}

sub weekly_days {
	my ($self, $year, $mon) = @_;
	my @weeks;
	my @mday  = (31,31,28,31,30,31,30,31,31,30,31,30,31);
	if (($year % 4 == 0) and ($year % 100) or ($year % 400 == 0)) { $mday[2]  = 29 };
	
	my $lastday = $mday[$mon];
	@mday = (1 .. $mday[$mon]);
	if($mon < 3){ $mon += 12; $year--; }
	my $first_day = ($year+int($year/4)-int($year/100)+int($year/400)+int((13*$mon+8)/5)+1)% 7;

	my $day = 1;
	for my $week (0 .. 7) {
		my @days;
		for(my $i = 0; $i < 7; $i++) {
			push @days, 
			(($week == 0 and $i < $first_day) or ($day > $lastday)) ? 
			'' : $day++;
		}
		$weeks[$week] = \@days;
	}
	
	return @weeks;
}

# Refer to:  http://lowlife.jp/yasusii/stories/8.html
sub discover_tb {
	my ($self, $url) = @_;
	my $ua = LWP::UserAgent->new;
	$ua->agent('TrackBack/1.0');  
	$ua->parse_head(0);   
	$ua->timeout(15);
	my $req = HTTP::Request->new(GET => $url);
	my $res = $ua->request($req);
	return unless $res->is_success;
	my $c = $res->content;
	(my $url_no_anchor = $url) =~ s/#.*$//;
	my $item;
	while ($c =~ m!(<rdf:RDF.*?</rdf:RDF>)!sg) {
		my $rdf = $1;
		my($perm_url) = $rdf =~ m!dc:identifier="([^"]+)"!;  
			next unless $perm_url eq $url || $perm_url eq $url_no_anchor;
		if ($rdf =~ m!trackback:ping="([^"]+)"!) {
			return $1;
		} elsif ($rdf =~ m!about="([^"]+)"!) {
			return $1;
		}
	}
}

1;
