package jp.sourceforge.acerola3d.a3;

import java.awt.*;
import java.awt.image.BufferedImage;
import javax.media.j3d.*;
import javax.vecmath.*;

import java.io.*;
import java.util.*;
import javax.imageio.ImageIO;
import com.sun.j3d.exp.swing.JCanvas3D;

/**
 * A3Objectを表示するためのLightWeightなGUIコンポーネントです。
 * このクラスのaddメソッドを用いてA3Objectを登録すると、
 * A3ObjectがこのGUIコンポーネントの中に表示されるように
 * なっています。また、3D仮想空間におけるカメラの操作
 * も、このクラスのメソッドを通じて行います。
 * それと、A3Listenerの登録先でもあります。
 * コンストラクタではなく、createA3Canvasメソッドを使って
 * 生成します。
 */
public class JA3Canvas extends JCanvas3D implements A3CanvasInterface {
    private static final long serialVersionUID = 1L;

    A3VirtualUniverse universe;

    /**
     * (500,500)の大きさのJA3Canvasを作成します。
     */
    public static JA3Canvas createJA3Canvas() {
        return createJA3Canvas(500,500);
    }

    /**
     * (w,h)の大きさのJA3Canvasを作ります。
     */
    public static JA3Canvas createJA3Canvas(int w,int h) {
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        GraphicsDevice gd = ge.getDefaultScreenDevice();
        GraphicsConfigTemplate3D gct3d = new GraphicsConfigTemplate3D();
        return createJA3Canvas(gct3d,gd,w,h);
    }

    /**
     * GraphicsConfigurationオブジェクトを指定して
     * (w,h)の大きさのA3Canvasを作ります。
     */
    public static JA3Canvas createJA3Canvas(GraphicsConfigTemplate3D t,GraphicsDevice d,int w,int h) {
        return new JA3Canvas(t,d,w,h);
    }

    JA3Canvas(GraphicsConfigTemplate3D t,GraphicsDevice d,int w,int h) {
        super(t,d);
        setPreferredSize(new Dimension(w,h));
        setSize(w,h);
        //本当は、以下の処理はここでやりたいところだけど、
        //NullPointerExceptionが出るので、addNotifyメソッドに
        //入れてあげたらうまくいった。でも場合によってはまずいかも。
        //universe = new A3VirtualUniverse(this);
    }
    public void addNotify() {
        super.addNotify();
        universe = new A3VirtualUniverse(this);
        enableEvents(java.awt.AWTEvent.MOUSE_EVENT_MASK);
        enableEvents(java.awt.AWTEvent.MOUSE_MOTION_EVENT_MASK);
        enableEvents(java.awt.AWTEvent.MOUSE_WHEEL_EVENT_MASK);
    }
    public TimerBehavior getTimerBehavior() {
        return universe.getTimerBehavior();
    }

    // A3Objectの追加と削除
    /**
     * A3Objectを追加して表示されるようにします。
     */
    public void add(A3Object a) {
        universe.add(a);
    }

    /**
     * 指定されたA3Objectの登録を削除して表示されないように
     * します。
     */
    public void del(A3Object a) {
        universe.del(a);
    }

    /**
     * 登録されている全てのA3Objectを削除して表示されないようにします。
     *
     */
    public void delAll() {
        universe.delAll();
    }

    /**
     * 背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a) {
        universe.setBackground(a);
    }

    /**
     * 背景を削除します。
     */
    public void delBackground() {
        universe.delBackground();
    }

    /**
     * アバタをセットします。
     */
    public void setAvatar(A3Object a) {
        universe.setAvatar(a);
    }

    /**
     * セットされたアバタを取得します。
     */
    public A3Object getAvatar() {
        return universe.getAvatar();
    }

    // リスナ設定のラッパーメソッド
    /**
     * A3Listenerを登録します。
     */
    public void addA3Listener(A3Listener l) {
        universe.addA3Listener(l);
    }

    /**
     * 指定されたA3Listenerの登録を抹消します。
     */
    public void removeA3Listener(A3Listener l) {
        universe.addA3Listener(l);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z) {
        universe.setDefaultCameraLoc(x,y,z);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc) {
        universe.setDefaultCameraLoc(loc);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w) {
        universe.setDefaultCameraQuat(x,y,z,w);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat) {
        universe.setDefaultCameraQuat(quat);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z) {
        universe.setDefaultCameraRot(x,y,z);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot) {
        universe.setDefaultCameraRot(rot);
    }

    /**
     * カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s) {
        universe.setDefaultCameraScale(s);
    }

    /**
     * カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera() {
        universe.resetCamera();
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z) {
        universe.setCameraLoc(x,y,z);
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc) {
        universe.setCameraLoc(loc);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z) {
        universe.setCameraLocImmediately(x,y,z);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc) {
        universe.setCameraLocImmediately(loc);
    }

    /**
     * カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc() {
        return universe.getCameraLoc();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w) {
        universe.setCameraQuat(x,y,z,w);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat) {
        universe.setCameraQuat(quat);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w) {
        universe.setCameraQuatImmediately(x,y,z,w);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat) {
        universe.setCameraQuatImmediately(quat);
    }

    /**
     * カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat() {
        return universe.getCameraQuat();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z) {
        universe.setCameraRot(x,y,z);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot) {
        universe.setCameraRot(rot);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z) {
        universe.setCameraRotImmediately(x,y,z);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot) {
        universe.setCameraRotImmediately(rot);
    }

    /**
     * カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s) {
        universe.setCameraScale(s);
    }

    /**
     * カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s) {
        universe.setCameraScaleImmediately(s);
    }

    /**
     * カメラの拡大率を返します。
     */
    public double getCameraScale() {
        return universe.getCameraScale();
    }

    /**
     * ヘッドライトのON,OFFを設定します。
     */
    public void setHeadLightEnable(boolean b) {
        universe.setHeadLightEnable(b);
    }
    // マウスナビゲーションのモード設定
    /**
     * ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m) {
        universe.setNavigationMode(m);
    }

    /**
     * ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s) {
        universe.setNavigationSpeed(s);
    }

    /**
     * ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed() {
        return universe.getNavigationSpeed();
    }

    /**
     * A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c) {
        universe.setA3Controller(c);
    }
//  ----------座標変換とピッキングのためのラッパーメソッド---------
    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算してます。
     */
    public Point3d canvasToVirtualCS(int x,int y) {
        return universe.canvasToVirtualCS(x,y);
    }

    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToVirtualCS(int x,int y,double dis) {
        return universe.canvasToVirtualCS(x,y,dis);
    }

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算しています。
     */
    public Point3d canvasToPhysicalCS(int x,int y) {
        return universe.canvasToPhysicalCS(x,y);
    }

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToPhysicalCS(int x,int y,double dis) {
        return universe.canvasToPhysicalCS(x,y,dis);
    }

    /**
     * 物理空間上の点(x,y,z)を仮想空間上の点(x,y,z)に変換します。
     */
    public Vector3d physicalCSToVirtualCS(Vector3d v) {
        return universe.physicalCSToVirtualCS(v);
    }

    /**
     * 仮想空間上の点(x,y,z)をA3Canvas上の点(x,y)に変換します。
     */
    public Point virtualCSToCanvas(Point3d p) {
        return universe.virtualCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)を物理空間上の点(x,y,z)に変換します。
     */
    public Vector3d virtualCSToPhysicalCS(Vector3d v) {
        return universe.virtualCSToPhysicalCS(v);
    }

    /**
     * カメラの座標系(物理空間の座標系)のX軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecX() {
        return universe.getCameraUnitVecX();
    }

    /**
     * カメラの座標系(物理空間の座標系)のY軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecY() {
        return universe.getCameraUnitVecY();
    }

    /**
     * カメラの座標系(物理空間の座標系)のZ軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecZ() {
        return universe.getCameraUnitVecZ();
    }

    /**
     * A3Canvas上の点(x,y)にあるA3Objectをピックアップします。
     */
    public A3Object pickA3(int x,int y) {
        return universe.pickingBehavior.pickA3(x,y);
    }
//  ----------J3DGraphics2D(文字描画など)---------
    ArrayList<Component2D> components2D = new ArrayList<Component2D>(); 
    public synchronized void add(Component2D c) {
        if (!components2D.contains(c))
            components2D.add(c);
    }
    public synchronized void del(Component2D c) {
        components2D.remove(c);
    }
    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D gg = (Graphics2D)g;
        for (Component2D c : components2D) {
            c.paint(gg,this);
        }
        //gg.flush(true);
    }
//  ----------おまけ機能---------
    volatile boolean check = false;
    GraphicsContext3D gc;
    Raster readRaster;

    public void postSwap() {
        //super.postSwap();
        if (check) {
            gc.readRaster(readRaster);
            check = false;
        }
    }

    /**
     * A3Canvasに表示されている内容をPNG画像としてファイルに保存します。
     */
    public void saveImage(File file) throws IOException {
        int width = getWidth();
        int height = getHeight();
        BufferedImage bImage = new BufferedImage(
                               width,height,BufferedImage.TYPE_INT_RGB);
        ImageComponent2D ic2d = new ImageComponent2D(
                                ImageComponent.FORMAT_RGB,bImage);
        //DepthComponentFloat dcf = new DepthComponentFloat(width,height);
        readRaster = new Raster(new Point3f(0.0f,0.0f,0.0f),
                            Raster.RASTER_COLOR,0,0,width,height,
                            ic2d,null);
        check = true;
        while(check) {
            try{Thread.sleep(300);}catch(Exception e){;}
        }

        ImageComponent2D ic = readRaster.getImage();
        BufferedImage image = ic.getImage();

        ImageIO.write(image,"png",file);
//        FileOutputStream out = new FileOutputStream(file);
//        JPEGImageEncoder e = JPEGCodec.createJPEGEncoder(out);
//        e.encode(image);
//        out.close();
    }
}
