package jp.sourceforge.acerola3d.a3;

import java.awt.*;
import java.awt.event.KeyListener;
import javax.vecmath.*;
import java.io.*;

/**
 * A3CanvasとJA3Canvasを統一的に扱うために
 * 導入されたインタフェース。
 */
public interface A3CanvasInterface {

    // A3Objectの追加と削除
    /**
     * A3Objectを追加して表示されるようにします。
     */
    public void add(A3Object a);

    /**
     * 指定されたA3Objectの登録を削除して表示されないように
     * します。
     */
    public void del(A3Object a);

    /**
     * 登録されている全てのA3Objectを削除して表示されないようにします。
     *
     */
    public void delAll();

    /**
     * 背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a);

    /**
     * 背景を削除します。
     */
    public void delBackground();

    /**
     * アバタをセットします。
     */
    public void setAvatar(A3Object a);

    /**
     * セットされたアバタを取得します。
     */
    public A3Object getAvatar();

    // リスナ設定のラッパーメソッド
    /**
     * A3Listenerを登録します。
     */
    public void addA3Listener(A3Listener l);

    /**
     * 指定されたA3Listenerの登録を抹消します。
     */
    public void removeA3Listener(A3Listener l);

    /**
     * KeyListenerを登録します。
     */
    public void addKeyListener(KeyListener l);
    /**
     * 指定されてKeyListenerの登録を抹消します。
     */
    public void removeKeyListener(KeyListener l);

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z);

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc);

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w);

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat);

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z);

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot);

    /**
     * カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s);

    /**
     * カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera();

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z);

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc);

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z);

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc);

    /**
     * カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc();

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w);

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat);

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w);

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat);

    /**
     * カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat();

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z);

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot);

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z);

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot);

    /**
     * カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s);

    /**
     * カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s);

    /**
     * カメラの拡大率を返します。
     */
    public double getCameraScale();

    /**
     * ヘッドライトのON,OFFを設定します。
     */
    public void setHeadLightEnable(boolean b);
    // マウスナビゲーションのモード設定
    /**
     * ナビゲーションモードです。主に3D仮想空間をマウスで観覧する時の
     * モードを設定するための列挙型ですが、A3オブジェクトの位置などを
     * 編集できるモードも含まれます。ナビゲーションモードを設定すると、
     * a3パッケージがカメラやA3Objectの座標や回転などをコントロール
     * するようになるので、A3Canvas.setCameraLoc()や
     * A3Object.setLoc()などのメソッドと併用する時は注意が必要です。
     * FLYモードは未実装です。将来他のモードが追加される可能性があります。
     */
    public enum NaviMode {
        /** a3パッケージがナビゲーションに関与しないモードです。 */
        NONE,
        /** 特定のa3オブジェクトに着目してそれを中心に観覧するモードです。 */
        EXAMINE,
        /** 平面を歩いているようなナビゲーションモードです。 */
        WALK,
        /** 空間を自由に飛ぶようなナビゲーションモードです。 */
        FLY,
        /** 選択されたオブジェクトの位置、回転、大きさを変更できるモードです。 */
        EDIT,
        /** EXAMINEモードと似ていますが、注視点が任意に変更できる観覧モードです。 */
        SIMPLE,
        /** アバタの後方からアバタを追跡するような観覧モードです。 */
        CHASE,
        /** ユーザ定義のナビゲーションモードです。 */
        USER
    }

    /**
     * ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m);

    /**
     * ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s);

    /**
     * ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed();

    /**
     * A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c);
//  ----------座標変換とピッキングのためのラッパーメソッド---------
    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算してます。
     */
    public Point3d canvasToVirtualCS(int x,int y);

    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToVirtualCS(int x,int y,double dis);

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算しています。
     */
    public Point3d canvasToPhysicalCS(int x,int y);

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToPhysicalCS(int x,int y,double dis);

    /**
     * 物理空間上の点(x,y,z)を仮想空間上の点(x,y,z)に変換します。
     */
    public Vector3d physicalCSToVirtualCS(Vector3d v);

    /**
     * 仮想空間上の点(x,y,z)をA3Canvas上の点(x,y)に変換します。
     */
    public Point virtualCSToCanvas(Point3d p);

    /**
     * 仮想空間上の点(x,y,z)を物理空間上の点(x,y,z)に変換します。
     */
    public Vector3d virtualCSToPhysicalCS(Vector3d v);

    /**
     * カメラの座標系(物理空間の座標系)のX軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecX();

    /**
     * カメラの座標系(物理空間の座標系)のY軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecY();

    /**
     * カメラの座標系(物理空間の座標系)のZ軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecZ();

    /**
     * A3Canvas上の点(x,y)にあるA3Objectをピックアップします。
     */
    public A3Object pickA3(int x,int y);
//  ----------J3DGraphics2D(文字描画など)---------
    void add(Component2D c);
    void del(Component2D c);
//  ----------Acerola3D内部で必要なもの---------
    TimerBehavior getTimerBehavior();
//  ----------おまけ機能---------
    /**
     * A3Canvasに表示されている内容をPNG画像としてファイルに保存します。
     */
    public void saveImage(File file) throws IOException;
}
