package jp.sourceforge.acerola3d.a3;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.*;
import javax.vecmath.*;

import java.io.*;
import java.awt.*;

/**
 * A3Objectを表示するためのWindowです。このウィンドウには
 * デフォルトでJA3Canvasが内包されており、
 * JA3Canvasと同名のメソッドは単に内包されているA3Canvas
 * の同じ名前のメソッドを呼び出しているだけです。
 * このクラスのaddメソッドを用いてA3Objectを登録すると、
 * A3ObjectがこのWindowの中に表示されるように
 * なっています。また、3D仮想空間におけるカメラの操作
 * も、このクラスのメソッドを通じて行うことができます。
 * それと、A3Listenerの登録先にもなります。
 */
public class JA3Window extends JFrame {
    private static final long serialVersionUID = 1L;
    JA3Canvas canvas;

    /**
     * (w,h)の大きさのA3Windowを生成します。
     */
    public JA3Window(int w,int h) {
        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent we) {
                dispose();
                System.exit(0);
            }
        });
        canvas = JA3Canvas.createJA3Canvas(w,h);
        getContentPane().add(canvas);
        pack();
        setVisible(true);
        EventQueue.invokeLater(new Runnable() {
            public void run() {
                System.out.println("invokeLater");
                canvas.requestFocusInWindow();
            }
        });
    }

    /**
     * このA3Windowに内蔵されるJA3Canvasを返します。
     */
    public JA3Canvas getJA3Canvas() {
        return canvas;
    }

    // A3Objectの追加と削除
    /**
     * A3Objectを追加して表示されるようにします。
     */
    public void add(A3Object a) {
        canvas.add(a);
    }

    /**
     * 指定されたA3Objectの登録を削除して表示されないように
     * します。
     */
    public void del(A3Object a) {
        canvas.del(a);
    }

    /**
     * 登録されている全てのA3Objectを削除して表示されないようにします。
     *
     */
    public void delAll() {
        canvas.delAll();
    }

    /**
     * 背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a) {
        canvas.setBackground(a);
    }

    /**
     * 背景を削除します。
     */
    public void delBackground() {
        canvas.delBackground();
    }

    /**
     * アバタをセットします。
     */
    public void setAvatar(A3Object a) {
        canvas.setAvatar(a);
    }

    // リスナ設定のラッパーメソッド
    /**
     * A3Listenerを登録します。
     */
    public void addA3Listener(A3Listener l) {
        canvas.addA3Listener(l);
    }

    /**
     * 指定されたA3Listenerの登録を抹消します。
     */
    public void removeA3Listener(A3Listener l) {
        canvas.removeA3Listener(l);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z) {
        canvas.setDefaultCameraLoc(x,y,z);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc) {
        canvas.setDefaultCameraLoc(loc);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w) {
        canvas.setDefaultCameraQuat(x,y,z,w);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat) {
        canvas.setDefaultCameraQuat(quat);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z) {
        canvas.setDefaultCameraRot(x,y,z);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot) {
        canvas.setDefaultCameraRot(rot);
    }

    /**
     * カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s) {
        canvas.setDefaultCameraScale(s);
    }

    /**
     * カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera() {
        canvas.resetCamera();
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z) {
        canvas.setCameraLoc(x,y,z);
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc) {
        canvas.setCameraLoc(loc);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z) {
        canvas.setCameraLocImmediately(x,y,z);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc) {
        canvas.setCameraLocImmediately(loc);
    }

    /**
     * カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc() {
        return canvas.getCameraLoc();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w) {
        canvas.setCameraQuat(x,y,z,w);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat) {
        canvas.setCameraQuat(quat);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w) {
        canvas.setCameraQuatImmediately(x,y,z,w);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat) {
        canvas.setCameraQuatImmediately(quat);
    }

    /**
     * カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat() {
        return canvas.getCameraQuat();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x, double y, double z) {
        canvas.setCameraRot(x,y,z);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot) {
        canvas.setCameraRot(rot);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x, double y, double z) {
        canvas.setCameraRotImmediately(x,y,z);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot) {
        canvas.setCameraRotImmediately(rot);
    }

    /**
     * カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。
     */
    public void setCameraScale(double s) {
        canvas.setCameraScale(s);
    }

    /**
     * カメラの拡大率を即時に指定します。
     */
    public void setCameraScaleImmediately(double s) {
        canvas.setCameraScaleImmediately(s);
    }

    /**
     * ヘッドライトのON,OFFを設定します。
     */
    public void setHeadLightEnable(boolean b) {
        canvas.setHeadLightEnable(b);
    }

    /**
     * ナビゲーションモードを指定します。
     */
    public void setNavigationMode(A3Canvas.NaviMode m) {
        canvas.setNavigationMode(m);
    }

    /**
     * ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s) {
        canvas.setNavigationSpeed(s);
    }

    /**
     * ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed() {
        return canvas.getNavigationSpeed();
    }

    /**
     * A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c) {
        canvas.setA3Controller(c);
    }
//  ----------座標変換とピッキングのためのラッパーメソッド---------
    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算してます。
     */
    public Point3d canvasToVirtualCS(int x,int y) {
        return canvas.canvasToVirtualCS(x,y);
    }

    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToVirtualCS(int x,int y,double dis) {
        return canvas.canvasToVirtualCS(x,y,dis);
    }

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算しています。
     */
    public Point3d canvasToPhysicalCS(int x,int y) {
        return canvas.canvasToPhysicalCS(x,y);
    }

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToPhysicalCS(int x,int y,double dis) {
        return canvas.canvasToPhysicalCS(x,y,dis);
    }

    /**
     * 物理空間上の点(x,y,z)を仮想空間上の点(x,y,z)に変換します。
     */
    public Vector3d physicalCSToVirtualCS(Vector3d v) {
        return canvas.physicalCSToVirtualCS(v);
    }

    /**
     * 仮想空間上の点(x,y,z)をA3Canvas上の点(x,y)に変換します。
     */
    public Point virtualCSToCanvas(Point3d p) {
        return canvas.virtualCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)を物理空間上の点(x,y,z)に変換します。
     */
    public Vector3d virtualCSToPhysicalCS(Vector3d v) {
        return canvas.virtualCSToPhysicalCS(v);
    }

    /**
     * カメラの座標系(物理空間の座標系)のX軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecX() {
        return canvas.getCameraUnitVecX();
    }

    /**
     * カメラの座標系(物理空間の座標系)のY軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecY() {
        return canvas.getCameraUnitVecY();
    }

    /**
     * カメラの座標系(物理空間の座標系)のZ軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecZ() {
        return canvas.getCameraUnitVecZ();
    }

    /**
     * A3Canvas上の点(x,y)にあるA3Objectをピックアップします。
     */
    public A3Object pickA3(int x,int y) {
        return canvas.pickA3(x,y);
    }

    /**
     * A3Canvasに表示されている内容をPNG画像としてファイルに保存します。
     */
    public void saveImage(File file) throws IOException {
        canvas.saveImage(file);
    }

}
