package jp.sourceforge.acerola3d.a3;

import java.awt.*;
import javax.media.j3d.*;
import javax.vecmath.*;

import java.util.*;
import java.lang.reflect.Constructor;
import jp.sourceforge.acerola3d.a3.A3CanvasInterface.NaviMode;

/**
 * A3CanvasとJA3Canvasの両方で共通して使うことが
 * できるVirtualUniverseのAcerola3Dバージョン。
 */
class A3VirtualUniverse extends VirtualUniverse {
    A3CanvasInterface canvas = null;
    Canvas3D canvas3d = null;
    javax.media.j3d.Locale locale;
    javax.media.j3d.View view;
    BranchGroup rootGroup;
    TransformGroup tGroup;
    Transform3D transform;
    BranchGroup vpGroup;
    ViewPlatform vp;
    BranchGroup mainGroup;
    BranchGroup bgGroup;
    TimerBehavior timerBehavior;
    PickingBehavior pickingBehavior;
    CameraBehavior cameraBehavior;
    Light headLight;

    A3Object background = null;
    A3Object avatar = null;
    Hashtable<A3Object,BranchGroup> a3Hash = new Hashtable<A3Object,BranchGroup>();

    A3VirtualUniverse(A3CanvasInterface c) {
        canvas = c;
        if (canvas instanceof A3Canvas)
            init((Canvas3D)c);
        else if (canvas instanceof JA3Canvas)
            init(((JA3Canvas)canvas).getOffscreenCanvas3D());
    }
    void init(Canvas3D c3d) {
        canvas3d = c3d;
        locale = new javax.media.j3d.Locale(this);

        PhysicalBody body = new PhysicalBody();
        PhysicalEnvironment environment = new PhysicalEnvironment();

        view = new javax.media.j3d.View();
        view.addCanvas3D(canvas3d);
        view.setPhysicalBody(body);
        view.setPhysicalEnvironment(environment);
//        view.setFrontClipDistance(0.001);
        view.setBackClipDistance(1000.0);
        view.setUserHeadToVworldEnable(true);

        vpGroup = new BranchGroup();
        tGroup = new TransformGroup();
        tGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        tGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
        vpGroup.addChild(tGroup);
        transform = new Transform3D();
        tGroup.setTransform(transform);
        timerBehavior = new TimerBehavior();
        BoundingSphere bs = new BoundingSphere(new Point3d(0.0,0.0,0.0),
                                               10.0);
        timerBehavior.setSchedulingBounds(bs);
        tGroup.addChild(timerBehavior);
        vp = new ViewPlatform();
//System.out.println(vp.getActivationRadius());
        tGroup.addChild(vp);
        headLight = new DirectionalLight();
        headLight.setCapability(Light.ALLOW_STATE_WRITE);
        headLight.setInfluencingBounds(new BoundingSphere(new Point3d(0.0,0.0,0.0),100.0));
        headLight.setEnable(true);
        tGroup.addChild(headLight);

        view.attachViewPlatform(vp);

//        vpGroup.setCapability(BranchGroup.ALLOW_DETACH);
//        vpGroup.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
//        vpGroup.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
//        vpGroup.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);

        pickingBehavior = new PickingBehavior(this,view,tGroup,locale);
        vpGroup.addChild(pickingBehavior);
        BoundingSphere sb = new BoundingSphere(new Point3d(0.0,0.0,0.0),10.0);
        pickingBehavior.setSchedulingBounds(sb);
        cameraBehavior = new CameraBehavior(this);
        vpGroup.addChild(cameraBehavior);
        cameraBehavior.setSchedulingBounds(sb);

        mainGroup = new BranchGroup();
        mainGroup.setCapability(BranchGroup.ALLOW_DETACH);
        mainGroup.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
        mainGroup.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
        mainGroup.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);

        bgGroup = new BranchGroup();
        bgGroup.setCapability(BranchGroup.ALLOW_DETACH);
        bgGroup.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
        bgGroup.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
        bgGroup.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);

        rootGroup = new BranchGroup();
//      rootGroup.setCapability(BranchGroup.ALLOW_DETACH);
//      rootGroup.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
//      rootGroup.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
//      rootGroup.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);

        rootGroup.addChild(vpGroup);
        rootGroup.addChild(mainGroup);
        rootGroup.addChild(bgGroup);
        
        locale.addBranchGraph(rootGroup);

        AudioDevice mixer = null;
        String mixerClassName = System.getProperty("j3d.audiodevice");
        System.out.println(mixerClassName);
        if (mixerClassName!=null) {
            try {
                Class<?> c = Class.forName(mixerClassName);
                Class<? extends AudioDevice> mixerClass = c.asSubclass(AudioDevice.class);
                Constructor<? extends AudioDevice> constructor = mixerClass.getConstructor(PhysicalEnvironment.class);
                mixer = constructor.newInstance(environment);
            } catch(Exception e) {
                e.printStackTrace();
            }
        }
        if (mixer==null) {
            try {
                mixer = new org.jdesktop.j3d.audioengines.joal.JOALMixer(environment);
            } catch(Exception e) {
                ;
            }
        }
        /*
        if (mixer==null) {
            try {
                //mixer = new com.sun.j3d.audioengines.headspace.HeadspaceMixer(environment);
            } catch(Exception e) {
                ;
            }
        }
        */
        if (mixer==null) {
            try {
                mixer = new com.sun.j3d.audioengines.javasound.JavaSoundMixer(environment);
            } catch(Exception e) {
                ;
            }
        }
        if (mixer == null) {
            System.out.println("null AudioDevice!");
        } else {
            System.out.println(mixer.getClass().getName());
            environment.setAudioDevice(mixer);
            mixer.initialize();
        }
    }

    TimerBehavior getTimerBehavior() {
        return timerBehavior;
    }

    void prepareVirtualUniverse() {
    }

    // A3Objectの追加と削除
    /**
     * A3Objectを追加して表示されるようにします。
     */
    public void add(A3Object a) {
        if (a3Hash.containsKey(a))
            return;
        if (canvas!=null)
            a.setA3CanvasInterface(canvas);
        a.init();
        A3BranchGroup bg = a.getA3BranchGroup();
        mainGroup.addChild(bg);
        a3Hash.put(a,bg);
    }

    /**
     * 指定されたA3Objectの登録を削除して表示されないように
     * します。
     */
    public void del(A3Object a) {
        BranchGroup bg = a3Hash.get(a);
        if (bg==null)
            return;
        bg.detach();
        a.dispose();
        a3Hash.remove(a);
    }

    /**
     * 登録されている全てのA3Objectを削除して表示されないようにします。
     *
     */
    public void delAll() {
        Enumeration<A3Object> e = a3Hash.keys();
        while (e.hasMoreElements()) {
            A3Object a3 = e.nextElement();
            BranchGroup bg = a3Hash.get(a3);
            bg.detach();
            a3.dispose();
        }
        a3Hash.clear();
    }

    BranchGroup backgroundBG = null;
    /**
     * 背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a) {
        if (backgroundBG!=null)
            backgroundBG.detach();
        Node n = a.getA3BranchGroup();
        backgroundBG = new BranchGroup();
        backgroundBG.setCapability(BranchGroup.ALLOW_DETACH);
        backgroundBG.addChild(n);
        mainGroup.addChild(backgroundBG);
        background = a;
    }

    /**
     * 背景を削除します。
     */
    public void delBackground() {
        if (backgroundBG!=null)
            backgroundBG.detach();
        background = null;
    }

    /**
     * アバタをセットします。
     */
    public void setAvatar(A3Object a) {
        avatar = a;
    }

    /**
     * セットされたアバタを取得します。
     */
    public A3Object getAvatar() {
        return avatar;
    }

    // リスナ設定のラッパーメソッド
    /**
     * A3Listenerを登録します。
     */
    public void addA3Listener(A3Listener l) {
        pickingBehavior.addA3Listener(l);
    }

    /**
     * 指定されたA3Listenerの登録を抹消します。
     */
    public void removeA3Listener(A3Listener l) {
        pickingBehavior.removeA3Listener(l);
    }

    // cameraの手動操作のための変数とメソッド
    Vector3d cameraNowV = new Vector3d(0.0,0.0,2.0);
    Quat4d cameraNowQ = new Quat4d(0.0,0.0,0.0,1.0);
    double cameraNowS = 1.0;
    Vector3d cameraNextV = new Vector3d(0.0,0.0,2.0);
    Quat4d cameraNextQ = new Quat4d(0.0,0.0,0.0,1.0);
    double cameraNextS = 1.0;
    Vector3d defaultV = new Vector3d(0.0,0.0,2.0);
    Quat4d defaultQ = new Quat4d(0.0,0.0,0.0,1.0);
    double defaultS = 1.0;

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z) {
        defaultV = new Vector3d(x,y,z);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc) {
        defaultV = new Vector3d(loc);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w) {
        defaultQ = new Quat4d(x,y,z,w);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat) {
        defaultQ = new Quat4d(quat);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z) {
        defaultQ = rot2quat(x,y,z);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot) {
        defaultQ = rot2quat(rot);
    }

    /**
     * カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s) {
        defaultS = s;
    }

    /**
     * カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera() {
        cameraNowV.set(defaultV);
        cameraNowQ.set(defaultQ);
        cameraNowS = defaultS;
        cameraNextV.set(defaultV);
        cameraNextQ.set(defaultQ);
        cameraNextS = defaultS;
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z) {
        cameraNextV.set(x,y,z);
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc) {
        cameraNextV.set(loc);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z) {
        cameraNowV.set(x,y,z);
        cameraNextV.set(x,y,z);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc) {
        cameraNowV.set(loc);
        cameraNextV.set(loc);
    }

    /**
     * カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc() {
        return new Vector3d(cameraNowV);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w) {
        cameraNextQ.set(x,y,z,w);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat) {
        cameraNextQ.set(quat);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w) {
        cameraNowQ.set(x,y,z,w);
        cameraNextQ.set(x,y,z,w);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat) {
        cameraNowQ.set(quat);
        cameraNextQ.set(quat);
    }

    /**
     * カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat() {
        return new Quat4d(cameraNowQ);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z) {
        setCameraQuat(rot2quat(x,y,z));
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot) {
        setCameraQuat(rot2quat(rot));
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z) {
        setCameraQuatImmediately(rot2quat(x,y,z));
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot) {
        setCameraQuatImmediately(rot2quat(rot));
    }

    /**
     * カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。
     */
    public void setCameraScale(double s) {
        cameraNextS = s;
    }

    /**
     * カメラの拡大率を即時に指定します。
     */
    public void setCameraScaleImmediately(double s) {
        cameraNowS = s;
        cameraNextS = s;
    }

    /**
     * カメラの拡大率を返します。
     */
    public double getCameraScale() {
        return cameraNowS;
    }

    Quat4d rot2quat(double x,double y,double z) {
        Transform3D t0 = new Transform3D();
        Transform3D t1 = new Transform3D();
        t1.rotX(x);
        t0.mul(t1);
        t1.rotY(y);
        t0.mul(t1);
        t1.rotZ(z);
        t0.mul(t1);
        Quat4d q = new Quat4d();
        t0.get(q);
        return q;
    }
    Quat4d rot2quat(Vector3d rot) {
        return rot2quat(rot.x,rot.y,rot.z);
    }

    /**
     * ヘッドライトのON,OFFを設定します。
     */
    public void setHeadLightEnable(boolean b) {
        headLight.setEnable(b);
    }
    // マウスナビゲーションのモード設定
    NaviMode naviMode = NaviMode.NONE;

    /**
     * ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m) {
        naviMode = m;
        if (naviMode == NaviMode.NONE)
            setA3Controller0(new NoneController());
        else if (naviMode == NaviMode.WALK)
            setA3Controller0(new WalkController());
        else if (naviMode == NaviMode.FLY)
            setA3Controller0(new FlyController());
        else if (naviMode == NaviMode.EXAMINE)
            setA3Controller0(new ExamController());
        else if (naviMode == NaviMode.EDIT)
            setA3Controller0(new EditController());
        else if (naviMode == NaviMode.SIMPLE)
            setA3Controller0(new SimpleController());
        else if (naviMode == NaviMode.CHASE)
            setA3Controller0(new ChaseController());
    }

    double naviSpeed = 1.0;
    /**
     * ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s) {
        naviSpeed = s;
    }

    /**
     * ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed() {
        return naviSpeed;
    }

    A3Controller controller = null;
    void setA3Controller0(A3Controller c) {
        if (canvas!=null)
            c.setA3CanvasInterface(canvas);
        if (controller!=null) {
            removeA3Listener(controller);
            canvas.removeKeyListener(controller);
            controller.stop();
        }
        controller = c;
        addA3Listener(controller);
        canvas.addKeyListener(controller);
        controller.init();
    }
    /**
     * A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c) {
        naviMode = NaviMode.USER;
        setA3Controller0(c);
    }
//  ----------座標変換とピッキングのためのラッパーメソッド---------
    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算してます。
     */
    public Point3d canvasToVirtualCS(int x,int y) {
        return pickingBehavior.canvasToVirtualCS(x,y);
    }

    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToVirtualCS(int x,int y,double dis) {
        return pickingBehavior.canvasToVirtualCS(x,y,dis);
    }

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算しています。
     */
    public Point3d canvasToPhysicalCS(int x,int y) {
        return pickingBehavior.canvasToPhysicalCS(x,y);
    }

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToPhysicalCS(int x,int y,double dis) {
        return pickingBehavior.canvasToPhysicalCS(x,y,dis);
    }

    /**
     * 物理空間上の点(x,y,z)を仮想空間上の点(x,y,z)に変換します。
     */
    public Vector3d physicalCSToVirtualCS(Vector3d v) {
        return pickingBehavior.physicalCSToVirtualCS(v);
    }

    /**
     * 仮想空間上の点(x,y,z)をA3Canvas上の点(x,y)に変換します。
     */
    public Point virtualCSToCanvas(Point3d p) {
        return pickingBehavior.virtualCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)を物理空間上の点(x,y,z)に変換します。
     */
    public Vector3d virtualCSToPhysicalCS(Vector3d v) {
        return pickingBehavior.virtualCSToPhysicalCS(v);
    }

    /**
     * カメラの座標系(物理空間の座標系)のX軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecX() {
        return physicalCSToVirtualCS(new Vector3d(1.0,0.0,0.0));
    }

    /**
     * カメラの座標系(物理空間の座標系)のY軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecY() {
        return physicalCSToVirtualCS(new Vector3d(0.0,1.0,0.0));
    }

    /**
     * カメラの座標系(物理空間の座標系)のZ軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecZ() {
        return physicalCSToVirtualCS(new Vector3d(0.0,0.0,1.0));
    }

    /**
     * A3Canvas上の点(x,y)にあるA3Objectをピックアップします。
     */
    public A3Object pickA3(int x,int y) {
        return pickingBehavior.pickA3(x,y);
    }
}
