#include "connection.h"
#include "descriptor.h"
#include "error.h"

typedef enum
{
	UP_SELECT     = 0x001,
	UP_UPDATE     = 0x002,
	UP_INSERT     = 0x004,
	UP_DELETE     = 0x008,
	UP_RULE       = 0x010,
	UP_REFERENCES = 0x020,
	UP_TRIGGER    = 0x040,

	UP_ALL = UP_SELECT | UP_UPDATE | UP_INSERT | UP_DELETE | UP_RULE | UP_REFERENCES | UP_TRIGGER
} UserPrivileges;

typedef enum
{
	TT_NO_ONE           = 0x00,

	TT_TABLE            = 0x01,
	TT_VIEW             = 0x02,
	TT_SYSTEM_TABLE     = 0x04,
	TT_GLOBAL_TEMPORARY = 0x08, /* unsupported */
	TT_LOCAL_TEMPORARY  = 0x10, /*    -//-     */
	TT_ALIAS            = 0x20, /*    -//-     */
	TT_SYNONYM          = 0x40, /*    -//-     */

	TT_DEFAULT = TT_TABLE | TT_VIEW | TT_SYSTEM_TABLE
} TableTypes;

typedef enum
{
	TC_AS_IS = 0,
	TC_LOWER = 1,
	TC_STOP  = 2

} TextConversion;

typedef enum {
	/* types with unknown size, they have SQL_VARCHAR SQL data type */
/*
	PTN__TIMETZ       =  0,
	PTN__BIT          =  1,
	PTN_VARBIT        =  2,
	PTN__VARBIT       =  3,
	PTN_REFCURSOR     =  4,
	PTN__REFCURSOR    =  5,
	PTN_REGPROCEDURE  =  6,
	PTN_REGOPER       =  7,
	PTN_REGOPERATOR   =  8,
	PTN_REGCLASS      =  9,
	PTN_REGTYPE       = 10,
	PTN__REGPROCEDURE = 11,
	PTN__REGOPER      = 12,
	PTN__REGOPERATOR  = 13,
	PTN__REGCLASS     = 14,
	PTN_BYTEA         = 15,
	PTN_REGPROC       = 16,
	PTN_TEXT          = 17,
	PTN_SMGR          = 18,
	PTN__LINE         = 19,
	PTN_CIDR          = 20,
	PTN__CIDR         = 21,
	PTN_UNKNOWN       = 22,
	PTN__CIRCLE       = 23,
	PTN__MONEY        = 24,
	PTN_INET          = 25,
	PTN__BOOL         = 26,
	PTN__BYTEA        = 27,
	PTN__CHAR         = 28,
	PTN__NAME         = 29,
	PTN__INT2         = 30,
	PTN__INT2VECTOR   = 31,
	PTN__INT4         = 32,
	PTN__REGPROC      = 33,
	PTN__TEXT         = 34,
	PTN__TID          = 35,
	PTN__XID          = 36,
	PTN__CID          = 37,
	PTN__OIDVECTOR    = 38,
	PTN__BPCHAR       = 39,
	PTN__VARCHAR      = 40,
	PTN__INT8         = 41,
	PTN__POINT        = 42,
	PTN__LSEG         = 43,
	PTN__PATH         = 44,
	PTN__BOX          = 45,
	PTN__FLOAT4       = 46,
	PTN__FLOAT8       = 47,
	PTN__ABSTIME      = 48,
	PTN__RELTIME      = 49,
	PTN__TINTERVAL    = 50,
	PTN__POLYGON      = 51,
	PTN__OID          = 52,
	PTN__ACLITEM      = 53,
	PTN__MACADDR      = 54,
	PTN__INET         = 55,
	PTN__REGTYPE      = 56,

	PTN_VARCHAR       = 57,

	PTN__TIMESTAMP    = 58,
	PTN__DATE         = 59,
	PTN__TIME         = 60,
	PTN__TIMESTAMPTZ  = 61,
	PTN__INTERVAL     = 62,
	PTN__NUMERIC      = 63,
	PTN_OID           = 64,
	
	PTN_BOOL          = 65,

	PTN_CHAR          = 66,

	PTN_NAME          = 67,
	PTN_MACADDR       = 68,
	PTN_TIMESTAMPTZ   = 69,
	PTN_TIMETZ        = 70,
	PTN_BPCHAR        = 71,

	PTN_TIME          = 72,
	PTN_BIT           = 73,

	PTN_INT8          = 74,
	PTN_INT2          = 75,
	PTN_INT4          = 76,

	PTN_XID           = 77,
	PTN_CID           = 78,
	PTN_TIMESTAMP     = 79,
	PTN_DATE          = 80,
	PTN_NUMERIC       = 81,
	PTN_INTERVAL      = 82,
*/
	PTN_CHAR          =   18,
	PTN_VARCHAR       = 1043,
	PTN_INT8          =   20,
	PTN_INT2          =   21,
	PTN_INT4          =   23,

	PTN_OID           =   49 /* PSQL_DATATYPES_NUMBER - 1 */

} PSQLDataTypeName;


#define PSQL_DATATYPES_NUMBER         51  /*83*/


typedef struct
{
	const TCHAR*      name;
	PSQLDataTypeName  postgresql_type_oid;
} ColumnNameType;

typedef struct
{
	const TCHAR*      name;
	const TCHAR*      local_name;
	const SQLSMALLINT sql_type;
	const SQLSMALLINT postgresql_type_oid;
	const BYTE        is_autoincrementing; /* 0 - null, 1 - SQL_TRUE, 2 - SQL_FALSE */
	const BYTE        is_nullable;
	const BYTE        is_unsigned;
	const SQLINTEGER  column_size;
	const TCHAR*      create_params;

	/*
	const TCHAR* name;
	const int    oid;
	SQLSMALLINT  sql_type;
	BYTE         is_unsigned;
	const int    length;*/
} PostgreSQLDataType;

/*-----------------------------------------------------------------------------

	Resulted rows:

	  We allocate memory by BLOCKs, every BLOCK's size is RESULTED_ROWS_BLOCK_SIZE,
	blockes link as list

-----------------------------------------------------------------------------*/
#define RESULTED_ROWS_BLOCK_SIZE           0x8000

typedef struct _block Block;

/*---------------------------------------------------------
	struct Block
---------------------------------------------------------*/
struct _block
{
	/* header */
	Block* next;
	Block* prev;
	unsigned int size;

	BYTE data[RESULTED_ROWS_BLOCK_SIZE];
};

Block* AllocBlock(Block* prev, int size);
void   FreeBlock(Block* block);
TCHAR* AddField(Statement* pStatement, unsigned int length);

/*---------------------------------------------------------------------------*/

typedef enum
{
	PS_ALLOCATED        = 0x01,
	PS_BOUND            = 0x02,
	PS_NEED_REBIND      = 0x04,
	PS_NEED_UNBIND      = 0x08,
	PS_BOUND_TO_BACKEND = 0x10

} ParameterState;

typedef enum 
{
	SS_ALLOCATED    = 0x1,
	SS_EXECUTED     = 0x2,
	SS_POSITIONED   = 0x4,

	/* mask for the above states */
	SS_EXEC_STATES  = SS_ALLOCATED | SS_EXECUTED | SS_POSITIONED,

	SS_PORTAL_DECLARED    = 0x10,
	SS_STATEMENT_DECLARED = 0x20,

	SS_STATEMENT_NEED_REDECLARE = 0x10000,
	SS_PORTAL_NEED_REDECLARE    = 0x20000,
	SS_TRANSACTION_OPENED       = 0x40000,

	SS_UNALLOCATED = SS_POSITIONED | SS_EXECUTED
} StatementState;

typedef enum
{
	CS_DECLARED     = 1
} CursorState;


/* WARNING! Do not change this values, they are defined as Postrges Data Types */
#define	PT_TEXT_ID       0
#define PT_BINARY_ID     1

typedef struct _result Result;
/*---------------------------------------------------------
	struct Parameter
---------------------------------------------------------*/
struct _parameter
{
	BOOL  bound;           /* bound into driver  */
	BOOL  boundToBackend;  /* bound into backend */

	BYTE*  value;   /* pointer to the malloced buffer, containing value */
	uint16 type;    /* can be Text or Binary */
	uint32 length;  /* length */
	uint32 oid;     /* object ID of the parameter data type */
};

/*---------------------------------------------------------
	struct _results
---------------------------------------------------------*/
struct _results
{
	/*---- resultsets' list -----------------------------------------------------
	 * we use explicitly allocated IRD, which fully describes any resultset
	 *---------------------------------------------------------------------------
	 */
	struct _list irds; /*   results affected after last execution,
	                   /* using SQLMoreResults application can get them all */

	int current;       /* index of currently used resultset */

	/*---- buffer for results' data --------------------------------------------*/
	Block* pHead;           /* List of the results, can be NULL or empty */
	Block* pTail;
	Block* pLastHead;       /* last recordset block */
	int*   pLastHeadData;   /* first field in last recordset */

	unsigned int unFreeSpaceInTailBlock;
	union{
		unsigned int* pTailField;    /* Last field (empty) in tail block */
		TCHAR*        pTailField_chr;
		};
};

/*---------------------------------------------------------
	struct DataAtExec
---------------------------------------------------------*/
struct _data_at_exec
{
	SQLSMALLINT index;       /* number of corresponding column or parameter - 1 */
	SQLSMALLINT c_type;
	SQLINTEGER  length;
	SQLINTEGER  free_size;
	BYTE*       data;
};

typedef struct _data_at_exec DataAtExec;
/*---------------------------------------------------------
	struct _query
---------------------------------------------------------*/
struct _query
{
	uint16  nParametersNumber;  /* number of parameters found in SQLPrepare query */
	TCHAR*  query;              /* query's text */

	TCHAR cursorName[MAX_CURSOR_NAME_LENGTH+1];

	QueryType   type;
	CursorState cursorState;    /* CS_DECLARED */
};

/*---------------------------------------------------------
	struct StmtAttributes
---------------------------------------------------------*/
struct _stmt_attributes
{
	SQLUINTEGER no_scan;
	SQLUINTEGER use_bookmarks;   /* SQL_UB_OFF or SQL_UB_VARIABLE */
	SQLUINTEGER metadata_id;     /* SQL_FALSE or SQL_TRUE */
	SQLUINTEGER enable_auto_ipd; /* SQL_FALSE or SQL_TRUE */
};

/*---------------------------------------------------------
	struct Statement
---------------------------------------------------------*/
struct _statement
{
	Connection* connection; /* parent connection, can't be NULL */
	StatementState state;   /* current statement's state */

	struct _stmt_attributes attributes;
	struct _results         results;
	struct _query           query;
	struct _diag            diag;

	TCHAR                   need_data;
	struct _data_at_exec    data_at_exec;

	/* fetch */
	int fetch_first;     /* first row of avaible results */
	int fetch_position;  /* current     -//-             */
	int fetch_last;      /* number of avaible rows       */

	BOOL use_buffering;

	/* descriptors */
	Descriptor* apd; /* Application Parameter Descriptor (APD)    */
	Descriptor* ard; /* Application Row Descriptor (ARD)          */
	Descriptor* ipd; /* Implementation Parameter Descriptor (IPD) */
	Descriptor* ird; /* Implementation Row Descriptor (IRD)       */

	struct _descriptor d_apd;
	struct _descriptor d_ard;
	struct _descriptor d_ipd;
	struct _descriptor d_ird;

	Statement* prev;
	Statement* next;  /* next statement in the statement's list */

	__CRITICAL_SECTION(cs);
};

/*---------------------------------------------------------
	functions
---------------------------------------------------------*/
Statement* AllocStatement(Connection* pConnection);
void       EmptyStatementResults(Statement* pStatement);
SQLRETURN  ResetStatement(Statement* pStatement);
SQLRETURN  ExecuteStatement(Statement* pStatement, BOOL ForceBuffering);
SQLRETURN  FreeStatement(Statement* pStatement, SQLUSMALLINT Option);
SQLRETURN  GetStatementError(Statement* pStatement, SQLSMALLINT RecNumber, SQLCHAR* MessageText, SQLSMALLINT BufferLength, SQLSMALLINT *TextLength);
SQLRETURN  PrepareStatement(Statement* pStatement, SQLTCHAR* StatementText, SQLINTEGER TextLength);
SQLRETURN  Fetch(Statement* pStatement, SQLSMALLINT FetchOrientation, SQLINTEGER FetchOffset);
SQLRETURN  PrepareCursor(Statement* pStatement);
SQLRETURN  SetCursorName(Statement* pStatement, SQLTCHAR* CursorName, SQLSMALLINT NameLength, BOOL bAutoGenerate);
SQLRETURN  PrepareQuery(Statement* pStatement, SQLTCHAR* StatementText, SQLINTEGER TextLength);
SQLRETURN  BindParameter(Statement* pStatement, SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, SQLSMALLINT ValueType, SQLSMALLINT ParameterType, SQLUINTEGER ColumnSize, SQLSMALLINT DecimalDigits, SQLPOINTER ParameterValuePtr, SQLINTEGER BufferLength, SQLINTEGER* StrLen_or_IndPtr);
SQLRETURN  SetApplicationDescriptor(Statement* pStatement, Descriptor** ppDescInUse, Descriptor* pDescImplicit, Descriptor* pDescToSet);
SQLRETURN  PopulateID(Descriptor* pDescriptor);
SQLRETURN  GetData(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, SQLPOINTER TargetValue, SQLINTEGER BufferLength, SQLINTEGER *StrLen_or_Ind);
SQLRETURN  ColAttribute(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLUSMALLINT FieldIdentifier, SQLPOINTER CharacterAttribute, SQLSMALLINT BufferLength, SQLSMALLINT* StringLength, SQLPOINTER NumericAttribute);
SQLRETURN  DescribeCol(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLTCHAR* ColumnName, SQLSMALLINT BufferLength, SQLSMALLINT* NameLength, SQLSMALLINT* DataType, SQLUINTEGER* ColumnSize, SQLSMALLINT* DecimalDigits, SQLSMALLINT* Nullable);
SQLRETURN  BindCol(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, SQLPOINTER TargetValue, SQLINTEGER BufferLength, SQLINTEGER* StrLen_or_Ind);
SQLRETURN  GetStmtAttr(Statement* pStatement, SQLINTEGER Attribute, SQLPOINTER Value);

TCHAR* CompileCSOCQuery(Statement* pStatement, SQLTCHAR* query, TextConversion Conversion, SQLTCHAR* CatalogName, SQLSMALLINT CatalogNameLength, SQLTCHAR* SchemaName, SQLSMALLINT SchemaNameLength, SQLTCHAR* ObjectName, SQLSMALLINT ObjectNameLength, SQLTCHAR* ColumnName, SQLSMALLINT ColumnNameLenght, SQLSMALLINT ObjectType);
void RenameColumns(Statement* pStatement, const ColumnNameType* pColumns, int nNumber);
int* PrepareParameter(Statement* pStatement, SQLPOINTER ParamValue, SQLUINTEGER ParamLength, SQLSMALLINT ParamType, SQLINTEGER* ParamIndPtr);
SQLTCHAR* GetSQLString(SQLTCHAR* Text, SQLINTEGER TextLength, TextConversion Conversion);



/* column names for Catalog functions */
extern const ColumnNameType c_TablesColumns[5];
extern const ColumnNameType c_ColumnsColumns[18];
extern const ColumnNameType c_ProceduresColumns[8];
extern const ColumnNameType c_StatisticsColumns[13];
extern const ColumnNameType c_PrimaryKeysColumns[6];
extern const ColumnNameType c_ForeignKeysColumns[14];
extern const ColumnNameType c_GetTypeInfoColumns[19];
extern const ColumnNameType c_SpecialColumnsColumns[8];
extern const ColumnNameType c_TablePrivilegesColumns[7];
extern const ColumnNameType c_ColumnPrivilegesColumns[8];
extern const ColumnNameType c_ProcedureColumnsColumns[19];


extern const PostgreSQLDataType c_PostgreSQLDataTypes[PSQL_DATATYPES_NUMBER];

