/******************************************************************************

	file: odbc_core_level_api.c

******************************************************************************/
#include "connection.h"
#include "statement.h"

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLAllocHandle
 *-----------------------------------------------------------------------------
 */
SQLRETURN	SQL_API
SQLAllocHandle(SQLSMALLINT HandleType, SQLHANDLE InputHandle, SQLHANDLE* OutputHandle)
{
	RETCODE nRet;

	BEGIN(_T("SQLAllocHandle"));
	switch (HandleType)
	{
		case SQL_HANDLE_ENV:
			nRet = (*OutputHandle = AllocEnvironment()) ? SQL_SUCCESS : SQL_ERROR;
			break;
		case SQL_HANDLE_DBC:
			ENTER_CRITICAL_SECTION((Environment*) InputHandle);
			nRet = (*OutputHandle = AllocConnection((Environment*) InputHandle)) ? SQL_SUCCESS : SQL_ERROR;
			LEAVE_CRITICAL_SECTION((Environment*) InputHandle);
			break;
		case SQL_HANDLE_STMT:
			ENTER_CRITICAL_SECTION((Connection*) InputHandle);
			nRet = (*OutputHandle = AllocStatement((Connection*) InputHandle)) ? SQL_SUCCESS : SQL_ERROR;
			LEAVE_CRITICAL_SECTION((Connection*) InputHandle);
			break;
		case SQL_HANDLE_DESC:
			ENTER_CRITICAL_SECTION((Connection*) InputHandle);
			nRet = (*OutputHandle = AllocDescriptor((Connection*) InputHandle)) ? SQL_SUCCESS : SQL_ERROR;
			LEAVE_CRITICAL_SECTION((Connection*) InputHandle);
			break;
		/* Driver Manager returnes SQL_ERROR for any other HandleType */
	}
	END(nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLBindCol
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLBindCol(SQLHSTMT StatementHandle, SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, 
		       SQLPOINTER TargetValue, SQLINTEGER BufferLength, SQLINTEGER* StrLen_or_Ind)
{
	SQLRETURN nRet = SQL_SUCCESS;

	ENTER_STMT(StatementHandle, _T("SQLBindCol"));
	nRet = BindCol((Statement*)StatementHandle, ColumnNumber, TargetType, TargetValue, BufferLength, StrLen_or_Ind);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLBindParameter
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLBindParameter(SQLHSTMT StatementHandle, SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType,
								 SQLSMALLINT ValueType, SQLSMALLINT ParameterType, SQLUINTEGER ColumnSize, SQLSMALLINT DecimalDigits,
								 SQLPOINTER ParameterValuePtr, SQLINTEGER BufferLength, SQLINTEGER* StrLen_or_IndPtr)
{
	SQLRETURN nRet;

	ENTER_STMT(StatementHandle, _T("SQLBindParameter"));
	nRet = BindParameter((Statement*)StatementHandle, ParameterNumber, InputOutputType, ValueType, ParameterType,
	                     ColumnSize, DecimalDigits, ParameterValuePtr, BufferLength, StrLen_or_IndPtr);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLCancel
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLCancel(SQLHSTMT StatementHandle)
{
	ENTER_STMT(StatementHandle, _T("SQLCancel"));
	LEAVE_STMT(StatementHandle, SQL_SUCCESS);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLColAttribute
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLColAttribute_(SQLHSTMT StatementHandle, SQLUSMALLINT ColumnNumber, SQLUSMALLINT FieldIdentifier,
                 SQLPOINTER CharacterAttribute, SQLSMALLINT BufferLength, SQLSMALLINT* StringLength,
                 SQLPOINTER NumericAttribute)
{
	SQLRETURN   nRet;
	ENTER_STMT(StatementHandle, _T("SQLColAttribute"));
	nRet = ColAttribute((Statement*)StatementHandle, ColumnNumber, FieldIdentifier, CharacterAttribute, (SQLSMALLINT)BYTES_TO_CHARACTERS(BufferLength), StringLength, NumericAttribute);
	LEAVE_STMT(StatementHandle, nRet);
}


/*-----------------------------------------------------------------------------
 * FUNCTION: SQLConnect
 *-----------------------------------------------------------------------------
 */
SQLRETURN	SQL_API
SQLConnect_(SQLHDBC ConnectionHandle,
            SQLTCHAR* ServerName,     SQLSMALLINT ServerNameLength,
            SQLTCHAR* UserName,       SQLSMALLINT UserNameLength,
            SQLTCHAR* Authentication, SQLSMALLINT AuthNameLength)
{
	SQLRETURN   nRet;
	SQLSMALLINT length;
	Connection* pConnection = (Connection*) ConnectionHandle;

	ENTER_CONN(ConnectionHandle, _T("SQLConnect"));
	/* check input data and fill connection parameters */
	if (NULL != UserName && 0 != UserNameLength && _T('\0') != UserName[0])
	{
		length = MIN((SQL_NTS == UserNameLength) ? _tcslen(UserName) : UserNameLength, UID_MAX_LENGTH);
		_tcsncpy(pConnection->parameters[UID_PARAM], UserName, length);
		pConnection->parameters[UID_PARAM][length] = _T('\0');
	}
	if (NULL != Authentication && 0 != AuthNameLength && _T('\0') != Authentication[0])
	{
		length = MIN((SQL_NTS == AuthNameLength) ? _tcslen(Authentication) : AuthNameLength, PWD_MAX_LENGTH);
		_tcsncpy(pConnection->parameters[PWD_PARAM], Authentication, length);
		pConnection->parameters[PWD_PARAM][length] = _T('\0');
	}
	ReadFromDS(pConnection->parameters, ServerName, ServerNameLength);
	/* comments:
	 *   ServerName here is always a valid DSN - no need for error checking
	 */
	nRet = Connect(pConnection);
	LEAVE_CONN(ConnectionHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLCopyDesc
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLCopyDesc(SQLHDESC SourceDescHandle, SQLHDESC TargetDescHandle)
{
	SQLRETURN   nRet;
	Descriptor* source;
	Descriptor* target;

	BEGIN(_T("SQLCopyDesc"));
	source = GET_DESCRIPTOR((Descriptor*) SourceDescHandle);
	target = GET_DESCRIPTOR((Descriptor*) TargetDescHandle);

	/* we can't modify IRD */
	if (0 != (DT_IRD & target->type))
	{
		SetError(SQL_HANDLE_DESC, TargetDescHandle, ERR_DESC_CANT_MODIFY_IRD, NULL);
		nRet = SQL_ERROR;
	}
	else
	{
		nRet = CopyDescriptor(source, target);
	}
	RET_DESCRIPTOR(target);
	RET_DESCRIPTOR(source);
	END(nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLDescribeCol
 *
 * DESCRIPTION: returnes info about columns in resultset (including bookmark)
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLDescribeCol_(SQLHSTMT StatementHandle, SQLUSMALLINT ColumnNumber,  SQLTCHAR* ColumnName,
                SQLSMALLINT BufferLength, SQLSMALLINT* NameLength, SQLSMALLINT* DataType,
                SQLUINTEGER* ColumnSize, SQLSMALLINT* DecimalDigits, SQLSMALLINT* Nullable)
{
	SQLRETURN nRet;
	ENTER_STMT(StatementHandle, _T("SQLDescribeCol"));
	nRet = DescribeCol((Statement*) StatementHandle, ColumnNumber, ColumnName, BufferLength, NameLength, DataType, ColumnSize, DecimalDigits, Nullable);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLDisconnect
 *-----------------------------------------------------------------------------
 */
SQLRETURN	SQL_API
SQLDisconnect(SQLHDBC ConnectionHandle)
{
	SQLRETURN nRet;

	ENTER_CONN(ConnectionHandle, _T("SQLDisconnect"));
	nRet = Disconnect((Connection*) ConnectionHandle);
	LEAVE_CONN(ConnectionHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLExecDirect
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLExecDirect_(SQLHSTMT StatementHandle, SQLTCHAR* StatementText, SQLINTEGER TextLength)
{
	SQLRETURN nRet, nRet1;

	ENTER_STMT(StatementHandle, _T("SQLExecDirect"));
	/* Driver Manager tracks for statement's state - we won't */
	if (SQL_ERROR != (nRet  = ResetStatement((Statement*) StatementHandle)) &&
	    SQL_ERROR != (nRet1 = PrepareQuery((Statement*) StatementHandle, StatementText, TextLength))
	   )
	{
		nRet  = MAX_ERROR(nRet, nRet1);
		nRet1 = ExecuteStatement((Statement*) StatementHandle, FALSE);
		nRet  = MAX_ERROR(nRet, nRet1);
	}
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUCTION: SQLExecute
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLExecute(SQLHSTMT StatementHandle)
{
	SQLRETURN nRet;

	ENTER_STMT(StatementHandle, _T("SQLExecute"));
	/* Driver Manager tracks for statement's state - we won't */
	EmptyStatementResults((Statement*)StatementHandle);
	nRet = ExecuteStatement((Statement*)StatementHandle, FALSE);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUCTION: SQLFetch
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLFetch(SQLHSTMT StatementHandle)
{
	SQLRETURN nRet;

	ENTER_STMT(StatementHandle, _T("SQLFetch"));
	nRet = Fetch((Statement*) StatementHandle, SQL_FETCH_NEXT, 0);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUCTION: SQLFetchScroll
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLFetchScroll(SQLHSTMT StatementHandle, SQLSMALLINT FetchOrientation, SQLINTEGER FetchOffset)
{
	SQLRETURN nRet;

	ENTER_STMT(StatementHandle, _T("SQLFetchScroll"));
	nRet = Fetch((Statement*) StatementHandle, FetchOrientation, FetchOffset);
	LEAVE_STMT(StatementHandle, nRet);
}


/*-----------------------------------------------------------------------------
 * FUCTION: SQLFreeHandle
 *-----------------------------------------------------------------------------
 */
SQLRETURN	SQL_API
SQLFreeHandle(SQLSMALLINT HandleType, SQLHANDLE Handle)
{
	RETCODE nRet;
	
	BEGIN(_T("SQLFreeHandle"));
	switch (HandleType)
	{
		case SQL_HANDLE_ENV:
			nRet = FreeEnvironment((Environment*) Handle);
			break;
		case SQL_HANDLE_DBC:
			nRet = FreeConnection((Connection*) Handle, SQL_DROP);
			break;
		case SQL_HANDLE_STMT:
			nRet = FreeStatement((Statement*) Handle, SQL_DROP);
			break;
		case SQL_HANDLE_DESC:
			ENTER_CRITICAL_SECTION((Descriptor*) Handle);
			nRet = FreeDescriptor((Descriptor*) Handle);
			break;
		/* Driver Manager returnes SQL_ERROR for any other HandleType */
	}
	END(nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLGetConnectAttr
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLGetConnectAttr_(SQLHDBC ConnectionHandle, SQLINTEGER Attribute, SQLPOINTER Value,
                   SQLINTEGER BufferLength, SQLINTEGER* StringLength)
{
	SQLRETURN   nRet        = SQL_SUCCESS;
	Connection* pConnection = (Connection*)ConnectionHandle;
	SQLINTEGER  uInteger;

	ENTER_CONN(ConnectionHandle, _T("SQLGetConnectAttr"));
	BufferLength = BYTES_TO_CHARACTERS(BufferLength);

	switch(Attribute)
	{
		case SQL_ATTR_AUTO_IPD:              /* 10001 */
			uInteger = SQL_TRUE;
			break;
		case SQL_ATTR_CONNECTION_DEAD:        /* 1209 */
			/* andyk possible enter here some connect test function */
			uInteger = pConnection->attributes.connection_dead;
			break;
		case SQL_ATTR_AUTOCOMMIT:              /* 102 */
			uInteger = pConnection->attributes.autocommit;
			break;
		case SQL_ATTR_LOGIN_TIMEOUT:          /* 103 */
			uInteger = 0;
			break;
		case SQL_ATTR_ACCESS_MODE:
		case SQL_ATTR_ASYNC_ENABLE:
		case SQL_ATTR_CONNECTION_TIMEOUT:
		case SQL_ATTR_CURRENT_CATALOG:
		case SQL_ATTR_METADATA_ID:
		case SQL_ATTR_ODBC_CURSORS:
		case SQL_ATTR_PACKET_SIZE:
		case SQL_ATTR_QUIET_MODE:
		case SQL_ATTR_TRACE:
		case SQL_ATTR_TRACEFILE:
		case SQL_ATTR_TRANSLATE_LIB:
		case SQL_ATTR_TRANSLATE_OPTION:
		case SQL_ATTR_TXN_ISOLATION:
		default:
			SetError(SQL_HANDLE_DBC, ConnectionHandle, ERR_UNKNOWN_ATTRIBUTE, NULL);
			nRet = SQL_ERROR;
	}
	if (SQL_SUCCESS == nRet)
		memcpy(Value, &uInteger, sizeof(uInteger));

	LEAVE_CONN(ConnectionHandle, nRet);
}


/*-----------------------------------------------------------------------------
 * SQLGetCursorName
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLGetCursorName_(SQLHSTMT StatementHandle, SQLTCHAR* CursorName, SQLSMALLINT BufferLength, SQLSMALLINT* NameLength)
{
	SQLRETURN  nRet;
	Statement* pStatement = (Statement*)StatementHandle;

	ENTER_STMT(StatementHandle, _T("SQLGetCursorName"));
	BufferLength = BYTES_TO_CHARACTERS(BufferLength);
	
	/* check cursor's avaibility */
	if (_T('\0') == pStatement->query.cursorName[0])
		nRet = SetCursorName(pStatement, NULL, 0, TRUE);

	/* return requested values */
	if (SQL_SUCCESS != (nRet = ReturnString(CursorName, BufferLength, NameLength, pStatement->query.cursorName, SQL_NTS)))
		SetError(SQL_HANDLE_STMT, pStatement, ERR_TOO_SMALL_BUFFER, _T("CursorName"), NULL);

	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLGetDescField
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLGetDescField_(SQLHDESC DescriptorHandle, SQLSMALLINT RecNumber, SQLSMALLINT FieldIdentifier,
                 SQLPOINTER Value, SQLINTEGER BufferLength, SQLINTEGER* StringLength)
{
	SQLRETURN   nRet;
	SQLSMALLINT length = 0;

	ENTER_DESC(DescriptorHandle, _T("SQLGetDescField"));
	nRet = GetDescField((Descriptor*) DescriptorHandle, RecNumber, FieldIdentifier, Value, BYTES_TO_CHARACTERS(BufferLength), &length, NULL);
	if (NULL != StringLength)
		*StringLength = length;
	LEAVE_DESC(DescriptorHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLGetDescRec
 *
 * DESCRIPTION: returnes SQL_DESC_NAME, SQL_DESC_TYPE, SQL_DESC_DATETIME_INTERVAL_CODE (for records
 *              whose type is SQL_DATETIME or SQL_INTERVAL), SQL_DESC_OCTET_LENGTH, SQL_DESC_PRECISION,
 *              SQL_DESC_SCALE and SQL_DESC_NULLABLE at once.
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLGetDescRec_(SQLHDESC DescriptorHandle, SQLSMALLINT RecNumber, SQLTCHAR* Name, SQLSMALLINT BufferLength,
               SQLSMALLINT* StringLength, SQLSMALLINT* Type, SQLSMALLINT* SubType, SQLINTEGER* Length,
							 SQLSMALLINT* Precision, SQLSMALLINT* Scale, SQLSMALLINT* Nullable)
{
	SQLRETURN   nRet;
	Descriptor* pDescriptor;

	ENTER_DESC(DescriptorHandle, _T("SQLGetDescRec"));
	BufferLength = BYTES_TO_CHARACTERS(BufferLength);	
	if ((pDescriptor = (Descriptor*)DescriptorHandle)->header.count < RecNumber)
	{
		/* no record with this number */
		nRet = SQL_NO_DATA;
	}
	else if (0 == RecNumber)
	{
		/* Driver Manager won't get us here if SQL_ATTR_USE_BOOKMARKS = SQL_UB_OFF */
		if (DT_IPD == pDescriptor->type)
		{
			SetError(SQL_HANDLE_DESC, pDescriptor, ERR_DESC_READ_ZERO_REC_IN_PD, NULL);
			nRet = SQL_ERROR;
		}
		else
		{
			/* bookmark column */

			/* SQL_DESC_NULLABLE */
			if (NULL != Nullable)
				*Nullable = pDescriptor->id_records[RecNumber].nullable;

			/* SQL_DESC_NAME */
			if (SQL_SUCCESS != (nRet = ReturnString(Name, BufferLength, StringLength, c_szBookmarkName, SQL_NTS)))
				SetError(SQL_HANDLE_DESC, pDescriptor, ERR_TOO_SMALL_BUFFER, _T("Name"), NULL);

			/* SQL_DESC_TYPE */
			if (NULL != Type)
				*Type = pDescriptor->bookmark.type;

			/* SQL_DESC_OCTET_LENGTH */
			if (NULL != Precision)
				*Length = pDescriptor->bookmark.octet_length;

			/* SQL_DESC_PRECISION */
			if (NULL != Precision)
				*Scale = 0;

			/* SQL_DESC_SCALE */
			if (NULL != Scale)
				*Scale = 0;

			nRet = SQL_SUCCESS;
		}
	}
	else
	{
		CD_REC* pCD_REC;

		RecNumber--;
		if (0 != (DT_ID & pDescriptor->type))
		{
			/* some fields can be retrieved only for implementation descriptors */

			/* SQL_DESC_NULLABLE */
			if (NULL != Nullable)
				*Nullable = pDescriptor->id_records[RecNumber].nullable;

			/* SQL_DESC_NAME */
			if (SQL_SUCCESS != (nRet = ReturnString(Name, BufferLength, StringLength, pDescriptor->id_records[RecNumber].name, SQL_NTS)))
				SetError(SQL_HANDLE_DESC, pDescriptor, ERR_TOO_SMALL_BUFFER, _T("Name"), NULL);

			pCD_REC = &pDescriptor->id_records[RecNumber].common;
		}
		else
		{
			pCD_REC = &pDescriptor->ad_records[RecNumber].common;
			nRet = SQL_SUCCESS;
		}

		/* SQL_DESC_TYPE */
		if (NULL != Type)
			*Type = pCD_REC->type;

		/* SQL_DESC_DATETIME_INTERVAL_CODE */
		if ((SQL_DATETIME == *Type || SQL_INTERVAL == *Type) && (NULL != SubType))
			*SubType = pCD_REC->datetime_interval_code;

		/* SQL_DESC_OCTET_LENGTH */
		if (NULL != Precision)
			*Length = pCD_REC->octet_length;

		/* SQL_DESC_PRECISION */
		if (NULL != Precision)
			*Scale = pCD_REC->precision;

		/* SQL_DESC_SCALE */
		if (NULL != Scale)
			*Scale = pCD_REC->scale;
	}
	
	LEAVE_DESC(DescriptorHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLGetEnvAttr
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLGetEnvAttr(SQLHENV EnvironmentHandle, SQLINTEGER Attribute, SQLPOINTER Value,
           SQLINTEGER BufferLength, SQLINTEGER *StringLength)
{
	SQLRETURN  nRet = SQL_SUCCESS;
	SQLINTEGER uInteger;

	ENTER_ENV(EnvironmentHandle, _T("SQLGetEnvAttr"));
	switch(Attribute)
	{
		case SQL_ATTR_ODBC_VERSION:       /*   200 */
			uInteger = ((Environment*) EnvironmentHandle)->attributes.odbc_version;
			break;
		case SQL_ATTR_CONNECTION_POOLING: /*   201 */
			uInteger = ((Environment*) EnvironmentHandle)->attributes.connection_pooling;
			break;
		case SQL_ATTR_CP_MATCH:           /*   202 */
			uInteger = SQL_CP_STRICT_MATCH;
			break;
		case SQL_ATTR_OUTPUT_NTS:         /* 10001 */
			uInteger = SQL_TRUE;
			break;
		default:
			SetError(SQL_HANDLE_ENV, EnvironmentHandle, ERR_UNKNOWN_ATTRIBUTE, NULL);
			nRet = SQL_ERROR;	
	}

	if (SQL_SUCCESS == nRet)
		memcpy(Value, &uInteger, sizeof(uInteger));

	LEAVE_ENV(EnvironmentHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLGetStmtAttr
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLGetStmtAttr_(SQLHSTMT StatementHandle, SQLINTEGER Attribute, SQLPOINTER Value,
                SQLINTEGER BufferLength, SQLINTEGER *StringLength)
{
	SQLRETURN nRet;
	ENTER_STMT(StatementHandle, _T("SQLGetStmtAttr"));
	BufferLength = BYTES_TO_CHARACTERS(BufferLength); /* currently unused */
	nRet = GetStmtAttr((Statement*)StatementHandle, Attribute, Value);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLNativeSQL
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLNativeSql_(SQLHDBC ConnectionHandle, SQLTCHAR* SourceValue, SQLINTEGER SourceLength,
              SQLTCHAR* TargetValue, SQLINTEGER TargetLength, SQLINTEGER* SourceLengthPtr)
{
	SQLRETURN nRet;
	SQLSMALLINT nSourceLength;

	ENTER_CONN(ConnectionHandle, _T("SQLNativeSQL"));

	/* for now return the same text */
	if (SQL_SUCCESS != (nRet = ReturnString(TargetValue, (SQLSMALLINT)BYTES_TO_CHARACTERS(TargetLength), &nSourceLength, SourceValue, (SQLSMALLINT)SourceLength)))
		SetError(SQL_HANDLE_DBC, ConnectionHandle, ERR_TOO_SMALL_BUFFER, _T("TargetValue"), NULL);

	*SourceLengthPtr = nSourceLength;
	LEAVE_CONN(ConnectionHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLNumParams
 *
 * DESCRIPTION: returnes the number of parameters in a prepared query, can be 
 *              called only after the SQLPrepare function.
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLNumParams(SQLHSTMT StatementHandle, SQLSMALLINT* ParameterCountPtr)
{
	Descriptor* pIPD;

	ENTER_STMT(StatementHandle, _T("SQLNumParams"));
	/* Driver Manager tracks for statement's state - we won't */
	pIPD = GET_DESCRIPTOR(((Statement*)StatementHandle)->ipd);
	*ParameterCountPtr = pIPD->header.count;
	RET_DESCRIPTOR(pIPD);
	LEAVE_STMT(StatementHandle, SQL_SUCCESS);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLNumResultCols
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLNumResultCols(SQLHSTMT StatementHandle, SQLSMALLINT *ColumnCount)
{
	Descriptor* pIRD;
	ENTER_STMT(StatementHandle, _T("SQLNumResultCols"));
	/* Driver Manager tracks for statement's state - we won't */
	pIRD = GET_DESCRIPTOR(((Statement*) StatementHandle)->ird);
	*ColumnCount = pIRD->header.count;
	RET_DESCRIPTOR(pIRD);

	LEAVE_STMT(StatementHandle, SQL_SUCCESS);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLParamData
 *
 * DESCRIPTION: checks parameters ('p') or columns ('c') for SQL_DATA_AT_EXEC
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLParamData(SQLHSTMT StatementHandle, SQLPOINTER* Value)
{
	SQLRETURN nRet = SQL_SUCCESS;
	ENTER_STMT(StatementHandle, _T("SQLParamData"));
	{
		Descriptor* pApplDesc;
		switch (((Statement*)StatementHandle)->need_data)
		{
			case _T('p'):
				pApplDesc = GET_DESCRIPTOR(((Statement*)StatementHandle)->apd);
			case _T('c'):
				pApplDesc = GET_DESCRIPTOR(((Statement*)StatementHandle)->ard);
			default:
				pApplDesc = NULL;
		}

		if (NULL != pApplDesc)
		{/* check buffer */
			DataAtExec* daexec = &((Statement*)StatementHandle)->data_at_exec;
			if (0 < daexec->length - daexec->free_size)
			{/* application haven't finished with previous buffer */
				nRet = SQL_NEED_DATA;
			}
			else
			{
				SQLSMALLINT i;

				for (i=daexec->index;i>=0;i--) /* look for DATA_AT_EXEC */
					if (SQL_DATA_AT_EXEC == *pApplDesc->ad_records[i].indicator_ptr)
					{
						nRet = SQL_NEED_DATA;
						break;
					}

				if (SQL_NEED_DATA == nRet)
				{
					CD_REC* cd_rec = &pApplDesc->ad_records[i].common;
					if (SQL_C_DEFAULT == (daexec->c_type = cd_rec->concise_type))
					{/* convert SQL-type to default C-type */
						Descriptor* pImplDesc = GET_DESCRIPTOR(_T('p') == (((Statement*)StatementHandle)->need_data) ? ((Statement*)StatementHandle)->ipd : ((Statement*)StatementHandle)->ird);
						daexec->c_type = GetCDefaultType(pImplDesc->id_records[i].common.concise_type);
						RET_DESCRIPTOR(pImplDesc);
					}
					daexec->free_size = daexec->length = GetCTypeLength(daexec->c_type, cd_rec->octet_length);
					daexec->data      = (BYTE*)cd_rec->data_ptr;
					daexec->index     = i;
				}
			}
			RET_DESCRIPTOR(pApplDesc);
		}
	}
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLPrepare
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLPrepare_(SQLHSTMT StatementHandle, SQLTCHAR* StatementText, SQLINTEGER TextLength)
{
	SQLRETURN nRet, nRet1;

	ENTER_STMT(StatementHandle, _T("SQLPrepare"));
	if (SQL_ERROR != (nRet = ResetStatement((Statement*) StatementHandle)))
	{
		nRet1 = PrepareStatement((Statement*) StatementHandle, StatementText, TextLength);
		nRet = MAX_ERROR(nRet, nRet1);
	}
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLRowCount
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLRowCount(SQLHSTMT StatementHandle, SQLINTEGER* RowCount)
{
	SQLRETURN nRet;
	Statement* pStatement;

	ENTER_STMT(StatementHandle, _T("SQLRowCount"));
	pStatement = (Statement*) StatementHandle;
	if (NULL != RowCount)
	{
		pStatement = (Statement*) StatementHandle;
		if (0 != (pStatement->query.type & (ST_UPDATE | ST_INSERT | ST_DELETE)))
		{
			*RowCount = pStatement->ird->header.rows_affected;
			nRet = SQL_SUCCESS;
		}
		else
		{
			if (0 != (pStatement->query.type & ST_SELECT))
			{
				/* we can't know the number of rows resulted in select query when no buffer used */
				if (STMT_USE_BUFFERING(pStatement))
				{
					*RowCount = pStatement->ird->header.array_size;
					nRet = SQL_SUCCESS;
				}
				else
				{
					SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_ROW_COUNT_UNAVAIBLE, NULL);
					nRet = SQL_SUCCESS_WITH_INFO;
					*RowCount = -1;
				}
			}
			else
			{
				/* wrong statement's state */
				SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_INVALID_STATE, NULL);
				nRet = SQL_SUCCESS_WITH_INFO;
				*RowCount = -1;
			}
		}
	}
	else
	{
		/* null 'RowCount' pointer */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_NULL_POINTER_BUFFER, NULL);
		nRet = SQL_ERROR;
	}
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLSetConnectAttr
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetConnectAttr_(SQLHDBC ConnectionHandle, SQLINTEGER Attribute, SQLPOINTER Value, SQLINTEGER StringLength)
{
	SQLRETURN   nRet;
	ENTER_CONN(ConnectionHandle, _T("SQLSetConnectAttr"));
	nRet = SetConnectAttr((Connection*)ConnectionHandle, Attribute, Value, StringLength);
	LEAVE_CONN(ConnectionHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLSetCursorName
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetCursorName_(SQLHSTMT StatementHandle, SQLTCHAR* CursorName, SQLSMALLINT NameLength)
{
	SQLRETURN nRet;

	ENTER_STMT(StatementHandle, _T("SQLSetCursorName"));
	nRet = SetCursorName((Statement*)StatementHandle, CursorName, NameLength, FALSE);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLSetDescField
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetDescField_(SQLHDESC DescriptorHandle, SQLSMALLINT RecNumber, SQLSMALLINT FieldIdentifier,
                 SQLPOINTER Value, SQLINTEGER BufferLength)
{
	SQLRETURN nRet;

	ENTER_DESC(DescriptorHandle, _T("SQLSetDescField"));
	nRet = SetDescData((Descriptor*) DescriptorHandle, RecNumber, Value, BufferLength, FieldIdentifier, 0, 0, 0, NULL, NULL, SQL_FALSE);
	LEAVE_DESC(DescriptorHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLSetDescRec
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetDescRec(SQLHDESC DescriptorHandle, SQLSMALLINT RecNumber, SQLSMALLINT Type, SQLSMALLINT SubType,
              SQLINTEGER Length, SQLSMALLINT Precision, SQLSMALLINT Scale, SQLPOINTER DataPtr,
              SQLINTEGER* StringLength, SQLINTEGER* Indicator)
{
	SQLRETURN   nRet;

	ENTER_DESC(DescriptorHandle, _T("SQLSetDescRec"));
	nRet = SetDescData((Descriptor*) DescriptorHandle, RecNumber, DataPtr, Length, Type, SubType, Precision, Scale, StringLength, Indicator, SQL_TRUE);
	LEAVE_DESC(DescriptorHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLSetEnvAttr
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetEnvAttr(SQLHENV EnvironmentHandle, SQLINTEGER Attribute, SQLPOINTER Value,
              SQLINTEGER StringLength)
{
	SQLRETURN      nRet;
	EnvAttributes* attributes;

	ENTER_ENV(EnvironmentHandle, _T("SQLSetEnvAttr"));
	attributes = &((Environment*) EnvironmentHandle)->attributes;
	nRet = SQL_SUCCESS;
	switch(Attribute)
	{
		case SQL_ATTR_ODBC_VERSION:        /* 200 */
			attributes->odbc_version = (SQLUINTEGER) Value;
			break;
		case SQL_ATTR_CONNECTION_POOLING:  /* 201 */
			attributes->connection_pooling = (SQLUINTEGER) Value;
			break;
		case SQL_ATTR_CP_MATCH:            /* 202 */
			attributes->cp_match = (SQLUINTEGER) Value;
			break;
		/* Driver Manager deals himself with SQL_ATTR_OUTPUT_NTS */
		default:
			SetError(SQL_HANDLE_ENV, EnvironmentHandle, ERR_UNKNOWN_ATTRIBUTE, NULL);
			nRet = SQL_ERROR;
	}

	LEAVE_ENV(EnvironmentHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLSetStmtAttr
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetStmtAttr_(SQLHSTMT StatementHandle, SQLINTEGER Attribute, SQLPOINTER Value, SQLINTEGER StringLength)
{
	SQLRETURN   nRet       = SQL_SUCCESS;
	Statement*  pStatement = (Statement*)StatementHandle;

	ENTER_STMT(StatementHandle, _T("SQLSetStmtAttr"));
	switch(Attribute)
	{
		case SQL_ATTR_APP_ROW_DESC:         /* 10010 */
			nRet = SetApplicationDescriptor(pStatement, &pStatement->ard, &pStatement->d_ard, Value);
			break;
		case SQL_ATTR_APP_PARAM_DESC:       /* 10011 */
			nRet = SetApplicationDescriptor(pStatement, &pStatement->apd, &pStatement->d_apd, Value);
			break;
		case SQL_ATTR_NOSCAN:                   /* 2 */
			pStatement->attributes.no_scan = (SQLUINTEGER) Value;
			break;
		case SQL_ATTR_PARAMS_PROCESSED_PTR:    /* 21 */
			nRet = SetDescField(pStatement->ipd, 0, SQL_DESC_ROWS_PROCESSED_PTR, Value, StringLength);
			break;
		case SQL_ATTR_ROWS_FETCHED_PTR:        /* 26 */
			nRet = SetDescField(pStatement->ird, 0, SQL_DESC_ROWS_PROCESSED_PTR, Value, StringLength);
			break;
		/* number of rows returned by each call to SQLFetch or SQLFetchScroll */
		case SQL_ATTR_ROW_ARRAY_SIZE:          /* 27 */
		/* ODBC 2.0 */
		case SQL_ROWSET_SIZE:                   /* 9 */
			nRet = SetDescField(pStatement->ard, 0, SQL_DESC_ARRAY_SIZE, Value, StringLength);
			break;
		case SQL_ATTR_USE_BOOKMARKS:           /* 12 */
			pStatement->attributes.use_bookmarks = (SQLUINTEGER) Value;
			break;
		case SQL_ATTR_ENABLE_AUTO_IPD:         /* 15 */
			pStatement->attributes.enable_auto_ipd = (SQLUINTEGER) Value;
			break;


		/* binding orientation to be used when SQLFetch or SQLFetchScroll is called */
		case SQL_ATTR_ROW_BIND_TYPE:
			nRet = SetDescField(pStatement->ard, 0, SQL_DESC_BIND_TYPE, Value, StringLength);
			break;
		/* array of SQLUSMALLINT values containing row status values after a call to SQLFetch or SQLFetchScroll */
		case SQL_ATTR_ROW_STATUS_PTR:
			nRet = SetDescField(pStatement->ird, 0, SQL_DESC_ARRAY_STATUS_PTR, Value, StringLength);
			break;
		/* array of SQLUSMALLINT values used to ignore a parameter during execution of an SQL statement */
		case SQL_ATTR_PARAM_OPERATION_PTR:
			nRet = SetDescField(pStatement->apd, 0, SQL_DESC_ARRAY_STATUS_PTR, Value, StringLength);
			break;

		/* unknown or unsupported attribute */
		default:
			SetError(SQL_HANDLE_STMT, pStatement, ERR_UNKNOWN_ATTRIBUTE, NULL);
			nRet = SQL_FALSE;
	}

	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * ---------------------------- Depracated functions --------------------------
 *-----------------------------------------------------------------------------
 */

/*-----------------------------------------------------------------------------
 * SQLAllocConnect
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLAllocConnect(SQLHENV EnvironmentHandle, SQLHDBC* ConnectionHandle)
{
	SQLRETURN nRet;

	ENTER_ENV(EnvironmentHandle, _T("SQLAllocConnect"));
	nRet = (*ConnectionHandle = AllocConnection((Environment*) EnvironmentHandle)) ? SQL_SUCCESS : SQL_ERROR;
	LEAVE_ENV(EnvironmentHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * SQLAllocEnv
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLAllocEnv(SQLHENV *EnvironmentHandle)
{
	BEGIN(_T("SQLAllocEnv"));
	END((*EnvironmentHandle = AllocEnvironment()) ? SQL_SUCCESS : SQL_ERROR);
}

/*-----------------------------------------------------------------------------
 * SQLAllocStmt
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLAllocStmt(SQLHDBC ConnectionHandle, SQLHSTMT* StatementHandle)
{
	SQLRETURN nRet;

	ENTER_CONN(ConnectionHandle, _T("SQLAllocStmt"));
	nRet = (*StatementHandle = AllocStatement((Connection*) ConnectionHandle)) ? SQL_SUCCESS : SQL_ERROR;
	LEAVE_CONN(ConnectionHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * SQLBindParam
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLBindParam(SQLHSTMT StatementHandle, SQLUSMALLINT ParameterNumber,
						 SQLSMALLINT ValueType, SQLSMALLINT ParameterType,
						 SQLUINTEGER LengthPrecision, SQLSMALLINT ParameterScale, 
						 SQLPOINTER ParameterValue, SQLINTEGER *StrLen_or_Ind)
{
	ENTER_STMT(StatementHandle, _T("SQLBindParam"));
	LEAVE_STMT(StatementHandle, BindParameter((Statement*)StatementHandle, ParameterNumber, SQL_PARAM_INPUT, ValueType, ParameterType, LengthPrecision, ParameterScale, ParameterValue, 0, StrLen_or_Ind));
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLColAttributes
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLColAttributes_(SQLHSTMT StatementHandle, SQLUSMALLINT ColumnNumber, SQLUSMALLINT fDescType,
                  SQLPOINTER rgbDesc, SQLSMALLINT cbDescMax, SQLSMALLINT* pcbDesc,
                  SQLINTEGER* pfDesc)
{
	SQLRETURN nRet = SQL_SUCCESS;
	ENTER_STMT(StatementHandle, _T("SQLColAttributes"));
        
	switch(fDescType)
	{
		case SQL_COLUMN_LENGTH:              /* 3 */
		case SQL_COLUMN_SCALE:               /* 5 */
		{
			SQLINTEGER*  length = NULL;
			SQLSMALLINT* scale  = NULL;

			switch(fDescType)
			{
				case SQL_COLUMN_LENGTH:          /* 3 */
					length = (SQLINTEGER*)pfDesc;
					break;
				case SQL_COLUMN_SCALE:           /* 5 */
					scale = (SQLSMALLINT*)pfDesc;
					break;
			}
                        
			nRet = DescribeCol((Statement*)StatementHandle, ColumnNumber, NULL, 0, NULL, NULL, length, scale, NULL);
                        break;
		}
		case SQL_COLUMN_PRECISION:           /* 4 */
			fDescType = SQL_DESC_PRECISION; /* 1005 */
			break;
		case SQL_COLUMN_NAME:                /* 1 */
			fDescType = SQL_DESC_NAME;      /* 1011 */
			break;
		case SQL_COLUMN_NULLABLE:            /* 7 */
			fDescType = SQL_DESC_NULLABLE;  /* 1008 */
			break;
		case SQL_COLUMN_COUNT:               /* 0 */
			fDescType = SQL_DESC_COUNT;     /* 1001 */
			break;
	}

        if (SQL_SUCCESS != nRet)
        {
                nRet = ColAttribute((Statement*)StatementHandle, ColumnNumber, fDescType, rgbDesc, cbDescMax, pcbDesc, (SQLPOINTER) pfDesc);
                if ((SQL_OV_ODBC2 == ((Statement*)StatementHandle)->connection->environment->attributes.odbc_version) &&
                (SQL_COLUMN_TYPE == fDescType)
                )
                {
                }
        }
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * SQLError (depracated)
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLError_(SQLHENV EnvironmentHandle, SQLHDBC ConnectionHandle, SQLHSTMT StatementHandle,
          SQLTCHAR* Sqlstate, SQLINTEGER* NativeError,
          SQLTCHAR* MessageText, SQLSMALLINT BufferLength,
          SQLSMALLINT *TextLength)
{
	SQLRETURN    nRet;
	SQLHANDLE    handle;
	SQLSMALLINT  hType;
	SQLSMALLINT* rec;

	BEGIN(_T("SQLError"));
	if (NULL != EnvironmentHandle)
	{
		handle = EnvironmentHandle;
		hType  = SQL_HANDLE_ENV;
		rec = &((Environment*) EnvironmentHandle)->diag.i;
	}
	else if (NULL != ConnectionHandle)
	{
		handle = ConnectionHandle;
		hType  = SQL_HANDLE_DBC;
		rec = &((Connection*) ConnectionHandle)->diag.i;
	}
	else if (NULL != StatementHandle)
	{
		handle = StatementHandle;
		hType  = SQL_HANDLE_STMT;
		rec = &((Statement*) StatementHandle)->diag.i;
	}
	
	nRet = GetDiagData(hType, handle, ++(*rec), 0, Sqlstate, NativeError, MessageText, BufferLength, TextLength);
	if (SQL_NO_DATA == nRet)
		*rec = 0;

	END(nRet);
}

/*-----------------------------------------------------------------------------
 * SQLFreeConnect
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLFreeConnect(SQLHDBC ConnectionHandle)
{
	BEGIN(_T("SQLFreeConnect"));
	END(FreeConnection((Connection*) ConnectionHandle, 0));
}

/*-----------------------------------------------------------------------------
 * SQLFreeEnv
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLFreeEnv(SQLHENV EnvironmentHandle)
{
	BEGIN(_T("SQLFreeEnv"));
	END(FreeEnvironment((Environment*) EnvironmentHandle));
}

/*-----------------------------------------------------------------------------
 * SQLFreeStmt
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLFreeStmt(SQLHSTMT StatementHandle, SQLUSMALLINT Option)
{
	BEGIN(_T("SQLFreeStmt"));
	END(FreeStatement((Statement*) StatementHandle, Option));
}

/*-----------------------------------------------------------------------------
 * SQLGetConnectOption
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLGetConnectOption_(SQLHDBC ConnectionHandle, SQLUSMALLINT Option, SQLPOINTER Value)
{
	ENTER_CONN(ConnectionHandle, _T("SQLGetConnectOption"));
	LEAVE_CONN(ConnectionHandle, SQL_ERROR);
}

/*-----------------------------------------------------------------------------
 * SQLGetStmtOption
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLGetStmtOption(SQLHSTMT StatementHandle, SQLUSMALLINT Option, SQLPOINTER Value)
{
	ENTER_STMT(StatementHandle, _T("SQLGetStmtOption"));
	LEAVE_STMT(StatementHandle, SQL_ERROR);
}

/*-----------------------------------------------------------------------------
 * SQLSetConnectOption
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetConnectOption_(SQLHDBC ConnectionHandle, SQLUSMALLINT Option, SQLUINTEGER Value)
{
	SQLRETURN nRet;
	ENTER_CONN(ConnectionHandle, _T("SQLSetConnectOption"));
	nRet = SetConnectAttr((Connection*) ConnectionHandle, Option, (SQLPOINTER)Value, SQL_NTS);
	LEAVE_CONN(ConnectionHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * SQLSetScrollOptions
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetScrollOptions(SQLHSTMT StatementHandle, SQLUSMALLINT fConcurrency, SQLINTEGER crowKeyset, SQLUSMALLINT crowRowset)
{
	ENTER_STMT(StatementHandle, _T("SQLSetScrollOptions"));
	LEAVE_STMT(StatementHandle, SQL_ERROR);
}

/*-----------------------------------------------------------------------------
 * SQLSetStmtOption
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetStmtOption(SQLHSTMT StatementHandle, SQLUSMALLINT Option, SQLUINTEGER Value)
{
	ENTER_STMT(StatementHandle, _T("SQLSetStmtOption"));
	LEAVE_STMT(StatementHandle, SQL_ERROR);
}

/*-----------------------------------------------------------------------------
 * SQLSetParam
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetParam(SQLHSTMT StatementHandle, SQLUSMALLINT ParameterNumber, SQLSMALLINT ValueType,
            SQLSMALLINT ParameterType, SQLUINTEGER LengthPrecision, SQLSMALLINT ParameterScale,
						SQLPOINTER ParameterValue, SQLINTEGER* StrLen_or_Ind)
{
	SQLRETURN nRet;

	ENTER_STMT(StatementHandle, _T("SQLSetParam"));
	nRet = BindParameter((Statement*)StatementHandle, ParameterNumber, SQL_PARAM_INPUT_OUTPUT, ValueType,
	       ParameterType, LengthPrecision, ParameterScale, ParameterValue, SQL_SETPARAM_VALUE_MAX, StrLen_or_Ind);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * SQLTransact
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLTransact(SQLHENV EnvironmentHandle, SQLHDBC ConnectionHandle, SQLUSMALLINT CompletionType)
{
	SQLHANDLE handle;
	SQLSMALLINT handleType;

	BEGIN(_T("SQLTransact"));
	if (NULL != ConnectionHandle)
	{
		handleType = SQL_HANDLE_DBC;
		handle = ConnectionHandle;
	}
	else
	{
		handleType = SQL_HANDLE_ENV;
		handle = EnvironmentHandle;	
	}

	END(EndTransaction(handleType, handle, CompletionType, TO_APPLICATION));
}

/*-----------------------------------------------------------------------------
 * SQLExtendedFetch
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLExtendedFetch(SQLHSTMT StatementHandle, SQLUSMALLINT FetchType, SQLINTEGER Row, SQLUINTEGER* pcrow, SQLUSMALLINT* rgfRowStatus)
{
	SQLRETURN nRet;
	ENTER_STMT(StatementHandle, _T("SQLExtendedFetch"));
	nRet = SQL_ERROR;
	LEAVE_STMT(StatementHandle, nRet);
}




/* SQLParamOptions */