#include "statement.h"
#include "descriptor.h"
#include "protocol.h"

/*-----------------------------------------------------------------------------
 * FUNCTION: AllocDescriptor
 *-----------------------------------------------------------------------------
 */
Descriptor* AllocDescriptor(Connection* pConnection)
{
	Descriptor* pDescriptor = (Descriptor*)malloc(sizeof(Descriptor));
	if (pDescriptor)
	{
		pDescriptor->connection = pConnection;
		pDescriptor->type       = SQL_DEFAULT;

		InitDescriptor(pDescriptor, SQL_DESC_ALLOC_USER);
	}

	return pDescriptor;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: FreeDescriptor
 *-----------------------------------------------------------------------------
 */
SQLRETURN FreeDescriptor(Descriptor* pDescriptor)
{
	_DELETE_CRITICAL_SECTION(pDescriptor);

	free(pDescriptor->records);
	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: EmptyDescriptor
 *-----------------------------------------------------------------------------
 */
void InitDescriptor(Descriptor* pDescriptor, SQLSMALLINT alloc_type)
{
	/* empty header */
	pDescriptor->header.alloc_type         = alloc_type;
	pDescriptor->header.bind_type          = SQL_BIND_BY_COLUMN;  /* implicitly or explicitly */
	pDescriptor->header.rows_processed_ptr = NULL;                /* implementation descriptors */
	pDescriptor->header.array_status_ptr   = NULL;
	pDescriptor->header.bind_offset_ptr    = NULL;                /* application descriptors */
	pDescriptor->records                   = NULL;
	pDescriptor->header.count              = 0;
	pDescriptor->header.array_size         = 1;                   /* application descriptors */
	pDescriptor->header.query_type         = 0;
	pDescriptor->header.rows_affected      = 0;
	pDescriptor->allocated                 = 0;

	InitDiag(&pDescriptor->diag);
	_INIT_CRITICAL_SECTION(pDescriptor);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: EmptyDescriptorRecord
 *
 * WARNING: here used native indexing (0-based)
 *-----------------------------------------------------------------------------
 */
SQLRETURN EmptyDescriptorRecord(Descriptor* pDescriptor, SQLSMALLINT number)
{
	/* record is in range? */
	if (number < pDescriptor->header.count)
	{
		AD_REC* pAD_REC;
		ID_REC* pID_REC;
		CD_REC* pCD_REC;

		if (0 != (DT_AD & pDescriptor->type))
		{
			/* application descriptor */
			pAD_REC = &pDescriptor->ad_records[number];
			pCD_REC = &pAD_REC->common;
			pAD_REC->indicator_ptr    = NULL;
			pAD_REC->octet_length_ptr = NULL;
			pAD_REC->bound            = SQL_FALSE;
		}
		else
		{
			if (0 != (DT_ID & pDescriptor->type))
			{
				/* implementation descriptor */
				pID_REC = &pDescriptor->id_records[number];
				pCD_REC = &pID_REC->common;
				pID_REC->parameter_type = SQL_PARAM_INPUT;
				pID_REC->unnamed        = SQL_UNNAMED;
				pID_REC->max_column_length = 0;

				if (0    != (DT_IPD & pDescriptor->type) &&
				    NULL != pCD_REC->data_ptr
				   )
					free (pCD_REC->data_ptr);
			}
			else
			{
				/* undefined type - impossible situation */
				return SQL_ERROR;
			}
		}

		/* set default values to common fields */
		pCD_REC->concise_type = SQL_C_DEFAULT;
		pCD_REC->type         = SQL_C_DEFAULT;
		pCD_REC->data_ptr     = NULL;
	}

	return SQL_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: ReallocDescriptorRecords
 *-----------------------------------------------------------------------------
 */
SQLRETURN ReallocDescriptorRecords(Descriptor* pDescriptor, SQLSMALLINT number)
{
	if (pDescriptor->header.count < number)
	{
		SQLSMALLINT i;

		if (pDescriptor->allocated < number)
		{
			int   nSizeOfRecord;
			void* pRecords;

			if (0 != (DT_AD & pDescriptor->type))
				nSizeOfRecord = sizeof(AD_REC);
			else
				if (0 != (DT_ID & pDescriptor->type))
					nSizeOfRecord = sizeof(ID_REC);
				else
				{
					/* andyk - now we can't determine descriptor's state */
					return SQL_ERROR;
				}

			/* we need to realloc memory for new records */
			if (pRecords = calloc(number + 10, nSizeOfRecord))
			{
				memcpy(pRecords, pDescriptor->records, pDescriptor->header.count*nSizeOfRecord);
				free(pDescriptor->records);
				pDescriptor->allocated = number;
				pDescriptor->records = pRecords;				
			}
			else
			{
				/* impossible to allocate enough memory */
				SetError(SQL_HANDLE_DESC, pDescriptor, ERR_NOT_ENOUGH_MEMORY, NULL);
				return SQL_ERROR;
			}
		}

		/* empty newly requested records (WARNING! 0-based indexing in function) */
		i = pDescriptor->header.count;
		pDescriptor->header.count = number;
		for(/* i set above */;i<number;i++)
			EmptyDescriptorRecord(pDescriptor, i);
	}

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetDescField
 *-----------------------------------------------------------------------------
 */
SQLRETURN GetDescField(Descriptor* pDescriptor, SQLSMALLINT RecNumber, SQLSMALLINT FieldIdentifier,
                       SQLPOINTER Value, SQLINTEGER BufferLength, SQLSMALLINT* StringLength,
                       Statement* pStatement)
{
	SQLRETURN       nRet       = SQL_SUCCESS;
	SQLSMALLINT     type       = pDescriptor->type;
	SQLSMALLINT     processed  = SQL_TRUE;
	const SQLTCHAR* pRetString = NULL;

	SQLSMALLINT uSmallInt;
	SQLINTEGER  uInteger;
	SQLSMALLINT ret_type;
	SQLPOINTER  pointer;


	switch (FieldIdentifier)
	{
		/*-------------------------------------------------------------------------
		 * header fields
		 *-------------------------------------------------------------------------
		 */
		case SQL_DESC_ARRAY_SIZE:         /* 20 */
			uInteger = (0 != (DT_AD & type)) ? pDescriptor->header.array_size : 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DESC_ARRAY_STATUS_PTR:   /* 21 */
			pointer  = pDescriptor->header.array_status_ptr;
			ret_type = SQL_IS_POINTER;
			break;
		case SQL_DESC_BIND_OFFSET_PTR:    /* 24 */
			pointer  = (0 != (DT_AD & type)) ? pDescriptor->header.bind_offset_ptr : NULL;
			ret_type = SQL_IS_POINTER;
			break;
		case SQL_DESC_BIND_TYPE:          /* 25 */
			uInteger = (0 != (DT_AD & type)) ? pDescriptor->header.bind_type : 0;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DESC_ROWS_PROCESSED_PTR: /* 34 */
			pointer  = (0 != (DT_ID & type)) ? pDescriptor->header.rows_processed_ptr : NULL;
			ret_type = SQL_IS_POINTER;
			break;
		case SQL_DESC_COUNT:            /* 1001 */
			uSmallInt = pDescriptor->header.count;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_DESC_ALLOC_TYPE:       /* 1099 */
			uSmallInt = pDescriptor->header.alloc_type;
			ret_type  = SQL_IS_SMALLINT;
			break;
		/*-------------------------------------------------------------------------
		 * constant or not implemented fields
		 *-------------------------------------------------------------------------
		 */
		case SQL_DESC_UPDATABLE:          /* 10 */
			/* now driver can't determine is the row updatable */
			uSmallInt = (DT_IRD == type) ? SQL_ATTR_READONLY : SQL_ATTR_READONLY;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_DESC_AUTO_UNIQUE_VALUE:  /* 11 */
			/* now driver can't determine is the column auto-incrementing */
			uInteger = (DT_IRD == type) ? SQL_FALSE : SQL_FALSE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DESC_CASE_SENSITIVE:     /* 12 */
			/* now driver can't determine is the column case sensitive */
			uInteger = (DT_IRD == type) ? SQL_TRUE : SQL_TRUE;
			ret_type = SQL_IS_INTEGER;
			break;
		case SQL_DESC_SEARCHABLE:         /* 13 */
			/* assume that every column is searchable */
			uSmallInt = (DT_IRD == type) ? SQL_PRED_SEARCHABLE : SQL_PRED_NONE;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_DESC_ROWVER:             /* 35 */
			/* now driver can't determine is the column auto modified by DBMS */
			uSmallInt = (0 != (DT_ID & type)) ? SQL_FALSE : SQL_FALSE;
			ret_type  = SQL_IS_SMALLINT;
			break;
		case SQL_DESC_NULLABLE:         /* 1008 */
			/* now driver can't determine is the column or parameter nullable */
			uSmallInt = (DT_IPD == type) ? SQL_NULLABLE : SQL_NULLABLE_UNKNOWN;
			ret_type  = SQL_IS_SMALLINT;
			break;
		default:
			processed = SQL_FALSE;
			if (0 == RecNumber)
			{
				/* bookmark column */
				if ((DT_IPD == type) || ((DT_APD == type) && (SQL_DESC_ALLOC_AUTO == pDescriptor->header.alloc_type)))
				{
					SetError(SQL_HANDLE_DESC, pDescriptor, ERR_DESC_READ_ZERO_REC_IN_PD, NULL);
					return SQL_ERROR;
				}

				switch (FieldIdentifier)
				{
					/*-------------------------------------------------------------------------
					 * pointer fields
					 *-------------------------------------------------------------------------
					 */
					case SQL_DESC_OCTET_LENGTH_PTR: /* 1004 */
						pointer  = NULL;
						ret_type = SQL_IS_POINTER;
						break;
					case SQL_DESC_INDICATOR_PTR:    /* 1009 */
						pointer  = pDescriptor->bookmark.indicator_ptr;
						ret_type = SQL_IS_POINTER;
						break;
					case SQL_DESC_DATA_PTR:         /* 1010 */
						pointer  = pDescriptor->bookmark.data_ptr;
						ret_type = SQL_IS_POINTER;
						break;

					/*-------------------------------------------------------------------------
					 * numeric fields
					 *-------------------------------------------------------------------------
					 */
					case SQL_DESC_CONCISE_TYPE:        /* 2 */
					case SQL_DESC_TYPE:             /* 1002 */
						uSmallInt = pDescriptor->bookmark.type;
						ret_type  = SQL_IS_SMALLINT;
						break;
					case SQL_DESC_DISPLAY_SIZE:        /* 6 */
					case SQL_DESC_LENGTH:           /* 1003 */
					case SQL_DESC_PRECISION:        /* 1005 */
					case SQL_DESC_OCTET_LENGTH:     /* 1013 */
						uInteger = (DT_IRD == type) ? pDescriptor->id_records[RecNumber].display_size : 0;
						ret_type = SQL_IS_INTEGER;
						break;
					case SQL_DESC_UNSIGNED:            /* 8 */
						uSmallInt = SQL_TRUE;
						ret_type  = SQL_IS_SMALLINT;
						break;
					case SQL_DESC_NUM_PREC_RADIX:     /* 32 */
						uInteger = 10;
						ret_type = SQL_IS_INTEGER;
						break;
					case SQL_DESC_PARAMETER_TYPE:     /* 33 */
						uSmallInt = SQL_PARAM_INPUT_OUTPUT;
						ret_type  = SQL_IS_SMALLINT;
						break;
					case SQL_DESC_FIXED_PREC_SCALE:    /* 9 */
					case SQL_DESC_DATETIME_INTERVAL_PRECISION: /* 26 */
					case SQL_DESC_SCALE:            /* 1006 */
					case SQL_DESC_DATETIME_INTERVAL_CODE: /* 1007 */
						uSmallInt = 0;
						ret_type  = SQL_IS_SMALLINT;
						break;
					case SQL_DESC_UNNAMED:          /* 1012 */
						uSmallInt = SQL_NAMED;
						ret_type  = SQL_IS_SMALLINT;
						break;

					/*-------------------------------------------------------------------------	
					 * string fields
					 *-------------------------------------------------------------------------
					 */
					case SQL_DESC_TYPE_NAME:        /* 14 */
					case SQL_DESC_LABEL:            /* 18 */
					case SQL_DESC_LOCAL_TYPE_NAME:  /* 29 */
					case SQL_DESC_NAME:           /* 1011 */
						pRetString = c_szBookmarkName;
						break;
					case SQL_DESC_CATALOG_NAME:     /* 17 */
						/* PostgreSQL 8.0.3 uses connected database name for catalog name */
						pRetString = pDescriptor->connection->parameters[DATABASE_PARAM];
						break;
					case SQL_DESC_LITERAL_PREFIX:   /* 27 */
					case SQL_DESC_LITERAL_SUFFIX:   /* 28 */
					case SQL_DESC_TABLE_NAME:       /* 15 */
					case SQL_DESC_SCHEMA_NAME:      /* 16 */
					case SQL_DESC_BASE_COLUMN_NAME: /* 22 */
					case SQL_DESC_BASE_TABLE_NAME:  /* 23 */
						pRetString = _T("");
						break;

					/*-------------------------------------------------------------------------	
					 * unknown field
					 *-------------------------------------------------------------------------
					 */
					default:
						SetError(SQL_HANDLE_DESC, pDescriptor, ERR_DESC_UNKNOWN_FIELD, NULL);
						return SQL_ERROR;
				}
				processed = SQL_TRUE;
			}
			else
				RecNumber--;
	}

	if (SQL_FALSE == processed)
		switch (FieldIdentifier)
		{
			/*-------------------------------------------------------------------------
			 * pointer fields
			 *-------------------------------------------------------------------------
			 */
			case SQL_DESC_OCTET_LENGTH_PTR: /* 1004 */
				pointer  = (0 != (DT_AD & type)) ? pDescriptor->ad_records[RecNumber].octet_length_ptr : NULL;
				ret_type = SQL_IS_POINTER;
				break;
			case SQL_DESC_INDICATOR_PTR:    /* 1009 */
				pointer  = (0 != (DT_AD & type)) ? pDescriptor->ad_records[RecNumber].indicator_ptr : NULL;
				ret_type = SQL_IS_POINTER;
				break;
			case SQL_DESC_DATA_PTR:         /* 1010 */
				pointer  = (0 != (DT_AD & type)) ? pDescriptor->ad_records[RecNumber].common.data_ptr : ((DT_IPD == type) ? pDescriptor->id_records[RecNumber].common.data_ptr : NULL);
				ret_type = SQL_IS_POINTER;
				break;
		
			/*-------------------------------------------------------------------------
			 * numeric fields
			 *-------------------------------------------------------------------------
			 */
			case SQL_DESC_CONCISE_TYPE:        /* 2 */
				uSmallInt = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].common.concise_type : pDescriptor->ad_records[RecNumber].common.concise_type;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_DISPLAY_SIZE:        /* 6 */
				uInteger = (DT_IRD == type) ? pDescriptor->id_records[RecNumber].display_size : 0;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_DESC_UNSIGNED:            /* 8 */
				uSmallInt = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].is_unsigned : SQL_FALSE;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_FIXED_PREC_SCALE:    /* 9 */
				uSmallInt = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].fixed_prec_scale : SQL_FALSE;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_DATETIME_INTERVAL_PRECISION: /* 26 */
				uInteger = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].common.datetime_interval_precision : pDescriptor->ad_records[RecNumber].common.datetime_interval_precision;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_DESC_NUM_PREC_RADIX:     /* 32 */
				uInteger = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].common.num_prec_radix : pDescriptor->ad_records[RecNumber].common.num_prec_radix;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_DESC_PARAMETER_TYPE:     /* 33 */
				uSmallInt = (DT_IPD == type) ? pDescriptor->id_records[RecNumber].parameter_type : SQL_PARAM_INPUT_OUTPUT;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_TYPE:             /* 1002 */
				uSmallInt = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].common.type : pDescriptor->ad_records[RecNumber].common.type;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_LENGTH:           /* 1003 */
				uInteger = (0 != (DT_AD & type)) ? pDescriptor->ad_records[RecNumber].common.length : pDescriptor->id_records[RecNumber].common.length;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_DESC_PRECISION:        /* 1005 */
				uSmallInt = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].common.precision : pDescriptor->ad_records[RecNumber].common.precision;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_SCALE:            /* 1006 */
				uSmallInt = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].common.scale : pDescriptor->ad_records[RecNumber].common.scale;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_DATETIME_INTERVAL_CODE: /* 1007 */
				uSmallInt = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].common.datetime_interval_code : pDescriptor->ad_records[RecNumber].common.datetime_interval_code;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_UNNAMED:          /* 1012 */
				uSmallInt = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].unnamed : SQL_DESC_UNNAMED;
				ret_type  = SQL_IS_SMALLINT;
				break;
			case SQL_DESC_OCTET_LENGTH:     /* 1013 */
				uInteger = (0 != (DT_ID & type)) ? pDescriptor->id_records[RecNumber].common.octet_length : pDescriptor->ad_records[RecNumber].common.octet_length;
				ret_type = SQL_IS_INTEGER;
				break;
			/*-------------------------------------------------------------------------	
			 * string fields
			 *-------------------------------------------------------------------------
			 */
			case SQL_DESC_TYPE_NAME:        /* 14 */
			case SQL_DESC_LOCAL_TYPE_NAME:  /* 29 */
				pRetString = ((0 != (DT_ID & type)) && (NULL != pDescriptor->id_records[RecNumber].type_name)) ? pDescriptor->id_records[RecNumber].type_name : _T("");
				break;
			case SQL_DESC_CATALOG_NAME:     /* 17 */
				/* PostgreSQL 8.0.3 uses connected database name for catalog name */
				pRetString = (DT_IRD == type) ? pDescriptor->connection->parameters[DATABASE_PARAM] : _T("");
				break;
			case SQL_DESC_LABEL:            /* 18 */
				/* driver doesn't support column labels - return column name instead */
				pRetString = ((DT_IRD == type) && (NULL != pDescriptor->id_records[RecNumber].name)) ? pDescriptor->id_records[RecNumber].name : _T("");
				break;
			case SQL_DESC_LITERAL_PREFIX:   /* 27 */
			case SQL_DESC_LITERAL_SUFFIX:   /* 28 */
				pRetString = (DT_IRD == type) ? _T("") : _T("");
				break;
			case SQL_DESC_NAME:           /* 1011 */
				pRetString = ((0 != (DT_ID & type)) && (NULL != pDescriptor->id_records[RecNumber].name)) ? pDescriptor->id_records[RecNumber].name : _T("");
				break;
			case SQL_DESC_TABLE_NAME:       /* 15 */
			case SQL_DESC_SCHEMA_NAME:      /* 16 */
			case SQL_DESC_BASE_COLUMN_NAME: /* 22 */
			case SQL_DESC_BASE_TABLE_NAME:  /* 23 */
			{
				TCHAR* pQuery;
				TCHAR  table_oid[SQLSMALLINT_LENGTH];
				TCHAR  column_attribute[SQLSMALLINT_LENGTH];

				_itot(pDescriptor->id_records[RecNumber].table_oid,        table_oid,        10);
				_itot(pDescriptor->id_records[RecNumber].column_attribute, column_attribute, 10);

				if (pQuery = GetText(_T("SELECT n.nspname,c.relname,a.attname,a.attnum FROM pg_catalog.pg_namespace n,pg_catalog.pg_class c,pg_catalog.pg_attribute a WHERE c.oid=a.attrelid AND n.oid=c.relnamespace AND c.oid=? AND a.attnum=?"), table_oid, column_attribute, NULL));
				{/* query successfully prepared - allocate new statement and execute it */
					Statement* pQueryStmt = AllocStatement(pDescriptor->connection);
					if (pQueryStmt)
					{
						SQLSMALLINT column;
						SQLINTEGER  res_length;
						PrepareQuery(pQueryStmt, pQuery, SQL_NTS);
						ExecuteStatement(pQueryStmt, SQL_TRUE);

						/* bind corresponding buffer */
						switch(FieldIdentifier)
						{
							case SQL_DESC_SCHEMA_NAME:      /* 16 */
								column = 1;
								break;
							case SQL_DESC_TABLE_NAME:       /* 15 */
							case SQL_DESC_BASE_TABLE_NAME:  /* 23 */
								column = 2;
								break;
							case SQL_DESC_BASE_COLUMN_NAME: /* 22 */
								column = 3;
								break;
						}

						BindCol(pQueryStmt, column, SQL_C_TCHAR, Value, BufferLength, &res_length);						
						/* fetch requested data */
						if (SQL_SUCCEEDED(Fetch(pQueryStmt, SQL_FETCH_FIRST, 0)) && StringLength)
							*StringLength = (SQLSMALLINT)res_length;

						FreeStatement(pQueryStmt, SQL_DROP);
					}

					free(pQuery);
				}
				return SQL_SUCCESS;
			}
			/*-------------------------------------------------------------------------	
			 * unknown field
			 *-------------------------------------------------------------------------
			 */
			default:
				SetError(SQL_HANDLE_DESC, pDescriptor, ERR_DESC_UNKNOWN_FIELD, NULL);
				return SQL_ERROR;
		}


	/* return string value */
	if ((NULL != pRetString) &&
	    (SQL_SUCCESS != (nRet = ReturnString(Value, (SQLSMALLINT) BufferLength, StringLength, pRetString, SQL_NTS)))
	   )
	{
		if (NULL == pStatement)
			SetError(SQL_HANDLE_DESC, pDescriptor, ERR_TOO_SMALL_BUFFER, _T("Value"), NULL);
		else
			SetError(SQL_HANDLE_STMT, pStatement, ERR_TOO_SMALL_BUFFER, _T("Value"), NULL);
	}
	else switch (ret_type)
	{
		case SQL_IS_SMALLINT:
			memcpy(Value, &uSmallInt, sizeof(uSmallInt));
			break;
		case SQL_IS_INTEGER:
			memcpy(Value, &uInteger, sizeof(uInteger));
			break;
		case SQL_IS_POINTER:
			memcpy(Value, &pointer, sizeof(pointer));
			break;
	}

	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SetDescData
 *
 * DESCRIPTION: record = SQL_TRUE means - called by SQLSetDescRec, else - by SQLSetDescField
 *-----------------------------------------------------------------------------
 */
SQLRETURN SetDescData(Descriptor* pDescriptor, SQLSMALLINT RecNumber, SQLPOINTER Value, SQLINTEGER Length,
                      SQLSMALLINT Type, SQLSMALLINT SubType, SQLSMALLINT Precision, SQLSMALLINT Scale,
                      SQLINTEGER* StringLength, SQLINTEGER* Indicator, BOOL record)
{
	if ((DT_IRD == pDescriptor->type) && 
	    ((SQL_TRUE == record) || ((SQL_DESC_ARRAY_STATUS_PTR != Type) && (SQL_DESC_ROWS_PROCESSED_PTR != Type)))
	   )
	{
		/* it's impossible to modify IRD */
		SetError(SQL_HANDLE_DESC, pDescriptor, ERR_DESC_CANT_MODIFY_IRD, NULL);
		return SQL_ERROR;
	}
	else if (SQL_ERROR == CheckDescriptorConsistency(pDescriptor, RecNumber))
	{
		/* inconsistent descriptor information */
		SetError(SQL_HANDLE_DESC, pDescriptor, ERR_DESC_INCONSISTENT_INFO, NULL);
		return SQL_ERROR;
	}
	else if (0 == RecNumber)
	{
		/* bookmarks */
		if ((DT_IPD == pDescriptor->type) ||
		    (DT_APD == pDescriptor->type && SQL_DESC_ALLOC_AUTO == pDescriptor->header.alloc_type)
		   )
		{
			/* it's impossible to set 0 record in parameter descriptor */
			SetError(SQL_HANDLE_DESC, pDescriptor, ERR_DESC_MODIFY_ZERO_REC_IN_PD, NULL);
			return SQL_ERROR;
		}
		else
		{
			Bookmark* pBookmark = &pDescriptor->bookmark;

			if (SQL_TRUE == record)
			{
				/* SQLSetDescRec */
				pBookmark->data_ptr      = Value;
				pBookmark->indicator_ptr = Indicator;
				pBookmark->octet_length  = Length;
				pBookmark->type          = Type;
			}
			else
			{
				/* SQLSetDescField */
				switch(Type)
				{
					case SQL_DESC_DATA_PTR:
					case SQL_DESC_INDICATOR_PTR:
					case SQL_DESC_OCTET_LENGTH:
					case SQL_DESC_TYPE:
						break;
					/* driver can't set other bookmarks' fields */
				}
			}
		}
		return SQL_SUCCESS;
	}
	else if (SQL_ERROR == ReallocDescriptorRecords(pDescriptor, RecNumber))
	{
		/* impossible to allocate memory for requested RecNumber */
		SetError(SQL_HANDLE_DESC, pDescriptor, ERR_NOT_ENOUGH_MEMORY, NULL);
		return SQL_ERROR;
	}
	else
	{
		/* ready to set fields */
		if (SQL_TRUE == record)
		{
			/* SQLSetDescRec */
			CD_REC* pCD_REC;

			RecNumber--;
			if (0 != (DT_AD & pDescriptor->type))
			{
				pCD_REC = &pDescriptor->ad_records[RecNumber].common;
				pDescriptor->ad_records[RecNumber].indicator_ptr    = Indicator;
				pDescriptor->ad_records[RecNumber].octet_length_ptr = StringLength;
			}
			else
				pCD_REC = &pDescriptor->id_records[RecNumber].common;

			if (SQL_DATETIME == Type || SQL_INTERVAL == Type)
				pCD_REC->datetime_interval_code = SubType;
			pCD_REC->precision    = Precision;
			pCD_REC->octet_length = Length;
			pCD_REC->data_ptr     = Value;
			pCD_REC->scale        = Scale;
			pCD_REC->type         = Type;
			return SQL_SUCCESS;
		}
		else
		{
			/* SQLSetDescField */
			return SetDescField(pDescriptor, RecNumber, Type, Value, Length);
		}
	}
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SetDescField
 *-----------------------------------------------------------------------------
 */
SQLRETURN SetDescField(Descriptor* pDescriptor, SQLSMALLINT RecNumber, SQLSMALLINT FieldIdentifier,
                       SQLPOINTER Value, SQLINTEGER BufferLength)
{
	SQLSMALLINT type       = pDescriptor->type;
	SQLSMALLINT processed  = SQL_TRUE;
	SQLCHAR*    pRetString = NULL;
	ERR         error      = ERR_CLEAR;

	switch (FieldIdentifier)
	{
		/*-------------------------------------------------------------------------
		 * header fields
		 *-------------------------------------------------------------------------
		 */
		case SQL_DESC_ARRAY_SIZE:         /* 20 */
			if (0 != (DT_AD & type))
				pDescriptor->header.array_size = (SQLUINTEGER) Value;
			else
				error = ERR_DESC_INVALID_FIELD;
			break;
		case SQL_DESC_ARRAY_STATUS_PTR:   /* 21 */
			pDescriptor->header.array_status_ptr = (SQLUSMALLINT*) Value;
			break;
		case SQL_DESC_BIND_OFFSET_PTR:    /* 24 */
			if (0 != (DT_AD & type))
				pDescriptor->header.bind_offset_ptr = (SQLINTEGER*) Value;
			else
				error = ERR_DESC_INVALID_FIELD;
			break;
		case SQL_DESC_BIND_TYPE:          /* 25 */
			if (0 != (DT_AD & type))
				pDescriptor->header.bind_type = (SQLUINTEGER) Value;
			else
				error = ERR_DESC_INVALID_FIELD;
			break;
		case SQL_DESC_ROWS_PROCESSED_PTR: /* 34 */
			if (0 != (DT_ID & type))
				pDescriptor->header.rows_processed_ptr = (SQLUINTEGER*) Value;
			else
				error = ERR_DESC_INVALID_FIELD;
			break;
		case SQL_DESC_COUNT:            /* 1001 */
			if (pDescriptor->header.count < (SQLSMALLINT) Value)
				return ReallocDescriptorRecords(pDescriptor, (SQLSMALLINT) Value);
			else
			{
				if (DT_IRD == pDescriptor->type)
				{
					int i = pDescriptor->header.count-1;
					for (;i>=0;i--)
					{
						free(pDescriptor->id_records[i].name);
						pDescriptor->id_records[i].name = NULL;
					}
				}
				pDescriptor->header.count = (SQLSMALLINT) Value;
			}
			break;

		/*-------------------------------------------------------------------------
		 * read-only fields
		 *-------------------------------------------------------------------------
		*/
		case SQL_DESC_DISPLAY_SIZE:        /* 6 */
		case SQL_DESC_UNSIGNED:            /* 8 */
		case SQL_DESC_FIXED_PREC_SCALE:    /* 9 */
		case SQL_DESC_UPDATABLE:          /* 10 */
		case SQL_DESC_AUTO_UNIQUE_VALUE:  /* 11 */
		case SQL_DESC_CASE_SENSITIVE:     /* 12 */
		case SQL_DESC_SEARCHABLE:         /* 13 */
		case SQL_DESC_TYPE_NAME:          /* 14 */
		case SQL_DESC_TABLE_NAME:         /* 15 */
		case SQL_DESC_SCHEMA_NAME:        /* 16 */
		case SQL_DESC_CATALOG_NAME:       /* 17 */
		case SQL_DESC_LABEL:              /* 18 */
		case SQL_DESC_BASE_COLUMN_NAME:   /* 22 */
		case SQL_DESC_BASE_TABLE_NAME:    /* 23 */
		case SQL_DESC_LITERAL_PREFIX:     /* 27 */
		case SQL_DESC_LITERAL_SUFFIX:     /* 28 */
		case SQL_DESC_LOCAL_TYPE_NAME:    /* 29 */
		case SQL_DESC_ROWVER:             /* 35 */
		case SQL_DESC_NULLABLE:         /* 1008 */
		case SQL_DESC_ALLOC_TYPE:       /* 1099 */
			error = ERR_DESC_READ_ONLY_FIELD;
			break;
		default:
			processed = SQL_FALSE;
			if (0 == RecNumber)
			{
				/* bookmark column */
				if ((DT_IPD == type) || ((DT_APD == type) && (SQL_DESC_ALLOC_AUTO == pDescriptor->header.alloc_type)))
					error = ERR_DESC_READ_ZERO_REC_IN_PD;
			}
			else
				RecNumber--;
	}

	if (SQL_FALSE == processed)
	{
		switch (FieldIdentifier)
		{
			/*-------------------------------------------------------------------------
			 * non-common fields
			 *-------------------------------------------------------------------------
			 */
			case SQL_DESC_PARAMETER_TYPE:     /* 33 */
				if (DT_IPD == type)
					pDescriptor->id_records[RecNumber].parameter_type = (SQLSMALLINT) Value;
				else
					error = ERR_DESC_INVALID_FIELD;
				break;
			case SQL_DESC_OCTET_LENGTH_PTR: /* 1004 */
				if (0 != (DT_AD & type))
					pDescriptor->ad_records[RecNumber].octet_length_ptr = (SQLINTEGER*) Value;
				else
					error = ERR_DESC_INVALID_FIELD;
				break;
			case SQL_DESC_INDICATOR_PTR:    /* 1009 */
				if (0 != (DT_AD & type))
					pDescriptor->ad_records[RecNumber].indicator_ptr = (SQLINTEGER*) Value;
				else
					error = ERR_DESC_INVALID_FIELD;
				break;
			case SQL_DESC_DATA_PTR:         /* 1010 */
				if (0 != (DT_AD & type))
					pDescriptor->ad_records[RecNumber].common.data_ptr = (SQLPOINTER) Value;
				else
					pDescriptor->id_records[RecNumber].common.data_ptr = (SQLPOINTER) Value;
				break;
			case SQL_DESC_NAME:            /* 1011 */
				if (DT_IPD == type)
				{
					free(pDescriptor->id_records[RecNumber].name);
					if (NULL == Value || '\0' == *(SQLCHAR*) Value)
					{
						pDescriptor->id_records[RecNumber].name    = NULL;
						pDescriptor->id_records[RecNumber].unnamed = SQL_UNNAMED;
					}
					else
					{
						pDescriptor->id_records[RecNumber].name    = (SQLTCHAR*) Value;
						pDescriptor->id_records[RecNumber].unnamed = SQL_NAMED;
					}
				}
				else
					error = ERR_DESC_INVALID_FIELD;
				break;
			case SQL_DESC_UNNAMED:          /* 1012 */
				if (SQL_UNNAMED == (SQLSMALLINT) Value)
				{
					pDescriptor->id_records[RecNumber].unnamed = (SQLSMALLINT) Value;
					free(pDescriptor->id_records[RecNumber].name);
					pDescriptor->id_records[RecNumber].name = NULL;
				}
				else
					error = ERR_DESC_CANT_SET_NAMED;
				break;
			default:
				processed = SQL_FALSE;
		}

		if (SQL_FALSE == processed)
		{
			CD_REC* pCD_REC = (0 != (DT_AD & type)) ? &pDescriptor->ad_records[RecNumber].common : &pDescriptor->id_records[RecNumber].common;
			switch (FieldIdentifier)
			{
				/*-------------------------------------------------------------------------
				 * common fields
				 *-------------------------------------------------------------------------
				 */
				case SQL_DESC_CONCISE_TYPE:                 /* 2 */
					pCD_REC->concise_type = (SQLSMALLINT) Value;
					break;
				case SQL_DESC_DATETIME_INTERVAL_PRECISION: /* 26 */
					pCD_REC->datetime_interval_precision = (SQLINTEGER) Value;
					break;
				case SQL_DESC_NUM_PREC_RADIX:              /* 32 */
					pCD_REC->num_prec_radix = (SQLINTEGER) Value;
					break;
				case SQL_DESC_TYPE:                      /* 1002 */
					pCD_REC->type = (SQLSMALLINT) Value;
					break;
				case SQL_DESC_LENGTH:                    /* 1003 */
					pCD_REC->length = (SQLUINTEGER) Value;
					break;
				case SQL_DESC_PRECISION:                 /* 1005 */
					pCD_REC->precision = (SQLSMALLINT) Value;
					break;
				case SQL_DESC_SCALE:                     /* 1006 */
					pCD_REC->scale = (SQLSMALLINT) Value;
					break;
				case SQL_DESC_DATETIME_INTERVAL_CODE:    /* 1007 */
					pCD_REC->datetime_interval_code = (SQLSMALLINT) Value;
					break;
				case SQL_DESC_OCTET_LENGTH:              /* 1013 */
					pCD_REC->octet_length = (SQLINTEGER) Value;
					break;
				default:
					error = ERR_DESC_UNKNOWN_FIELD;
			}
		}
	}

	if (ERR_CLEAR != error)
	{
		SetError(SQL_HANDLE_DESC, pDescriptor, error, NULL);
		return SQL_ERROR;
	}

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: CopyDescriptor
 *-----------------------------------------------------------------------------
 */
SQLRETURN CopyDescriptor(Descriptor* SourceDescHandle, Descriptor* TargetDescHandle)
{
	SQLRETURN   nRet;
	SQLSMALLINT sourceCount;
	SQLSMALLINT i;

	/* if Target descriptor wasn't typed - type it the same as Source descriptor */
	if ((DT_AD == TargetDescHandle->type) && (0 != (DT_AD & SourceDescHandle->type)))
		TargetDescHandle->type = SourceDescHandle->type;

	sourceCount = SourceDescHandle->header.count;

	/* check Source descriptor consistency and possible realloc records in Target descriptor */
	if (SQL_SUCCESS != (nRet = CheckDescriptorConsistency(SourceDescHandle, 0)) ||
	    SQL_SUCCESS != (nRet = ReallocDescriptorRecords(TargetDescHandle, sourceCount))
	   )
		return nRet;

	/* -- copy header -- */
	memcpy(&TargetDescHandle->header, &SourceDescHandle->header, sizeof(struct _descriptor_header));

	/* -- copy bookmark -- */
	memcpy(&TargetDescHandle->bookmark, &SourceDescHandle->bookmark, sizeof(struct _bookmark));


	/* -- copy descriptor records -- */
	/* if descriptors are of the same class (application or implementation) - use quick copying,
	 * if not - copy common fields and set default values to other */
	if (0 != (DT_AD & TargetDescHandle->type))
	{
		if (0 != (DT_AD & SourceDescHandle->type))
			memcpy(TargetDescHandle->ad_records, SourceDescHandle->ad_records, sizeof(struct _application_desc_record)*sourceCount);
		else
			for (i=0;i<sourceCount;i++)
			{
				EmptyDescriptorRecord(TargetDescHandle, i);
				memcpy(&TargetDescHandle->ad_records[i].common, &SourceDescHandle->id_records[i].common, sizeof(struct _common_desc_record));
			}
	}
	else
	{
		if (0 != (DT_ID & SourceDescHandle->type))
			memcpy(TargetDescHandle->id_records, SourceDescHandle->id_records, sizeof(struct _implementation_desc_record)*sourceCount);
		else
			for (i=0;i<sourceCount;i++)
			{
				EmptyDescriptorRecord(TargetDescHandle, i);
				memcpy(&TargetDescHandle->id_records[i].common, &SourceDescHandle->ad_records[i].common, sizeof(struct _common_desc_record));
			}
	}

	return SQL_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: CheckDescriptorConsistency
 *
 * DESCRIPTION: This check is always performed when SQLBindParameter or SQLBindCol
 *              is called or when SQLSetDescRec is called for an APD, ARD, or IPD.
 *-----------------------------------------------------------------------------
 */
SQLRETURN CheckDescriptorConsistency(Descriptor* pDescriptor, SQLSMALLINT RecNumber)
{
	SQLRETURN nRet;
	CD_REC*   pCD_REC;

	/* check RecNumber and get common part of the descriptor */
	if (0 == RecNumber)
	{
		return SQL_SUCCESS;
	}
	else
		RecNumber--;

	if (0 != (DT_AD & pDescriptor->type))
		pCD_REC = &pDescriptor->ad_records[RecNumber].common;
	else
		pCD_REC = &pDescriptor->id_records[RecNumber].common;

	/* check SQL_DESC_TYPE and SQL_DESC_CONCISE_TYPE */
			switch(pCD_REC->type)
			{
				/* simple types */
				case SQL_GUID:
				case SQL_CHAR:
				case SQL_WCHAR:
				case SQL_BINARY:
				case SQL_VARCHAR:
				case SQL_WVARCHAR:
				case SQL_VARBINARY:
				case SQL_LONGVARCHAR:
				case SQL_WLONGVARCHAR:
				case SQL_LONGVARBINARY:
					nRet = (pCD_REC->type == pCD_REC->concise_type) ? SQL_SUCCESS : SQL_ERROR;
					break;
				/* numeric types */
				case SQL_DECIMAL:
				case SQL_NUMERIC:
				case SQL_SMALLINT:
				case SQL_INTEGER:
				case SQL_FLOAT:
				case SQL_REAL:
				case SQL_DOUBLE:
				case SQL_BIT:
				case SQL_TINYINT:
				case SQL_BIGINT:
					nRet = (pCD_REC->type == pCD_REC->concise_type && 0 <= pCD_REC->precision && 0 <= pCD_REC->scale) ? SQL_SUCCESS : SQL_ERROR;
					break;

				case SQL_INTERVAL:
					if (/* interval precision must be valid */
					    ((SQL_INTERVAL_DAY == pCD_REC->concise_type && SQL_CODE_DAY == pCD_REC->datetime_interval_code) ||
					     (SQL_INTERVAL_DAY_TO_HOUR == pCD_REC->concise_type && SQL_CODE_DAY_TO_HOUR == pCD_REC->datetime_interval_code) ||
					     (SQL_INTERVAL_DAY_TO_MINUTE == pCD_REC->concise_type && SQL_CODE_DAY_TO_MINUTE == pCD_REC->datetime_interval_code) ||
							 (SQL_INTERVAL_MONTH == pCD_REC->concise_type && SQL_CODE_MONTH == pCD_REC->datetime_interval_code) ||
							 (SQL_INTERVAL_YEAR == pCD_REC->concise_type && SQL_CODE_YEAR == pCD_REC->datetime_interval_code) ||
					     (SQL_INTERVAL_YEAR_TO_MONTH == pCD_REC->concise_type && SQL_CODE_YEAR_TO_MONTH == pCD_REC->datetime_interval_code) ||
					     (SQL_INTERVAL_HOUR == pCD_REC->concise_type && SQL_CODE_HOUR == pCD_REC->datetime_interval_code) ||
					     (SQL_INTERVAL_MINUTE == pCD_REC->concise_type && SQL_CODE_MINUTE == pCD_REC->datetime_interval_code) ||
					     (SQL_INTERVAL_HOUR_TO_MINUTE == pCD_REC->concise_type && SQL_CODE_HOUR_TO_MINUTE == pCD_REC->datetime_interval_code) ||
					     /* types with avaible seconds precision */
							 ((SQL_INTERVAL_DAY_TO_SECOND == pCD_REC->concise_type && SQL_CODE_DAY_TO_SECOND == pCD_REC->datetime_interval_code) ||
					      (SQL_INTERVAL_SECOND == pCD_REC->concise_type && SQL_CODE_SECOND == pCD_REC->datetime_interval_code) ||
					      (SQL_INTERVAL_HOUR_TO_SECOND == pCD_REC->concise_type && SQL_CODE_HOUR_TO_SECOND == pCD_REC->datetime_interval_code) ||
					      (SQL_INTERVAL_MINUTE_TO_SECOND == pCD_REC->concise_type && SQL_CODE_MINUTE_TO_SECOND == pCD_REC->datetime_interval_code)
							 ) && (0 <= pCD_REC->precision)
					    ) && (0 <= pCD_REC->datetime_interval_precision)
						 )
						nRet = SQL_SUCCESS;
					else
						nRet = SQL_ERROR;
					break;
				case SQL_DATETIME:
					if ((SQL_TYPE_DATE == pCD_REC->concise_type && SQL_CODE_DATE == pCD_REC->datetime_interval_code) ||
					    /* types with avaible seconds precision */
					    ((SQL_TYPE_TIME      == pCD_REC->concise_type && SQL_CODE_TIME      == pCD_REC->datetime_interval_code) ||
					     (SQL_TYPE_TIMESTAMP == pCD_REC->concise_type && SQL_CODE_TIMESTAMP == pCD_REC->datetime_interval_code)
					    ) && (0 <= pCD_REC->precision)
					   )
						nRet = SQL_SUCCESS;
					else
						nRet = SQL_ERROR;
					break;
				default:
					nRet = SQL_ERROR;
			}

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: RenameColumns
 *
 * DESCRIPTION: directly renames resultset's columns from 0 till nNumber
 *-----------------------------------------------------------------------------
 */
void RenameColumns(Statement* pStatement, const ColumnNameType* pColumns, int nNumber)
{
	int         i;
	Descriptor* pIRD = GET_DESCRIPTOR(pStatement->ird);

	for (i=0;i<nNumber;i++)
	{
		free(pIRD->id_records[i].name);
		pIRD->id_records[i].name     = _tcsdup(pColumns[i].name);
		pIRD->id_records[i].type_oid = pColumns[i].postgresql_type_oid;
	}
	PopulateID(pIRD);
	RET_DESCRIPTOR(pIRD);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: PrepareResultset
 *-----------------------------------------------------------------------------
 */
int PrepareResultset(Statement* pStatement, int nRows)
{
	int          i;
	unsigned int j;

	union {
		BYTE* bPtr;
		int*  iPtr;
	} data;

	Descriptor* pIRD = GET_DESCRIPTOR((0 < pStatement->results.irds.used) ? (Descriptor*)pStatement->results.irds.handles[pStatement->results.irds.used-1]: pStatement->ird);
	if (0 < nRows)
	{
		/* realloc array of pointers */
		SQLPOINTER* array = (SQLPOINTER*) malloc(sizeof(SQLPOINTER) * pIRD->header.count * (pIRD->header.array_size = nRows));
		
		/* divide array into columns */
		for(i=0;i<pIRD->header.count;i++,array+=pIRD->header.array_size)
			pIRD->id_records[i].common.data_ptr = array;
	}
	else
	{
		pIRD->header.array_size = 0;
		for(i=0;i<pIRD->header.count;i++)
			pIRD->id_records[i].common.data_ptr = NULL;
	}

	/* fill with poiters to data */
	data.iPtr = pStatement->results.pLastHeadData;
	for(j=0;j<pIRD->header.array_size;j++) /* rows */
		for(i=0;i<pIRD->header.count;i++)    /* columns */
		{
			if (FIELD_IN_NEXT_BLOCK == *data.iPtr)
			{
				data.bPtr = (pStatement->results.pLastHead = pStatement->results.pLastHead->next)->data;
				pIRD->id_records[i].common.data_ptr[j] = data.iPtr;
			}
			else
			{
				pIRD->id_records[i].common.data_ptr[j] = data.iPtr;
			}
			data.bPtr += (0 < *data.iPtr)? (*data.iPtr)*sizeof(TCHAR)+sizeof(unsigned int) : sizeof(unsigned int);
		}

	/* remove LastHead */
	pStatement->results.pLastHead = pStatement->results.pTail;
	pStatement->results.pLastHeadData = pStatement->results.pTailField;

	RET_DESCRIPTOR(pIRD);
	return nRows;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: NewResultset
 *-----------------------------------------------------------------------------
 */
Descriptor* NewResultset(Statement* pStatement)
{
	if (pStatement->results.irds.used < ++pStatement->results.current)
	{
		Descriptor* pIRD;
		if (pIRD = AllocDescriptor(pStatement->connection))
		{
			InitDescriptor(pIRD, SQL_DESC_ALLOC_AUTO);
			pIRD->type = DT_IRD;
							
			AddItem(&pStatement->results.irds, pIRD);
		}
		return pIRD;
	}
	else
		return pStatement->ird;
}