;;;
;;; $Id: irchat-commands.el,v 1.11 1994/02/23 15:05:00 tmo Exp $
;;;
;;; see file irchat-copyright.el for change log and copyright info

(provide 'irchat-commands)
(require 'irchat-inlines)
(require 'irchat-globals)
(require 'irchat-dcc)
(require 'irchat-pj-jisx0201)

(defun irchat-Command-describe-briefly ()
  (message (substitute-command-keys "Type \\[describe-mode] for help")))


(defun irchat-Command-redisplay (&optional center)
  "Un-freezes and re-selects the Dialogue buffer in another window.
   With argument, recenter with that argument."
  (interactive "P")
  (if irchat-channel-buffer-mode
      (progn
	(if (null (get-buffer-window irchat-Channel-buffer))
	    (irchat-configure-windows))
	(set-buffer irchat-Channel-buffer)
	(goto-char (point-max))
	(set-window-point (get-buffer-window irchat-Channel-buffer)
			  (point-max)))
    (setq irchat-freeze nil)
    (if (get-buffer-window irchat-Dialogue-buffer)
	(let ((owin (selected-window)) win)
	  (if (one-window-p)
	      (irchat-configure-windows)
	    (display-buffer irchat-Dialogue-buffer))
	  (if (setq win (get-buffer-window irchat-Dialogue-buffer))
	      (let ((obuf (current-buffer)))
		(set-buffer irchat-Dialogue-buffer)
		(goto-char (point-max))
		(select-window win)
		(recenter (- (window-height) 1))
		(select-window owin)
		(set-buffer obuf)))))))

(defun irchat-Command-send-message (xmsg)
  "Send MESSAGE to current chat partner of current channel."
  (if (eq irchat-command-buffer-mode 'chat)
      (if irchat-current-chat-partner
	  (irchat-send-privmsg irchat-current-chat-partner xmsg)
	(message 
	 (substitute-command-keys 
	  "Type \\[irchat-Command-join] to start private conversation")))
    (if (not irchat-current-channel)
	(progn
	  ;; modified by simm@irc.fan.gr.jp, Mon, 20 Dec 1999 21:44:10 +0900
	  (funcall irchat-pj-sound-error-function)
	  (message 
	   (substitute-command-keys 
	    "Type \\[irchat-Command-join] to join a channel")))
      (irchat-send-privmsg irchat-current-channel xmsg))))

(defun irchat-Command-send-line ()
  "Send the current line to the current channel."
  (interactive)
  (let (xmsg start)
    (beginning-of-line)
    (setq start (point))
    (end-of-line)
    (setq xmsg (buffer-substring start (point)))
    (irchat-next-line 1)
    (irchat-Command-send-message xmsg)))

;; add by simm@irc.fan.gr.jp, Mon, 24 Jan 2000 22:51:04 +0900
(defun irchat-pj-Command-send-line ()
  "Send the current line to the current channel with SKK mode considered."
  (interactive)
  (cond ((not (boundp 'skk-mode))
	 (irchat-Command-send-line))
	((not skk-mode)
	 (irchat-Command-send-line))
	((and (boundp 'skk-henkan-mode) (not skk-henkan-mode))
	 (irchat-Command-send-line))
	((and (boundp 'skk-henkan-on) (not skk-henkan-on))
	 (irchat-Command-send-line))
	(skk-egg-like-newline
	 (skk-kakutei))
	(t
	 (skk-kakutei)
	 (irchat-Command-send-line))))

(defun irchat-Command-enter-message ()
  "Ask for a line as an entry in the IRC dialogue on the current channel."
  (interactive)
  (let ((xmsg "x")
	(to (if (eq irchat-command-buffer-mode 'chat)
		irchat-current-chat-partner irchat-current-channel)))
    (if to
	(while (not (string= xmsg ""))
	  (setq xmsg (read-string (format "to %s> "
					  (irchat-chan-virtual to))))
	  (or (string= xmsg "")
	      (irchat-Command-send-message xmsg)))
      nil)))


(defun irchat-Command-debug ()
  "Start debugging irchat."
  (interactive)
  (if irchat-debug-buffer
      (progn
	(setq irchat-debug-buffer nil)
	(other-window 1)
	(delete-window)
	(other-window -1))
    (if irchat-use-full-window
	(delete-other-windows))
    (irchat-configure-windows)
    (split-window-horizontally)
    (other-window 1)
    (setq irchat-debug-buffer (get-buffer-create "*IRC Debugging*"))
    (switch-to-buffer irchat-debug-buffer)
    (other-window -1)))


(defun irchat-Command-inline ()
  "Send current line as a message to the IRC server."
  (interactive)
  (let (message start stop)
    (beginning-of-line)
    (setq start (point))
    (end-of-line)
    (setq stop (point))
    (setq message (buffer-substring start stop))
    (newline)
    (irchat-send "%s" message)))


(defun irchat-Command-join (chan &optional key)
  "Join a channel or private conversation.
If user nicname is given, join the same set of channels as 
the specified user. 
If Command-buffer is in chat-mode, start private conversation 
with specified user."

  (interactive (let (chan key (completion-ignore-case t))
                 ;; modified by simm@irc.fan.gr.jp, Fri, 23 Jul 1999
		 (setq chan 
		       (if (numberp current-prefix-arg)
                           current-prefix-arg
                         (irchat-completing-default-read
                          "Join channel/nick: "
                          (if (eq irchat-command-buffer-mode 'chat)
                              (append irchat-nick-alist irchat-channel-alist)
                            (append irchat-channel-alist irchat-nick-alist))
                          '(lambda (s) t)
                          nil
                          (or (irchat-chan-virtual irchat-invited-channel)
                              irchat-privmsg-partner))))
                 (and current-prefix-arg
                      (not (numberp current-prefix-arg))
                      (setq key
                            (if (eq current-prefix-arg '-)
                                (read-string (concat "Key for channel " chan ": "))
                              (irchat-read-passwd (concat "Key for channel " chan ": ")))))
		 (list chan key)))
  (if (numberp chan)
      (irchat-Channel-jump chan)
    ;; begin: add by simm@irc.fan.gr.jp, on Sat, 16 Jan 1999
    (if irchat-pj-fix-gaga-problem
	(let ((pt 0) (cur "") (enc "") (tmp (concat chan ",")))
	  (setq chan "")
	  (while (setq pt (string-match "," tmp))
	    (setq cur (substring tmp 0 pt)
		  pt  (1+ pt)
		  tmp (if (< pt (length tmp)) (substring tmp pt) "")
		  enc (irchat-pj-string-out-convert cur))
	    (cond ((string-match "," enc)
		   ;; modified by simm@irc.fan.gr.jp, Mon, 20 Dec 1999 21:44:10 +0900
		   (funcall irchat-pj-sound-error-function)
		   (message "Invalid channel name. Cannot join %s" cur))
		  (t
		   (setq chan (if (string= "" chan) cur (concat chan "," cur))))))))
    ;; end
    (setq chan (irchat-chan-real chan)
          irchat-invited-channel nil)
    (if (irchat-ischannel chan)
        (progn
          (setq irchat-command-buffer-mode 'channel)
          (or key
              (setq key (get (intern chan) 'key)))
          (put (intern chan) 'key key)
          (or key
              (setq key ""))
          (irchat-Command-join-channel chan key))
      (setq irchat-command-buffer-mode 'chat)
      (irchat-Command-join-partner chan))
    ;; refresh mode line
    (set-buffer-modified-p (buffer-modified-p))))


(defun irchat-Command-join-channel (chan key)
  (and (catch 'found
	 (mapcar 
	  '(lambda (elem)
	     (and (string-ci-equal chan elem)
		  (setq chan elem)
		  (throw 'found t)))
	  irchat-current-channels)
	 (irchat-send "JOIN %s %s" chan key)
	 nil)
       (irchat-Channel-join chan)))


(defun irchat-Command-join-partner (partner)
  (setq irchat-current-chat-partners (append 
				      (list partner)
				      (string-list-ci-delete
				       partner
				       irchat-current-chat-partners))
	irchat-current-chat-partner partner
	irchat-chat-partner-alist (list-to-assoclist 
				   irchat-current-chat-partners))
  (irchat-Channel-join irchat-current-chat-partner))


(defun irchat-Command-part (chan &optional reason)
  "Part a channel or private conversation."
  (interactive (let (chan (completion-ignore-case t)
			  (reason (or irchat-channel-signoff-msg "bye...")))
		 (setq chan
		       (if (eq irchat-command-buffer-mode 'chat)
			   (irchat-completing-default-read	
			    "End private conversation with: "
			    irchat-chat-partner-alist
			    '(lambda (s) t) nil 
			    irchat-current-chat-partner)
			 (irchat-completing-default-read	
			  "Part channel: "
			  (list-to-assoclist irchat-current-channels)
			  '(lambda (s) t) nil 
			  (irchat-chan-virtual irchat-current-channel))))
		 (if current-prefix-arg
		     (setq reason (read-string "Reason: ")))
		 (list chan reason)))
  (setq chan (irchat-chan-real chan))
  (if (not (irchat-ischannel chan))
      (setq irchat-current-chat-partners (string-list-ci-delete
					  chan
					  irchat-current-chat-partners)
	    irchat-current-chat-partner (car irchat-current-chat-partners)
	    irchat-chat-partner-alist (list-to-assoclist 
				       irchat-current-chat-partners))
    (progn
      (if (string-list-ci-memberp chan irchat-current-channels)
	  (setq irchat-current-channel chan)) ; just refocusing
      (irchat-send "PART %s :%s" chan reason)))
  (if (not (irchat-ischannel chan))
      (irchat-Channel-part chan)))


(defun irchat-Command-ignore (nick)
  "Ignore messages from this user.  If already ignoring him/her, toggle."
  (interactive (let (nick (completion-ignore-case t))
		 (setq nick
		       (irchat-pj-completing-read 
			"Ignore nickname: " 
			irchat-nick-alist
			'(lambda (s) t) nil nil))
		 (list nick)))
  (if (string= "" nick)
      (let ((mylist irchat-ignore-nickname) str)
	(setq str "*** Currently ignored people:")
	(while mylist
	  (setq str (format "%s %s" str (car mylist)))
	  (setq mylist (cdr mylist)))
	(irchat-insert0 (format "%s\n" str)))
    (if (memq (intern nick) irchat-ignore-nickname)
	(progn
	  (irchat-insert0 (format "*** Ignore OFF: %s\n" nick))
	  (setq irchat-ignore-nickname 
		(delq (intern nick) irchat-ignore-nickname)))
      (irchat-insert0 (format "*** Ignore ON: %s\n" nick))
      (setq irchat-ignore-nickname 
	    (cons (intern nick) irchat-ignore-nickname)))))


(defun irchat-Command-kick (nick &optional reason)
  "Kick this user out."
  (interactive (let (nick (completion-ignore-case t)
			  (reason "heh..."))
		 (setq nick 
		       (irchat-pj-completing-read 
			"Kick out nickname: " 
			irchat-nick-alist
			'(lambda (s) t) nil nil))
		 (if current-prefix-arg
		     (setq reason (read-string "Reason: ")))
		 (list nick reason)))
  (irchat-send "KICK %s %s :%s"
	       irchat-current-channel nick reason))


(defun irchat-Command-servers (regexp)
  "List the given regexp servers.
With Control-U as argument, show server conecting association."
  (interactive "sServer name: ")
  (setq irchat-how-to-show-links-reply current-prefix-arg)
  (irchat-send "LINKS %s" regexp))


(defun irchat-Command-list (&optional chan)
  "List the given channel and its topics.
If you enter only Control-U as argument, list the current channel.
With - as argument, list all channels."
  (interactive
   (if (or current-prefix-arg (null irchat-current-channel))
       (if (eq current-prefix-arg '-)
	   (list current-prefix-arg)
	 (list
	  (let ((completion-ignore-case t))
	    (irchat-pj-completing-read 
	     "LIST channel: " 
	     irchat-channel-alist
	     '(lambda (s) t) nil nil))))
     nil))
  (if (null chan)
      (if irchat-current-channel
	  (irchat-send "LIST %s" irchat-current-channel)
	(irchat-send "LIST %s" (irchat-Channel-lists)))
    (if (eq chan '-)
	(irchat-send "LIST")
      (if (not (string= chan ""))
	  (irchat-send "LIST %s" (irchat-chan-real chan))
	(irchat-send "LIST %s" (irchat-Channel-lists))))))

(defun irchat-Channel-lists ()
  (if (null irchat-current-channels)
      ""
    (let ((chans (cdr irchat-current-channels))
	  (str (car irchat-current-channels)))
      (while chans
	(setq str (format "%s,%s" str (car chans)))
	(setq chans (cdr chans)))
      str)))
  
(defun irchat-Command-users ()
  "List the number of users and servers"
  (interactive)
  (irchat-send "LUSERS"))

(defun irchat-Command-version ()
  "Ask server version"
  (interactive)
  (irchat-send "VERSION"))

(defun irchat-Command-admin ()
  "Ask server admin"
  (interactive)
  (irchat-send "ADMIN"))

(defun irchat-Command-modec (&optional chan)
  "Send/Check the mode for you/channel."
  (interactive
   (if current-prefix-arg
       (if (eq current-prefix-arg '-)
	   (list current-prefix-arg)
	 (list
	  (let ((completion-ignore-case t))
	    (irchat-completing-default-read 
	     "MODE channel: " 
	     irchat-channel-alist
	     '(lambda (s) t) nil nil))))
     nil))
  (if (null chan)
      (setq chan irchat-current-channel))
  (if (or (string= "" chan) (eq chan '-))
      (setq chan nil))
  (setq chan (irchat-chan-real chan))
  (let (value)
    (if chan
	(setq value (read-string (format "Mode for channel %s: "
					 (irchat-chan-virtual chan))))
      (setq value (read-string "Mode for you: ")))
    (irchat-send "MODE %s %s" (or chan irchat-nickname) value)))

(defun irchat-Command-send-minibuffer (chan xmsg)
  "Send a message to another user/channel from minibuffer."
  (interactive (let (chan (completion-ignore-case t))
		 (setq chan
		       (irchat-chan-real
			(irchat-completing-default-read 
			 "Private message to: "
			 (append irchat-nick-alist irchat-channel-alist)
			 '(lambda (s) t) nil
			 (irchat-chan-virtual irchat-privmsg-partner))))
		 (list chan
		       (read-string 
			(format "Private message to %s: "
				(irchat-chan-virtual chan))))))
  (setq irchat-privmsg-partner chan)
  (irchat-send-privmsg chan xmsg))

(defun irchat-send-privmsg (to xmsg)
  (irchat-insert-private nil to xmsg)
  (if (string-match "^=\\([^ ]+\\)" to)
      (irchat-dcc-chat-send (matching-substring to 1) xmsg)
    (irchat-send "PRIVMSG %s :%s" to xmsg)))

(defun irchat-Command-send-private ()
  "Send a private message (current line) to another user."
  (interactive)
  (let ((completion-ignore-case t) xmsg start stop)
    (setq irchat-privmsg-partner
	  (irchat-chan-real
	   (irchat-completing-default-read 
	    "To whom: "
	    (append irchat-nick-alist irchat-channel-alist)
	    '(lambda (s) t) 
	    nil (irchat-chan-virtual irchat-privmsg-partner))))
    (beginning-of-line)
    (setq start (point))
    (end-of-line)
    (setq stop (point))
    (setq xmsg (buffer-substring start stop))
    (irchat-next-line 1)
    (irchat-send-privmsg irchat-privmsg-partner xmsg)))

;; add by simm@irc.fan.gr.jp, Mon, 14 Jun 1999
(defun irchat-pj-send-broadcast (xmsg)
  (let (chanlist string count)
    (mapcar
     '(lambda (to)
	(cond ((member to irchat-pj-broadcast-accept-list)
	       (setq chanlist (cons to chanlist)))
	      ((member to irchat-pj-broadcast-ignore-list)
	       nil)
	      ((or (eq ?# (elt to 0))
		   (eq ?! (elt to 0))
		   (eq ?% (elt to 0)))
	       (and irchat-pj-broadcast-channel
		    (setq chanlist (cons to chanlist))))
	      (t
	       (and irchat-pj-broadcast-private
		    (setq chanlist (cons to chanlist))))))
     irchat-chanbuf-list)
    (setq count 1)
    (mapcar
     '(lambda (chan)
	(irchat-insert-private nil chan xmsg)
	(cond ((eq 1 count)
	       (setq count (1+ count)
		     string chan))
	      ((eq irchat-pj-maximum-privmsg-channels count)
	       (irchat-send "PRIVMSG %s,%s :%s" chan string xmsg)
	       (setq count 1
		     string nil))
	      (t
	       (setq count (1+ count)
		     string (format "%s,%s" chan string)))))
     chanlist)
    (and string
	 (irchat-send "PRIVMSG %s :%s" string xmsg))))

(defun irchat-pj-Command-broadcast-minibuffer (xmsg)
  "Broadcast a message from minibuffer."
  (interactive (list (read-string "Broadcast Message: ")))
  (irchat-pj-send-broadcast xmsg))

(defun irchat-pj-Command-broadcast-message ()
  "Send a broadcast message (current line)."
  (interactive)
  (let (xmsg start stop)
    (beginning-of-line)
    (setq start (point))
    (end-of-line)
    (setq stop (point))
    (setq xmsg (buffer-substring start stop))
    (irchat-next-line 1)
    (irchat-pj-send-broadcast xmsg)))
;; end

(defun irchat-Command-names (&optional chan)
  "List the nicknames of the current IRC users on given channel.
With an Control-U as argument, only the current channel is listed.
With - as argument, list all channels."
  (interactive
   (if (or current-prefix-arg (null irchat-current-channel))
       (if (eq current-prefix-arg '-)
	   (list current-prefix-arg)
	 (list
	  (let ((completion-ignore-case t))
	    (irchat-pj-completing-read 
	     "Names on channel: " 
	     irchat-channel-alist
	     '(lambda (s) t) nil nil))))
     nil))
  (if (null chan)
      (if irchat-current-channel
	  (irchat-send "NAMES %s" irchat-current-channel)
	(irchat-send "NAMES %s" (irchat-Channel-lists)))
    (if (eq chan '-)
	(irchat-send "NAMES")
      (if (not (string= chan ""))
	  (irchat-send "NAMES %s" (irchat-chan-real chan))
	(irchat-send "NAMES %s" (irchat-Channel-lists))))))

(defun irchat-Command-nickname (nick)
  "Set your nickname."
  (interactive "sEnter your nickname: ")
  (setq irchat-trying-nickname nick)
  (irchat-send "NICK %s" nick))

(defun irchat-Command-who (&optional chan)
  "Lists tue users that match the given expression.
If you enter only Control-U as argument, list the current channel.
With - as argument, list all users."
  (interactive 
   (if (or current-prefix-arg (null irchat-current-channel))
       (if (eq current-prefix-arg '-)
	   (list current-prefix-arg)
	 (list
	  (let ((completion-ignore-case t))
	    (irchat-pj-completing-read 
	     "WHO expression: " 
	     irchat-channel-alist
	     '(lambda (s) t) nil nil))))
     nil))
  (if (null chan)
      (if irchat-current-channel
	  (irchat-send "WHO %s" irchat-current-channel)
	(irchat-send "WHO %s" (irchat-Channel-lists)))
    (if (eq chan '-)
	(irchat-send "WHO")
      (if (not (string= chan ""))
	  (irchat-send "WHO %s" (irchat-chan-real chan))
	(irchat-send "WHO %s" irchat-nickname)))))


(defun irchat-Command-wait (nick &optional greeting)
  "Wait for NICK to enter IRC.  When this person appears, you will
be informed. If the optional argument GREETING is non-nil, it should 
be a string to send NICK upon entering."
  (interactive 
   (progn (setq nick (read-string "Wait for: ")
		greeting (read-string 
			  (format "Message to send %s upon entering: " nick)))
	  (if (string= greeting "")
	      (setq greeting nil))
	  (list nick greeting)))
  (put (intern nick) 'irchat-waited-for t)
  (if greeting 
      (put (intern nick) 'irchat-greeting greeting)))


(defun irchat-Command-finger (finger-nick-var)
  "Get information about a specific user."
  (interactive (let (finger-nick-var (completion-ignore-case t))
		 (setq finger-nick-var 
		       (irchat-pj-completing-read 
			"Finger whom: " irchat-nick-alist
			'(lambda (s) t) nil nil))
		 (list finger-nick-var)))
  (irchat-send "WHOIS %s" finger-nick-var))


(defun irchat-Command-trace (trace-nick-var)
  "Get information about a specific user."
  (interactive (let (trace-nick-var (completion-ignore-case t))
		 (setq trace-nick-var 
		       (irchat-pj-completing-read 
			"Trace whom: " irchat-nick-alist
			'(lambda (s) t) nil nil))
		 (list trace-nick-var)))
  (irchat-send "TRACE %s" trace-nick-var))


(defun irchat-Command-finger-direct (finger-nick-var)
  "Get information about a specific user."
  (interactive (let (finger-nick-var (completion-ignore-case t))
		 (setq finger-nick-var 
		       (irchat-pj-completing-read 
			"Finger whom: " irchat-nick-alist
			'(lambda (s) t) nil nil))
		 (list finger-nick-var)))
  (irchat-send "WHOIS %s %s" finger-nick-var finger-nick-var))


(defun irchat-Command-topic ()
  "Change topic/userinfo of channel/you."
  (interactive)
  (let (value)
    (if irchat-current-channel
	(setq value (read-string (format "Topic for channel %s: "
			        (irchat-chan-virtual irchat-current-channel))))
      (setq value (read-string "Userinfo for you: " irchat-ctcp-userinfo)))
    (if irchat-current-channel
	(irchat-send "TOPIC %s :%s" irchat-current-channel value)
      (setq irchat-ctcp-userinfo value))))


(defun irchat-Command-invite (&optional chan nick)
  "Invite user to channel."
  (interactive 
   (list
    (if current-prefix-arg
	(let ((completion-ignore-case t))
	  (irchat-pj-completing-read 
	   "Invite channel: "
	   (mapcar '(lambda (x)
		      (list x))
		   irchat-current-channels)
	   '(lambda (s) t) nil nil))
	  nil)
    (let ((completion-ignore-case t)) 
      (irchat-pj-completing-read "Invite whom: " 
		       irchat-nick-alist
		       '(lambda (s) t) nil nil))))
  (if chan
      (setq chan (irchat-chan-real chan))
    (if irchat-current-channel
	(setq chan irchat-current-channel)
      (setq chan irchat-nickname)))
  (irchat-send "INVITE %s %s" nick chan))


(defun irchat-Command-away (awaymsg)
  "Mark/unmark yourself as being away."
  (interactive "sAway message: ")
  (irchat-send "AWAY :%s" awaymsg))


(defun irchat-Current-scroll-down ()
  "Scroll down current buffer"
  (interactive)
  (if (pos-visible-in-window-p (point-min))
      (message "Beginning of buffer")
    (scroll-down)))


(defun irchat-Command-scroll-down ()
  "Scroll Dialogue-buffer down from Command-buffer."
  (interactive)
  (if irchat-channel-buffer-mode
      (pop-to-buffer irchat-Channel-buffer)
    (pop-to-buffer irchat-Dialogue-buffer))
  (if (pos-visible-in-window-p (point-min))
      (message "Beginning of buffer")
    (scroll-down))
  (pop-to-buffer irchat-Command-buffer))

(defun irchat-Current-scroll-up ()
  "Scroll up current buffer."
  (interactive)
  (if (pos-visible-in-window-p (point-max))
      (progn
	(goto-char (point-max))
	(recenter 1))
    (scroll-up)))


(defun irchat-Command-scroll-up ()
  "Scroll Dialogue-buffer up from Command-buffer."
  (interactive)
  (let ((obuf (current-buffer)) owin win)
    (if irchat-channel-buffer-mode
	(set-buffer irchat-Channel-buffer)
      (set-buffer irchat-Dialogue-buffer))
    (if (setq win (get-buffer-window (current-buffer)))
	(progn
	  (setq owin (selected-window))
	  (select-window win)
	  (if (not (pos-visible-in-window-p (point-max)))
	      (scroll-up 1))
	  (if (pos-visible-in-window-p (point-max))
	      (progn
		(goto-char (point-max))
		(recenter 1))
	    (scroll-up))
	  (select-window owin)))
    (set-buffer obuf)))

(defun irchat-Command-scroll-freeze ()
  "Toggle the automatic scrolling of the Current/Dialogue window."
  (interactive)
  (if irchat-channel-buffer-mode
      (irchat-Channel-freeze)
    (irchat-Dialogue-freeze)))

(defun irchat-Dialogue-freeze ()
  "Toggle the automatic scrolling of the Dialogue window."
  (interactive)
  (setq irchat-freeze (not irchat-freeze))
  (set-buffer-modified-p (buffer-modified-p)))

(defun irchat-Channel-freeze (&optional value)
  "Toggle the automatic scrolling of the Channel window."
  (interactive)
  (save-excursion
    (set-buffer irchat-Channel-buffer)
    (cond ((eq value 'on)
	   (setq irchat-freeze-local t))
	  ((eq value 'off)
	   (setq irchat-freeze-local nil)))
    (setq irchat-freeze-local (not irchat-freeze-local)))
  (set-buffer-modified-p (buffer-modified-p)))


(defun irchat-Command-beep-on-message (&optional value)
  "Toggle the automatic beep notice when the channel mesage is received."
  (interactive)
  (save-excursion
    (set-buffer irchat-Channel-buffer)
    (cond ((eq value 'on)
	   (setq irchat-beep-local t))
	  ((eq value 'off)
	   (setq irchat-beep-local nil)))
    (setq irchat-beep-local (not irchat-beep-local)))
  (set-buffer-modified-p (buffer-modified-p)))


(defun irchat-Command-suppress-others (&optional value)
  "Toggle to suppress this channel messages display to Others-buffer."
  (interactive)
  (save-excursion
    (set-buffer irchat-Channel-buffer)
    (cond ((eq value 'on)
	   (setq irchat-suppress-local t))
	  ((eq value 'off)
	   (setq irchat-suppress-local nil)))
    (setq irchat-suppress-local (not irchat-suppress-local)))
  (set-buffer-modified-p (buffer-modified-p)))


(defun irchat-quit ()
  (irchat-Command-quit 'quit))

(defun irchat-Command-quit (&optional quit-msg)
  "Quit irchat-pj."
  (interactive "P")
  (if (or (not (irchat-server-opened))
	  quit-msg
	  (y-or-n-p "Quit irchat-pj? "))
      (let (quit-string)
	(message "")
	(if (and (get-buffer-process irchat-server-buffer)
		 (irchat-server-opened))
	    (progn
	      (if (and quit-msg (not (eq quit-msg 'quit)))
		  (setq quit-string (read-string "Signoff message: "))
		(setq quit-string (or irchat-signoff-msg "nil")))
	      (irchat-send "QUIT :%s" quit-string)))
        ;; modified by simm@irc.fan.gr.jp, Thu, 10 Jun 1999
	(if (not (and quit-msg (or (eq quit-msg 'quit) (eq quit-msg 'error))))
	    (irchat-handle-quit irchat-nickname irchat-pj-my-userhost quit-string))
	(irchat-clear-system)
	(if irchat-use-full-window
	    (delete-other-windows))
	(irchat-close-server)
	(if (not (and quit-msg (or (eq quit-msg 'quit) (eq quit-msg 'error))))
	    (if irchat-old-window-configuration
		(progn
		  (set-window-configuration irchat-old-window-configuration)
		  (setq irchat-old-window-configuration nil))))
	(run-hooks 'irchat-Exit-hook)
        ;; begin: add by simm@irc.fan.gr.jp, Sat, 5 Jun 1999
        (if irchat-pj-save-log-channel-alist
            (irchat-pj-save-log))
        (if irchat-pj-quit-with-kill-buffer
            (let (name)
              (mapcar
               '(lambda (item)
                  (setq name (buffer-name item))
                  ;; modified by simm@irc.fan.gr.jp, Wed, 21 Jul 1999
                  (if (or (string-match irchat-buffer-base name)
                          (string= irchat-Command-buffer name))
                      (kill-buffer item)))
               (buffer-list))
              ;;(kill-buffer irchat-Command-buffer)
              (setq irchat-Private-buffer (concat irchat-buffer-base " Private"))))
        ;; end
	(setq irchat-polling nil)
	(setq irchat-current-channel nil)
	(setq irchat-current-channels nil))))


(defun irchat-Command-generic (message)
  "Enter a generic IRC message, which is sent to the server.
 A ? lists the useful generic messages."
  (interactive "sIRC Command: ")
  (if (string= message "?")
      (with-output-to-temp-buffer "*IRC Help*"
	(princ "The following generic IRC messages may be of interest to you:
TOPIC <channel> <new topic>	set the topic of your channel
INVITE <nickname> <channel>	invite another user to join your channel
LINKS <mask>			lists the currently reachable IRC servers
NAMES <channel>			lists users per channel
")
	(message 
	 (substitute-command-keys 
	  "Type \\[irchat-Command-redisplay] to continue")))
    (irchat-send "%s" message)))


(defun irchat-Command-irc-compatible ()
  "If entered at column 0, allows you to enter a generic IRC message to
be sent to the server.  For a list of messages, see irchat-Command-generic."
  (interactive)
  (if (eq (current-column) 0)
      (call-interactively (function irchat-Command-generic))
    (self-insert-command 1)))


(defun irchat-Command-send-exec (command)
  "Execute command, and send it to the current channel."
  (interactive "sShell Command: ")
  (shell-command command t)
  (let ((opoint (point)))
    (while (< (point) (mark))
      (progn
	(irchat-Command-send-line)
	(set-buffer irchat-Command-buffer)))
    (push-mark opoint t)))

;;;
;;; CTCP (client-to-client protocol) queries
;;;
(defun irchat-Channel-ctcp-ping ()
  (interactive)
  (irchat-send "PRIVMSG %s :\001PING %s %s\001" irchat-current-channel
	       (irchat-current-time) irchat-current-channel))

(defun irchat-Channel-ctcp-version ()
  (interactive)
  (irchat-send "PRIVMSG %s :\001VERSION\001" irchat-current-channel))

(defun irchat-Channel-ctcp-userinfo ()
  (interactive)
  (irchat-send "PRIVMSG %s :\001USERINFO\001" irchat-current-channel))

(defun irchat-Channel-ctcp-time ()
  (interactive)
  (irchat-send "PRIVMSG %s :\001TIME\001" irchat-current-channel))

(defun irchat-Command-ctcp (command)
  (interactive)
  (irchat-read-ctcp-nick command)
  (irchat-send "PRIVMSG %s :\001%s\001" irchat-ctcp-lastnick command))

(defun irchat-Command-ctcp-version ()
  "Ask about someones client version."
  (interactive)
  (irchat-Command-ctcp "VERSION"))

(defun irchat-Command-ctcp-time ()
  "Ask about someones time."
  (interactive)
  (irchat-Command-ctcp "TIME"))

(defun irchat-Command-ctcp-ping ()
  "Ask about someones ping."
  (interactive)
  (irchat-read-ctcp-nick "PING")
  (irchat-send "PRIVMSG %s :\001PING %s\001" irchat-ctcp-lastnick
	       (irchat-current-time)))

(defun irchat-Command-ctcp-finger ()
  "Ask about someones finger."
  (interactive)
  (irchat-Command-ctcp "FINGER"))

(defun irchat-Command-ctcp-userinfo ()
  "Ask about someones userinfo."
  (interactive)
  (irchat-Command-ctcp "USERINFO"))

(defun irchat-Command-ctcp-clientinfo ()
  "Ask about someones available ctcp commands."
  (interactive)
  (irchat-Command-ctcp "CLIENTINFO"))

(defun irchat-Command-ctcp-action ()
  "Action."
  (interactive)
  (irchat-read-ctcp-nick "ACTION")
  (let ((xmsg (read-string (format "CTCP ACTION argument: "))))
    (irchat-send "PRIVMSG %s :\001ACTION %s\001" irchat-ctcp-lastnick xmsg)
    (irchat-insert-private nil irchat-ctcp-lastnick
			   (format "*** %s %s" irchat-nickname xmsg))))

;; add by simm@irc.fan.gr.jp, Sun, 29 Aug 1999 22:55:43 +0900
(defun irchat-Command-ctcp-caesar ()
  "Action."
  (interactive)
  (irchat-read-ctcp-nick "CAESAR")
  (let ((xmsg (read-string (format "CTCP CAESAR argument: "))))
    (irchat-send "PRIVMSG %s :\001CAESAR %s\001"
                 irchat-ctcp-lastnick
                 (irchat-pj-caesar-string xmsg))
    (irchat-insert-private nil irchat-ctcp-lastnick
			   (format "*** %s %s" irchat-nickname xmsg))))

(defun irchat-Command-ctcp-clientinfo-generic ()
  "Ask about someones available ctcp commands."
  (interactive)
  (irchat-read-ctcp-nick "clientinfo generic")
  (let ((completion-ignore-case t))
    (setq irchat-ctcp-lastcommand
	  (irchat-completing-default-read 
	   "What CTCP command: " 
	   irchat-ctcp-alist '(lambda (s) t) nil 
	   irchat-ctcp-lastcommand)))
  (irchat-send "PRIVMSG %s :\001CLIENTINFO%s%s\001"
	       irchat-ctcp-lastnick
	       (if (string= irchat-ctcp-lastcommand "") "" " ")
	       irchat-ctcp-lastcommand))

(defun irchat-Command-ctcp-generic ()
  "Generic CTCP"
  (interactive)
  (irchat-read-ctcp-nick "generic")
  (let ((completion-ignore-case t) arg)
    (setq irchat-ctcp-lastcommand
	  (irchat-completing-default-read 
	   "What CTCP command: " 
	   irchat-ctcp-alist '(lambda (s) t) nil 
	   irchat-ctcp-lastcommand))
    (if current-prefix-arg
	(setq arg (read-string (format "CTCP %s argument: "
				       irchat-ctcp-lastcommand))))
    (irchat-send "PRIVMSG %s :\001%s%s\001"
		 irchat-ctcp-lastnick irchat-ctcp-lastcommand
		 (if arg (format " %s" arg) ""))))

(defun irchat-Command-ctcp-userinfo-from-minibuffer ()
  "Set my userinfo from minibuffer."
  (interactive)
  (setq irchat-ctcp-userinfo
	(read-from-minibuffer "New userinfo: "
			      irchat-ctcp-userinfo)))

(defun irchat-Command-ctcp-userinfo-from-commandbuffer ()
  "Set my userinfo from commandbuffer."
  (interactive)
  (beginning-of-line)
  (setq start (point))
  (end-of-line)
  (setq stop (point))
  (setq irchat-ctcp-userinfo (buffer-substring start stop))
  (irchat-next-line 1))

(defun irchat-read-ctcp-nick (type)
  (let ((completion-ignore-case t))
    (setq irchat-ctcp-lastnick
	  (irchat-chan-real
	   (irchat-completing-default-read 
	    (format "CTCP %s query to: " type) irchat-nick-alist
	    '(lambda (s) t) nil
	    (irchat-chan-virtual irchat-ctcp-lastnick))))))

(defvar irchat-ctcp-alist
  '(("ACTION") ("CAESAR") ("CLIENTINFO") ("DCC") ("ECHO") ("ERRMSG")
    ("PING") ("TIME") ("USERINFO") ("VERSION"))
  "*CTCP commands alist")

(defun irchat-ctcp-add-to-list (command)
  (setq irchat-ctcp-alist (append irchat-ctcp-alist (list (list command)))))


;;;
;;; send text in kill-buffer
;;;
(defun irchat-Command-send-yank (&optional howmany)
  "Yank kill-buffer, and send it to the current channel."
  (interactive)
  (let ((beg (point)) end)
    (insert (car kill-ring-yank-pointer))
    (setq end (point))
    (goto-char beg)
    (while (< (point) end)
      (progn
	(irchat-Command-send-line)
	(set-buffer irchat-Command-buffer)))))

;;;
;;; send rot-5-13-47-48 encrypted data
;;; modified by simm@irc.fan.gr.jp, Sun, 29 Aug 1999 22:27:45 +0900
;;;
(defun irchat-Command-caesar-line ()
  "*Caesar encrypt current line."
  (interactive)
  (let (beg end)
    (beginning-of-line nil)
    (setq beg (point))
    (push-mark (point))
    (end-of-line)
    (setq end (point))
    (irchat-pj-caesar-region beg end)))


;;;
;;;
;;;
(defun get-word-left ()
  "Return word left from point."
  (save-excursion
    (let (point-now)
      (setq point-now (point))
      (backward-word 1)
      (buffer-substring (point) point-now))))


(defun irchat-Command-complete ()
  "Complete word before point from userlist."
  (interactive)
  (insert
   (save-excursion
     (let ((completion-ignore-case t) point-now word result)
       (setq point-now (point)
	     word (get-word-left)
	     result (try-completion word irchat-nick-alist))
       (backward-word 1)
       (delete-region (point) point-now)
       (if (or (eq result t) (eq result nil))
           word
         result)))))


(defun irchat-Command-load-vars ()
  "Load configuration from irchat-variables-file."
  (interactive)
  (let ((file (expand-file-name irchat-variables-file)))
    (if (file-exists-p file)
	(progn
	  (load-file file)
	  (irchat-Command-reconfigure-windows)))))


(defun irchat-Command-reconfigure-windows ()
  (interactive)
  (let ((command-window (get-buffer-window irchat-Command-buffer))
	(dialogue-window (get-buffer-window irchat-Dialogue-buffer)))
    (if (and command-window dialogue-window)
	(let ((c-height (window-height command-window))
	      (d-height (window-height dialogue-window)))
	  (delete-window command-window)
	  (pop-to-buffer irchat-Dialogue-buffer)
	  (enlarge-window (+ c-height d-height
			     (- (window-height dialogue-window)))))
      (pop-to-buffer irchat-Dialogue-buffer))
    (irchat-configure-windows)
    (if irchat-one-buffer-mode
	(pop-to-buffer irchat-Dialogue-buffer)
      (pop-to-buffer irchat-Command-buffer))))


;;;
;;; command to get end of the Dialogue buffer
;;;
(defun irchat-Command-eod-buffer ()
  (interactive)
  (let ((saved-buffer (current-buffer)))
    (set-buffer irchat-Dialogue-buffer)
    (goto-char (point-max))
    (set-buffer saved-buffer)))

(defun irchat-Command-toggle-display-mode ()
  (interactive)
  (setq irchat-channel-buffer-mode (not irchat-channel-buffer-mode))
  (irchat-configure-windows))


(defun irchat-Command-next-channel ()
  "Select next channel or chat partner."
  (interactive)
  (if (numberp current-prefix-arg)
      (irchat-Channel-jump current-prefix-arg)
    (irchat-Channel-next)))


(defun irchat-Command-previous-channel ()
  "Select previous channel or chat partner."
  (interactive)
  (if (numberp current-prefix-arg)
      (irchat-Channel-jump current-prefix-arg)
    (irchat-Channel-previous)))


(defun irchat-Channel-jump (num)
  (if (or (= 0 num) (nth1 num irchat-chanbuf-list))
      (irchat-Channel-select num)
    (message (format "No binding at %d" num))))

(defun irchat-Command-alternative-channel ()
  (interactive)
  (if (or (= 0 irchat-chanbuf-alternative-number)
	  (nth1 irchat-chanbuf-alternative-number irchat-chanbuf-list))
      (irchat-Channel-select irchat-chanbuf-alternative-number)))

(defsubst irchat-channel-key (chan key)
  (put (intern chan) 'key key))

(defvar irchat-chanbuf-current-number 0)
(defvar irchat-chanbuf-alternative-number 0)

(defun irchat-Channel-buffer-create (chan)
  (let (nbuf obuf tmp)
    (setq obuf (current-buffer))
    (setq nbuf (get-buffer-create  (concat irchat-buffer-base chan)))
    (set-buffer nbuf)
    (setq tmp irchat-Channel-buffer)
    (setq irchat-Channel-buffer nbuf)
    (irchat-Channel-freeze (get (intern chan) 'freeze))
    (irchat-Command-beep-on-message (get (intern chan) 'beep))
    (irchat-Command-suppress-others (get (intern chan) 'suppress))
    (setq irchat-Channel-buffer tmp)
    (insert (current-time-string) " Created.\n")
    (irchat-Channel-mode)
    (set-buffer obuf)
    nbuf))

(defun irchat-Channel-select (num)
  (if (/= irchat-chanbuf-current-number num)
      (setq irchat-chanbuf-alternative-number irchat-chanbuf-current-number
	    irchat-chanbuf-current-number num))
  (let (chan buf win)
    (if (/= num 0)
	(setq chan (nth1 num irchat-chanbuf-list)
	      buf  (or (get-buffer (concat irchat-buffer-base chan))
		       (irchat-Channel-buffer-create chan)))
      (setq buf irchat-Private-buffer))
    (cond ((= 0 num)
	   (setq irchat-chanbuf-indicator    "Private"
		 irchat-current-channel      nil
		 irchat-command-buffer-mode  'chat
		 irchat-current-chat-partner irchat-nickname))
	  ((irchat-ischannel chan)
	   (setq irchat-chanbuf-indicator   (concat "Channel " chan)
		 irchat-command-buffer-mode 'channel
		 irchat-current-channel     chan))
	  (t
	   (setq irchat-chanbuf-indicator   (concat "With " chan)
		 irchat-current-channel     nil
		 irchat-command-buffer-mode 'chat
		 irchat-current-chat-partner chan)))
    (if (setq win (get-buffer-window irchat-Channel-buffer t))
	(let (obuf)
	  (setq obuf (current-buffer))
	  (set-window-buffer win buf)
	  (set-buffer buf)
	  (if (not irchat-freeze-local)
	      (set-window-point win (point-max)))
	  (set-buffer obuf)))
    (setq irchat-Channel-buffer buf)
    (setq irchat-chanbuf-num num)
    (set-buffer-modified-p (buffer-modified-p))))

(defun irchat-Channel-exist (chan)
  (if (stringp chan)
      (let ((rest irchat-chanbuf-list) found)
	(while rest
	  (if (and (car rest) (string-ci-equal chan (car rest)))
	      (setq found (car rest)))
	  (setq rest (cdr rest)))
	(if found
	    (get-buffer (concat irchat-buffer-base found))
	  nil))
    nil))

(defun irchat-Channel-next ()
  (let ((rest (nthcdr irchat-chanbuf-num irchat-chanbuf-list))
	(num (+ irchat-chanbuf-num 1))
	(found nil))
    (while (and rest (not found))
      (if (car rest)
	  (setq found t)
	(setq rest (cdr rest))
	(setq num (+ num 1))))
    (if found
	(irchat-Channel-select num)
      (irchat-Channel-select 0))))

(defun irchat-Channel-previous ()
  (let ((num (- irchat-chanbuf-num 1)))
    (if (< num 0)
	(setq num (length irchat-chanbuf-list)))
    (while (and (/= num 0)
		(null (nth1 num irchat-chanbuf-list)))
      (setq num (- num 1)))
    (irchat-Channel-select num)))

(defun irchat-Channel-join (chan)
  (let ((rest irchat-chanbuf-list)
	(num 1)
	(found nil))
    (while (and rest (not found))
      (if (and (car rest) (string-ci-equal chan (car rest)))
	  (setq found t)
	(setq num (+ num 1))
	(setq rest (cdr rest))))
    (if (not found)
	(progn
	  (setq rest irchat-default-channel-binding)
	  (setq num 1)
	  (while (and rest (not found))
	    (if (and (car rest) (string-ci-equal chan (car rest)))
		(setq found t)
	      (setq num (+ num 1))
	      (setq rest (cdr rest))))
	  (if (or (not found)
		  (nth1 num irchat-chanbuf-list))
	      (progn
		(setq rest irchat-chanbuf-list)
		(setq num 1)
		(while (and rest (not found))
		  (if (and (null (car rest))
			   (null (nth1 num irchat-default-channel-binding)))
		      (setq found t)
		    (setq num (+ num 1))
		    (setq rest (cdr rest))))
		(if (not found)
		    (progn
		      (setq num (+ (length irchat-chanbuf-list) 1))
		      (while (nth1 num irchat-default-channel-binding)
			(setq num (+ num 1)))))))))
    (if (> num (length irchat-chanbuf-list))
	(setq irchat-chanbuf-list
	      (append irchat-chanbuf-list
		      (make-list (- num (length irchat-chanbuf-list)) nil))))
    (if (null (nth1 num irchat-chanbuf-list))
	(progn
	  (setcar (nth1cdr num irchat-chanbuf-list) chan)
	  (irchat-Channel-change)))
    (irchat-Channel-select num)))

(defun irchat-Channel-part (chan)
  (if chan
      (let ((rest irchat-chanbuf-list)
	    (num 1)
	    (found nil))
	(while (and rest (not found))
	  (if (and (car rest) (string-ci-equal chan (car rest)))
	      (setq found t)
	    (setq num (+ num 1))
	    (setq rest (cdr rest))))
	(if (not found)
	    (message (format "Not found [%s]" chan))
	  (setcar (nth1cdr num irchat-chanbuf-list) nil)
	  (irchat-Channel-change)
	  (irchat-Channel-select 0)))
    (setq irchat-chanbuf-list nil)
    (irchat-Channel-change)
    (irchat-Channel-select 0)))

(defun irchat-Channel-change ()
  (let ((rest irchat-chanbuf-list) (string "") (n 1))
    (while rest
      (if (car rest)
	  (setq string (format "%s,%d%s%s" string n 
			       (if (irchat-ischannel (car rest)) "" ":")
			       (irchat-chan-virtual (car rest)))))
      (setq n (+ n 1))
      (setq rest (cdr rest)))
    (if (string= string "")
	(setq irchat-chanbufs-indicator "No channel")
      (setq irchat-chanbufs-indicator (substring string 1 (length string))))))

(defun irchat-chan-virtual (chan)
  (if chan
      (let ((rest irchat-channel-conversion-map) match)
	(while rest
	  (if (string= (car (car rest)) chan)
	      (setq match (cdr (car rest))))
	  (setq rest (cdr rest)))
	(if match
	    match
	  (if (and (string-match "^\\([#+]\\)\\(.*\\):\\(.*\\)$" chan)
		   (string= (matching-substring chan 3)
			    irchat-channel-conversion-default-mask))
	      (if (string= (matching-substring chan 1) "#")
		  (format "%%%s" (matching-substring chan 2))
		(format "-%s" (matching-substring chan 2)))
	    chan)))
    nil))

(defun irchat-chan-real (chan)
  (if chan
      (let ((rest irchat-channel-conversion-map) match)
	(while rest
	  (if (string= (cdr (car rest)) chan)
	      (setq match (car (car rest))))
	  (setq rest (cdr rest)))
	(if match
	    match
	  (if (string-match "^%\\(.*\\)$" chan)
	      (format "#%s:%s" (matching-substring chan 1)
		      irchat-channel-conversion-default-mask)
	    (if (string-match "^-\\(.*\\)$" chan)
		(format "+%s:%s" (matching-substring chan 1)
			irchat-channel-conversion-default-mask)
	      chan))))
    nil))

(defun irchat-Command-jump-channel0 ()
  (interactive)
  (irchat-Channel-jump 0))

(defun irchat-Command-jump-channel1 ()
  (interactive)
  (irchat-Channel-jump 1))

(defun irchat-Command-jump-channel2 ()
  (interactive)
  (irchat-Channel-jump 2))

(defun irchat-Command-jump-channel3 ()
  (interactive)
  (irchat-Channel-jump 3))

(defun irchat-Command-jump-channel4 ()
  (interactive)
  (irchat-Channel-jump 4))

(defun irchat-Command-jump-channel5 ()
  (interactive)
  (irchat-Channel-jump 5))

(defun irchat-Command-jump-channel6 ()
  (interactive)
  (irchat-Channel-jump 6))

(defun irchat-Command-jump-channel7 ()
  (interactive)
  (irchat-Channel-jump 7))

(defun irchat-Command-jump-channel8 ()
  (interactive)
  (irchat-Channel-jump 8))

(defun irchat-Command-jump-channel9 ()
  (interactive)
  (irchat-Channel-jump 9))

(defun irchat-Command-jump-channel10 ()
  (interactive)
  (irchat-Channel-jump 10))

(defun irchat-Command-jump-channel11 ()
  (interactive)
  (irchat-Channel-jump 11))

(defun irchat-Command-jump-channel12 ()
  (interactive)
  (irchat-Channel-jump 12))

(defun irchat-Command-jump-channel13 ()
  (interactive)
  (irchat-Channel-jump 13))

(defun irchat-Command-jump-channel14 ()
  (interactive)
  (irchat-Channel-jump 14))

(defun irchat-Command-jump-channel15 ()
  (interactive)
  (irchat-Channel-jump 15))

(defun irchat-Command-jump-channel16 ()
  (interactive)
  (irchat-Channel-jump 16))

(defun irchat-Command-jump-channel17 ()
  (interactive)
  (irchat-Channel-jump 17))

(defun irchat-Command-jump-channel18 ()
  (interactive)
  (irchat-Channel-jump 18))

(defun irchat-Command-jump-channel19 ()
  (interactive)
  (irchat-Channel-jump 19))

(defun irchat-Command-jump-channel (num)
  (interactive "nChannel Number: ")
  (irchat-Channel-jump num))

;;;
;;; Cannot use completing read, user may want to query many names
;;;
(defun irchat-Command-ison (nick)
  "IsON user."
  (interactive "sIsON: ")
  (irchat-send "ISON %s" nick))


(defun irchat-Command-userhost (nick)
  "Ask for userhost."
  (interactive "sUserhost nick(s): ")
  (irchat-send "USERHOST %s" nick))


;;;
;;; eof
;;;
