# WMLCard.pm - The Class that provides a WMLCard Object
# Created by James Pattie, 12/21/2000.

# Copyright (c) 2000 PC & Web Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package HTMLObject::WMLCard;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Exporter AutoLoader);
@EXPORT = qw();
	
$VERSION = '2.30';

# new
# instantiates an instance of a WMLCard Object
# takes: id
# optional: title, newcontext, ordered, onenterforward, onenterbackword, ontimer
sub new
{
  my $that = shift;
  my $class = ref($that) || $that;
  my $self = bless {}, $class;
  my %args = ( id => "card0", title => "", newcontext => "false", ordered => "true", onenterforward => "", onenterbackward => "", ontimer => "", @_ );
  my $errStr = "HTMLObject::WMLCard->new() - Error!<br/>\n";
  
  $self->{id} = $args{id};
  $self->{title} = $args{title};
  $self->{newcontext} = $args{newcontext};
  $self->{ordered} = $args{ordered};
  $self->{onenterforward} = $args{onenterforward};
  $self->{onenterbackward} = $args{onenterbackward};
  $self->{ontimer} = $args{ontimer};
  $self->{content} = "";  # This is the body of the card.

  # do validation
  $self->{error} = !$self->isValid;  
  if ($self->{error})
  {
    $self->{errorString} = $errStr . $self->{errorString};
  }

  return $self;
}

# isValid - Returns 0 or 1 to indicate if the object is valid.
sub isValid
{
  my $self = shift;
  my $error = 0;
  my $errorString = "";
  my $errStr = "HTMLObject::WMLCard->isValid() - Error!<br/>\n";

  # do validation code here.
  if (length $self->{id} == 0)
  {
    $error = 1;
    $errorString .= "id = '$self->{id}' is invalid!<br/>\n";
  }
  if ($self->{newcontext} !~ /^(true|false)$/)
  {
    $error = 1;
    $errorString .= "newcontext = '$self->{newcontext}' is invalid!<br/>\n";
  }
  if ($self->{ordered} !~ /^(true|false)$/)
  {
    $error = 1;
    $errorString .= "ordered = '$self->{ordered}' is invalid!<br/>\n";
  }

  $self->{error} = $error;
  $self->{errorString} = $errStr if $error;
  $self->{errorString} .= $errorString;

  return !$error;
}

sub DESTROY
{
  my $self = shift;
}

sub AUTOLOAD
{
  my $self = shift;
  my $type = ref($self) || die "$self is not an object";
  my $name = $AUTOLOAD;
  $name =~ s/.*://;	# strip fully-qualified portion
  unless (exists $self->{$name})
  {
    die "Can't access `$name' field in object of class $type";
  }
  if (@_)
  {
    return $self->{$name} = shift;
  }
  else
  {
    return $self->{$name};
  }
}

# setError
# requires: errorString
# returns: nothing
sub setError
{
  my $self = shift;
  my %args = ( errorString => "", @_ );
  my $errorString = $args{errorString};
  
  $self->{errorString} = $errorString;
  $self->{error} = 1;
}

# didErrorOccur
# returns the value of error.
sub didErrorOccur
{
  my $self = shift;

  return $self->{error};
}

# errorMessage
# returns the value of errorString
sub errorMessage
{
  my $self = shift;
  
  return $self->{errorString};
}

# print
# takes: string of text to add to content variable
# returns: nothing
sub print
{
  my $self = shift;
  my $content = shift;
  
  $self->{content} .= $content;
}

# display
# takes: nothing
# returns: string that represents the WML Card
sub display
{
  my $self = shift;
  my $output = "";
  
  $output .= "  <card id=\"$self->{id}\"";
  $output .= (length $self->{title} > 0 ? " title=\"$self->{title}\"" : "");
  $output .= " newcontext=\"$self->{newcontext}\"";
  $output .= " ordered=\"$self->{ordered}\"";
  $output .= (length $self->{onenterforward} > 0 ? " onenterforward=\"$self->{onenterforward}\"" : "");
  $output .= (length $self->{onenterbackward} > 0 ? " onenterbackward=\"$self->{onenterbackward}\"" : "");
  $output .= (length $self->{ontimer} > 0 ? " ontimer=\"$self->{ontimer}\"" : "");
  $output .= ">\n";
  (my $content = $self->{content}) =~ s/^(.*)$/    $1/mg;  # indent 4 spaces to fit inside the <card>..</card> and inside the overall <wml>..</wml> tags.
  $output .= $content;
  $output .= "  </card>\n";
  
  return $output;
}

1;
__END__

=head1 NAME

WMLCard - Object used to build a WMLCard Object Class.

=head1 SYNOPSIS

  use HTMLObject::WMLCard;
  my $obj = HTMLObject::WMLCard->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

WMLCard is a WMLCard class.

=head1 Exported FUNCTIONS

  scalar new(id, title, newcontext, ordered, onenterforward, 
             onenterbackword, ontimer)
    Creates a new instance of the HTMLObject::WMLCard
    object.

  bool isValid(void)
    Returns 0 or 1 to indicate if the object is valid.  The error will 
    be available via errorMessage().

  void setError(errorString)
    sets error and errorString to indicate an error occurred.

  scalar didErrorOccur(void)
    Returns the value of error.

  scalar errorMessage(void)
    Returns the value of errorString.
    
  void print(scalar)
    Adds the contents of the passed in string to the content string.
    
  scalar display(void)
    Returns the <card id=""...> content </card> for this object.


  NOTE:  All data fields are accessible by specifying the object
         and pointing to the data member to be modified on the
         left-hand side of the assignment.
         Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:htmlobject@pcxperience.com)

=head1 SEE ALSO

perl(1), HTMLObject::WAP(3)

=cut
