/*
 * xmeter.c - Display histogram of rstat(3) output for multiple hosts.
 *
 * Copyright (c) 1991, Bob Schwartzkopf
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both the
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the RAND Corporation not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The RAND Corporation
 * makes no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * The X Consortium, and any party obtaining a copy of these files from
 * the X Consortium, directly or indirectly, is granted, free of charge, a
 * full and unrestricted irrevocable, world-wide, paid up, royalty-free,
 * nonexclusive right and license to deal in this software and
 * documentation files (the "Software"), including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons who receive
 * copies from any such party to do so.  This license includes without
 * limitation a license to do the foregoing actions under any patents of
 * the party supplying this software to the X Consortium.
 *
 * Author: Bob Schwartzkopf, The RAND Corporation
 *
 * Suggestions for improvements and bug fixes can be sent to "bobs@rand.org".
 * As my schedule permits I'll try to incorporate them.
 */

#ifndef lint
static char	*RCSid="$Header: /net/src/rand/bin/xmeter/RCS/xmeter.c,v 1.19 1994/05/25 01:01:08 bobs Exp $";
#endif lint

/*
 * $Log: xmeter.c,v $
 * Revision 1.19  1994/05/25  01:01:08  bobs
 * Add X copyright.
 *
 * Revision 1.18  1994/02/12  02:46:40  bobs
 * Port to solaris 2.
 *
 * Revision 1.17  1994/02/10  20:54:17  bobs
 * Fixed bug in quit() that caused core dumps.
 * Removed bcopy() call in newshmeter().
 *
 * Revision 1.16  1993/12/09  20:10:26  bobs
 * Patches 12 and 13 (see README for details).
 *
 * Revision 1.15  1993/08/18  01:42:33  bobs
 * Fixed bug in fdisks(), dk_xfer dimensioned is 4.
 * Include <X11/Xos.h>, remove some other includes and external declarations.
 * Changed vfork to vork.
 * Don't destroy popup widget in popdownscale if it's null.
 * Include <sys/dk.h> under hpux.
 *
 * Revision 1.14  1993/05/25  00:06:27  bobs
 * Use sigset(3) under SVR4.
 *
 * Revision 1.13  1993/01/25  22:32:18  bobs
 * Redo previous 2 edits.
 *
 * Revision 1.12  1993/01/19  00:24:22  bobs
 * Have child processes exit if parent has gone away.
 *
 * Revision 1.11  1993/01/19  00:12:33  bobs
 * Kill children on INT signal too.
 *
 * Revision 1.10  1993/01/19  00:08:56  bobs
 * Kill child processes on TERM signal.
 *
 * Revision 1.9  1992/05/21  00:11:17  bobs
 * Add option to display current value in label.
 * Add quit option to stat menu.
 * Redo application resource handling in more portable way.
 * Use complete hostname for comparisons when shortnames has been specified.
 * Redo child process cleanup code in a hopefully more portable way.
 * Add stat name as fourth arg to alert programs.
 * Fix divide by zero problem in functions that return stat values.
 *
 * Revision 1.8  1991/09/16  21:01:04  bobs
 * Fix bugs handling down hosts.
 * Add composite stats swap, disks, page and pkts.
 * Add popup window with version number, scale and downtime (if applicable).
 *
 * Revision 1.7  1991/06/03  22:21:43  root
 * Add -v option to print xmeter version number.
 * Add defstat option.
 * Add code to clear stripchart when watched stat is changed.
 * When host is down fork process to wait for it to come up instead of
 * having toplevel process wait for timeouts.
 *
 * Revision 1.6  90/09/28  20:32:34  root
 * Add explicit closes of sockets back in, apparently some versions of
 * clnt_destroy() don't do it automatically.
 * 
 * Revision 1.5  90/09/18  20:47:03  root
 * Allow user specification of foreground colors.
 * 
 * Revision 1.4  90/08/20  14:18:01  root
 * Added menus, column and row options, background bitmaps, and user
 * specifiable programs to be invoked when graphs change state.
 * 
 * Revision 1.3  90/06/07  16:17:06  bobs
 * Removed "retries".
 * Changed name of paned widgets to host name displayed in that widget.
 * Used actual time between rstat calls instead of "update" interval in
 * computing rates in functions that return values to stripchart widgets.
 * Removed "ost" variable, rewrote functions that return values to
 * stripcharts.
 * Fixed bug in fsys, fcpu and fuser that could cause divide by 0 errors.
 * Fixed rpc timeout handling in getmeter and getport.
 * Use RSTATVERS_TIME instead of RSTATVERS, which isn't defined in SunOS 4.1.
 * 
 * Revision 1.2  90/05/04  12:31:52  bobs
 * Fix memory leak in getport().  Wasn't freeing resources allocated by
 * clntudp_create when clnt_call failed.  Also removed explicit calls
 * to close sockets since clnt_destroy() does this.
 * 
 * Revision 1.1  90/04/30  14:33:59  bobs
 * Initial revision
 * 
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <signal.h>
#include <netdb.h>
#define PORTMAP		/* Get right function declarations on Solaris 2	*/
#include <rpc/rpc.h>
#include <rpc/pmap_prot.h>
#include <rpcsvc/rstat.h>
/*
 * We need the definitions of CP_USER, CP_NICE and CP_SYS.  Some systems
 * declare them in <sys/dk.h>, which may or may not be included by
 * <rpcsvc/rstat.h>, others define them in <rpcsvc/rstat.h>, and others 
 * don't seem to define them anywhere.  To simplify things I'm just
 * going to define them here if I didn't get them from <rpcsvc/rstat.h>.
 * If they ever change this is going to break.
 */
#ifndef CP_USER
#define CP_USER		0
#define CP_NICE		1
#define CP_SYS		2
#endif
#ifndef CPUSTATES
#ifdef RSTAT_CPUSTATES
#define CPUSTATES	RSTAT_CPUSTATES
#else
#define CPUSTATES	4
#endif
#endif
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/StripChart.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xmu/Xmu.h>
#include "patchlevel.h"

#define MAJORVERSION	1
#define DMSG		"down"
#define OK		0
#define WARN		1
#define ERROR		2
#define FATAL		3
#define MAXBACKS	(FATAL+1)
#define STATMENU	"statmenu"
#ifndef FSCALE
#define FSCALE		(1 << 8)
#endif

/*
 * There is one METER structure per strip chart.  If multiple stats
 * on a single host are being watched we save rstatd calls by sharing
 * part of this structure (SHMETER) between the different meters watching
 * the same host.
 */
typedef struct _shmeter {
    char		*name;	/* Host name				*/
    char		*label;	/* Displayed host name			*/
    struct sockaddr_in	addr;
    CLIENT		*clnt;
    int			s;	/* Socket				*/
    struct statstime	st[2];	/* Keep last 2 stats to compute diffs	*/
    int			idx;	/* Index into st array			*/
    int			refcnt;	/* Meters sharing this structure	*/
    int			curcnt;	/* Meters who've displayed current data	*/
    int			first;	/* TRUE when only 1 rstat has been done	*/
    int			pid;	/* Pid of proc waiting for host		*/
    time_t		when;	/* Time we first notice host down	*/
    struct _shmeter	*nxt;	/* Link these together			*/
} SHMETER;

typedef struct _meter {
    char		*label;
    int			stat;		/* Which stat to watch		*/
    Widget		pdwidget;	/* Paned widget			*/
    Widget		mbwidget;	/* MenuButton widget		*/
    Widget		scwidget;	/* StripChart widget		*/
    int			oldstate;	/* Save state so know when to	*/
					/* change backgrounds		*/
    SHMETER		*sh;		/* Shared info			*/
    Pixmap		pm;		/* Current background pixmap	*/
    int			oldjumpscroll;	/* Old jumpscroll value		*/
    struct _meter	*nxt;
} METER;

/*
 * There is a STATDATA structure for each statistic that can be monitored.
 */
typedef struct {
    int		(*val)();	/* Function that computes this stat	*/
    char	*name;		/* Name of stat for menu widget		*/
} STATDATA;

#if defined(SVR4) || defined(sgi)
#define SIGTYPE	void
#else
#define SIGTYPE	int
#endif

int		getstatus();
int		changestat();
int		selecthost();
SIGTYPE		freechild();
METER		*initmeter();
SHMETER		*newshmeter();
int		fcoll(), fcpu(), fierr(), fintr(), fipkt(), fload();
int		foerr(), fopkt(), fpage(), fpgpgin(), fpgpgout();
int		fpswpin(), fpswpout(), fswt(), fsys(), fuser();
int		fpkts(), fdisks(), fswap(), ferr();
void		popupscale();
void		popdownscale();
void		quit();
char		*mystrdup();

static STATDATA		sd[] = {
#define S_COLL		0
  {fcoll, "coll"},
#define S_CPU		S_COLL+1
  {fcpu, "cpu"},
#define S_DISKS		S_CPU+1
  {fdisks, "disks"},
#define S_ERR		S_DISKS+1
  {ferr, "err"},
#define S_IERR		S_ERR+1
  {fierr, "ierr"},
#define S_INTR		S_IERR+1
  {fintr, "intr"},
#define S_IPKT		S_INTR+1
  {fipkt, "ipkt"},
#define S_LOAD		S_IPKT+1
  {fload, "load"},
#define S_OERR		S_LOAD+1
  {foerr, "oerr"},
#define S_OPKT		S_OERR+1
  {fopkt, "opkt"},
#define S_PAGE		S_OPKT+1
  {fpage, "page"},
#define S_PGPGIN	S_PAGE+1
  {fpgpgin, "pgpgin"},
#define S_PGPGOUT	S_PGPGIN+1
  {fpgpgout, "pgpgout"},
#define S_PKTS		S_PGPGOUT+1
  {fpkts, "pkts"},
#define S_PSWPIN	S_PKTS+1
  {fpswpin, "pswpin"},
#define S_PSWPOUT	S_PSWPIN+1
  {fpswpout, "pswpout"},
#define S_SWAP		S_PSWPOUT+1
  {fswap, "swap"},
#define S_SWT		S_SWAP+1
  {fswt, "swt"},
#define S_SYS		S_SWT+1
  {fsys, "sys"},
#define S_USER		S_SYS+1
  {fuser, "user"},
};
#define DEFSTATNAME	"load"
#define MAXSTAT		(sizeof(sd) / sizeof(STATDATA))
#define MAXSTATNAME	10		/* Max length of stat name	*/
#define MAXSTATVALUE	10		/* Max length of stat value	*/

/*
 * xmeter supports 4 background colors and pixmaps that it will switch
 * between as a meter changes state, called OK, WARN, ERROR and FATAL.
 * The background bitmaps are stored in BITMAP structures.
 */
typedef struct {
  int		w;				/* Width of bitmap	*/
  int		h;				/* Height of bitmap	*/
  Pixmap	bm;				/* Bitmap		*/
} BITMAP;

/*
 * Application specific resources are stored in an APPRESOURCES structure.
 **/
typedef struct {
  Pixel			back[MAXBACKS];		/* Meter backgrounds	*/
  Pixel			bd[MAXBACKS];		/* Meter borders	*/
  BITMAP		bitmap[MAXBACKS];	/* Stripchart bg pixmaps*/
  int			columns;		/* Columns to display	*/
  String		defStat;		/* Default stat		*/
  Boolean		displayValue;		/* Show value in label	*/
  int			errorLevel;		/* Error level		*/
  Pixel			fore[MAXBACKS];		/* Meter foregrounds	*/
  Pixel			hl[MAXBACKS];		/* Meter highlights	*/
  Pixel			ibd[MAXBACKS];		/* Meter internal bds	*/
  Pixel			lback[MAXBACKS];	/* Label backgrounds	*/
  Pixel			lfore[MAXBACKS];	/* Label foregrounds	*/
  String		prog[MAXBACKS];		/* Alert programs	*/
  int			retries;		/* RPC retries		*/
  int			rows;			/* Rows	to display	*/
  int			scales[MAXSTAT];	/* Scale for each stat	*/
  Boolean		shortName;		/* Short hostname flag	*/
  int			timeout;		/* RPC timeout		*/
  int			warnLevel;		/* Warning level	*/
} APPRESOURCES;

/*
 * Global variables.
 */
static char		*progname;
static Pixmap		bitmapdef = XtUnspecifiedPixmap;
static struct timeval	ptto = {0, 0};		/* RPC timeout interval	*/
static struct timeval	tto = {0, 0};		/* Total timeout	*/
static SHMETER		*shmeters = NULL;	/* List of SHMETERs	*/
static METER		*selected;		/* Current meter	*/
static METER		*meterlist = NULL;	/* List of METERs	*/
static int		loadscaledef = FSCALE;	/* Scale for load ave.	*/
static Widget		popup;		/* Used by pop[up|down]scale	*/
static APPRESOURCES	ar;		/* App specific resources	*/

#define offset(f)	XtOffsetOf(APPRESOURCES, f)
static XtResource	resources[] = {
    {"columns", "Columns", XtRInt, sizeof(int),
	offset(columns), XtRString, "0"},
    {"defStat", "DefStat", XtRString, sizeof(char *),
	offset(defStat), XtRString, DEFSTATNAME},
    {"displayValue", "DisplayValue", XtRBoolean, sizeof(Boolean),
	offset(displayValue), XtRString, "False"},
    {"errorBack", XtCBackground, XtRPixel, sizeof(Pixel),
	offset(back[ERROR]), XtRString, XtDefaultBackground},
    {"errorBd", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(bd[ERROR]), XtRString, XtDefaultForeground},
    {"errorBitmap", XtCBitmap, XtRBitmap, sizeof(Pixmap),
	offset(bitmap[ERROR].bm), XtRBitmap,(caddr_t) &bitmapdef},
    {"errorFore", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(fore[ERROR]), XtRString, XtDefaultForeground},
    {"errorHl", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(hl[ERROR]), XtRString, XtDefaultForeground},
    {"errorIbd", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(ibd[ERROR]), XtRString, XtDefaultForeground},
    {"errorLevel", "ErrorLevel", XtRInt, sizeof(int),
	offset(errorLevel), XtRString, "6"},
    {"errorProg", "Program", XtRString, sizeof(char *),
	offset(prog[ERROR]), XtRString, NULL},
    {"fatalBack", XtCBackground, XtRPixel, sizeof(Pixel),
	offset(back[FATAL]), XtRString, XtDefaultBackground},
    {"fatalBd", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(bd[FATAL]), XtRString, XtDefaultForeground},
    {"fatalBitmap", XtCBitmap, XtRBitmap, sizeof(Pixmap),
	offset(bitmap[FATAL].bm), XtRBitmap,(caddr_t) &bitmapdef},
    {"fatalFore", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(fore[FATAL]), XtRString, XtDefaultForeground},
    {"fatalHl", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(hl[FATAL]), XtRString, XtDefaultForeground},
    {"fatalIbd", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(ibd[FATAL]), XtRString, XtDefaultForeground},
    {"fatalProg", "Program", XtRString, sizeof(char *),
	offset(prog[FATAL]), XtRString, NULL},
    {"lErrorBack", XtCBackground, XtRPixel, sizeof(Pixel),
	offset(lback[ERROR]), XtRString, XtDefaultBackground},
    {"lErrorFore", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(lfore[ERROR]), XtRString, XtDefaultForeground},
    {"lFatalBack", XtCBackground, XtRPixel, sizeof(Pixel),
	offset(lback[FATAL]), XtRString, XtDefaultBackground},
    {"lFatalFore", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(lfore[FATAL]), XtRString, XtDefaultForeground},
    {"lOkBack", XtCBackground, XtRPixel, sizeof(Pixel),
	offset(lback[OK]), XtRString, XtDefaultBackground},
    {"lOkFore", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(lfore[OK]), XtRString, XtDefaultForeground},
    {"lWarnBack", XtCBackground, XtRPixel, sizeof(Pixel),
	offset(lback[WARN]), XtRString, XtDefaultBackground},
    {"lWarnFore", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(lfore[WARN]), XtRString, XtDefaultForeground},
    {"okBack", XtCBackground, XtRPixel, sizeof(Pixel),
	offset(back[OK]), XtRString, XtDefaultBackground},
    {"okBd", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(bd[OK]), XtRString, XtDefaultForeground},
    {"okBitmap", XtCBitmap, XtRBitmap, sizeof(Pixmap),
	offset(bitmap[OK].bm), XtRBitmap, (caddr_t) &bitmapdef},
    {"okFore", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(fore[OK]), XtRString, XtDefaultForeground},
    {"okHl", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(hl[OK]), XtRString, XtDefaultForeground},
    {"okIbd", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(ibd[OK]), XtRString, XtDefaultForeground},
    {"okProg", "Program", XtRString, sizeof(char *),
	offset(prog[OK]), XtRString, NULL},
    {"retries", "Retries", XtRInt, sizeof(int),
	offset(retries), XtRString, "2"},
    {"rows", "Rows", XtRInt, sizeof(int),
	offset(rows), XtRString, "0"},
    {"scaleColl", "ScaleColl", XtRInt, sizeof(int),
	offset(scales[S_COLL]), XtRString, "1"},
    {"scaleCpu", "ScaleCpu", XtRInt, sizeof(int),
	offset(scales[S_CPU]), XtRString, "20"},
    {"scaleDisks", "ScaleDisks", XtRInt, sizeof(int),
	offset(scales[S_DISKS]), XtRString, "40"},
    {"scaleErr", "ScaleErr", XtRInt, sizeof(int),
	offset(scales[S_ERR]), XtRString, "1"},
    {"scaleIerr", "ScaleIerr", XtRInt, sizeof(int),
	offset(scales[S_IERR]), XtRString, "1"},
    {"scaleIntr", "ScaleIntr", XtRInt, sizeof(int),
	offset(scales[S_INTR]), XtRString, "50"},
    {"scaleIpkt", "ScaleIpkt", XtRInt, sizeof(int),
	offset(scales[S_IPKT]), XtRString, "20"},
    {"scaleLoad", "ScaleLoad", XtRInt, sizeof(int),
	offset(scales[S_LOAD]), XtRInt, (caddr_t) &loadscaledef},
    {"scaleOerr", "ScaleOerr", XtRInt, sizeof(int),
	offset(scales[S_OERR]), XtRString, "1"},
    {"scaleOpkt", "ScaleOpkt", XtRInt, sizeof(int),
	offset(scales[S_OPKT]), XtRString, "20"},
    {"scalePage", "ScalePage", XtRInt, sizeof(int),
	offset(scales[S_PAGE]), XtRString, "10"},
    {"scalePgpgin", "ScalePgpgin", XtRInt, sizeof(int),
	offset(scales[S_PGPGIN]), XtRString, "10"},
    {"scalePgpgout", "ScalePgpgout", XtRInt, sizeof(int),
	offset(scales[S_PGPGOUT]), XtRString, "10"},
    {"scalePkts", "ScalePkts", XtRInt, sizeof(int),
	offset(scales[S_PKTS]), XtRString, "40"},
    {"scalePswpin", "ScalePswpin", XtRInt, sizeof(int),
	offset(scales[S_PSWPIN]), XtRString, "5"},
    {"scalePswpout", "ScalePswpout", XtRInt, sizeof(int),
	offset(scales[S_PSWPOUT]), XtRString, "5"},
    {"scaleSwap", "ScaleSwap", XtRInt, sizeof(int),
	offset(scales[S_SWAP]), XtRString, "5"},
    {"scaleSwt", "ScaleSwt", XtRInt, sizeof(int),
	offset(scales[S_SWT]), XtRString, "30"},
    {"scaleSys", "ScaleSys", XtRInt, sizeof(int),
	offset(scales[S_SYS]), XtRString, "20"},
    {"scaleUser", "ScaleUser", XtRInt, sizeof(int),
	offset(scales[S_USER]), XtRString, "20"},
    {"shortName", "Shortname", XtRBoolean, sizeof(Boolean),
	offset(shortName), XtRString, "False"},
    {"timeout", "Timeout", XtRInt, sizeof(int),
	offset(timeout), XtRString, "5"},
    {"warnBack", XtCBackground, XtRPixel, sizeof(Pixel),
	offset(back[WARN]), XtRString, XtDefaultBackground},
    {"warnBd", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(bd[WARN]), XtRString, XtDefaultForeground},
    {"warnBitmap", XtCBitmap, XtRBitmap, sizeof(Pixmap),
	offset(bitmap[WARN].bm), XtRBitmap, (caddr_t) &bitmapdef},
    {"warnFore", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(fore[WARN]), XtRString, XtDefaultForeground},
    {"warnHl", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(hl[WARN]), XtRString, XtDefaultForeground},
    {"warnIbd", XtCForeground, XtRPixel, sizeof(Pixel),
	offset(ibd[WARN]), XtRString, XtDefaultForeground},
    {"warnLevel", "WarnLevel", XtRInt, sizeof(int),
	offset(warnLevel), XtRString, "3"},
    {"warnProg", "Program", XtRString, sizeof(char *),
	offset(prog[WARN]), XtRString, NULL},
};
static XrmOptionDescRec	options[] = {
    {"-cols",		"columns",		XrmoptionSepArg, NULL},
    {"-dv",		"displayValue",		XrmoptionNoArg, "True"},
    {"-ebd",		"errorBd",		XrmoptionSepArg, NULL},
    {"-ebg",		"errorBack",		XrmoptionSepArg, NULL},
    {"-efg",		"errorFore",		XrmoptionSepArg, NULL},
    {"-ehl",		"errorHl",		XrmoptionSepArg, NULL},
    {"-eibd",		"errorIbd",		XrmoptionSepArg, NULL},
    {"-elevel",		"errorLevel",		XrmoptionSepArg, NULL},
    {"-ep",		"errorProg",		XrmoptionSepArg, NULL},
    {"-fbd",		"fatalBd",		XrmoptionSepArg, NULL},
    {"-fbg",		"fatalBack",		XrmoptionSepArg, NULL},
    {"-ffg",		"fatalFore",		XrmoptionSepArg, NULL},
    {"-fhl",		"fatalHl",		XrmoptionSepArg, NULL},
    {"-fibd",		"fatalIbd",		XrmoptionSepArg, NULL},
    {"-fp",		"fatalProg",		XrmoptionSepArg, NULL},
    {"-h",		"*Paned.height",	XrmoptionSepArg, NULL},
    {"-lebg",		"lErrorBack",		XrmoptionSepArg, NULL},
    {"-lefg",		"lErrorFore",		XrmoptionSepArg, NULL},
    {"-lfbg",		"lFatalBack",		XrmoptionSepArg, NULL},
    {"-lffg",		"lFatalFore",		XrmoptionSepArg, NULL},
    {"-lobg",		"lOkBack",		XrmoptionSepArg, NULL},
    {"-lofg",		"lOkFore",		XrmoptionSepArg, NULL},
    {"-lwbg",		"lWarnBack",		XrmoptionSepArg, NULL},
    {"-lwfg",		"lWarnFore",		XrmoptionSepArg, NULL},
    {"-obd",		"okBd",			XrmoptionSepArg, NULL},
    {"-obg",		"okBack",		XrmoptionSepArg, NULL},
    {"-ofg",		"okFore",		XrmoptionSepArg, NULL},
    {"-ohl",		"okHl",			XrmoptionSepArg, NULL},
    {"-oibd",		"okIbd",		XrmoptionSepArg, NULL},
    {"-op",		"okProg",		XrmoptionSepArg, NULL},
    {"-rows",		"rows",			XrmoptionSepArg, NULL},
    {"-rt",		"retries",		XrmoptionSepArg, NULL},
    {"-scoll",		"scaleColl",		XrmoptionSepArg, NULL},
    {"-scpu",		"scaleCpu",		XrmoptionSepArg, NULL},
    {"-sdisks",		"scaleDisks",		XrmoptionSepArg, NULL},
    {"-serr",		"scaleErr",		XrmoptionSepArg, NULL},
    {"-sierr",		"scaleIerr",		XrmoptionSepArg, NULL},
    {"-sintr",		"scaleIntr",		XrmoptionSepArg, NULL},
    {"-sipkt",		"scaleIpkt",		XrmoptionSepArg, NULL},
    {"-sload",		"scaleLoad",		XrmoptionSepArg, NULL},
    {"-sn",		"shortName",		XrmoptionNoArg, "True"},
    {"-soerr",		"scaleOerr",		XrmoptionSepArg, NULL},
    {"-sopkt",		"scaleOpkt",		XrmoptionSepArg, NULL},
    {"-spage",		"scalePage",		XrmoptionSepArg, NULL},
    {"-spgpgin",	"scalePgpgin",		XrmoptionSepArg, NULL},
    {"-spgpgout",	"scalePgpgout",		XrmoptionSepArg, NULL},
    {"-spkts",		"scalePkts",		XrmoptionSepArg, NULL},
    {"-spswpin",	"scalePswpin",		XrmoptionSepArg, NULL},
    {"-spswpout",	"scalePswpout",		XrmoptionSepArg, NULL},
    {"-sswap",		"scaleSwap",		XrmoptionSepArg, NULL},
    {"-sswt",		"scaleSwt",		XrmoptionSepArg, NULL},
    {"-ssys",		"scaleSys",		XrmoptionSepArg, NULL},
    {"-suser",		"scaleUser",		XrmoptionSepArg, NULL},
    {"-to",		"timeout",		XrmoptionSepArg, NULL},
    {"-update",		"*StripChart.update",	XrmoptionSepArg, NULL},
    {"-w",		"*Paned.width",		XrmoptionSepArg, NULL},
    {"-wbd",		"warnBd",		XrmoptionSepArg, NULL},
    {"-wbg",		"warnBack",		XrmoptionSepArg, NULL},
    {"-wfg",		"warnFore",		XrmoptionSepArg, NULL},
    {"-whl",		"warnHl",		XrmoptionSepArg, NULL},
    {"-wibd",		"warnIbd",		XrmoptionSepArg, NULL},
    {"-wlevel",		"warnLevel",		XrmoptionSepArg, NULL},
    {"-wp",		"warnProg",		XrmoptionSepArg, NULL},
};

static XtActionsRec	scact[] = {
    {"popupscale", popupscale},
    {"popdownscale", popdownscale}
};
#define NUMSCACT	(sizeof(scact) / sizeof(XtActionsRec))

static XtActionsRec	actions[] = {
    {"quit", quit},
};
Atom			wm_delete_window;

main(argc, argv)

int	argc;
char	**argv;

{
  Widget		toplevel;
  Widget		form;
  Widget		pane = NULL;
  int			i;
  int			n;
  Arg			args[10];
  XtAppContext		appcon;
  String		mbtrans =
      "<EnterWindow>:	highlight()\n\
       <LeaveWindow>:	reset()\n\
       <BtnDown>:	set() notify() PopupMenu()";
  XtTranslations	mbtt;
  String		sctrans =
      "<BtnDown>:	popupscale()\n\
       <BtnUp>:		popdownscale()";
  XtTranslations	scacttt;
  Widget		*lw;	/* Last column or row of widgets	*/
  int			numwgt;

  progname = argv[0];
  toplevel = XtAppInitialize(&appcon, "XMeter", options,
		XtNumber(options), &argc, argv, NULL, NULL, ZERO);
  if (argc < 2)
      usage();
  XtGetApplicationResources(toplevel, (XtPointer) &ar,
			    resources, XtNumber(resources), NULL, ZERO);
  form = XtCreateManagedWidget("form", formWidgetClass, toplevel, NULL, ZERO);
	/* Handle f.delete	*/
  XtAppAddActions(XtWidgetToApplicationContext(toplevel), actions,
		  XtNumber(actions));
  XtOverrideTranslations(toplevel,
	XtParseTranslationTable("<Message>WM_PROTOCOLS: quit()"));
  if (strcmp("-v", argv[1]) == 0)
       printversion();
  init(toplevel, &lw);
  createmenus(form);
  mbtt = XtParseTranslationTable(mbtrans);
  scacttt = XtParseTranslationTable(sctrans);
  XtAppAddActions(appcon, scact, NUMSCACT);
  /*
   * Each meter consists of a paned widget, each paned widget has two
   * children, which are a menubutton widget and stripchart widget.
   * The following loop creates and initializes the appropriate widgets.
   */
  for (i = 1, numwgt = 0; i < argc; i++, numwgt++) {
      meterlist = initmeter(meterlist, &i, argc, argv);
	/*
	 * Create paned widget.
	 */
      n = 0;
      if (ar.columns > 0) {		/* Locate pane in form ...	*/
	  XtSetArg(args[n], XtNfromVert, lw[numwgt % ar.columns]); n++;
	  if (numwgt % ar.columns != 0) {
	      XtSetArg(args[n], XtNfromHoriz, pane); n++;
	  }
      } else {
	  XtSetArg(args[n], XtNfromHoriz, lw[numwgt % ar.rows]); n++;
	  if (numwgt % ar.rows != 0) {
	      XtSetArg(args[n], XtNfromVert, pane); n++;
	  }
      }
      XtSetArg(args[n], XtNborder, ar.bd[OK]); n++;
      XtSetArg(args[n], XtNinternalBorderColor, ar.ibd[OK]); n++;
      pane = XtCreateManagedWidget(meterlist->sh->label, panedWidgetClass,
				   form, args, n);
      lw[numwgt % (ar.columns > 0 ? ar.columns : ar.rows)] = pane;
      meterlist->pdwidget = pane;
	/*
	 * Create menubutton widget.
	 */
      n = 0;
      XtSetArg(args[n], XtNshowGrip, XtEno); n++;
      XtSetArg(args[n], XtNlabel, meterlist->label); n++;
      XtSetArg(args[n], XtNbackground, ar.lback[OK]); n++;
      XtSetArg(args[n], XtNforeground, ar.lfore[OK]); n++;
      XtSetArg(args[n], XtNmenuName, STATMENU); n++;
      XtSetArg(args[n], XtNtranslations, mbtt); n++;
      meterlist->mbwidget = XtCreateManagedWidget("menu",
				menuButtonWidgetClass, pane, args, n);
      XtRemoveAllCallbacks(meterlist->mbwidget, XtNcallback);
      XtAddCallback(meterlist->mbwidget, XtNcallback,
		    (XtCallbackProc) selecthost, (XtPointer) meterlist);
	/*
	 * Create stripchart widget.
	 */
      n = 0;
      XtSetArg(args[n], XtNfromVert, meterlist->mbwidget); n++;
      XtSetArg(args[n], XtNresizable, TRUE); n++;
      XtSetArg(args[n], XtNbackground, ar.back[OK]); n++;
      XtSetArg(args[n], XtNforeground, ar.fore[OK]); n++;
      XtSetArg(args[n], XtNhighlight, ar.hl[OK]); n++;
      XtSetArg(args[n], XtNtranslations, scacttt); n++;
      meterlist->scwidget = XtCreateManagedWidget("load",
				stripChartWidgetClass, pane, args, n);
      XtRemoveAllCallbacks(meterlist->scwidget, XtNgetValue);
      XtAddCallback(meterlist->scwidget, XtNgetValue,
		    (XtCallbackProc) getstatus, (XtPointer) meterlist);
  }
  free(lw);
#ifdef SVR4
  sigset(SIGCHLD, freechild);
#else
  signal(SIGCHLD, freechild);		/* Clean up after our children	*/
#endif
  XtRealizeWidget(toplevel);
  wm_delete_window = XInternAtom(XtDisplay(toplevel), "WM_DELETE_WINDOW",
				 False);
  XSetWMProtocols(XtDisplay(toplevel), XtWindow(toplevel),
		  &wm_delete_window, 1);
  setokbackgrounds(XtDisplay(toplevel), XtScreen(toplevel));
  XtAppMainLoop(appcon);
  return(0);
}

usage()

{
  int	i;

  fprintf(stderr, "Usage: %s [toolkit options] [options] [stat] host ...\n",
	   progname);
  fprintf(stderr, "  Options are:");
  for (i = 0; i < XtNumber(options); i++) {
      if (i % 8 == 0)
          fprintf(stderr, "\n    ");
      else
	  fprintf(stderr, ", ");
      fprintf(stderr, "%s", options[i].option);
  }
  fprintf(stderr, ", -v\n  Stats to watch are:");
  for (i = 0; i < MAXSTAT; i++) {
      if (i % 8 == 0)
          fprintf(stderr, "\n    ");
      else
	  fprintf(stderr, ", ");
      fprintf(stderr, "-%s", sd[i].name);
  }
  fprintf(stderr, "\n");
  exit(1);
}

/*
 * printversion - Print version number and exit.
 */
printversion()

{
  printf("XMeter version %d.%d\n", MAJORVERSION, PATCHLEVEL);
  exit(0);
}

/*
 * init - Initialze various globals.
 */
init(toplevel, lw)

Widget	toplevel;
Widget	**lw;

{
  int		i;
  Window	rw;
  int		x;
  int		y;
  int		bwidth;
  int		depth;

  ptto.tv_sec = ar.timeout;
  tto.tv_sec = ar.timeout * ar.retries;
  if (ar.columns <= 0 && ar.rows <= 0)
      ar.columns = 1;
  if (ar.columns > 0) {
      if ((*lw = (Widget *) calloc(ar.columns, sizeof(Widget))) == NULL)
	  fatal("lw");	
  } else
      if ((*lw = (Widget *) calloc(ar.rows, sizeof(Widget))) == NULL)
	  fatal("lw");
  for (i = 0; i < MAXBACKS; i++)
      if (ar.bitmap[i].bm != bitmapdef)	/* Get bitmap dimensions	*/
	  XGetGeometry(XtDisplay(toplevel), ar.bitmap[i].bm, &rw, &x, &y,
		(unsigned int *) &ar.bitmap[i].w,
		(unsigned int *) &ar.bitmap[i].h,
		(unsigned int *) &bwidth, (unsigned int *) &depth);
}

/*
 * createmenus - Create menus used by each meter.  Currently there's just
 *   one which consists of the stats that can be monitored, and a quit
 *   command.
 */
createmenus(parent)

Widget	parent;

{
  int		i;
  int		n;
  Widget	menu;
  Widget	me;
  Arg		args[10];

  menu = XtCreatePopupShell(STATMENU, simpleMenuWidgetClass, parent,
			    NULL, ZERO);
  for (i = 0; i < MAXSTAT; i++) {
      n = 0;
      XtSetArg(args[n], XtNlabel, sd[i].name); n++;
      me = XtCreateManagedWidget(sd[i].name, smeBSBObjectClass, menu,
				 args, n);
      XtRemoveAllCallbacks(me, XtNcallback);
      XtAddCallback(me, XtNcallback,
		    (XtCallbackProc) changestat, (XtPointer) i);
  }
  n = 0;
  XtSetArg(args[n], XtNlabel, "quit"); n++;
  me = XtCreateManagedWidget("quit", smeBSBObjectClass, menu, args, n);
  XtRemoveAllCallbacks(me, XtNcallback);
  XtAddCallback(me, XtNcallback, (XtCallbackProc) quit, (XtPointer) NULL);
}

void
popupscale(w, event, params, num)

Widget		w;
XEvent		*event;
String		*params;
Cardinal	num;

{
  int		n;
  Arg		args[10];
  Position	x;
  Position	y;
  static char	buf[100];
  METER		*h;
  time_t	tdown;

  for (h = meterlist; h; h = h->nxt)
      if (h->scwidget == w)
	  break;
  if (!h) {
      fprintf(stderr, "popupscale: Bad stripchart widget\n");
      exit(1);
  }
  XtTranslateCoords(w, (Position) 0, (Position) 0, &x, &y);
  n = 0;
  XtSetArg(args[n], XtNx, x); n++;
  XtSetArg(args[n], XtNy, y); n++;
  popup = XtCreatePopupShell("popup", transientShellWidgetClass, w, args, n);
  if (h->sh->pid != -1) {
      tdown = time(0) - h->sh->when;
      sprintf(buf, "%s\nversion %d.%d\nscale %d\ndown %02d:%02d:%02d",
	      h->sh->name,
	      MAJORVERSION, PATCHLEVEL,
	      h->stat == S_LOAD ? ar.scales[h->stat] / FSCALE :
				  ar.scales[h->stat],
	      tdown / (60*60), (tdown / 60) % 60, tdown % 60);
  } else
      sprintf(buf, "%s\nversion %d.%d\nscale %d",
	      h->sh->name,
	      MAJORVERSION, PATCHLEVEL,
	      h->stat == S_LOAD ? ar.scales[h->stat] / FSCALE :
				  ar.scales[h->stat]);
  n = 0;
  XtSetArg(args[n], XtNlabel, buf); n++;
  XtCreateManagedWidget("label", labelWidgetClass, popup, args, n);
  XtPopup(popup, XtGrabNone);
}

void
popdownscale(w, event, params, num)

Widget		w;
XEvent		*event;
String		*params;
Cardinal	num;

{
  if (popup)			/* Can be NULL in some cases apparently	*/
      XtDestroyWidget(popup);
}

/*
 * setokbackgrounds - Set initial background pixmaps.  Can't do this
 *   when the stripchart widgets are created as they have to be
 *   realized before the pixmaps can be created.
 */
setokbackgrounds(d, s)

Display	*d;
Screen	*s;

{
  METER		*h;
  int		n;
  Arg		args[10];

  if (ar.bitmap[OK].bm != bitmapdef) {
      for (h = meterlist; h; h = h->nxt) {
	  h->pm = XmuCreatePixmapFromBitmap(d, XtWindow(h->scwidget),
					    ar.bitmap[OK].bm,
					    ar.bitmap[OK].w, ar.bitmap[OK].h,
					    DefaultDepthOfScreen(s),
					    ar.fore[OK],
					    ar.back[OK]);
	  n = 0;
	  XtSetArg(args[n], XtNbackgroundPixmap, h->pm); n++;
	  XtSetValues(h->scwidget, args, n);
      }
  }
}

/*
 * freechild - Clean up child processes.
 */
SIGTYPE
freechild(sig)

int	sig;

{
  int	pid;
  METER	*h;

  pid = wait(NULL);
#ifdef SYSV
#ifdef SVR4
  sigset(SIGCHLD, freechild);
#else !SVR4
  signal(SIGCHLD, freechild);
#endif SVR4
#endif SYSV
  if (pid <= 0)
      return;
  for (h = meterlist; h; h = h->nxt)
      if (h->sh->pid == pid) {		/* Start updating host again	*/
	  h->sh->pid = -1;
	  break;
      }
}

/*
 * getstatus - Get status from remote host, update meter appropriately,
 *   including changing colors, background pixmap and label if necessary.
 */
getstatus(w, h, data)

Widget	w;
METER	*h;
char	*data;

{
  int			l;
  int			n;
  int			s;
  register SHMETER	*sh;
  Arg			args[10];

  sh = h->sh;
  if (h->oldstate == FATAL && sh->pid != -1) {	/* Ignore dead hosts	*/
      *(double *) data = 0.0;
      return;
  }
  if (h->oldjumpscroll) {		/* Restore old jumpscroll value	*/
      n = 0;
      XtSetArg(args[n], XtNjumpScroll, h->oldjumpscroll); n++;
      XtSetValues(h->scwidget, args, n);
      h->oldjumpscroll = 0;
  }
  s = state(l = getmeter(h), h);
  if (ar.displayValue) {
      n = 0;				/* Update menubutton		*/
      XtSetArg(args[n], XtNbackground, ar.lback[s]); n++;
      XtSetArg(args[n], XtNforeground, ar.lfore[s]); n++;
      sprintf(h->label, "%s %s %.2f", sh->label,
	      s == FATAL ? DMSG : sd[h->stat].name,
	      s == FATAL ? 0.0 : (float) l / ar.scales[h->stat]);
      XtSetArg(args[n], XtNlabel, h->label); n++;
      XtSetValues(h->mbwidget, args, n);
  }
  *(double *) data = (s == FATAL) ? 0.0 : (double) l / ar.scales[h->stat];
  if (s != h->oldstate) {
      sh->when = time(0);
      if (ar.bitmap[h->oldstate].bm != bitmapdef)
	  XFreePixmap(XtDisplay(w), h->pm);
      n = 0;				/* Update stripchart widget	*/
      XtSetArg(args[n], XtNbackground, ar.back[s]); n++;
      XtSetArg(args[n], XtNforeground, ar.fore[s]); n++;
      XtSetArg(args[n], XtNhighlight, ar.hl[s]); n++;
      if (ar.bitmap[s].bm != bitmapdef) {
	  h->pm = XmuCreatePixmapFromBitmap(XtDisplay(w),
					    XtWindow(w),
					    ar.bitmap[s].bm,
					    ar.bitmap[s].w, ar.bitmap[s].h,
					    DefaultDepthOfScreen(XtScreen(w)),
					    ar.fore[s],
					    ar.back[s]);
	  XtSetArg(args[n], XtNbackgroundPixmap, h->pm); n++;
      } else if (ar.bitmap[h->oldstate].bm != bitmapdef) {
	  XtSetArg(args[n], XtNbackgroundPixmap, XtUnspecifiedPixmap); n++;
      }
      XtSetValues(w, args, n);
      n = 0;				/* Update menubutton widget	*/
      XtSetArg(args[n], XtNbackground, ar.lback[s]); n++;
      XtSetArg(args[n], XtNforeground, ar.lfore[s]); n++;
      if (ar.displayValue) {
	  sprintf(h->label, "%s %s %.2f", sh->label,
		  s == FATAL ? DMSG : sd[h->stat].name,
		  s == FATAL ? 0.0 : (float) l / ar.scales[h->stat]);
	  XtSetArg(args[n], XtNlabel, h->label); n++;
      } else if (s == FATAL || h->oldstate == FATAL) {
	  sprintf(h->label, "%s %s", sh->label,
		   s == FATAL ? DMSG : sd[h->stat].name);
	  XtSetArg(args[n], XtNlabel, h->label); n++;
      }
      XtSetValues(h->mbwidget, args, n);
      n = 0;				/* Update paned widget		*/
      XtSetArg(args[n], XtNborder, ar.bd[s]); n++;
      XtSetArg(args[n], XtNinternalBorderColor, ar.ibd[s]); n++;
      XtSetValues(h->pdwidget, args, n);
      if (ar.prog[s])
	  runprog(h, s);
      if (s == FATAL)
	  sh->pid = waitforhost(h);
  }
  h->oldstate = s;
}

/*
 * waitforhost - Fork process which will wait for host to come back up.
 *   Child exits if parent is init (pid 1), to handle case where parent
 *   has gone away.
 */
int
waitforhost(h)

METER	*h;

{
  int	pid;

  if (pid = fork())
      return(pid);
#ifdef SVR4
  sigset(SIGTERM, SIG_DFL);
#else !SVR4
  signal(SIGTERM, SIG_DFL);
#endif !SVR4
  while (1) {
      sleep(10);
      if (getport(h) > 0 || getppid() < 2)
	  exit(0);
  }
}

/*
 * runprog - Run user specified alert program.
 */
runprog(h, s)

METER	*h;
int	s;

{
  char	oldstate[4];
  char	state[4];

  sprintf(oldstate, "%d", h->oldstate);
  sprintf(state, "%d", s);
  if (vfork())
      return;			/* Parent just returns		*/
  execlp(ar.prog[s], ar.prog[s], oldstate, state,
	 h->sh->label, sd[h->stat].name, 0);
  fatal(ar.prog[s]);
}

/*
 * selecthost - Select host for next changestat().
 */
selecthost(w, h, data)

Widget	w;
METER	*h;
char	*data;

{
  selected = h;
}

/*
 * changestat - Change statistic we're looking at.  To clear the stripchart
 *   it's necessary to poke the "interval" field in the StripChartWidget
 *   structure, and then set "jumpScroll" to the width of the stripchart.
 *   jumpScroll is saved here and restored in getstatus().
 */
#include <X11/IntrinsicP.h>
#include <X11/Xaw/StripCharP.h>
changestat(w, statidx, data)

Widget	w;
int	statidx;
char	*data;

{
  int	n;
  Arg	args[10];
  int	width;

  selected->stat = statidx;
  sprintf(selected->label, "%s %s", selected->sh->label, sd[statidx].name);
  n = 0;
  XtSetArg(args[n], XtNlabel, selected->label); n++;
  XtSetValues(selected->mbwidget, args, n);
  n = 0;
  XtSetArg(args[n], XtNwidth, &width); n++;
  XtSetArg(args[n], XtNjumpScroll, &selected->oldjumpscroll); n++;
  XtGetValues(selected->scwidget, args, n);
  n = 0;
  XtSetArg(args[n], XtNjumpScroll, width); n++;
  XtSetValues(selected->scwidget, args, n);
  ((StripChartWidget) selected->scwidget)->strip_chart.interval = width;
}

/*
 * state - Return state of current stat.
 */
int
state(l, h)

int	l;
METER	*h;

{
  if (l < 0)
      return(FATAL);
  else if (l < ar.warnLevel * ar.scales[h->stat])
      return(OK);
  else if (l < ar.errorLevel * ar.scales[h->stat])
      return(WARN);
  else
      return(ERROR);
}

/*
 * The following functions return the value of the specified stat, which
 * is normally computed by taking the difference between the current
 * value and the previous value, and dividing by the update interval in
 * order to get the current rate.
 */
#define DIF(m,fld)	(m->sh->st[m->sh->idx].fld - \
			 m->sh->st[m->sh->idx ^ 1].fld)
#define INT(m)		(DIF(m,curtime.tv_sec) <= 0 ? 1 : \
						      DIF(m,curtime.tv_sec))

int
fcoll(h)

METER	*h;

{
  return(DIF (h, if_collisions) / INT(h));
}

int
fcpu(h)

METER	*h;

{
  int	i;
  int	t;
  int	d[CPUSTATES];

  for (t = 0, i= 0; i < CPUSTATES; i++)
      t += (d[i] = DIF (h, cp_time[i]));
  return(t ? (100 * (d[CP_USER]+d[CP_NICE]+d[CP_SYS])) / t : 0);
}

int
fdisks(h)

METER	*h;

{
  int	i;
  int	t;

  for (t = 0, i= 0; i < 4; i++)
      t += (DIF (h, dk_xfer[i]));
  return(t / INT(h));
}

int
ferr(h)

METER	*h;

{
  return(fierr(h) + foerr(h));
}

int
fierr(h)

METER	*h;

{
  return(DIF (h, if_ierrors) / INT(h));
}

int
fintr(h)

METER	*h;

{
  return(DIF (h, v_intr) / INT(h));
}

int
fipkt(h)

METER	*h;

{
  return(DIF (h, if_ipackets) / INT(h));
}

int
fload(h)

METER	*h;

{
  return(h->sh->st[h->sh->idx].avenrun[0]);
}

int
foerr(h)

METER	*h;

{
  return(DIF (h, if_oerrors) / INT(h));
}

int
fopkt(h)

METER	*h;

{
  return(DIF (h, if_opackets) / INT(h));
}

int
fpage(h)

METER	*h;

{
  return((DIF (h, v_pgpgin) + DIF (h, v_pgpgout)) / INT(h));
}

int
fpgpgin(h)

METER	*h;

{
  return(DIF (h, v_pgpgin) / INT(h));
}

int
fpkts(h)

METER	*h;

{
  return(fipkt(h) + fopkt(h));
}

int
fpgpgout(h)

METER	*h;

{
  return(DIF (h, v_pgpgout) / INT(h));
}

int
 fswap(h)

METER	*h;

{
  return(fpswpin(h) + fpswpout(h));
}

int
 fpswpin(h)

METER	*h;

{
  return(DIF (h, v_pswpin) / INT(h));
}

int
fpswpout(h)

METER	*h;

{
  return(DIF (h, v_pswpout) / INT(h));
}

int
fswt(h)

METER	*h;

{
  return(DIF (h, v_swtch) / INT(h));
}

int
fsys(h)

METER	*h;

{
  int	i;
  int	t;
  int	d[CPUSTATES];

  for (t = 0, i= 0; i < CPUSTATES; i++)
      t += (d[i] = DIF (h, cp_time[i]));
  return(t ? (100 * d[CP_SYS]) / t : 0);
}

int
fuser(h)

METER	*h;

{
  int	i;
  int	t;
  int	d[CPUSTATES];

  for (t = 0, i= 0; i < CPUSTATES; i++)
      t += (d[i] = DIF (h, cp_time[i]));
  return(t ? (100 * (d[CP_USER] + d[CP_NICE])) / t : 0);
}

/*
 * getmeter - Executes rstat(3) call to read statistics for specified host.
 *   I do all the rpc junk myself so that I have better control over timeouts
 *   than rstat(3) gives me.  If we're watching multiple stats
 *   on the same host I only do one rstat(3) call (refcnt and curcnt are
 *   used for this).
 */
int
getmeter(h)

register METER	*h;

{
  enum clnt_stat	cs;
  register SHMETER	*sh;
  int			p;

  sh = h->sh;
  if (sh->curcnt >= sh->refcnt)
      sh->curcnt = 0;
  if (!sh->curcnt++) {
      if (sh->clnt == NULL) {
          if ((p = getport(h)) <= 0)
	      return(-1);
	  sh->addr.sin_port = htons(p);
          sh->s = RPC_ANYSOCK;
          if (!(sh->clnt = clntudp_create(&sh->addr, RSTATPROG, RSTATVERS_TIME,
					  ptto, &sh->s)))
	      return(-1);
	  sh->first = 1;
          sh->idx = 0;
      } else {
	  sh->first = 0;
	  sh->idx ^= 1;
      }
      cs = clnt_call(sh->clnt, RSTATPROC_STATS, xdr_void, 0, xdr_statstime,
		     (caddr_t) &sh->st[sh->idx], tto);
      if (cs != RPC_SUCCESS) {
          clnt_destroy(sh->clnt);
	  close(sh->s);	/* Some clnt_destroy's don't do this	*/
          sh->clnt = NULL;
          return(-1);
      }
  }
  return(sh->first ? 0 : sh->clnt == NULL ? -1 : (sd[h->stat].val)(h));
}

/*
 * getport - Get port rstatd is listening on.
 */
int
getport(h)

METER	*h;

{
  CLIENT		*c;
  enum clnt_stat	cs;
  static struct pmap	pm = {RSTATPROG, RSTATVERS_TIME, IPPROTO_UDP, 0};
  unsigned short	p;
  register SHMETER	*sh;

  sh = h->sh;
  sh->s = RPC_ANYSOCK;
  sh->addr.sin_port = htons(PMAPPORT);
  if (!(c = clntudp_create(&sh->addr, PMAPPROG, PMAPVERS, ptto, &sh->s)))
      return(-1);
  cs = clnt_call(c, PMAPPROC_GETPORT, xdr_pmap, (caddr_t) &pm,
		 xdr_u_short, (caddr_t) &p, tto);
  clnt_destroy(c);
  close(sh->s);		/* Some clnt_destroys don't do this	*/
  return(cs == RPC_SUCCESS ? p : -1);
}

/*
 * initmeter - Fill in METER and SHMETER structures for this stat.
 */
METER
*initmeter(meterlist, idx, argc, argv)

METER	*meterlist;
int	*idx;
int	argc;
char	**argv;

{
  register METER	*h;
  struct hostent	*he;
  int			i;
  char			*cp;

  /*
   * Create and fill in METER struct.
   */
  if (!(h = (METER *) malloc(sizeof(METER))))
      fatal("METER");
  h->nxt = meterlist;
  cp = (argv[*idx][0] == '-') ? &argv[*idx][1] : ar.defStat;
  for (i = 0; i < MAXSTAT; i++)
      if (strcmp(cp, sd[i].name) == 0) {
	  h->stat = i;
	  break;
      }
  if ((cp != ar.defStat) && (++(*idx) == argc || i >= MAXSTAT))
      usage();
  if ((he = gethostbyname(argv[*idx])) == NULL)
       fatal(argv[*idx]);
  h->oldstate = OK;
  h->oldjumpscroll = 0;
  h->sh = newshmeter(he);
  if (!(h->label = (char *) malloc(strlen(h->sh->label) + 2 + MAXSTATNAME +
				   MAXSTATVALUE + 6)))
      fatal("label");
  sprintf(h->label, "%s %s", h->sh->label, sd[h->stat].name);
  return(h);
}

/*
 * newshmeter - Return SHMETER structure for host.  Keeps
 *   reference count for each SHMETER.  If an SHMETER for this host
 *   doesn't already exist then one is created and saved on the shmeter
 *   linked list.  Note that he->h_name gets stepped on by this routine
 *   after it is saved in sh->name.
 *   
 */
SHMETER
*newshmeter(he)

struct hostent	*he;

{
  SHMETER	*sh;
  char		*cp;

  /*
   * If we're already looking at this host then just return existing
   * structure.
   */
  for (sh = shmeters; sh; sh = sh->nxt)
      if (strcmp(sh->name, he->h_name) == 0) {
	  sh->refcnt++;		/* Count how many references		*/
	  return(sh);
      }
  /*
   * Not looking at this host yet, create and fill in SHMETER struct.
   */
  if (!(sh = (SHMETER *) malloc(sizeof(SHMETER))))
      fatal("SHMETER");
  sh->nxt = shmeters;		/* Save this struct in linked list	*/
  shmeters = sh;
  if (!(sh->name = mystrdup(he->h_name)))
      fatal("newshmeter");
  if (ar.shortName && (cp = strchr(he->h_name, '.'))) {
      *cp = '\0';
      if (!(sh->label = mystrdup(he->h_name)))
	  fatal("initmeter");
  } else
      sh->label = sh->name;
  sh->addr.sin_family = AF_INET;
  sh->addr.sin_addr.s_addr = *(int *) he->h_addr;
  sh->clnt = NULL;
  sh->refcnt = 1;
  sh->curcnt = 0;
  sh->pid = -1;
  return(sh);
}

/*
 * mystrdup - Not everyone has strdup() in their C library, so provide
 *   our own.
 */
char *
mystrdup(s)

char	*s;

{
  char	*t;

  if (t = (char *) malloc(strlen(s) + 1))
      return(strcpy(t, s));
  else
      return(NULL);
}

void
quit(w)

Widget	w;

{
  XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
  exit(0);
}

fatal(m)

char	*m;

{
  perror(m);
  exit(1);
}
