/*----------------------------------------------------------------------------
--
--  Module:           xitWinInfo
--
--  Project:          Xtools
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Fetch information for a window tree under X-Windows.
--
--  Filename:         xitWinInfo.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-11-29
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitWinInfo.c, Version: 1.1, Date: 95/02/18 15:10:52";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Xatom.h>

#include "System.h"
#include "LstLinked.h"

#include "xitWinInfo.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xitFetchWindowInfo( LST_DESC_TYPE  win_info_list,
                      Display        *display,
                      Window         window,
                      UINT32         flags,
                      int            current_depth,
                      int            list_depth,
                      int            max_depth )
{

  /* Variables. */
  int           index;
  int           trans_x;
  int           trans_y;
  int           x;
  int           y;
  unsigned int  border_width;
  unsigned int  depth;
  unsigned int  num_children;
  unsigned int  height;
  unsigned int  width;
  char          *name;
  Status        okStatus;
  Window        child;
  Window        *children;
  Window        parent;
  Window        root_window;
  XClassHint    class_hint;
  LST_STATUS    lst_status;
  XIT_WI_INFO   win_info;


  /* Code. */

  if( win_info_list == NULL )
    return;

  /* Have we reached the max depth? */
  if( current_depth > max_depth )
    return;

  /* Are we on the correct depth? */
  if( list_depth != XIT_WI_ALL_DEPTHS && list_depth != current_depth ) {

    if( current_depth + 1 > max_depth )
      return;

    okStatus = XQueryTree( display, window, &root_window,
                           &parent, &children, &num_children );
    if( ! okStatus )
      return;

    /* Check the children. */
    for( index = 0; index < num_children; index++ ) {
      xitFetchWindowInfo( win_info_list,
                          display, children[ index ],
                          flags, current_depth + 1, 
                          list_depth, max_depth );
    }

    if( children != NULL )
      SysFree( children );

    return;

  } /* if */


  /* Initialization. */
  win_info.window_id      = window;
  win_info.level          = current_depth;
  win_info.res_class[ 0 ] = '\0';
  win_info.res_name[ 0 ]  = '\0';
  win_info.name[ 0 ]      = '\0';
  win_info.pos_x          = -1;
  win_info.pos_y          = -1;
  win_info.width          = -1;
  win_info.height         = -1;
  win_info.width_inc      = -1;
  win_info.height_inc     = -1;
  win_info.flags          = 0;

  class_hint.res_name  = NULL;
  class_hint.res_class = NULL;


  /* Fetch resource class and name? */
  if( flagIsSet( flags, (XIT_WI_RES_NAME | XIT_WI_RES_CLASS) ) ) {

    /* Get the class hint. */
    (void) XGetClassHint( display, window, &class_hint );

    /* Save application name? */
    if( class_hint.res_name != NULL && flagIsSet( flags, XIT_WI_RES_NAME ) )
      strcpy( win_info.res_name, class_hint.res_name );

    /* Save class name? */
    if( class_hint.res_class != NULL && flagIsSet( flags, XIT_WI_RES_CLASS ) )
      strcpy( win_info.res_class, class_hint.res_class );

  } /* if */    


  /* Fetch the window name (WM_NAME). */
  XFetchName( display, window, &name );

  if( flagIsSet( flags, XIT_WI_WIN_NAME ) && name != NULL ) {
    strcpy( win_info.name, name );

    SysFree( name );
  } /* if */


  /* Fetch the position and geometry. */
  okStatus = XGetGeometry( display, window, 
                           &root_window, 
                           &x, &y, &width, &height,
                           &border_width, &depth );

  if( flagIsSet( flags, XIT_WI_GEOMETRY ) && okStatus ) {

    /* Translate into 'real world' coordinates (relative to root). */
    okStatus = XTranslateCoordinates( display, window, root_window,
                                      0, 0, &trans_x, &trans_y, &child );
    if( okStatus ) {
      win_info.pos_x  = trans_x - border_width;
      win_info.pos_y  = trans_y - border_width;
      win_info.width  = width;
      win_info.height = height;
    }

  } /* if */


  /* Fetch atom for Window Manager state and size hints. */
  {

    int            actual_format;
    int            status;
    unsigned long  bytes_after;
    unsigned long  nitems;
    unsigned char  *property;
    Atom           actual_type;
    Atom           wm_state;

    wm_state = XInternAtom( display, "WM_STATE", True );

    status = XGetWindowProperty( display, window, 
                                 wm_state, 0L, 1L,
                                 False, AnyPropertyType, &actual_type,
                                 &actual_format, &nitems, &bytes_after,
                                 &property );

    if( property != NULL ) {

      int  *int_ref;

      int_ref = (int *)property;

      if( (*int_ref & 0x3) == WithdrawnState )
        flagSet( win_info.flags, XIT_WI_WITHDRAWN );

      if( (*int_ref & 0x3) == NormalState )
        flagSet( win_info.flags, XIT_WI_NORMAL );

      if( (*int_ref & 0x3) == IconicState )
        flagSet( win_info.flags, XIT_WI_ICONIC );

    } /* if */

    status = XGetWindowProperty( display, window, 
                                 XA_WM_NORMAL_HINTS, 0L, 512L,
                                 False, XA_WM_SIZE_HINTS, &actual_type,
                                 &actual_format, &nitems, &bytes_after,
                                 &property );
    if( status == 0 && actual_type == XA_WM_SIZE_HINTS && property != NULL ) {

      XSizeHints  *size_ref;

      size_ref = (XSizeHints *) property;

      if( flagIsSet( size_ref -> flags, PResizeInc ) ) {
        win_info.width_inc  = win_info.width  / size_ref -> width_inc;
        win_info.height_inc = win_info.height / size_ref -> height_inc;
      }

    } /* if */

  } /* block */

	
  /* Save the window information. */
  lst_status = LstLinkInsertLast( win_info_list, &win_info );


  /* Check the children window. */
  if( current_depth + 1 > max_depth )
    return;

  okStatus = XQueryTree( display, window, &root_window,
                         &parent, &children, &num_children );
  if( ! okStatus )
    return;

  for( index = 0; index < num_children; index++ ) {
    xitFetchWindowInfo( win_info_list,
                        display, children[ index ],
                        flags, current_depth + 1, 
                        list_depth, max_depth );
  }

  if( children != NULL )
    SysFree( children );


  return;

} /* listWindowTree */
