/*----------------------------------------------------------------------------
--
--  Module:           xitInfoBook
--
--  Project:          xit - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Select book in the information database.
--
--  Filename:         xitInfoBook.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-17
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitInfoBook.c, Version: 1.1, Date: 95/02/18 15:10:35";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <limits.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/List.h>

#include "System.h"
#include "Message.h"

#include "msgTopic.h"
#include "xitTools.h"
#include "xitInfoBook.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the print window. */
#define bookLa                dataLocalW[  0 ]
#define bookLi                dataLocalW[  1 ]

/* Max number of books we can handle. */
#define MAX_BOOKS      500


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record while searching entries. */
typedef struct {

  /* The book window. */
  Widget  bookW;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XIT_IB_ACTION_CB  actionCB;

} BOOK_REC, *BOOK_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xitInfoBook";

/* Name of text domain. */
static char  *td = "Topic";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  applyCB( Widget        widget,
           BOOK_REC_REF  book_ref,
           XtPointer     call_data );

static void 
  cancelCB( Widget        widget, 
            BOOK_REC_REF  book_ref,
            XtPointer     call_data );

static Widget
  createBookWindow( BOOK_REC_REF  book_ref,
                    Widget        parent );

static void 
  destroyCB( Widget        widget,
             BOOK_REC_REF  book_ref,
             XtPointer     call_data );

static void 
  displayBooks( BOOK_REC_REF     book_ref,
                XIT_IB_BOOK_REC  *user_books,
                int              user_books_no,
                XIT_IB_BOOK_REC  *system_books,
                int              system_books_no );

static Boolean
  fetchSelectedBook( BOOK_REC_REF  book_ref );

static void 
  okCB( Widget        widget,
        BOOK_REC_REF  book_ref,
        XtPointer     call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XIT_IB_HANDLE
  xitIbInitialize( Widget            parent,
                   XIT_IB_ACTION_CB  actionCB,
                   void              *user_data )
{

  /* Variables. */
  BOOK_REC_REF  book_ref;


  /* Code. */

  /* Create and initialize our private data. */
  book_ref = SysNew( BOOK_REC );
  if( book_ref == NULL )
    return( NULL );

  book_ref -> actionCB  = actionCB;
  book_ref -> user_data = user_data;


  /* Create the book window. */
  book_ref -> bookW = createBookWindow( book_ref, parent );

  if( book_ref -> bookW == NULL ) {
    SysFree( book_ref );

    return( NULL );
  }


  return( (XIT_IB_HANDLE) book_ref );

} /* xitIbInitialize */


/*----------------------------------------------------------------------*/

void
  xitIbDestroy( XIT_IB_HANDLE  book_handle )
{

  /* Variables. */
  BOOK_REC_REF  book_ref;


  /* Code. */

  if( book_handle == NULL )
    return;

  /* Our private data. */
  book_ref = (BOOK_REC_REF) book_handle;


  /* Destroy the window. */
  XtDestroyWidget( book_ref -> bookW );


  return;

} /* xitIbDestroy */


/*----------------------------------------------------------------------*/

void 
  xitIbSelectBook( XIT_IB_HANDLE    book_handle,
                   XIT_IB_BOOK_REC  *user_books,
                   int              user_books_no,
                   XIT_IB_BOOK_REC  *system_books,
                   int              system_books_no )
{

  /* Variables. */
  Arg           args[ 10 ];
  Cardinal      n;
  BOOK_REC_REF  book_ref;


  /* Code. */

  book_ref = (BOOK_REC_REF) book_handle;
  if( book_ref == NULL )
    return;

  /* Make sure the window is visible. */
  XtManageChild( book_ref -> bookW );


  /* Don't let our children spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( book_ref -> bookW, args, n );


  /* Display the books. */
  displayBooks( book_ref,
                user_books,   user_books_no,
                system_books, system_books_no );


  return;

} /* xitIbSelectBook */


/*----------------------------------------------------------------------*/

static Widget
  createBookWindow( BOOK_REC_REF  book_ref,
                    Widget        parent )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 2 ];
  Widget    bookFd;
  Widget    workFo;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   okCB,     NULL },
    { NULL, NULL,     NULL },
    { "",   applyCB,  NULL },
    { NULL, NULL,     NULL },
    { "",   cancelCB, NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgDGetText( td, MINF_OK_BUTTON );
  action_buttons[ 0 ].data  = book_ref;
  action_buttons[ 2 ].label = msgDGetText( td, MINF_APPLY_BUTTON );
  action_buttons[ 2 ].data  = book_ref;
  action_buttons[ 4 ].label = msgDGetText( td, MINF_CANCEL_BUTTON );
  action_buttons[ 4 ].data  = book_ref;


  /* Create a form dialog with buttons. */
  bookFd = xitCreateFormDialog( parent, "BookFd",
                                1, 0,
                                action_buttons,
                                XtNumber( action_buttons ) );

  XtAddCallback( bookFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) book_ref );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgDGetText( td, MINF_BOOK_TITLE ) ); n++;
  XtSetValues( XtParent( bookFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmUNMAP ); n++;
  XtSetValues( bookFd, args, n );

  /* Container for the contents of the window. */
  workFo = XtNameToWidget( bookFd, "BookFdFo" );


  /* Create books list. */
  bookLa = xitCreateLabel( workFo, "BookLa",
                           msgDGetText( td, MINF_BOOK_LIST_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  bookLi = XmCreateScrolledList( workFo, "BookLi", args, n );

  XtAddCallback( bookLi, XmNdefaultActionCallback, 
                 (XtCallbackProc) okCB, (XtPointer) book_ref );


  /* Place the various elements together. */
  xitAttachWidget( bookLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( bookLi ),
                   XmATTACH_WIDGET, bookLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( bookLa,             args, n );
  XtSetValues( XtParent( bookLi ), args, n );


  /* Manage all the children except the search form. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW  ) );


  /* Set the size of the window. */
  xitSetSizeFormDialog( bookFd, True );


  /* Make the final attachments. */
  xitAttachWidget( XtParent( bookLi ),
                   XmATTACH_WIDGET, bookLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_FORM, NULL );


  return( bookFd );

} /* createBookWindow */


/*----------------------------------------------------------------------*/

static void 
  displayBooks( BOOK_REC_REF     book_ref,
                XIT_IB_BOOK_REC  *user_books,
                int              user_books_no,
                XIT_IB_BOOK_REC  *system_books,
                int              system_books_no )
{

  /* Variables. */
  int       index;
  int       list_index = 0;
  char      buffer[ 200 ];
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;
  XmString  list_items[ MAX_BOOKS ];


  /* Code. */

  mainW = XtNameToWidget( book_ref -> bookW, "BookFdFo" );


  /* The user books. */
  for( index = 0; index < user_books_no; index++ ) {
    sprintf( buffer, "%-15.15s %s",
             (user_books + index) -> name,
             (user_books + index) -> description );

    list_items[ list_index ] = XmStringCreate( buffer, CS );
    list_index++;

    if( list_index >= MAX_BOOKS )
      return;
  }

  /* The system books. */
  for( index = 0; index < system_books_no; index++ ) {
    sprintf( buffer, "%-15.15s %s",
             (system_books + index) -> name,
             (system_books + index) -> description );

    list_items[ list_index ] = XmStringCreate( buffer, CS );
    list_index++;

    if( list_index >= MAX_BOOKS )
      return;
  }


  tempW = XtNameToWidget( mainW, "BookLiSW.BookLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, list_index ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  /* Free the match records. */
  for( index = 0; index < list_index; index++ )
    XmStringFree( list_items[ index ] );

  if( list_index > 0 )
    XmListSelectPos( tempW, 1, False );


  return;

} /* displayBooks */


/*----------------------------------------------------------------------*/

static Boolean
  fetchSelectedBook( BOOK_REC_REF  book_ref )
{

  /* Variables. */
  int       item_count;
  char      book_name[ 50 ];
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    bookLiW;
  Widget    mainW;
  XmString  *items;


  /* Code. */

  mainW   = XtNameToWidget( book_ref -> bookW, "BookFdFo" );
  bookLiW = XtNameToWidget( mainW, "BookLiSW.BookLi" );


  /* Fetch the selected book. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &item_count ); n++;
  XtGetValues( bookLiW, args, n );

  if( item_count != 1 )
    return( False );


  /* Get the selected book. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItems, &items ); n++;
  XtGetValues( bookLiW, args, n );

  char_ref = xitStringGetString( *items, CS );


  /* Extract header, chapter id and header level. */
  sscanf( char_ref, "%s", book_name );

  SysFree( char_ref );


  /* Do we have a user action callback registered? */
  if( book_ref -> actionCB != NULL )
    (* book_ref -> actionCB)( XIT_IB_REASON_NEW_BOOK, 
                              book_ref -> user_data, book_name );


  return( True );

} /* fetchSelectedBook */


/*----------------------------------------------------------------------*/

static void 
  applyCB( Widget        widget,
           BOOK_REC_REF  book_ref,
           XtPointer     call_data )
{

  /* Variables. */
  Boolean  ok;


  /* Code. */

  /* Fetch selected book. */
  ok = fetchSelectedBook( book_ref );
  if( ! ok )
    return;


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget        widget, 
            BOOK_REC_REF  book_ref,
            XtPointer     call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( book_ref -> actionCB != NULL )
    (* book_ref -> actionCB)( XIT_IB_REASON_POPDOWN, 
                              book_ref -> user_data, NULL );


  XtUnmanageChild( book_ref -> bookW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             BOOK_REC_REF  book_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( book_ref -> actionCB != NULL )
    (* book_ref -> actionCB)( XIT_IB_REASON_DESTROY, 
                              book_ref -> user_data, NULL );

  /* Free allocated data. */
  SysFree( book_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget        widget,
        BOOK_REC_REF  book_ref,
        XtPointer     call_data )
{

  /* Variables. */
  Boolean  ok;


  /* Code. */

  /* Fetch selected book. */
  ok = fetchSelectedBook( book_ref );
  if( ! ok )
    return;

  XtUnmanageChild( book_ref -> bookW );


  return;

} /* okCB */
