/*----------------------------------------------------------------------------
--
--  Module:           xtmNotify
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Watch one or more diary databaes for changes and when we find one,
--    give the user a message with the actions/new entries added.
--
--  Filename:         xtmNotify.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-01-27
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmNotify.c, Version: 1.1, Date: 95/02/18 16:07:58";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/ScrolledW.h>
#include <Xm/Text.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmFormat.h"
#include "xtmIcons.h"
#include "xtmResource.h"
#include "xitError.h"
#include "xitStickyMsg.h"
#include "xitTools.h"
#include "xtmTools.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME        "xdnotify"

/* Program class (also the name of the application defaults file). */
#define PROGRAM_CLASS       "XDiary"

/* File to keep 'last checked' information.*/
#define NOTIFY_STATUS_FILE  ".XDiary.notify"

/* Max number of database checks we can do. */
#define MAX_DB_CHECK        200


/* Local widgets in the message window. */
#define notifyLa        dataLocalW[ 0 ]
#define textTx          dataLocalW[ 1 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* One single database to check. */
typedef struct {
  char          name[ XTM_GL_MAX_CAL_NAME + 1 ];
  TIM_TIME_REF  last_check;
  TIM_TIME_REF  last_change;
} DB_INFO;


/* Basic data structure for the XDnotify appliaction. */
typedef struct {

  /* Use these databases instaed of these defined in customize. */
  char  *use_these_db;

  /* Default status file and current status file. */
  char  *def_status_file;
  char  status_file[ 150 ];

  /* Minutes between two checks. */
  int  check_delta;

  /* Give notification for events you did youself. */
  Boolean  check_self;

  /* Command line flags. */
  Boolean  startup_help;
  Boolean  version_help;

  /* Toplevel widget. */
  Widget  toplevel;

  /* The X application context. */
  XtAppContext  context;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data;

  /* Check these databases. */
  int      no_db_check;
  DB_INFO  db_check[ MAX_DB_CHECK ];

} XTM_NO_BASE_DATA, *XTM_NO_BASE_DATA_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;

/* Name of text domain. */
static char  *text_domain = "XDiary";

/* Name of module. */
static char  *module_name = "xtmNotify";

/* Application data. */
static  XTM_NO_BASE_DATA  appl_data;

/* Program options. */
static XrmOptionDescRec options[] = {
  { "-calendars",    "*.DbName",          XrmoptionSepArg, NULL },
  { "-checkSelf",    "*.CheckSelf",       XrmoptionNoArg,  "True" },
  { "-checkself",    "*.CheckSelf",       XrmoptionNoArg,  "True" },
  { "-dbName",       "*.DbName",          XrmoptionSepArg, NULL },
  { "-dbname",       "*.DbName",          XrmoptionSepArg, NULL },
  { "-delta",        "*.DeltaMin",        XrmoptionSepArg, NULL },
  { "-geometry",     "NotifyTl.geometry", XrmoptionSepArg, NULL },
  { "-h",            "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-help",         "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-iconic",       "NotifyTl.iconic",   XrmoptionNoArg,  "True" },
  { "-lan",          "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-language",     "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-noFileLock",   "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-nofilelock",   "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-statusFile",   "*.StatusFile",      XrmoptionSepArg, NULL },
  { "-statusfile",   "*.StatusFile",      XrmoptionSepArg, NULL },
  { "-usage",        "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-version",      "*.VersionHelp",     XrmoptionNoArg,  "True" },
};

/* Application resources. */
static XtResource  base_resources[] = {

  { "checkSelf", "CheckSelf", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_NO_BASE_DATA_REF, check_self ), 
    XtRString, "False" },

  { "dbName", "DbName", XtRString, sizeof( String ),
    XtOffset( XTM_NO_BASE_DATA_REF, use_these_db ), 
    XtRString, "" },

  { "deltaMin", "DeltaMin", XtRInt, sizeof( int ),
    XtOffset( XTM_NO_BASE_DATA_REF, check_delta ), 
    XtRString, "15" },

  { "startupHelp", "StartupHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_NO_BASE_DATA_REF, startup_help ), 
    XtRString, "False" },

  { "statusFile", "StatusFile", XtRString, sizeof( String ),
    XtOffset( XTM_NO_BASE_DATA_REF, def_status_file ), 
    XtRString, "" },

  { "versionHelp", "VersionHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_NO_BASE_DATA_REF, version_help ), 
    XtRString, "False" },

};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  closeCB( Widget     widget,
           Widget     baseW,
           XtPointer  call_data );

static void
  displayUsage();

static Boolean
  fetchChangedEntries( XTM_NO_BASE_DATA_REF  appl_data_ref,
                       char                  *db_name,
                       XTM_DB_LOG_RECORD     changes[],
                       int                   no_changes,
                       char                  **entry_text );

static Boolean
  initDbCheckRecord( XTM_NO_BASE_DATA_REF  appl_data_ref );

static void
  displayChangeText( XTM_NO_BASE_DATA_REF  appl_data_ref,
                     char                  *message );

static void
  startCheck( XTM_NO_BASE_DATA_REF  appl_data_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  main( unsigned int argc, char *argv[] )
{

  /* Variables. */
  Boolean             ok;
  int                 index;
  char                *char_ref;
  Arg                 args[ 10 ];
  Cardinal            n;
  Display             *display;
  TIM_STATUS_TYPE     time_status;
  XTM_CB_STATUS       custom_status;
  XTM_GL_CUSTOM_DATA  custom_data;


  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;
  xitErSetApplicationName( program_name );


  /* Save the original command parameters. */
  custom_data.orig_argc = argc;
  custom_data.orig_argv = (char**) XtMalloc( argc * sizeof( char* ) );

  for( index = 0; index < argc; index++ )
    custom_data.orig_argv[ index ] = XtNewString( argv[ index ] );
  
  /* NLS enabled. */
  xtmToSetLocale( program_name );


  /* Initialization. */
  SysInitializeEnvironment();
  xtmDbInitializeProcessId();

  
  /* Initialize toolkit and open display. */
  XtToolkitInitialize();

  appl_data.context = XtCreateApplicationContext();
  display = XtOpenDisplay( appl_data.context, NULL,
                           NULL, PROGRAM_CLASS,
                           options, XtNumber( options ),
#if XtSpecificationRelease < 5
                           (Cardinal *) &argc,
#else
                           (int *) &argc,
#endif
                           argv );

  if( display == NULL )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "Cannot open display, check your DISPLAY variable." );

  /* Resource mapping.*/
  xtmToInitializeResourceMapping( argc, argv, display );
  

  /* Create application shell. */
  n = 0;
  appl_data.toplevel = XtAppCreateShell( NULL, PROGRAM_CLASS,
                                         applicationShellWidgetClass,
                                         display,
                                         args, n );

  /* Get base application resources. */
  XtGetApplicationResources( appl_data.toplevel, (XtPointer) &appl_data, 
                             base_resources, 
                             XtNumber( base_resources ), 
                             NULL, 0 );

  /* Get customize resources. */
  xtmRsFetchCustomResources( &custom_data, appl_data.toplevel );

  
  /* A valid resource file? */
  if( ! custom_data.valid_resource_file )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "XDiary will only run with a correct X-Windows resource "
                  "file.\nPlease check the XDiary installation." );


  /* Initialize application data. */
  custom_data.cal_db_handle   = NULL;
  custom_data.group_db_handle = NULL;
  custom_data.archive_files   = NULL;
  custom_data.include_files   = NULL;

  appl_data.custom_data = &custom_data;


  /* Display current version? */
  if( appl_data.version_help ) {
    printf( "%s: Version: %s\n", program_name, VERSION_ID );
    exit( 0 );
  }

  /* Help requested? */
  if( appl_data.startup_help ) {
    displayUsage();
    exit( 0 );
  }


  /* Get customized data from file. */
  custom_status = xtmCbGetDataFromFile( appl_data.custom_data );

  if( custom_status == XTM_CB_WRONG_VERSION ) {
    char_ref = (char *) 
      SysMalloc( strlen( msgGetText( MXDI_CUST_WRONG_VERSION ) ) + 50 );

    sprintf( char_ref, msgGetText( MXDI_CUST_WRONG_VERSION ),
             xtmCbWhatVersion() );

    xitStDisplaySticky( appl_data.toplevel,
                        char_ref, XmUNSPECIFIED_PIXMAP,
                        msgGetText( MXDI_OK_BUTTON ), NULL,
                        NULL, NULL, NULL,
                        NULL );
    SysFree( char_ref );
  }


  /* Initialize necessary text domains. */
  msgInitialize();
  msgInitCatalogue( text_domain, NULL, custom_data.msg_language,
                    msgXdiaryTexts );

  /* Default catalogue Xdiary. */
  msgTextDomain( text_domain );


  /* Initialize the time formats. */
  time_status = TimInitializeFormat( custom_data.date_format,
                                     custom_data.time_format );
  if( time_status != TIM_OK )
    xitErMessage( appl_data.toplevel, XIT_ER_ERROR, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  /* Set colors and fonts in the resource database. */
  xtmRsFetchColors( &custom_data, appl_data.toplevel );


  /* Use file locking? */
  xtmDbUseFileLock( custom_data.use_file_lock );


  /* Name of status file to use. */
  if( strlen( appl_data.def_status_file ) > 0 ) {
    strcpy( appl_data.status_file, appl_data.def_status_file );

  } else {  
    char_ref = getenv( "HOME" );

    if( char_ref == NULL )
      sprintf( appl_data.status_file, "./%s", NOTIFY_STATUS_FILE );
    else
      sprintf( appl_data.status_file, "%s/%s", char_ref, NOTIFY_STATUS_FILE );

  } /* if */


  /* Initialize the db check record. */
  ok = initDbCheckRecord( &appl_data );
  if( ! ok )
    exit( 1 );


  /* Start the check procedure. */
  startCheck( &appl_data );


  /* Enter the event loop. */
  XtAppMainLoop( appl_data.context );


} /* main */


/*----------------------------------------------------------------------*/

static void
  displayChangeText( XTM_NO_BASE_DATA_REF  appl_data_ref,
                     char                  *message )
{

  /* Variables. */
  char      buffer[ 100 ];
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 2 ];
  Widget    notifyTl;
  Widget    workFo;

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "TextTx", NULL, 2, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { NULL,   NULL, NULL },
    { "",     NULL, NULL },
    { NULL,   NULL, NULL },
  };


  /* Code. */

  action_buttons[ 1 ].label = msgGetText( MXDI_OK_BUTTON );

  /* Create a separate window. */
  notifyTl = xitCreateToplevelDialog( 
               appl_data_ref -> toplevel, "NotifyTl",
               2, 0,
               action_buttons, XtNumber( action_buttons ) );


  /* Create the message window. */
  sprintf( buffer, "%s", msgGetText( MXDI_NOTIFY_TITLE ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle,    buffer ); n++;
  XtSetArg( args[ n ], XmNiconName, buffer ); n++;
  XtSetValues( notifyTl, args, n );


  /* Exit the application if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( notifyTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols(        notifyTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( notifyTl, wm_delete_window, 
                             (XtCallbackProc) closeCB, (XtPointer) notifyTl );
  } /* block */


  /* Form to hold the alarm message. */
  workFo = XtNameToWidget( notifyTl, "NotifyTlBase.NotifyTlFo" );


  /* What kind of window is this. */
  notifyLa = xitCreateLabel( workFo, "NotifyLa", 
                             msgGetText( MXDI_NOTIFY_DESC_LABEL ),
                             XmALIGNMENT_BEGINNING );


  /* Display the messages in a scrolled text. */
  textTx = xitCreateTextScrolled( workFo, &text_buffer_def[ 0 ] );

  n = 0;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( textTx, args, n );

  XmTextSetString( textTx, message );


  /* Put the Parts together. */
  xitAttachWidget( notifyLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( textTx ),
                   XmATTACH_WIDGET, notifyLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( notifyLa,           args, n );
  XtSetValues( XtParent( textTx ), args, n );

  xitManageChildren( dataLocalW,  XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( notifyTl, workFo, XTM_IC_ICON_DEFAULT );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( notifyTl, True );


  /* Make the final attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( XtParent( textTx ), args, n );


  XtPopup( notifyTl, XtGrabNone );


  /* Make our little beep. */
  XBell( XtDisplay( notifyTl ), 100 );


  return;

} /* displayChangeText */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Give notification when one or more databases are updated.\n" 
    "\n"
    "Usage:\n"
    "  %s [flags]\n"
    "\n"
    "Flags:\n"
    "  -checkSelf         : Give notification also when databases are\n"
    "                       changed by you.\n"
    "  -dbName 'name ...' : Use this diary database(s). Default is to use \n"
    "                       databases selected in the customize tool.\n"
    "                       If you have more database names, separate the\n"
    "                       names with spaces.\n"
    "  -delta <min>       : Minutes between database checks. Should be\n"
    "                       >15 min.\n"
    "  -fmap large        : Use a large font.\n"
    "  -fmap medium       : Use a medium font.\n"
    "  -fmap small        : Use a small font.\n"
    "  -help              : Display this help.\n"
    "  -h                 : See above.\n"
    "  -language <lan>    : Use the language <lan>.\n"
    "  -lan <lan>         : See above.\n"
    "  -noFileLock        : Don't use any file locking.\n"
    "  -palette gray      : Use color palette Gray.\n"
    "  -palette lila      : Use color palette Lila.\n"
    "  -palette motif     : Use color palette Motif.\n"
    "  -palette neon      : Use color palette Neon.\n"
    "  -palette nordic    : Use color palette Nordic.\n"
    "  -palette red       : Use color palette Red.\n"
    "  -palette sea       : Use color palette Sea.\n"
    "  -palette sky       : Use color palette Sky.\n"
    "  -palette wood      : Use color palette Wood.\n"
    "  -statusFile <file> : Use this status file. Default is the file\n"
    "                       .XDiary.notify in your home directory.\n"
    "  -usage             : Display this help.\n"
    "  -version           : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );

  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static Boolean
  fetchChangedEntries( XTM_NO_BASE_DATA_REF  appl_data_ref,
                       char                  *db_name,
                       XTM_DB_LOG_RECORD     changes[],
                       int                   no_changes,
                       char                  **entry_text )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  int                     length;
  char                    buffer[ 100 ];
  char                    entry_buffer[ 100 ];
  char                    intro_text[ 250 ];
  char                    *char_ref;
  char                    *text_ref;
  char                    *text;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;
  *entry_text = NULL;

  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( custom_data -> cal_db_handle, 
                          db_name,
                          &db_info, NULL );
  if( ! ok )
    return( False );

  /* Open the database for read. */
  open_request.name       = db_info.short_name;
  open_request.directory  = db_info.directory;
  open_request.operations = XTM_DB_FLAG_MODE_READ;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;


  /* Try to open the database. */
  db_status = xtmDbOpenEntryDb( &open_request, &database );

  if( db_status != XTM_DB_OK ) {
    if( db_status == XTM_DB_LOCKED )
      xitErMessage( NULL, XIT_ER_ERROR, 
                    module_name, "fetchChangedEntries",
                    msgGetText( MXDI_ERRMSG_DB_LOCKED ) );
    else
      xitErMessage( NULL, XIT_ER_ERROR, 
                    module_name, "fetchChangedEntries",
                    msgGetText( MXDI_ERRMSG_CANNOT_OPEN_DB ) );

    return( False );
  } /* if */


  /* Fetch the entries. */
  for( index = no_changes - 1; index > 0; index-- ) {

    /* Interested in our own entries? */
    if(   changes[ index ].changed_by == getuid() &&
        ! appl_data_ref -> check_self )
      continue;

    /* Insert entry? */
    if( flagIsSet( changes[ index ].flags, XTM_DB_FLAG_LOG_SAVE ) ) {

      /* Fetch the changed entry from the database. */
      db_status = xtmDbFetchEntry( &database,
                                   changes[ index ].entry_id,
                                   &entry_record, &text_ref );

      if( db_status == XTM_DB_OK ) {

        char           date_buffer[ 50 ];
        char           time_buffer[ 50 ];
        char           changed_by[ 50 ];
        TIM_TIME_REF   log_time;
        struct passwd  *password_ref;

        /* Introduction text. */
        intro_text[ 0 ] = '\0';
        log_time = TimLocalTime( (TIM_TIME_REF) changes[ index ].time_stamp );

        xtmFoFormatDate( log_time, date_buffer, sizeof( date_buffer ) );
        xtmFoFormatTime( log_time, time_buffer, sizeof( time_buffer ) );

        password_ref = getpwuid( entry_record.entry.owner );
        if( password_ref != NULL )
          strcpy( changed_by, password_ref -> pw_name );
        else
          strcpy( changed_by, "?" );

        sprintf( intro_text,  msgGetText( MXDI_NOTIFY_INTRO_TEXT ),
                 date_buffer, time_buffer, changed_by );

        /* Time for the entry. */
        entry_buffer[ 0 ] = '\0';

        xtmFoFormatDate( entry_record.entry.date_stamp,
                         buffer, sizeof( buffer ) );

        strcat( entry_buffer, buffer );
        strcat( entry_buffer, " " );

        if( entry_record.entry.entry_type != XTM_DB_DAY_NOTE ) {
          xtmFoFormatEntryTimes( &entry_record,
                                 buffer, sizeof( buffer ) );

          strcat( entry_buffer, buffer );
          strcat( entry_buffer, " " );
        }

        /* Database name. */
        sprintf( buffer, "(%s) ", db_name );
        strcat( entry_buffer, buffer );

        if( entry_record.entry.entry_type != XTM_DB_DAY_NOTE )
          strcat( entry_buffer, "\n" );

        /* Format entry text. */
        if( text_ref != NULL )
          char_ref = text_ref;
        else
          char_ref = entry_record.entry.text;

        if( entry_record.entry.entry_type == XTM_DB_DAY_NOTE )
          text = xtmFoFormatText( char_ref, 2, 25, 80 );
        else
          text = xtmFoFormatText( char_ref, 2, 25, 80 );

        /* Make a nice package out of this. */
        if( *entry_text == NULL ) {
          length = strlen( intro_text ) + strlen( text ) +
                   strlen( entry_buffer );

          *entry_text  = SysMalloc( length + 50 );
          **entry_text = '\0';

        } else {
          length = strlen( *entry_text ) + strlen( text ) + 
                   strlen( intro_text )  + strlen( entry_buffer );

          *entry_text = SysRealloc( *entry_text, length + 50 );
        }

        strcat( *entry_text, "-----------------------------------------\n" );
        strcat( *entry_text, intro_text );
        strcat( *entry_text, entry_buffer );
        strcat( *entry_text, "\n" );
        strcat( *entry_text, text );
        strcat( *entry_text, "\n\n" );

        if( text_ref != NULL )
          SysFree( text_ref );

        SysFree( text );

      } /* if */

    } /* if */

  } /* lopp */

  xtmDbCloseEntryDb( &database );


  return( True );

} /* fetchChangedEntries */


/*----------------------------------------------------------------------*/

static Boolean
  initDbCheckRecord( XTM_NO_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  int                     char_read;
  int                     index;
  int                     items;
  int                     last_check;
  char                    buffer[ 100 ];
  char                    db_name[ 100 ];
  char                    *char_ref;
  FILE                    *file_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref -> no_db_check = 0;

  custom_data = appl_data_ref -> custom_data;


  /* Do we have specific databases to check? */
  if( strlen( appl_data_ref -> use_these_db ) > 0 ) {

    index    = 0;
    char_ref = appl_data_ref -> use_these_db;

    do {
      while( isspace( *char_ref ) )
        char_ref++;

      if( *char_ref == '\0' )
        break;

      char_read = strlen( char_ref );
      items = sscanf( char_ref, "%s%n", buffer, &char_read );
      if( items != 1 )
        break;

      (void) xtmCdFetchNamedDb( custom_data -> cal_db_handle, 
                                buffer, 
                                &db_info, NULL );

      if( flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {
        strcpy( appl_data_ref -> db_check[ index ].name, buffer );

        appl_data_ref -> db_check[ index ].last_check = TimMakeTimeNow();

        index++;
      }

      char_ref = char_ref + char_read;

    } while( index < MAX_DB_CHECK );

    appl_data_ref -> no_db_check = index;


  /* Fetch from customize data. */
  } else {

    char  *char_ref;
    char  *db_names;

    (void) xtmCdFetchDbNames( custom_data -> cal_db_handle, &db_names );
    char_ref = db_names;
    index    = 0;

    do {

      int              char_read;
      char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
      XTM_CD_CAL_INFO  db_info;

      while( isspace( *char_ref ) )
        char_ref++;

      if( *char_ref == '\0' )
        break;

      char_read = strlen( char_ref );
      sscanf( char_ref, "%s%n", db_name, &char_read );
      char_ref = char_ref + char_read;


      /* Fetch information about the database. */
      (void) xtmCdFetchNamedDb( custom_data -> cal_db_handle, db_name,
                                &db_info, NULL );

      /* Do we want notify event for this database? */
      if( flagIsSet( db_info.flags, XTM_CD_FLAG_NOTIFY ) &&
          flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {

        strcpy( appl_data_ref -> db_check[ index ].name,
                db_info.short_name );

        appl_data_ref -> db_check[ index ].last_check = TimMakeTimeNow();

        index++;

      } /* if */

    } while( True );

    if( db_names != NULL )
      SysFree( db_names );

    appl_data_ref -> no_db_check = index;

  } /* if */


  /* Do we have any databases to check? */
  if( appl_data_ref -> no_db_check == 0 ) {
    xitErMessage( NULL, XIT_ER_ERROR,
                  module_name, "initDbCheckRecord",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );

    return( False );
  }

  /* Get the last check time for the databases. */
  file_ref = fopen( appl_data_ref -> status_file, "r" );
  if( file_ref == NULL )
    return( True );


  /* Read all database definitions. */
  char_ref = fgets( buffer, sizeof( buffer ), file_ref );

  while( char_ref != NULL ) {

    items = sscanf( char_ref, "%s %d", db_name, &last_check );

    /* Was this a valid db definition? */
    if( items == 2 ) {

      /* Are we checking this database? */
      for( index = 0; index < appl_data_ref -> no_db_check; index++ ) {

        if( strcmp( appl_data_ref -> db_check[ index ].name, db_name ) == 0 )
          appl_data_ref -> db_check[ index ].last_check = 
            (TIM_TIME_REF ) last_check;

      } /* loop */

    } /* if */

    char_ref = fgets( buffer, sizeof( buffer ), file_ref );

  } /* while */


  fclose( file_ref );


  return( True );  

} /* initDbCheckRecord */


/*----------------------------------------------------------------------*/

static void
  startCheck( XTM_NO_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  Boolean                 changes_found;
  Boolean                 found;
  int                     index;
  int                     items;
  int                     last_check;
  int                     no_changes;
  int                     old_db_no;
  char                    buffer[ 100 ];
  char                    db_name[ 100 ];
  char                    *change_text = NULL;
  char                    *char_ref;
  DB_INFO                 old_db_check[ 200 ];
  FILE                    *file_ref;
  XTM_DB_LOG_RECORD       changes[ 200 ];
  XTM_DB_STATUS           db_status;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data   = appl_data_ref -> custom_data;
  changes_found = False;


  /* Schedule the next wakeup call, wait n minutes. */
  XtAppAddTimeOut( appl_data_ref -> context,
                   appl_data_ref -> check_delta * 1000 * 60,
                   (XtTimerCallbackProc) startCheck,
                   (XtPointer) appl_data_ref );


  /* Check all our databases. */
  for( index = 0; index < appl_data_ref -> no_db_check; index++ ) {

    (void) xtmCdFetchNamedDb( custom_data -> cal_db_handle, 
                              appl_data_ref -> db_check[ index ].name,
                              &db_info, NULL );

    /* Any changes since last check? */
    db_status = xtmDbChangesInLog( 
                  db_info.directory,
                  appl_data_ref -> db_check[ index ].last_check,
                  changes, sizeof( changes ), &no_changes );

    appl_data_ref -> db_check[ index ].last_check = TimMakeTimeNow();

    /* If we had changes, show what has been done. */
    if( no_changes > 0 ) {

      Boolean  ok;
      int      length;
      char     *entry_text;

      changes_found = True;

      /* Fetch the text for the changed entries. */
      ok = fetchChangedEntries( appl_data_ref, db_info.short_name,
                                changes, no_changes, &entry_text );

      if( ok && entry_text != NULL ) {

        /* Save the message text for later. */
        if( change_text == NULL ) {
          length       = strlen( entry_text );
          change_text  = SysMalloc( length + 2 );
          *change_text = '\0';

        } else {
          length      = strlen( change_text ) + strlen( entry_text );
          change_text = SysRealloc( change_text, length + 2 );
        }

        strcat( change_text, entry_text );

        SysFree( entry_text );

      } /* if */

    } /* if */

  } /* loop */


  if( ! changes_found )
    return;


  /* Display the changes we found. */
  if( change_text != NULL ) {
    displayChangeText( appl_data_ref, change_text );

    SysFree( change_text );
  } /* if */


  /* If we have found changes, update the status file. */
  old_db_no = 0;

  file_ref = fopen( appl_data_ref -> status_file, "r" );

  /* Read the old data saved. */
  if( file_ref != NULL ) {

    char_ref = fgets( buffer, sizeof( buffer ), file_ref );

    while( char_ref != NULL ) {

      items = sscanf( char_ref, "%s %d", db_name, &last_check );

      /* Was this a valid db definition? */
      if( items == 2 ) {

        /* Are we checking this database? */
        found = False;

        for( index = 0; index < appl_data_ref -> no_db_check; index++ ) {
          if( strcmp( appl_data_ref -> db_check[ index ].name, db_name ) == 0 )
            found = True;
        }

        /* If we were not checking this database, save it. */
        if( ! found ) {
          strcpy( old_db_check[ old_db_no ].name, db_name );

          old_db_check[ old_db_no ].last_check = (TIM_TIME_REF) last_check;

          old_db_no++;
        }

      } /* if */

      char_ref = fgets( buffer, sizeof( buffer ), file_ref );

    } /* while */

    fclose( file_ref );

  } /* if */


  /* If we have found changes, update the status file. */
  file_ref = fopen( appl_data_ref -> status_file, "w+" );

  if( file_ref != NULL ) {

    /* First, save the new data. */
    for( index = 0; index < appl_data_ref -> no_db_check; index++ ) {

      fprintf( file_ref, "%-20.20s ", 
               appl_data_ref -> db_check[ index ].name );

      fprintf( file_ref, "%d ", 
               (int)(appl_data_ref -> db_check[ index ].last_check) );

      TimFormatIsoDate( TimLocalTime( appl_data_ref -> 
                                      db_check[ index ].last_check ),
                        buffer, sizeof( buffer ) );
      fprintf( file_ref, "%s ", buffer );

      TimFormatIsoTime( TimLocalTime( appl_data_ref -> 
                                      db_check[ index ].last_check ),
                         buffer, sizeof( buffer ) );
      fprintf( file_ref, "%s\n", buffer );

    } /* loop */


    /* Now, the old data. */
    for( index = 0; index < old_db_no; index++ ) {

      fprintf( file_ref, "%-20.20s ", old_db_check[ index ].name );

      fprintf( file_ref, "%d ", (int)(old_db_check[ index ].last_check) );

      TimFormatIsoDate( TimLocalTime( old_db_check[ index ].last_check ), 
                        buffer, sizeof( buffer ) );
      fprintf( file_ref, "%s ", buffer );

      TimFormatIsoTime( TimLocalTime( old_db_check[ index ].last_check ), 
                        buffer, sizeof( buffer ) );
      fprintf( file_ref, "%s\n", buffer );

    } /* loop */


    fclose( file_ref );

  } /* if */


  return;

} /* startCheck */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget     widget,
           Widget     baseW,
           XtPointer  call_data )
{

  /* Code. */

  XtDestroyWidget( baseW );


  return;

} /* closeCB */
