/*----------------------------------------------------------------------------
--
--  Module:           xtmDiaryLgt
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Main program for XDiaryLgt.
--
--  Filename:         xtmDiaryLgt.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-11-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmDiaryLgt.c, Version: 1.1, Date: 95/02/18 16:07:16";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

#include <Xm/Xm.h>

#include "System.h"
#include "Message.h"
#include "SigHandler.h"
#include "TimDate.h"

#include "msgTopic.h"
#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmAlarmCtrl.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbMisc.h"
#include "xtmHelp.h"
#include "xtmHoliday.h"
#include "xtmResource.h"
#include "xtmSetup.h"
#include "xtmSchedMain.h"
#include "xtmShowToDo.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitMsgWin.h"
#include "xitStickyMsg.h"
#include "xitTools.h"
#include "xitInfo.h"
#include "xtmFormat.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdlight"

/* Program class (also name of application resource file). */
#define PROGRAM_CLASS  "XDiary"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;

/* Name of text domain. */
static char  *text_domain      = "XDiary";
static char  *info_text_domain = "Topic";

/* Name of module. */
static char  *module_name = "xtmDiaryLight";

/* Keyboard translations for newline in multi-line editor. */
XtTranslations  xtm_di_newline_trans;

/* Handle to the 'today' summary window. */
static XTM_SD_HANDLE  today_show_handle = NULL;

/* Application data. */
static  XTM_GL_BASE_DATA  appl_data;


/* Key translations. */
static char newlineTrans[] = 
  "<Key>Return:    newline()";


/* Program options. */
static XrmOptionDescRec options[] = {
  { "-alarm",        "*.giveAlarm",         XrmoptionNoArg,  "True" },
  { "-noalarm",      "*.giveAlarm",         XrmoptionNoArg,  "False" },
  { "-noAlarm",      "*.giveAlarm",         XrmoptionNoArg,  "False" },
  { "-custom",       "*.customFile",        XrmoptionSepArg, NULL },
  { "-debug",        "*.debug",             XrmoptionNoArg,  "True" },
  { "-delta",        "*.entryDelta",        XrmoptionSepArg, NULL },
  { "-geometry",     "*.SchedTl.geometry",  XrmoptionSepArg, NULL },
  { "-h",            "*.StartupHelp",       XrmoptionNoArg,  "True" },
  { "-help",         "*.StartupHelp",       XrmoptionNoArg,  "True" },
  { "-iconic",       "*.SchedTl.iconic",    XrmoptionNoArg,  "True" },
  { "-lan",          "*.msgLanguage",       XrmoptionSepArg, NULL },
  { "-language",     "*.msgLanguage",       XrmoptionSepArg, NULL },
  { "-noFileLock",   "*.useFileLock",       XrmoptionNoArg,  "False" },
  { "-nofilelock",   "*.useFileLock",       XrmoptionNoArg,  "False" },
  { "-readOnly",     "*.readOnlyMode",      XrmoptionNoArg,  "True" },
  { "-readonly",     "*.readOnlyMode",      XrmoptionNoArg,  "True" },
  { "-ro",           "*.readOnlyMode",      XrmoptionNoArg,  "True" },
  { "-startDate",    "*.startDate",         XrmoptionSepArg, NULL },
  { "-startdate",    "*.startDate",         XrmoptionSepArg, NULL },
  { "-sd",           "*.startDate",         XrmoptionSepArg, NULL },
  { "-title",        "*.title",             XrmoptionSepArg, NULL },
  { "-usage",        "*.StartupHelp",       XrmoptionNoArg,  "True" },
  { "-version",      "*.VersionHelp",       XrmoptionNoArg,  "True" },
};


/* Base program resources. */
static XtResource  base_resources[] = {

  { "defaultDbDir", "DefaultDbDir", XtRString, sizeof( String ),
    XtOffset( XTM_GL_BASE_DATA_REF, default_db_dir ), 
    XtRString, "." },

  { "startDate", "StartDate", XtRString, sizeof( String ),
    XtOffset( XTM_GL_BASE_DATA_REF, start_date ), 
    XtRString, "" },

  { "startupHelp", "StartupHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_GL_BASE_DATA_REF, startup_help ), 
    XtRString, "False" },

  { "title", "Title", XtRString, sizeof( String ),
    XtOffset( XTM_GL_BASE_DATA_REF, title ), 
    XtRString, "XDiary" },

  { "versionHelp", "VersionHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_GL_BASE_DATA_REF, version_help ), 
    XtRString, "False" },

};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  displayDefaultDayView( XTM_GL_BASE_DATA_REF  appl_data_ref );

static void
  displayUsage();

static void
  mainRedisplayCB( int   this_signal,
                   void  *user_data );

static void
  mainRedisplayDo( XTM_GL_BASE_DATA_REF appl_data_ref );

static void
  refreshCB( UINT32  flags,
             void    *user_data,
             void    *call_data );

static void 
  scheduleActionCB( XTM_SM_REASON reason,
                    void          *user_data );

static void
  summaryToday( XTM_GL_BASE_DATA_REF appl_data_ref );

static void 
  summaryActionCB( XTM_SD_REASON reason,
                   void          *user_data );

static void
  terminateCB( int   this_signal,
               void  *user_data );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  main( unsigned int argc, char **argv )
{

  /* Variables. */
  int                   index;
  int                   temp;
  char                  *char_ref;
  Arg                   args[ 10 ];
  Cardinal              n;
  Display               *display;
  Widget                fake_mainW;
  TIM_STATUS_TYPE       time_status;
  TIM_TIME_REF          now;
  XTM_CD_CAL_INFO       default_db;
  XTM_CB_STATUS         custom_status;
  XTM_GL_CUSTOM_DATA    custom_data;
  XTM_GL_PROC_INFO      process_info;



  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;
  xitErSetApplicationName( PROGRAM_CLASS );

  now = TimMakeTimeNow();

  
  /* Save the original command parameters. */
  custom_data.orig_argc = argc;
  custom_data.orig_argv = (char**) XtMalloc( argc * sizeof( char* ) );

  for( index = 0; index < argc; index++ )
    custom_data.orig_argv[ index ] = XtNewString( argv[ index ] );
  

  /* NLS enabled. */
  xtmToSetLocale( program_name );


  /* Initialization. */
  SysInitializeEnvironment();
  xtmDbInitializeProcessId();
    

  /* Initialize toolkit and open display. */
  XtToolkitInitialize();

  appl_data.context = XtCreateApplicationContext();
  display = XtOpenDisplay( appl_data.context, NULL,
                           NULL, PROGRAM_CLASS,
                           options, XtNumber( options ),
#if XtSpecificationRelease < 5
                           (Cardinal *) &argc,
#else
                           (int *) &argc,
#endif
                           argv );

  if( display == NULL )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "Cannot open display, check your DISPLAY variable." );

  /* Resource mapping.*/
  xtmToInitializeResourceMapping( argc, argv, display );
  
  /* Default translations */
  xtm_di_newline_trans   = XtParseTranslationTable( newlineTrans );


  /* Create application shell. */
  n = 0;
  appl_data.toplevel = XtAppCreateShell( NULL, PROGRAM_CLASS,
                                         applicationShellWidgetClass,
                                         display,
                                         args, n );

  /* Get base application resources. */
  XtGetApplicationResources( appl_data.toplevel, (XtPointer) &appl_data, 
                             base_resources, 
                             XtNumber( base_resources ), 
                             NULL, 0 );

  /* Get customize resources. */
  xtmRsFetchCustomResources( &custom_data, appl_data.toplevel );


  /* A valid resource file? */
  if( ! custom_data.valid_resource_file )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "XDiary will only run with a correct X-Windows resource "
                  "file.\nPlease check the XDiary installation." );


  /* Initialize the global data records. */
  custom_data.cal_db_handle   = NULL;
  custom_data.group_db_handle = NULL;
  custom_data.archive_files   = NULL;
  custom_data.include_files   = NULL;

  process_info.alarm_pid     = 0;

  appl_data.flags            = 0;
  appl_data.alarm_fifo_ref   = NULL;
  appl_data.alarm_fifo_name  = NULL;
  appl_data.msg_send_handle  = NULL;
  appl_data.info_handle      = NULL;
  appl_data.msg_win_ref      = NULL;
  appl_data.custom_data      = &custom_data;
  appl_data.process_info     = &process_info;

  for( index = 0; index < XTM_GL_MAX_ALARM_PRC; index++ )
    appl_data.process_info -> processes[ index ].db_name[ 0 ] = '\0';


  /* Display current version? */
  if( appl_data.version_help ) {
    printf( "%s: Version: %s\n", program_name, VERSION_ID );
    exit( 0 );
  }

  /* Help requested? */
  if( appl_data.startup_help ) {
    displayUsage();
    exit( 0 );
  }


  /* Get customized data from file. */
  custom_status = xtmCbGetDataFromFile( appl_data.custom_data );

  /* Initialize necessary text domains. */
  msgInitialize();
  msgInitCatalogue( text_domain, NULL, custom_data.msg_language,
                    msgXdiaryTexts );
  msgInitCatalogue( info_text_domain, NULL, custom_data.msg_language,
                    msgTopicTexts );

  /* Default catalogue Xdiary. */
  msgTextDomain( text_domain );

  /* What version are we using. */
  if( custom_status == XTM_CB_WRONG_VERSION ) {
    char_ref = (char *) 
      SysMalloc( strlen( msgGetText( MXDI_CUST_WRONG_VERSION ) ) + 50 );

    sprintf( char_ref, msgGetText( MXDI_CUST_WRONG_VERSION ),
             xtmCbWhatVersion() );

    xitStDisplaySticky( appl_data.toplevel,
                        char_ref, XmUNSPECIFIED_PIXMAP,
                        msgGetText( MXDI_OK_BUTTON ), NULL,
                        NULL, NULL, NULL,
                        NULL );
    SysFree( char_ref );
    exit( 1 );
  }


  /* Initialize the time formats. */
  time_status = TimInitializeFormat( custom_data.date_format,
                                     custom_data.time_format );
  if( time_status != TIM_OK )
    xitErMessage( appl_data.toplevel, XIT_ER_ERROR, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  /* Set colors and fonts in the resource database. */
  xtmRsFetchColors( &custom_data, appl_data.toplevel );


  /* Initialize holidays. */
  xtmHoInitialize( custom_data.holidays_db_dir, custom_data.workdays );


  /* Use file locking? */
  xtmDbUseFileLock( custom_data.use_file_lock );

  /* Run in debug mode? */
  if( custom_data.debug_mode ) {
    SysDebugOn( True );
  }


  /* Check the delta time. */
  if( custom_data.default_entry_delta < 5 ||
      custom_data.default_entry_delta > 360 )
    custom_data.default_entry_delta = 60;
  else
    custom_data.default_entry_delta = custom_data.default_entry_delta;


  /* Check list start and stop time. */
  if( custom_data.start_hour < 0 || custom_data.start_hour > 23 )
    custom_data.start_hour = 8;

  if( custom_data.stop_hour < 0 || custom_data.stop_hour > 23 )
    custom_data.stop_hour = 18;

  if( custom_data.start_hour > custom_data.stop_hour ) {
    temp = custom_data.start_hour;
    custom_data.start_hour = custom_data.stop_hour;
    custom_data.stop_hour  = temp;
  }


  /* Make sure the XDiary setup is coreect. */
  xtmStCheckSetup( appl_data.custom_data, 
                   appl_data.toplevel,
                   appl_data.context );


  /* Initialize the signal handler module. */
  SigInitialize();

  /* Initialize the update module. */
  if( custom_data.do_refresh )
    xtmUpInitialize( appl_data.context, custom_data.refresh_rate );
  else
    xtmUpInitialize( appl_data.context, 0 );


  /* Do we have a default database? */
  (void) xtmCdFetchDefaultDb( custom_data.cal_db_handle, &default_db, NULL );

  /* We do require read access to your default database. */
  if( ! flagIsSet( default_db.operations, XTM_DB_FLAG_MODE_READ ) )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );


  /* Initialize the error message window. */
  {
    XIT_MSW_LABEL_REC  msg_win_labels;

    msg_win_labels.title        = msgGetText( MXDI_MSG_WIN_TITLE );
    msg_win_labels.list_label   = msgGetText( MXDI_MSG_WIN_LIST_LABEL );
    msg_win_labels.close_button = msgGetText( MXDI_CLOSE_BUTTON );
    msg_win_labels.clear_button = msgGetText( MXDI_CLEAR_BUTTON );

    appl_data.msg_win_ref = xitCreateMsgWindow( appl_data.toplevel,
                                                50, custom_data.error_log_file,
                                                &msg_win_labels );
  } /* block */

  /* Initialize the help system. */
  appl_data.info_handle = xitInInitialize( appl_data.toplevel,
                                           NULL, NULL );


  /* Display the default day view. */
  displayDefaultDayView( &appl_data );


  /* Signal handlers. */
  (void) SigRegister( SIGHUP,  0, terminateCB,     &appl_data );
  (void) SigRegister( SIGTERM, 0, terminateCB,     &appl_data );
  (void) SigRegister( SIGINT,  0, terminateCB,     &appl_data );
  (void) SigRegister( SIGILL,  0, terminateCB,     &appl_data );
  (void) SigRegister( SIGQUIT, 0, terminateCB,     &appl_data );
  (void) SigRegister( SIGABRT, 0, terminateCB,     &appl_data );
  (void) SigRegister( SIGUSR1, 0, mainRedisplayCB, &appl_data );

  /* Tell the user what's up today? */
  if( appl_data.custom_data -> show_day_to_do )
    summaryToday( &appl_data );


  /* Are we running a new release? */
  if( custom_data.new_release )
    xtmHlDisplayReleaseNotes( appl_data.info_handle, appl_data.calW );


  /* We need a fake main window. */
  fake_mainW = xitCreateToplevelDialog( appl_data.toplevel, "", 
                                        0, 0, NULL, 0 );
  XtRealizeWidget( fake_mainW );

  
  /* Session management. */
  xtmToSessionMgmt( fake_mainW,
                    custom_data.orig_argc, custom_data.orig_argv );


  /* Start the alarm process. */
  xtmAcStartDefaultAlarmPrc( fake_mainW, &appl_data );


  /* Register for refresh updates. */
  (void) xtmUpRegister( XTM_UP_REFRESH_TICK, refreshCB, (void *) &appl_data );


  /* Enter the event loop. */
  XtAppMainLoop( appl_data.context );

} /* main */


/*----------------------------------------------------------------------*/

static void
  displayDefaultDayView( XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  Boolean                 ok;
  TIM_TIME_REF            start_date = 0;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  XTM_CD_CAL_INFO         db_info;
  XTM_SM_HANDLE           sched_handle;
  XTM_SM_INIT_REC         sched_init;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;


  /* Schedule init record. */
  sched_init.entry_start = TimMakeTime( 1970, 1, 1, 
                                        custom_data -> start_hour, 0, 0 );
  sched_init.entry_stop  = TimMakeTime( 1970, 1, 1,
                                        custom_data -> stop_hour,  0, 0 );
  sched_init.entry_delta = custom_data -> default_entry_delta;

  sched_init.default_flags = (XTM_SM_USE_GRID | 
                              XTM_SM_ENTRY_HANDLES | 
                              XTM_SM_ONLY_ONE);

  if( custom_data -> confirm_actions )
    flagSet( sched_init.default_flags, XTM_SM_CONFIRM );
  if( custom_data -> list_layout )
    flagSet( sched_init.default_flags, XTM_SM_LIST_LAYOUT );
  if( custom_data -> display_nav_cal )
    flagSet( sched_init.default_flags, XTM_SM_AUTO_NAV_CAL );
  if( custom_data -> display_entry_flags )
    flagSet( sched_init.default_flags, XTM_SM_ENTRY_FLAGS );
  if( custom_data -> read_only_mode )
    flagSet( sched_init.default_flags, XTM_SM_READ_ONLY );
  if( custom_data -> true_color_included )
    flagSet( sched_init.default_flags, XTM_SM_TRUE_COLOR_INC );


  /* Display the default day view. */
  (void) xtmCdFetchDefaultDb( custom_data -> cal_db_handle, 
                              &db_info, NULL );


  /* Create the schedule window. */
  sched_handle = xtmSmInitialize( appl_data_ref,
                                  appl_data.toplevel,
                                  db_info.short_name, &sched_init,
                                  scheduleActionCB, (void *) appl_data_ref );

  /* Give an error message if the schedule could not be initialized. */
  if( sched_handle == NULL )
    xitErMessage( NULL, XIT_ER_FATAL,
                  module_name, "displayStartup",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );


  /* A start date? */
  /* Orig: if( strlen( appl_data_ref -> start_date ) > 0 ) {
     ok = xtmFoFetchDate( appl_data_ref -> start_date,
     False, NULL, &start_date ); */
  /* I'm absolutely not sure about this.... */
  if( strlen( appl_data_ref -> start_date ) > 0 ) {
    ok = xtmFoFetchDate( NULL, appl_data_ref -> calW,
                         XTM_FO_START_DATE, False, &start_date );
    if( ! ok )
      start_date = 0;
  }


  /* Start as icon? */
  if( flagIsSet( db_info.flags, XTM_CD_FLAG_DEFAULT_START_ICON ) )
    xtmSmViewSchedule( sched_handle, start_date, True );
  else
    xtmSmViewSchedule( sched_handle, start_date, False );


  return;

} /* displayDefaultDayView */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Your personal calendar.\n" 
    "\n"
    "XDiary is your personal organizer that combines the functions of a\n"
    "desktop calendar, an appointment book and an alarm clock. XDiary will\n"
    "help you keep track of your meetings, appointments and plan your time.\n"
    "\n"
    "Usage:\n"
    "  %s [flags]\n"
    "\n"
    "Flags:\n"
    "  -fmap large      : Use a large font.\n"
    "  -fmap medium     : Use a medium font.\n"
    "  -fmap small      : Use a small font.\n"
    "  -help            : Display this help.\n"
    "  -h               : See above.\n"
    "  -language <lan>  : Use the language <lan>.\n"
    "  -lan <lan>       : See above.\n"
    "  -noFileLock      : Don't use any file locking.\n"
    "  -palette gray    : Use color palette Gray.\n"
    "  -palette lila    : Use color palette Lila.\n"
    "  -palette motif   : Use color palette Motif.\n"
    "  -palette neon    : Use color palette Neon.\n"
    "  -palette nordic  : Use color palette Nordic.\n"
    "  -palette red     : Use color palette Red.\n"
    "  -palette sea     : Use color palette Sea.\n"
    "  -palette sky     : Use color palette Sky.\n"
    "  -palette wood    : Use color palette Wood.\n"
    "  -readOnly        : Run in read-only mode.\n"
    "  -ro              : See above.\n"
    "  -setDate <date>  : Start with this date.\n"
    "  -sd              : See above.\n"
    "  -usage           : Display this help.\n"
    "  -version         : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );


  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static void
  mainRedisplayDo( XTM_GL_BASE_DATA_REF appl_data_ref )
{

  /* Code. */

  /* Update the calendars. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;

} /* mainRedisplayDo */


/*----------------------------------------------------------------------*/

static void
  summaryToday( XTM_GL_BASE_DATA_REF appl_data_ref )
{

  /* Variables. */
  Boolean                 entry_defined[ 1 ];
  TIM_TIME_REF            now;
  TIM_TIME_REF            today;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         default_db;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  now = TimLocalTime( TimMakeTimeNow() );

  today = TimMakeTime( TimIndexOfYear(  now ),
                       TimIndexOfMonth( now ),
                       TimIndexOfDay(   now ),
                       0, 0, 0 );

  (void) xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle, 
                              &default_db, NULL );

  /* Check current month. */
  xtmDmEntriesDefined( appl_data_ref,
                       default_db.short_name,
                       today, 1, True, True, 
                       entry_defined );

  /* Any entries defined today? */
  if( ! entry_defined[ 0 ] )
    return;


  /* Create summary window? */
  today_show_handle = xtmSdInitialize( appl_data_ref, 
                                       appl_data_ref -> toplevel,
                                       default_db.short_name,
                                       summaryActionCB, NULL );

  if( today_show_handle == NULL )
    return;


  /* Display the summary window. */
  xtmSdDisplaySummary( today_show_handle,
                       today, today,
                       custom_data_ref -> filter_day_view_tags );


  return;

} /* summaryToday */


/*----------------------------------------------------------------------*/

static void
  mainRedisplayCB( int   this_signal,
                   void  *user_data )
{

  /* Variables. */
  XTM_GL_BASE_DATA_REF appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_GL_BASE_DATA_REF) user_data;

  /* Start a 'NULL' timer to do the update. */
  XtAppAddTimeOut( appl_data_ref -> context, 0, 
                   (XtTimerCallbackProc) mainRedisplayDo,
                   (XtPointer) appl_data_ref );


  return;

} /* mainRedisplayCB */


/*----------------------------------------------------------------------*/

static void
  refreshCB( UINT32  flags,
             void    *user_data,
             void    *call_data )
{

  /* Variables. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_GL_BASE_DATA_REF) user_data;

  /* Update the calendar and all day views. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;

} /* refreshCB */


/*----------------------------------------------------------------------*/

static void 
  scheduleActionCB( XTM_SM_REASON reason,
                    void          *user_data )
{

  /* Variables. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_GL_BASE_DATA_REF) user_data;


  if( reason == XTM_SM_REASON_EXIT ||
      reason == XTM_SM_REASON_DESTROY )
    terminateCB( SIGTERM, (void *) appl_data_ref );


  return;

} /* scheduleActionCB */


/*----------------------------------------------------------------------*/

static void 
  summaryActionCB( XTM_SD_REASON reason,
                   void          *user_data )
{

  /* Code. */

  if( today_show_handle == NULL )
    return;

  if( reason == XTM_SD_REASON_DESTROY )
    today_show_handle = NULL;


  return;

} /* summaryActionCB */


/*----------------------------------------------------------------------*/

static void
  terminateCB( int   this_signal,
               void  *user_data )
{

  /* Variables. */
  int                  status;
  XTM_GL_BASE_DATA_REF appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_GL_BASE_DATA_REF) user_data;

  /* Kill all alarm processes. */
  if( appl_data_ref -> process_info -> alarm_pid != 0 )
    kill( appl_data_ref -> process_info -> alarm_pid, SIGTERM );

  if( appl_data_ref -> alarm_fifo_ref != NULL ) {
    fclose( appl_data_ref -> alarm_fifo_ref );

    status = unlink( appl_data_ref -> alarm_fifo_name );
  }

  exit( 0 );

} /* terminateCB */
