/*----------------------------------------------------------------------------
--
--  Module:           xtmAlarmProc
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Process that alarms the user of evebts defined in the XDiary
--    database. Each process controls a single diary database and can be
--    started as a standalone program or be integrated in the XDiary
--    environment.
--
--    Syntax:
--      XDalarm [flags]
--
--    Flags:
--      -alarmTags : Display alarms containing one of these tags.
--      -dbName    : Name of the database(s) to use.
--      -usage     : Displays some help.
--      -help      : Same as above.
--
--  Filename:         xtmAlarmProc.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-07-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmAlarmProc.c, Version: 1.1, Date: 95/02/18 16:01:50";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/signal.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

#include <Xm/Xm.h>
#include <Xm/BulletinB.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/Scale.h>
#include <Xm/SeparatoG.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "SigHandler.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmIcons.h"
#include "xtmResource.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitStickyMsg.h"
#include "xitTools.h"
#include "xtmTools.h"
#include "xtmHoliday.h"
#include "xtmFormat.h"
#include "xtmDbTools.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdalarm"

/* Program class (also the name of the application defaults file). */
#define PROGRAM_CLASS  "XDiary"


/* Local widgets in the day list view window. */
#define markerPx        dataLocalW[ 0 ]
#define messageLa       dataLocalW[ 1 ]

/* Local widgets in the snooze window. */
#define snoozeLa        dataLocalW[ 0 ]
#define snoozeSc        dataLocalW[ 1 ]


/* Number of alarms per entry. */
#define MAX_ALARMS          5

/* Number of tune files. */
#define MAX_TUNES           5

/* The number of database we can watch simultaneously. */
#define MAX_DATABASE_CHECK  50

/* Maximun number of entry alarms that can be ignored. */
#define MAX_ENTRIES_IGNORE  100

/* Maximum number of snooze alarms. */
#define MAX_SNOOZE_ALARM    100

/* Maximum number of alarms in the cache. */
#define MAX_ALARM_CACHE     200

/* Maximum number of displayed alarm windows we keep track of. */
#define MAX_TRACK_ALARM_WIN  100

/* Name of database file (we should actually no know this!). */
#define DATABASE_FILE       "idDB.pag"



/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* A single alarm record. */
typedef struct {
  Boolean       can_do_action;
  UINT32        id;
  UINT32        flags;
  char          db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  TIM_TIME_REF  entry_date;
  TIM_TIME_REF  time;
} ALARM_CACHE;


/* Information about the databases we check. */
typedef struct {
  char    db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  time_t  last_modified;  
} DATABASE_CHECK;


/* Entries to ignore. */
typedef struct {
  UINT32  id;
  char    db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
} IGNORE_ENTRY;


/* Basic data structure for the XDalarm appliaction. */
typedef struct {

  /* Command line flags. */
  Boolean  startup_help;
  Boolean  version_help;

  /* Do animation when displaying alarms. */
  Boolean  do_animate_alarm;

  /* Volume for alarms. */
  int  alarm_volume;

  /* Default snooze time. */
  int  snooze_minutes;

  /* Pipe to use when communicating with the parent. */
  char  *cmd_pipe_id;

  /* Alarms for these databases. */
  char  *default_db_alarms;

  /* Display alarms inluding one of these tags. */
  char  *filter_alarm_tags;

  /* Default script (program) to use. */
  char  *launch_def_script;

  /* Shell to use when starting alarm action. */
  char  *launch_use_shell;

  /* We keep track of displayed windows. */
  Widget  *track_alarm_win;

  /* The X application context. */
  XtAppContext  context;

  /* The alarm cache containing alarms to check. */
  ALARM_CACHE  alarms[ MAX_ALARM_CACHE ];

  /* Snooze alarms. */
  ALARM_CACHE  snooze[ MAX_SNOOZE_ALARM ];

  /* The databases we check. */
  DATABASE_CHECK  db_check[ MAX_DATABASE_CHECK ];

  /* File pointer for the command pipe. */
  int  cmd_pipe_ref;

  /* Entries that can be ignored. */
  IGNORE_ENTRY  ignore_entry[ MAX_ENTRIES_IGNORE ];

  /* The current date. */
  TIM_TIME_REF  current_date;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data;

} XTM_AP_BASE_DATA, *XTM_AP_BASE_DATA_REF;


/* User data necessary for alarm window. */
typedef struct {
  Boolean               important;
  int                   curr_pixmap;
  UINT32                id;
  char                  db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  Widget                alarmW;
  Widget                snoozeW;
  XtIntervalId          timer_id;
  TIM_TIME_REF          entry_date;
  XTM_AP_BASE_DATA_REF  appl_data_ref;
} USER_DATA, *USER_DATA_REF;



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;

/* Name of text domain. */
static char  *text_domain = "XDiary";

/* Name of module. */
static char  *module_name = "xtmAlarmProcess";

/* Application data. */
static  XTM_AP_BASE_DATA  appl_data;

static  Widget  toplevel;

/* Pixmaps and pixmap sequence used. */
static  Boolean  init_pixmap = True;
static  Pixmap   ani_pixmap[ 10 ];

static  int ani_pixmap_blink[] = {
  0, 5, 0, 5, 0, 5, 0, -1
};

static  int ani_pixmap_seq[] = {
  1, 2, 3, 4, 5, 6, 7, 8, 9,
  8, 7, 6, 5, 4, 3, 2, 1,
  2, 3, 4, 5, 6, 7, 8,
  7, 6, 5, 4, 3, 2,
  3, 4, 5, 6, 7,
  6, 5, 4,
  3, 4, 5, 6, 7,
  6, 5, 4, 3,
  4, 5, 6,
  6, 5,
  4, 5, 6,
  6, 5, 4,
  5, -1
};


/* Program options. */
static XrmOptionDescRec options[] = {
  { "-alarmTags",   "*.FilterAlarmTags", XrmoptionSepArg, NULL },
  { "-alarmtags",   "*.FilterAlarmTags", XrmoptionSepArg, NULL },
  { "-cmdPipeId",   "*.CmdPipeId",       XrmoptionSepArg, NULL },
  { "-dbName",      "*.DefaultDbAlarms", XrmoptionSepArg, NULL },
  { "-dbname",      "*.DefaultDbAlarms", XrmoptionSepArg, NULL },
  { "-calendars",   "*.DefaultDbAlarms", XrmoptionSepArg, NULL },
  { "-debug",       "*.debug",           XrmoptionNoArg,  "True" },
  { "-h",           "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-help",        "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-lan",         "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-language",    "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-noFileLock",  "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-nofilelock",  "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-usage",       "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-version",     "*.VersionHelp",     XrmoptionNoArg,  "True" },
};

/* Application resources. */
static XtResource  base_resources[] = {

  { "cmdPipeId", "CmdPipeId", XtRString, sizeof( String ),
    XtOffset( XTM_AP_BASE_DATA_REF, cmd_pipe_id ), 
    XtRString, "" },

  { "defaultDbAlarms", "DefaultDbAlarms", XtRString, sizeof( String ),
    XtOffset( XTM_AP_BASE_DATA_REF, default_db_alarms ), 
    XtRString, "" },

  { "doAnimateAlarm", "DoAnimateAlarm", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_AP_BASE_DATA_REF, do_animate_alarm ), 
    XtRString, "True" },

  { "filterAlarmTags", "FilterAlarmTags", XtRString, sizeof( String ),
    XtOffset( XTM_AP_BASE_DATA_REF, filter_alarm_tags ), 
    XtRString, "" },

  { "launchDefScript", "LaunchDefScript", XtRString, sizeof( String ),
    XtOffset( XTM_AP_BASE_DATA_REF, launch_def_script ), 
    XtRString, "XDiary.alarmdo" },

  { "launchUseShell", "LaunchUseShell", XtRString, sizeof( String ),
    XtOffset( XTM_AP_BASE_DATA_REF, launch_use_shell ), 
    XtRString, "/bin/sh" },

  { "startupHelp", "StartupHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_AP_BASE_DATA_REF, startup_help ), 
    XtRString, "False" },

  { "snoozeMinutes", "SnoozeMinutes", XtRInt, sizeof( int ),
    XtOffset( XTM_AP_BASE_DATA_REF, snooze_minutes ), 
    XtRString, "5" },

  { "versionHelp", "VersionHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_AP_BASE_DATA_REF, version_help ), 
    XtRString, "False" },

};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  alarmConfirmCB( Widget               widget, 
                  USER_DATA_REF        user_data,
                  XmAnyCallbackStruct  *call_data );

static void 
  alarmDeleteCB( Widget         widget, 
                 USER_DATA_REF  user_data,
                 XtPointer      call_data );

static void 
  alarmNoMoreCB( Widget         widget, 
                 USER_DATA_REF  user_data,
                 XtPointer      call_data );

static void 
  alarmSnoozeCB( Widget         widget, 
                 USER_DATA_REF  user_data,
                 XtPointer      call_data );

static void 
  alarmViewCB( Widget         widget, 
               USER_DATA_REF  user_data,
               XtPointer      call_data );

static void
  checkAlarms( XTM_AP_BASE_DATA_REF  appl_data_ref,
               TIM_TIME_REF          time_now );

static void 
  deleteEntryCB( Widget         widget, 
                 USER_DATA_REF  user_data,
                 XtPointer      call_data );

static void 
  destroyCB( Widget         widget, 
             USER_DATA_REF  user_data,
             XtPointer      call_data );

static void
  displayAlarm( XTM_AP_BASE_DATA_REF  appl_data_ref,
                ALARM_CACHE           *alarm_ref );

static void
  displayUsage();

static void
  doActionAlarm( XTM_AP_BASE_DATA_REF  appl_data_ref,
                 ALARM_CACHE           *alarm_ref );

static void 
  doAnimateCB( USER_DATA_REF  user_data );

static void 
  doBlinkCB( USER_DATA_REF  user_data );

static Boolean
  fetchAlarms( XTM_AP_BASE_DATA_REF  appl_data_ref,
               XTM_CD_CAL_INFO       *db_info );

static void
  processCommandCB( XTM_AP_BASE_DATA_REF  appl_data_ref,
                    int                   *fd,
                    XtInputId             *id );

static Boolean
  setupCmdPipe( XTM_AP_BASE_DATA_REF  appl_data_ref );

static void 
  snoozeOkCB( Widget         widget, 
              USER_DATA_REF  user_data,
              XtPointer      call_data );

static void
  startAlarmForCal( XTM_AP_BASE_DATA_REF  appl_data_ref,
                    char                  *db_name,
                    char                  *db_location );

int
  startProcess( Widget  parent,
                char    *process_args[] );

static void
  stopAlarmForCal( XTM_AP_BASE_DATA_REF  appl_data_ref,
                   char                  *db_name );

static void
  terminateCB( int   this_signal,
               void  *user_data );

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  main( unsigned int argc, char *argv[] )
{

  /* Variables. */
  int                 index;
  char                *char_ref;
  Arg                 args[ 10 ];
  Cardinal            n;
  Display             *display;
  TIM_STATUS_TYPE     time_status;
  XTM_CB_STATUS       custom_status;
  XTM_GL_CUSTOM_DATA  custom_data;


  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;
  xitErSetApplicationName( program_name );
    

  /* Save the original command parameters. */
  custom_data.orig_argc = argc;
  custom_data.orig_argv = (char**) XtMalloc( argc * sizeof( char* ) );

  for( index = 0; index < argc; index++ )
    custom_data.orig_argv[ index ] = XtNewString( argv[ index ] );
  
  /* NLS enabled. */
  xtmToSetLocale( program_name );


  /* Initialize the signal handler module. */
  SigInitialize();

  /* Initialize. */
  SysInitializeEnvironment();
  xtmDbInitializeProcessId();

  
  /* Initialize toolkit and open display. */
  XtToolkitInitialize();

  appl_data.context = XtCreateApplicationContext();
  display = XtOpenDisplay( appl_data.context, NULL,
                           NULL, PROGRAM_CLASS,
                           options, XtNumber( options ),
#if XtSpecificationRelease < 5
                           (Cardinal *) &argc,
#else
                           (int *) &argc,
#endif
                           argv );

  if( display == NULL )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "Cannot open display, check your DISPLAY variable." );

  /* Resource mapping.*/
  xtmToInitializeResourceMapping( argc, argv, display );
  
  /* Create application shell. */
  n = 0;
  toplevel = XtAppCreateShell( NULL, PROGRAM_CLASS,
                               applicationShellWidgetClass,
                               display,
                               args, n );


  /* Get base application resources. */
  XtGetApplicationResources( toplevel, (XtPointer) &appl_data, 
                             base_resources, 
                             XtNumber( base_resources ), 
                             NULL, 0 );

  /* Get customize resources. */
  xtmRsFetchCustomResources( &custom_data, toplevel );

  /* Reset application data. */
  custom_data.cal_db_handle   = NULL;
  custom_data.group_db_handle = NULL;
  custom_data.archive_files   = NULL;
  custom_data.include_files   = NULL;

  appl_data.current_date = 0;
  appl_data.cmd_pipe_ref = -1;
  appl_data.custom_data  = &custom_data;

  for( index = 0; index < MAX_DATABASE_CHECK; index++ )
    appl_data.db_check[ index ].db_name[ 0 ] = '\0';

  for( index = 0; index < MAX_ALARM_CACHE; index++ )
    appl_data.alarms[ index ].id = 0;

  for( index = 0; index < MAX_ENTRIES_IGNORE; index++ )
    appl_data.ignore_entry[ index ].id = 0;

  for( index = 0; index < MAX_SNOOZE_ALARM; index++ )
    appl_data.snooze[ index ].id = 0;


  appl_data.track_alarm_win = (Widget *)
    SysMalloc( sizeof( Widget ) * MAX_TRACK_ALARM_WIN );

  for( index = 0; index < MAX_TRACK_ALARM_WIN; index++ )
    *(appl_data.track_alarm_win + index) = NULL;


  /* Display current version? */
  if( appl_data.version_help ) {
    printf( "%s: Version: %s\n", program_name, VERSION_ID );
    exit( 0 );
  }

  /* Help requested? */
  if( appl_data.startup_help ) {
    displayUsage();
    exit( 0 );
  }


  /* Get customized data from file. */
  custom_status = xtmCbGetDataFromFile( appl_data.custom_data );

  if( custom_status == XTM_CB_WRONG_VERSION ) {
    char_ref = (char *) 
      SysMalloc( strlen( msgGetText( MXDI_CUST_WRONG_VERSION ) ) + 50 );

    sprintf( char_ref, msgGetText( MXDI_CUST_WRONG_VERSION ),
             xtmCbWhatVersion() );

    xitStDisplaySticky( toplevel, char_ref, XmUNSPECIFIED_PIXMAP,
                        msgGetText( MXDI_OK_BUTTON ), NULL,
                        NULL, NULL, NULL,
                        NULL );
    SysFree( char_ref );
  }


  /* Initialize necessary text domains. */
  msgInitialize();
  msgInitCatalogue( text_domain, NULL, custom_data.msg_language,
                    msgXdiaryTexts );

  /* Default catalogue Xdiary. */
  msgTextDomain( text_domain );


  /* Initialize the time formats. */
  time_status = TimInitializeFormat( custom_data.date_format,
                                     custom_data.time_format );
  if( time_status != TIM_OK )
    xitErMessage( toplevel, XIT_ER_ERROR, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  /* Set colors and fonts in the resource database. */
  xtmRsFetchColors( &custom_data, toplevel );


  /* Initialize holidays. */
  xtmHoInitialize( custom_data.holidays_db_dir, custom_data.workdays );


  /* Use file locking? */
  xtmDbUseFileLock( custom_data.use_file_lock );

  /* Initialize the update module. */
  xtmUpInitialize( appl_data.context, 0 );


  /* If controlled by the XDiary process, watch out for commands. */
  if( *appl_data.cmd_pipe_id != '\0' )
    (void) setupCmdPipe( &appl_data );


  /* Any default databases given? */
  if( strlen( appl_data.default_db_alarms ) > 0 )
    startAlarmForCal( &appl_data, appl_data.default_db_alarms, "" );


  /* Signal handlers. */
  (void) SigRegister( SIGHUP,  0, terminateCB, &appl_data );
  (void) SigRegister( SIGTERM, 0, terminateCB, &appl_data );
  (void) SigRegister( SIGINT,  0, terminateCB, &appl_data );
  (void) SigRegister( SIGILL,  0, terminateCB, &appl_data );
  (void) SigRegister( SIGQUIT, 0, terminateCB, &appl_data );
  (void) SigRegister( SIGABRT, 0, terminateCB, &appl_data );

  /* Register for updates? */
  (void) xtmUpRegister( XTM_UP_MINUTE_TICK, updateCB, (void *) &appl_data );


  /* Enter the event loop. */
  XtAppMainLoop( appl_data.context );


} /* main */


/*----------------------------------------------------------------------*/

static void
  checkAlarms( XTM_AP_BASE_DATA_REF  appl_data_ref,
               TIM_TIME_REF          time_now )
{

  /* Variables. */
  Boolean  give_alarm;
  int      index;
  int      index1;


  /* Code. */

  /* Debug information? */
  if( appl_data_ref -> custom_data -> debug_mode ) {
    char  buffer[ 50 ];

    xtmFoFormatTime( time_now, buffer, sizeof( buffer ) );

    printf( "Time now: %s\n", buffer );
  }

  /* Check all alarms in the list. */
  for( index = 0; index < MAX_ALARM_CACHE; index++ ) {

    ALARM_CACHE  *alarm_ref;

    alarm_ref = &appl_data_ref -> alarms[ index ];

    /* A valid alarm? */
    if( alarm_ref -> id == 0 )
      continue;

    give_alarm = False;

    /* Debug information? */
    if( appl_data_ref -> custom_data -> debug_mode ) {
      char  buffer[ 50 ];

      xtmFoFormatTime( alarm_ref -> time, buffer, sizeof( buffer ) );
      printf( "  Checking time: %s\n", buffer );
    }

    if( TimTimeInSecondsRange( alarm_ref -> time, time_now, 50 ) == TIM_YES ) {

      give_alarm = True;

      /* Ignore this alarm? */
      for( index1 = 0; index1 < MAX_ENTRIES_IGNORE; index1++ ) {

        if( appl_data_ref -> ignore_entry[ index1 ].id == alarm_ref -> id ) {
          if( strcmp( appl_data_ref -> ignore_entry[ index1 ].db_name, 
                      alarm_ref -> db_name ) == 0 ) {

            give_alarm = False;
            break;
          }
        }

      } /* loop */


      if( give_alarm ) {

        /* Visual alarm? */
        if( flagIsSet( alarm_ref -> flags, XTM_DB_FLAG_ALARM ) )
          displayAlarm( appl_data_ref, alarm_ref );

        /* Action alarm (call script)? */
        if( flagIsSet( alarm_ref -> flags, XTM_DB_FLAG_ACTION_SCRIPT ) )
          doActionAlarm( appl_data_ref, alarm_ref );

        /* Action alarm (use text as script)? */
        if( flagIsSet( alarm_ref -> flags, XTM_DB_FLAG_ACTION_TEXT ) )
          doActionAlarm( appl_data_ref, alarm_ref );

      } /* if */

    } /* if */

  } /* loop */


  /* Check all snooze alarms. */
  for( index = 0; index < MAX_SNOOZE_ALARM; index++ ) {

    ALARM_CACHE  *snooze_ref;

    snooze_ref = &appl_data_ref -> snooze[ index ];

    /* Valid snooze alarm? */
    if( snooze_ref -> id != 0 ) {

      if( TimTimeInSecondsRange( snooze_ref -> time, 
          time_now, 50 ) == TIM_YES )
        displayAlarm( appl_data_ref, snooze_ref );

    }

  } /* loop */


  return;

} /* checkAlarms */


/*----------------------------------------------------------------------*/

static void
  displayAlarm( XTM_AP_BASE_DATA_REF  appl_data_ref,
                ALARM_CACHE           *alarm_ref )
{

  /* Variables. */
  int                     alarm_melody;
  int                     index;
  int                     lines_read;
  int                     size;
  UINT32                  can_do;
  UINT32                  operations;
  char                    buffer[ 100 ];
  char                    *char_ref;
  char                    *entry_text_ref = NULL;
  char                    *message;
  char                    *tune_file;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  alarmTl;
  Widget                  dataLocalW[ 2 ];
  Widget                  tempW;
  Widget                  workFo;
  USER_DATA_REF           user_data;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;
  XTM_CD_CAL_INFO         db_info;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", alarmConfirmCB, NULL },
    { "", alarmNoMoreCB,  NULL },
    { "", alarmSnoozeCB,  NULL },
    { "", alarmViewCB,    NULL },
    { "", alarmDeleteCB,  NULL },
  };


  /* Code. */

  /* Alarm is hidden? */
  if( flagIsSet( alarm_ref -> flags, XTM_DB_FLAG_HIDE_IN_ALARM ) )
    return;


  /* Fetch database information. */
  (void) xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle, 
                            alarm_ref -> db_name,
                            &db_info, NULL );

  /* Check operations. */
  xtmDbCheckDbOperations( db_info.directory, True, &operations );


  /* Open the database. */
  open_request.name       = db_info.short_name;
  open_request.directory  = db_info.directory;
  open_request.operations = XTM_DB_FLAG_MODE_READ;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;


  /* Try to open the database. */
  db_status = xtmDbOpenEntryDb( &open_request, &database );

  if( db_status != XTM_DB_OK ) {

    if( db_status == XTM_DB_LOCKED )
      xitErMessage( toplevel, XIT_ER_ERROR, 
                    module_name, "displayAlarm",
                    msgGetText( MXDI_ERRMSG_DB_LOCKED ) );
    else
      xitErMessage( toplevel, XIT_ER_ERROR, 
                    module_name, "displayAlarm",
                    msgGetText( MXDI_ERRMSG_CANNOT_OPEN_DB ) );

    raise exception;

  } /* if */


  /* Fetch the entry to display. */
  db_status = xtmDbFetchEntry( &database, alarm_ref -> id, 
                               &entry_record, &entry_text_ref );

  xtmDbCloseEntryDb( &database );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( NULL, XIT_ER_ERROR, 
                  module_name, "displayAlarm",
                  msgGetText( MXDI_ERRMSG_FETCH_ENTRY ) );

    raise exception;
  }


  /* Does the entry have the correct tag? */
  if( *appl_data_ref -> filter_alarm_tags != '\0' &&
      strstr( appl_data_ref -> filter_alarm_tags, 
              entry_record.entry.tag ) == NULL )
    raise exception;


  /* No messages for private entries (if you are not privileged). */
  if(   flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_PRIVATE ) &&
      ! flagIsSet( operations, XTM_DB_FLAG_MODE_PRIV ) )
    raise exception;
  

  /* Allocate memory for our user data. */
  user_data = SysNew( USER_DATA );

  user_data -> appl_data_ref = appl_data_ref;
  user_data -> entry_date    = alarm_ref -> entry_date;
  user_data -> id            = alarm_ref -> id;

  strcpy( user_data -> db_name, alarm_ref -> db_name );

  if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_IMPORTANT ) )
    user_data -> important = True;
  else
    user_data -> important = False;


  /* Create a separate window. */
  action_buttons[ 0 ].label = msgGetText( MXDI_ALARM_CONFIRM );
  action_buttons[ 0 ].data  = user_data;
  action_buttons[ 1 ].label = msgGetText( MXDI_ALARM_NO_MORE );
  action_buttons[ 1 ].data  = user_data;
  action_buttons[ 2 ].label = msgGetText( MXDI_ALARM_SNOOZE );
  action_buttons[ 2 ].data  = user_data;
  action_buttons[ 3 ].label = msgGetText( MXDI_ALARM_VIEW );
  action_buttons[ 3 ].data  = user_data;
  action_buttons[ 4 ].label = msgGetText( MXDI_ALARM_DELETE );
  action_buttons[ 4 ].data  = user_data;

  alarmTl = xitCreateToplevelDialog( toplevel, "AlarmTl",
                                     1, 0,
                                     action_buttons, 
                                     XtNumber( action_buttons ) );

  XtAddCallback( alarmTl, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) user_data );


  /* Create the message window. */
  sprintf( buffer, "%s %s", 
           alarm_ref -> db_name, msgGetText( MXDI_ALARM_TITLE ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer    ); n++;
  XtSetArg( args[ n ], XmNiconName, buffer ); n++;
  XtSetValues( alarmTl, args, n );


  /* Form to hold the alarm message. */
  workFo = XtNameToWidget( alarmTl, "AlarmTlBase.AlarmTlFo" );



  /* Pixelmap label. */
  markerPx = xitCreateLabel( workFo, "MarkerPx", "", -1 );


  /* Initialize the pixmaps? */
  if( init_pixmap ) {
    ani_pixmap[ 0 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK5INV, False );
    ani_pixmap[ 1 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK1, False );
    ani_pixmap[ 2 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK2, False );
    ani_pixmap[ 3 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK3, False );
    ani_pixmap[ 4 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK4, False );
    ani_pixmap[ 5 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK5, False );
    ani_pixmap[ 6 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK6, False );
    ani_pixmap[ 7 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK7, False );
    ani_pixmap[ 8 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK8, False );
    ani_pixmap[ 9 ] = xtmIcFetchSimplePixmap(
                        workFo, XTM_IC_ICON_ALARM_CLOCK9, False );

    init_pixmap = False;
  }

  n = 0;
  XtSetArg( args[ n ], XmNlabelPixmap, ani_pixmap[ 1 ] ); n++;
  XtSetArg( args[ n ], XmNlabelType, XmPIXMAP ); n++;
  XtSetValues( markerPx, args, n );


  /* We cannot delete entries in a read only db. */
  xtmDbGetEntryPermissions( operations,
                            entry_record.entry.owner,
                            entry_record.entry.flags,
                            &can_do );

  if( flagIsClear( can_do, XTM_DB_PROT_DELETE ) ) {
    tempW = XtNameToWidget( alarmTl, "AlarmTlBase.Bu5" );

    XtSetSensitive( tempW, False );
  }


  /* Make a copy of the entry text. */
  if( entry_text_ref != NULL )
    size = strlen( entry_text_ref ) + 75;
  else
    size = strlen( entry_record.entry.text ) + 75;

  message = SysMalloc( size );
  *message = '\0';

  xtmFoFormatDate( alarm_ref -> entry_date,
                   buffer, sizeof( buffer ) );

  strcat( message, buffer );
  strcat( message, ", " );

  TimFormatStrTime( alarm_ref -> entry_date, "%A",
                    buffer, sizeof( buffer ) );

  strcat( message, buffer );
  strcat( message, " " );

  xtmFoFormatTime( entry_record.entry.time_stamp,
                   buffer, sizeof( buffer ) );

  strcat( message, buffer );
  strcat( message, "\n" );


  if( entry_text_ref != NULL )
    strcat( message, entry_text_ref );
  else
    strcat( message, entry_record.entry.text );



  /* Number of lines to write? */
  lines_read = 0;
  char_ref   = message;

  while( lines_read < entry_record.entry.alarm_lines + 1 ) {
    char_ref = strchr( char_ref, '\n' );
    if( char_ref == NULL )
      break;
    lines_read++;
    char_ref++;
  }

  if( char_ref != NULL )
    *char_ref = '\0';


  /* Message label. */
  messageLa = xitCreateLabel( workFo,  "MessageLa", 
                              message, XmALIGNMENT_BEGINNING );


  /* Put the Parts together. */
  xitAttachWidget( markerPx,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( messageLa,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, markerPx,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( markerPx,  args, n );
  XtSetValues( messageLa, args, n );

  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( alarmTl, workFo, XTM_IC_ICON_DEFAULT );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( alarmTl, True );


  XtPopup( alarmTl, XtGrabNone );


  /* Free allocated data. */
  if( entry_text_ref != NULL )
    SysFree( entry_text_ref );


  /* Give an alarm, a boring bell or a nice melody. */
  alarm_melody = entry_record.entry.alarm_melody;

  if( alarm_melody > 0 && alarm_melody < 6 )
    tune_file = appl_data_ref -> custom_data -> tune_file[ alarm_melody - 1 ];
  else
    tune_file = NULL;

  xtmToPlayMelody( alarmTl,
                   appl_data_ref -> custom_data -> tune_player,
                   appl_data_ref -> custom_data -> tune_player_param,
                   tune_file,
                   appl_data_ref -> alarm_volume,
                   appl_data_ref -> custom_data -> tune_dur_delta );

  /* Display animation pixmap. */
  user_data -> timer_id = (XtIntervalId) 0;

  if( appl_data_ref -> do_animate_alarm ) {
    user_data -> curr_pixmap = 0;
    user_data -> timer_id = XtAppAddTimeOut(
                              appl_data_ref -> context, 0L,
                              (XtTimerCallbackProc) doAnimateCB,
                              (XtPointer) user_data );
  }


  /* We keep track of windows on the screen. */
  for( index = 0; index < MAX_TRACK_ALARM_WIN; index++ ) {
    if( *(appl_data_ref -> track_alarm_win + index) == NULL ) {
      *(appl_data_ref -> track_alarm_win + index) = alarmTl;
      break;
    }
  }

  user_data -> alarmW = alarmTl;

  return;


  /* Exception handler. */
  exception:

    if( entry_text_ref != NULL )
      SysFree( entry_text_ref );

    return;

} /* displayAlarm */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Alarm process for a single diary database\n" 
    "\n"
    "Start an alarm process for one or more diary databases. The\n"
    "default database in Diary in your home directory.\n"
    "\n"
    "Usage:\n"
    "  %s [flags]\n"
    "\n"
    "Flags:\n"
    "  -alarmTags       : Display alarms containing one of these tags.\n"
    "  -fmap large      : Use a large font.\n"
    "  -fmap medium     : Use a medium font.\n"
    "  -fmap small      : Use a small font.\n"
    "  -help            : Display this help.\n"
    "  -h               : See above.\n"
    "  -language <lan>  : Use the language <lan>.\n"
    "  -lan <lan>       : See above.\n"
    "  -noFileLock      : Don't use any file locking.\n"
    "  -palette gray    : Use color palette Gray.\n"
    "  -palette lila    : Use color palette Lila.\n"
    "  -palette motif   : Use color palette Motif.\n"
    "  -palette neon    : Use color palette Neon.\n"
    "  -palette nordic  : Use color palette Nordic.\n"
    "  -palette red     : Use color palette Red.\n"
    "  -palette sea     : Use color palette Sea.\n"
    "  -palette sky     : Use color palette Sky.\n"
    "  -palette wood    : Use color palette Wood.\n"
    "  -usage           : Display this help.\n"
    "  -version         : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );


  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static void
  doActionAlarm( XTM_AP_BASE_DATA_REF  appl_data_ref,
                 ALARM_CACHE           *alarm_ref )
{

  /* Variables. */
  int                     pid;
  UINT32                  operations;
  char                    date_buffer[ 50 ];
  char                    time_buffer[ 50 ];
  char                    *process_args[ 10 ];
  char                    *entry_text_ref;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  /* Should we do action alarm? */
  if( ! alarm_ref -> can_do_action )
    return;

  /* Fetch database information. */
  (void) xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle, 
                            alarm_ref -> db_name,
                            &db_info, NULL );

  /* Check operations. */
  xtmDbCheckDbOperations( db_info.directory, True, &operations );


  /* Open the database. */
  open_request.name       = db_info.short_name;
  open_request.directory  = db_info.directory;
  open_request.operations = XTM_DB_FLAG_MODE_READ;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  /* Try to open the database. */
  db_status = xtmDbOpenEntryDb( &open_request, &database );

  if( db_status != XTM_DB_OK ) {

    if( db_status == XTM_DB_LOCKED )
      xitErMessage( toplevel, XIT_ER_ERROR, 
                    module_name, "doActionAlarm",
                    msgGetText( MXDI_ERRMSG_DB_LOCKED ) );
    else
      xitErMessage( toplevel, XIT_ER_ERROR, 
                    module_name, "doActionAlarm",
                    msgGetText( MXDI_ERRMSG_CANNOT_OPEN_DB ) );
    return;

  } /* if */


  /* Fetch the entry to use. */
  db_status = xtmDbFetchEntry( &database, alarm_ref -> id, 
                               &entry_record, &entry_text_ref );

  xtmDbCloseEntryDb( &database );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( NULL, XIT_ER_ERROR, 
                  module_name, "doActionAlarm",
                  msgGetText( MXDI_ERRMSG_FETCH_ENTRY ) );

    return;
  }

  /* Does the entry have the correct tag? */
  if( *appl_data_ref -> filter_alarm_tags != '\0' &&
      strstr( appl_data_ref -> filter_alarm_tags, 
              entry_record.entry.tag ) == NULL )
    return;


  /* No actions for private entries (if you are not privileged). */
  if(   flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_PRIVATE ) &&
      ! flagIsSet( operations, XTM_DB_FLAG_MODE_PRIV ) )
    return;


  /* Entry date and time. */
  xtmFoFormatDate( entry_record.entry.time_stamp,
                   date_buffer, sizeof( date_buffer ) );

  xtmFoFormatTime( entry_record.entry.time_stamp,
                   time_buffer, sizeof( time_buffer ) );


  /* Start entry action shell script? */
  if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_ACTION_TEXT ) ) {

    /* Process arguments. */
    process_args[ 0 ] = appl_data_ref -> launch_use_shell;
    process_args[ 1 ] = "-c";

    if( entry_text_ref == NULL )
      process_args[ 2 ] = entry_record.entry.text;
    else
      process_args[ 2 ] = entry_text_ref;

    process_args[ 3 ] = NULL;

    pid = startProcess( toplevel, process_args );

  } /* if */ 


  /* Start entry action with entry text as shell script? */
  if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_ACTION_SCRIPT ) ) {

    /* Process arguments. */
    process_args[ 0 ] = appl_data_ref -> launch_def_script;
    process_args[ 1 ] = entry_record.entry.tag;
    process_args[ 2 ] = date_buffer;
    process_args[ 3 ] = time_buffer;

    if( entry_text_ref != NULL )
      process_args[ 4 ] = entry_text_ref;
    else
      process_args[ 4 ] = entry_record.entry.text;

    process_args[ 5 ] = NULL;

    pid = startProcess( toplevel, process_args );

  } /* if */ 

  if( entry_text_ref != NULL )
    SysFree( entry_text_ref );


  return;

} /* doActionAlarm */


/*----------------------------------------------------------------------*/

static Boolean
  fetchAlarms( XTM_AP_BASE_DATA_REF  appl_data_ref,
               XTM_CD_CAL_INFO       *db_info )
{

  /* Variables. */
  int                     index;
  UINT32                  search_flags;
  LST_DESC_TYPE           list_ref[ 2 ];
  LST_STATUS              lst_status;
  TIM_TIME_REF            current_day;
  TIM_TIME_REF            end_day;
  TIM_TIME_REF            now;
  TIM_TIME_REF            start_day;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;


  /* Code. */

  if( appl_data_ref -> custom_data -> debug_mode )
    printf( "Fetching db: %s\n", db_info -> short_name );


  /* Remove all old alarms for this database. */
  for( index = 0; index < MAX_ALARM_CACHE; index++ ) {
    if( strcmp( appl_data_ref -> alarms[ index ].db_name, 
                db_info -> short_name ) == 0 )
      appl_data_ref -> alarms[ index ].id = 0;
  }
    

  /* Open the database. */
  open_request.name       = db_info -> short_name;
  open_request.directory  = db_info -> directory;
  open_request.operations = XTM_DB_FLAG_MODE_READ;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  db_status = xtmDbOpenEntryDb( &open_request, &database );

  if( db_status != XTM_DB_OK ) {

    if( db_status == XTM_DB_LOCKED )
      xitErMessage( toplevel, XIT_ER_ERROR, 
                    module_name, "fetchAlarms",
                    msgGetText( MXDI_ERRMSG_DB_LOCKED ) );
    else
      xitErMessage( toplevel, XIT_ER_ERROR, 
                    module_name, "fetchAlarms",
                    msgGetText( MXDI_ERRMSG_CANNOT_OPEN_DB ) );

    return( False );

  } /* if */

  /* Fetch entries from today and 7 days before/after. */
  now       = TimLocalTime( TimMakeTimeNow() );
  start_day = TimMakeTime( TimIndexOfYear(  now ),
                           TimIndexOfMonth( now ),
                           TimIndexOfDay(   now ),
                           0, 0, 0 );

  TimAddDays( &start_day, -7 );
  current_day = start_day;
  end_day     = start_day;
  TimAddDays( &end_day, 14 );

  /* Fetch all entries. */
  while( current_day <= end_day ) {

    db_status = xtmDbFetchEntriesInDay( &database, 
                                        current_day, 
                                        XTM_DB_FETCH_STANDING,
                                        &list_ref[ 0 ], &list_ref[ 1 ] );

    if( db_status != XTM_DB_OK ) {
      xitErMessage( NULL, XIT_ER_ERROR, 
                    module_name, "displayAlarm",
                    msgGetText( MXDI_ERRMSG_FETCH_ENTRY ) );

      return( False );
    }

    search_flags = (XTM_DB_FLAG_ALARM | 
                    XTM_DB_FLAG_ACTION_SCRIPT |
                    XTM_DB_FLAG_ACTION_TEXT);

    /* Ignore any notes and save the alarms. */
    lst_status = LstLinkCurrentFirst( list_ref[ 0 ] );

    while( lst_status == LST_OK ) {

      lst_status = LstLinkGetCurrent( list_ref[ 0 ], &entry_record );

      /* Do we want alarms for this entry? */
      if( flagIsClear( entry_record.entry.flags, search_flags ) ) {
        lst_status = LstLinkCurrentNext( list_ref[ 0 ] );

        continue;
      }


      /* Process all alarms. */
      for( index = 0; index < MAX_ALARMS; index++ ) {

        int           minute_offset;
        int           slot;
        UINT32        entry_id;
        UINT32        flags;
        TIM_TIME_REF  alarm_time;

        /* Is the alarm valid? */
        if( entry_record.entry.alarm_valid[ index ] ) {
          minute_offset = entry_record.entry.alarm_offset[ index ];
          entry_id      = entry_record.entry.id;
          flags         = entry_record.entry.flags;

          /* Calculate the absolute alarm time. */
          alarm_time = current_day;
          alarm_time = TimAddTime( alarm_time, 
                                   entry_record.entry.time_stamp );
          TimAddMinutes( &alarm_time, minute_offset );

          /* Find a free slot where the alarm can be stored. */
          for( slot = 0; slot < MAX_ALARM_CACHE; slot++ ) {
            if( appl_data_ref -> alarms[ slot ].id == 0 )
              break;
          }


          /* Save alarm time and other data. */
          if( slot < MAX_ALARM_CACHE ) {

            UINT32  action_flag = 0;

            /* Can do action alarm? */
            switch( index + 1 ) {
              case 1:
                action_flag = XTM_DB_FLAG_ACTION_ALARM1;
                break;
              case 2:
                action_flag = XTM_DB_FLAG_ACTION_ALARM2;
                break;
              case 3:
                action_flag = XTM_DB_FLAG_ACTION_ALARM3;
                break;
              case 4:
                action_flag = XTM_DB_FLAG_ACTION_ALARM4;
                break;
              case 5:
                action_flag = XTM_DB_FLAG_ACTION_ALARM5;
                break;
            }

            appl_data_ref -> alarms[ slot ].entry_date    = current_day;
            appl_data_ref -> alarms[ slot ].time          = alarm_time;
            appl_data_ref -> alarms[ slot ].id            = entry_id;
            appl_data_ref -> alarms[ slot ].flags         = flags;
            appl_data_ref -> alarms[ slot ].can_do_action = False;

            if( flagIsSet( flags, action_flag ) )
              appl_data_ref -> alarms[ slot ].can_do_action = True;

            strcpy( appl_data_ref -> alarms[ slot ].db_name, 
                    db_info -> short_name );

            if( appl_data_ref -> custom_data -> debug_mode )
              printf( "Alarm time %d (%d) '%s'\n",
                      (int) alarm_time, slot,
                      entry_record.entry.text );
          }

        } /* if */

      } /* loop */


      /* Next entry. */
      lst_status = LstLinkCurrentNext( list_ref[ 0 ] );

    } /* while */


    /* Free any data occupied. */
    LstLinkClear( list_ref[ 0 ] );
    LstLinkClear( list_ref[ 1 ] );

    /* Next day. */
    TimAddDays( &current_day, 1 );

  } /* while */


  /* Close the database. */
  xtmDbCloseEntryDb( &database );


  return( True );

} /* fetchAlarms */


/*----------------------------------------------------------------------*/

static Boolean
  setupCmdPipe( XTM_AP_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  int  fd;


  /* Code. */

  if( *appl_data_ref -> cmd_pipe_id == '\0' )
    return( False );


  fd = open( appl_data_ref -> cmd_pipe_id, O_RDONLY );
  if( fd < 0 )
    return( False );

  appl_data_ref -> cmd_pipe_ref = fd;


  /* Add callback to call when input is pending. */
  XtAppAddInput( appl_data_ref -> context, 
                 appl_data_ref -> cmd_pipe_ref,
                 (XtPointer) XtInputReadMask,
                 (XtInputCallbackProc) processCommandCB,
                 (XtPointer) appl_data_ref );


  return( True );

} /* setupCmdPipe */


/*----------------------------------------------------------------------*/

static void
  startAlarmForCal( XTM_AP_BASE_DATA_REF  appl_data_ref,
                    char                  *db_name,
                    char                  *db_location )
{

  /* Variables. */
  Boolean          ok;
  UINT32           operations;
  int              save_to_index;
  int              index;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle, 
                          db_name, &db_info, NULL );

  /* If we did not find the database, see if we can add it 'on the fly'. */
  if( ! ok ) {

    /* We need at least read access. */
    xtmDbCheckDbOperations( db_location, True, &operations );

    if( flagIsClear( operations, XTM_DB_FLAG_MODE_READ ) )
      return;

    /* Add the calendar. */
    xtmCdAddDatabase( appl_data_ref -> custom_data -> cal_db_handle,
                      db_name, db_location, 0 );

    (void) xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle, 
                              db_name, &db_info, NULL );

  } /* if */


  save_to_index = -1;

  /* Save the database name (if already there, ignore it). */
  for( index = 0; index < MAX_DATABASE_CHECK; index++ ) {

    DATABASE_CHECK  *db_ref;

    db_ref = &appl_data_ref -> db_check[ index ];

    /* Already there? */
    if( strcmp( db_ref -> db_name, db_name ) == 0 ) {
        save_to_index = -1;
        break;
      }

    /* Free slot? */
    if( db_ref -> db_name[ 0 ] == '\0' && save_to_index == -1 )
      save_to_index = index;

  } /* loop */


  /* Save the database? */
  if( save_to_index >= 0 ) {
    if( appl_data_ref -> custom_data -> debug_mode )
      printf( "Starting alarm for: %s\n", db_name );

    strcpy( appl_data_ref -> db_check[ save_to_index ].db_name, db_name );
    appl_data_ref -> db_check[ save_to_index ].last_modified = 0;
  }


  return;

} /* startAlarmForCal */


/*----------------------------------------------------------------------*/

static void
  stopAlarmForCal( XTM_AP_BASE_DATA_REF  appl_data_ref,
                   char                  *db_name )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Remove from databases db. */
  for( index = 0; index < MAX_DATABASE_CHECK; index++ ) {
    if( strcmp( appl_data_ref -> db_check[ index ].db_name, db_name ) == 0 )
      appl_data_ref -> db_check[ index ].db_name[ 0 ] = '\0';
  }


  /* Remove from alarm_cache. */
  for( index = 0; index < MAX_ALARM_CACHE; index++ ) {
    if( strcmp( appl_data_ref -> alarms[ index ].db_name, db_name ) == 0 )
      appl_data_ref -> alarms[ index ].id = 0;
  }


  /* Remove from ignore entries. */
  for( index = 0; index < MAX_ENTRIES_IGNORE; index++ ) {
    if( strcmp( appl_data_ref -> ignore_entry[ index ].db_name, 
                db_name ) == 0 )
      appl_data_ref -> ignore_entry[ index ].id = 0;
  }


  /* Remove from snooze. */
  for( index = 0; index < MAX_SNOOZE_ALARM; index++ ) {
    if( strcmp( appl_data_ref -> snooze[ index ].db_name, db_name ) == 0 )
      appl_data_ref -> snooze[ index ].id = 0;
  }


  return;

} /* stopAlarmForCal */


/*----------------------------------------------------------------------*/

int
  startProcess( Widget  parent,
                char    *process_args[] )
{

  /* Variables. */
  int      pid;
  int      status;
  Display  *display = NULL;


  /* Code. */

  if( parent != NULL )
    display = XtDisplay( parent );

  /* Does the executable file exist? */
  status = access( process_args[ 0 ], (R_OK | X_OK) );

  if( status != 0 ) {
    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "startProcess",
                  msgGetText( MXDI_PROCESS_NO_FILE ) );
    return( 0 );  
  }

  /* Start the alarm process. */
  pid = fork();

  switch( pid ) {

    /* Child running. */
    case 0:
      if( display != NULL )
        close( ConnectionNumber( display ) );

      /* Start the process. */
      execv( process_args[ 0 ], process_args );

      {
        char  buffer[ 200 ];

        sprintf( buffer, 
                 msgGetText( MXDI_ERRMSG_EXECUTE_PROCESS ),
                 process_args[ 0 ] );

        xitErMessage( NULL, XIT_ER_ERROR, 
                      module_name, "startProcess",
                      buffer );
      } /* block */

      exit( 1 );

    /* Error in fork. */
    case -1:
      xitErMessage( parent, XIT_ER_ERROR,
                    module_name, "startProcess",
                    msgGetText( MXDI_PROCESS_CANNOT_FORK ) );
      return( 0 );

    /* Parent */
    default:
      break;

  } /* switch */

  /* Catch this process when we return. */
  (void) SigRegister( SIGCHLD, pid, NULL, NULL );


  return( pid );

} /* startProcess */


/*----------------------------------------------------------------------*/

static void 
  alarmConfirmCB( Widget               widget, 
                  USER_DATA_REF        user_data,
                  XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  int                   index;
  XTM_AP_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = user_data -> appl_data_ref;


  /* The shift modifier key confirms all alarm windows on screen. */
  if( (call_data -> event -> xbutton.state & ShiftMask) != 0 ) {

    for( index = 0; index < MAX_TRACK_ALARM_WIN; index++ ) {
      if( *(appl_data_ref -> track_alarm_win + index) == user_data -> alarmW )
        ;
      else if( *(appl_data_ref -> track_alarm_win + index) != NULL )
        XtDestroyWidget( *(appl_data_ref -> track_alarm_win + index) );
    }

  }


  XtDestroyWidget( user_data -> alarmW );


  return;

} /* alarmConfirmCB */


/*----------------------------------------------------------------------*/

static void 
  alarmDeleteCB( Widget         widget, 
                 USER_DATA_REF  user_data,
                 XtPointer      call_data )
{

  /* Variables. */
  Widget  tempW;


  /* Code. */

  /* Does the user confirm? */
  if( user_data -> appl_data_ref -> custom_data -> confirm_actions ) {

    tempW = xitCreateQuestionDialog( 
              user_data -> alarmW, "QuestionDialog",
              msgGetText( MXDI_DELETE_ENTRY_TITLE ),
              msgGetText( MXDI_DELETE_ENTRY_CONF ),
              deleteEntryCB, user_data,
              NULL, NULL );

    return;
  }

  /* Remove the entry. */
  deleteEntryCB( widget, user_data, NULL );


  return;

} /* alarmDeleteCB */


/*----------------------------------------------------------------------*/

static void 
  alarmNoMoreCB( Widget         widget, 
                 USER_DATA_REF  user_data,
                 XtPointer      call_data )
{

  /* Variables. */
  int     index;


  /* Code. */

  /* Mark this entry as ignored. */
  for( index = 0; index < MAX_ENTRIES_IGNORE; index++ ) {

    if( user_data -> appl_data_ref -> ignore_entry[ index ].id == 0 ) {
      user_data -> appl_data_ref -> ignore_entry[ index ].id = 
        user_data -> id;
      strcpy( user_data -> appl_data_ref -> ignore_entry[ index ].db_name,
              user_data -> db_name );
      break;
    }

  } /* loop */

  XtDestroyWidget( user_data -> alarmW );


  return;

} /* alarmNoMoreCB */


/*----------------------------------------------------------------------*/

static void 
  alarmSnoozeCB( Widget         widget, 
                 USER_DATA_REF  user_data,
                 XtPointer      call_data )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 2 ];
  Widget    snoozeFd;
  Widget    snoozeFo;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   snoozeOkCB, NULL },
    { NULL, NULL,       NULL },
    { NULL, NULL,       NULL },
    { NULL, NULL,       NULL },
    { NULL, NULL,       NULL },
    { "",   NULL,       NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = user_data;
  action_buttons[ 5 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 5 ].data  = user_data;


  /* Create a form dialog with buttons. */
  snoozeFd = xitCreateFormDialog( user_data -> alarmW, "SnoozeFd",
                                  1, 0,
                                  action_buttons,
                                  XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_SNOOZE_TITLE ) ); n++;
  XtSetValues( XtParent( snoozeFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( snoozeFd, args, n );


  /* Container for the contents of the snooze window. */
  snoozeFo = XtNameToWidget( snoozeFd, "SnoozeFdFo" );


  snoozeLa = xitCreateLabel( snoozeFo, "SnoozeLa", 
                             msgGetText( MXDI_SNOOZE_LABEL ), -1 );

  /* Create a slide bar with the time in minutes to wait. */
  n = 0;
  XtSetArg( args[ n ], XmNvalue, 1 ); n++;
  XtSetArg( args[ n ], XmNminimum, 1 ); n++;
  XtSetArg( args[ n ], XmNmaximum, 120 ); n++;
  XtSetArg( args[ n ], XmNprocessingDirection, XmMAX_ON_RIGHT ); n++;
  XtSetArg( args[ n ], XmNshowValue, True ); n++;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  snoozeSc = XmCreateScale( snoozeFo, "SnoozeSc", args, n );


  /* Put the elements together. */
  xitAttachWidget( snoozeLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( snoozeSc,
                   XmATTACH_WIDGET, snoozeLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( snoozeLa,  args, n );
  XtSetValues( snoozeSc,  args, n );


  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  XtManageChild( snoozeFd );


  /* Default snooze minutes. */
  XmScaleSetValue( snoozeSc, user_data -> appl_data_ref -> snooze_minutes );


  user_data -> snoozeW = snoozeFd;


  return;

} /* alarmSnoozeCB */


/*----------------------------------------------------------------------*/

static void 
  alarmViewCB( Widget         widget, 
               USER_DATA_REF  user_data,
               XtPointer      call_data )
{

  /* Variables. */
  char                    *char_ref;
  char                    *entry_text_ref;
  XTM_AP_BASE_DATA_REF    appl_data_ref;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref = user_data -> appl_data_ref;


  /* Fetch database information. */
  (void) xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                            user_data -> db_name,
                            &db_info, NULL );

  /* Open the database. */
  open_request.name       = db_info.short_name;
  open_request.directory  = db_info.directory;
  open_request.operations = XTM_DB_FLAG_MODE_READ;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  db_status = xtmDbOpenEntryDb( &open_request, &database );

  if( db_status != XTM_DB_OK ) {

    if( db_status == XTM_DB_LOCKED )
      xitErMessage( user_data -> alarmW, XIT_ER_ERROR, 
                    module_name, "alarmViewCB",
                    msgGetText( MXDI_ERRMSG_DB_LOCKED ) );
    else
      xitErMessage( user_data -> alarmW, XIT_ER_ERROR, 
                    module_name, "alarmViewCB",
                    msgGetText( MXDI_ERRMSG_CANNOT_OPEN_DB ) );

    return;

  } /* if */


  /* Fetch the entry to display. */
  db_status = xtmDbFetchEntry( &database, user_data -> id, 
                               &entry_record, &entry_text_ref );

  xtmDbCloseEntryDb( &database );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( user_data -> alarmW, XIT_ER_ERROR, 
                  module_name, "alarmViewCB",
                  msgGetText( MXDI_ERRMSG_FETCH_ENTRY ) );
    return;
  }


  /* View the entry. */
  if( entry_text_ref != NULL )
    char_ref = entry_text_ref;
  else
    char_ref = entry_record.entry.text;

  xitViewText( user_data -> alarmW, char_ref,
               " ", msgGetText( MXDI_CLOSE_BUTTON ) );


  return;

} /* alarmViewCB */


/*----------------------------------------------------------------------*/

static void 
  deleteEntryCB( Widget         widget, 
                 USER_DATA_REF  user_data,
                 XtPointer      call_data )
{

  /* Variables. */
  Widget                  tempW;
  XTM_AP_BASE_DATA_REF    appl_data_ref;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref = user_data -> appl_data_ref;


  /* Fetch database information. */
  (void) xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                            user_data -> db_name,
                            &db_info, NULL );

  /* Open the database. */
  open_request.name       = db_info.short_name;
  open_request.directory  = db_info.directory;
  open_request.operations = XTM_DB_FLAG_MODE_WRITE;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  db_status = xtmDbOpenEntryDb( &open_request, &database );

  if( db_status != XTM_DB_OK ) {

    if( db_status == XTM_DB_LOCKED )
      xitErMessage( user_data -> alarmW, XIT_ER_ERROR, 
                    module_name, "deleteEntryCB",
                    msgGetText( MXDI_ERRMSG_DB_LOCKED ) );
    else
      xitErMessage( user_data -> alarmW, XIT_ER_ERROR, 
                    module_name, "deleteEntryCB",
                    msgGetText( MXDI_ERRMSG_CANNOT_OPEN_DB ) );

    return;

  } /* if */


  /* Delete the entry. */
  db_status = xtmDbDeleteEntry( &database, user_data -> id );

  xtmDbCloseEntryDb( &database );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( user_data -> alarmW, XIT_ER_ERROR, 
                  module_name, "deleteEntryCB",
                  msgGetText( MXDI_NO_ENTRY ) );
    return;
  }


  /* If the entry could be deleted, invalidate some buttons. */
  if( db_status == XTM_DB_OK ) {
    tempW = XtNameToWidget( user_data -> alarmW, "AlarmTlBase.Bu2" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( user_data -> alarmW, "AlarmTlBase.Bu3" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( user_data -> alarmW, "AlarmTlBase.Bu4" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( user_data -> alarmW, "AlarmTlBase.Bu5" );
    XtSetSensitive( tempW, False );
  }

  /* Inform the user that it's time to refresh the day list. */
  tempW = xitCreateInformationDialog( 
            user_data -> alarmW, "InformationDialog", 
            msgGetText( MXDI_INFORMATION_LABEL ),
            msgGetText( MXDI_ALARM_ENTRY_DELETE_INFO ),
            NULL, NULL );


  return;

} /* deleteEntryCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget         widget, 
             USER_DATA_REF  user_data,
             XtPointer      call_data )
{

  /* Variables. */
  int                   index;
  XTM_AP_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = user_data -> appl_data_ref;


  for( index = 0; index < MAX_TRACK_ALARM_WIN; index++ ) {
    if( *(appl_data_ref -> track_alarm_win + index) == user_data -> alarmW )
      *(appl_data_ref -> track_alarm_win + index) = NULL;
  }

  SysFree( user_data );

  /* Remove timeout ID? */
  if( user_data -> timer_id != (XtIntervalId) 0 )
    XtRemoveTimeOut( user_data -> timer_id );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  doAnimateCB( USER_DATA_REF  user_data )
{

  /* Variables. */
  int                   pixmap_no;
  Arg                   args[ 10 ];
  Cardinal              n;
  Widget                tempW;
  XTM_AP_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = user_data -> appl_data_ref;
  pixmap_no = ani_pixmap_seq[ user_data -> curr_pixmap ];

  /* Are we done? */
  if( pixmap_no < 0 ) {
    if( user_data -> important ) {
      user_data -> curr_pixmap = 0;
      user_data -> timer_id = XtAppAddTimeOut(
                                appl_data_ref -> context, 0L,
                                (XtTimerCallbackProc) doBlinkCB,
                                (XtPointer) user_data );
    } else {
      user_data -> timer_id = (XtIntervalId) 0;
    }

    return;
  }

  /* Display the pixmap. */
  tempW = XtNameToWidget( user_data -> alarmW,
                          "AlarmTlBase.AlarmTlFo.MarkerPx" );

  n = 0;
  XtSetArg( args[ n ], XmNlabelPixmap, ani_pixmap[ pixmap_no ] ); n++;
  XtSetValues( tempW, args, n );

  /* Next pixmap in the sequence. */
  user_data -> curr_pixmap++;

  user_data -> timer_id = XtAppAddTimeOut(
                              appl_data_ref -> context, 150L,
                              (XtTimerCallbackProc) doAnimateCB,
                              (XtPointer) user_data );


  return;

} /* doAnimateCB */


/*----------------------------------------------------------------------*/

static void 
  doBlinkCB( USER_DATA_REF  user_data )
{

  /* Variables. */
  int                   pixmap_no;
  Arg                   args[ 10 ];
  Cardinal              n;
  Widget                tempW;
  XTM_AP_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = user_data -> appl_data_ref;
  pixmap_no = ani_pixmap_blink[ user_data -> curr_pixmap ];

  /* Are we done? */
  if( pixmap_no < 0 ) {
    user_data -> timer_id = (XtIntervalId) 0;
    return;
  }

  /* Display the pixmap. */
  tempW = XtNameToWidget( user_data -> alarmW,
                          "AlarmTlBase.AlarmTlFo.MarkerPx" );

  n = 0;
  XtSetArg( args[ n ], XmNlabelPixmap, ani_pixmap[ pixmap_no ] ); n++;
  XtSetValues( tempW, args, n );

  /* Next pixmap in the sequence. */
  user_data -> curr_pixmap++;

  user_data -> timer_id = XtAppAddTimeOut(
                            appl_data_ref -> context, 500L,
                            (XtTimerCallbackProc) doBlinkCB,
                            (XtPointer) user_data );


  return;

} /* doBlinkCB */


/*----------------------------------------------------------------------*/

static void
  processCommandCB( XTM_AP_BASE_DATA_REF  appl_data_ref,
                    int                   *fd,
                    XtInputId             *id )
{

  /* Variables. */
  int   bytes;
  int   items;
  char  buffer[ 5000 ];
  char  cal_name[ 50 ];
  char  command[ 60 ];
  char  location[ 150 ];
  char  line[ 300 ];
  char  *buffer_ref;


  /* Code. */

  /* Read the command pipe. */
  bytes = read( appl_data_ref -> cmd_pipe_ref, buffer, sizeof( buffer ) );

  if( bytes == 0 )
    return;

  if( bytes == -1 ) {
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "processCommandCB", "Exiting!" );

    terminateCB( SIGTERM, (void *) appl_data_ref );
  }

  buffer[ bytes ] = '\0';

  if( appl_data_ref -> custom_data -> debug_mode )
    printf( "Command over pipe:\n" );


  /* There can be one or more commands in the buffer. */
  buffer_ref = buffer;

  while( True ) {

    while( isspace( *buffer_ref ) )
      buffer_ref++;

    if( *buffer_ref == '\0' )
      return;

    bytes = strlen( buffer_ref );
    sscanf( buffer_ref, "%[^\n]%n", line, &bytes );
    buffer_ref = buffer_ref + bytes + 1;

    /* Fetch the command (if any). */
    items = sscanf( line, "%50s", command );
    if( items == 0 )
      return;


    /* Start alarm process? */
    if( strcmp( command, "start:" ) == 0 ) {

      cal_name[ 0 ] = '\0';
      location[ 0 ] = '\0';

      sscanf( line, "%s %s %s", command, cal_name, location );

      if( appl_data_ref -> custom_data -> debug_mode )
        printf( "  command: '%s' '%s' '%s'\n", command, cal_name, location );

      startAlarmForCal( appl_data_ref, cal_name, location );


    /* Stop running process. */
    } else if( strcmp( command, "stop:" ) == 0 ) {

      cal_name[ 0 ] = '\0';

      sscanf( line, "%s %s", command, cal_name );

      if( appl_data_ref -> custom_data -> debug_mode )
        printf( "  command: '%s' '%s'\n", command, cal_name );

      stopAlarmForCal( appl_data_ref, cal_name );


    /* Unknown command. */
    } else {
      sprintf( line, "Unexpected command: %s\n", command );

      xitErMessage( NULL, XIT_ER_FATAL, 
                    module_name, "processCommandCB", line );

    } /* if */

  } /* while */


} /* processCommandCB */


/*----------------------------------------------------------------------*/

static void 
  snoozeOkCB( Widget         widget, 
              USER_DATA_REF  user_data,
              XtPointer      call_data )
{

  /* Variables. */
  int                   index;
  int                   minutes;
  Widget                tempW;
  TIM_TIME_REF          alarm_time;
  TIM_TIME_REF          now;
  XTM_AP_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = user_data -> appl_data_ref;


  /* Fetch the slider value. */
  tempW = XtNameToWidget( user_data -> snoozeW, "SnoozeFdFo.SnoozeSc" );

  XmScaleGetValue( tempW, &minutes );

  if( minutes <= 0 )
    return;


  /* Calculate the alarm time. */
  now = TimMakeTimeNow();

  alarm_time = TimMakeTime( TimIndexOfYear(  now ),
                            TimIndexOfMonth( now ),
                            TimIndexOfDay(   now ),
                            TimHour(         now ),
                            TimMinute(       now ),
                            0 );
  alarm_time = TimLocalTime( alarm_time );

  TimAddMinutes( &alarm_time, minutes );


  /* Store the time. */
  for( index = 0; index < MAX_SNOOZE_ALARM; index++ ) {

    if( appl_data_ref -> snooze[ index ].id == 0 ) {

      appl_data_ref -> snooze[ index ].id         = user_data -> id;
      appl_data_ref -> snooze[ index ].entry_date = user_data -> entry_date;
      appl_data_ref -> snooze[ index ].time       = alarm_time;

      strcpy( appl_data_ref -> snooze[ index ].db_name, 
              user_data -> db_name );
      break;
    }

  } /* loop */

  XtDestroyWidget( user_data -> snoozeW );


  return;

} /* snoozeOkCB */


/*----------------------------------------------------------------------*/

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data )
{

  /* Variables. */
  Boolean               new_day = False;
  int                   parent_id;
  int                   index;
  TIM_TIME_REF          time_now;
  XTM_AP_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_AP_BASE_DATA_REF) user_data;


  if( appl_data_ref -> custom_data -> debug_mode )
    printf( "UpdateCB called\n" );

  /* If our parent process is 1 the our real parent died. */
  parent_id = getppid();
  if( parent_id <= 1 )
    terminateCB( SIGTERM, (void *) appl_data_ref );

  time_now = TimLocalTime( TimMakeTimeNow() );

  /* Has the date changed (over midnight)? */
  if( TimIsSameDate( appl_data_ref -> current_date, time_now ) != TIM_YES ) {

    appl_data_ref -> current_date = time_now;

    new_day = True;
  }


  /* If this is a new day, cancel all 'no more' entries. */
  if( new_day ) {
    for( index = 0; index < MAX_ENTRIES_IGNORE; index++ )
      appl_data.ignore_entry[ index ].id = 0;
  }


  /* Did any of our databases change since last checked? */
  for( index = 0; index < MAX_DATABASE_CHECK; index++ ) {

    Boolean          ok;
    char             buffer[ 200 ];
    struct stat      file_status;
    XTM_DB_STATUS    db_status;
    XTM_CD_CAL_INFO  db_info;

    if( appl_data_ref -> db_check[ index ].db_name[ 0 ] == '\0' )
      continue;

    if( appl_data_ref -> custom_data -> debug_mode )
      printf( "Check db changed: '%s'\n", 
              appl_data_ref -> db_check[ index ].db_name );

    /* Fetch complete information for the database. */
    ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                            appl_data_ref -> db_check[ index ].db_name,
                            &db_info, NULL );
    if( ! ok )
      continue;

    /* Build the file name for the database. */
    sprintf( buffer, "%s/%s",
             db_info.directory, DATABASE_FILE );

    /* If the database changed, fetch new information. */
    db_status = xtmDbFetchFileInfo( buffer, &file_status );

    if( new_day || 
        (db_status == XTM_DB_OK && 
         file_status.st_mtime !=
         appl_data_ref -> db_check[ index ].last_modified) ) {

      fetchAlarms( appl_data_ref, &db_info );

      appl_data_ref -> db_check[ index ].last_modified = 
        file_status.st_mtime;
    }

  } /* loop */


  /* Check if any alarms to give? */
  checkAlarms( appl_data_ref, time_now );


  return;

} /* updateCB */


/*----------------------------------------------------------------------*/

static void
  terminateCB( int   this_signal,
               void  *user_data )
{

  /* Variables. */
  XTM_AP_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_AP_BASE_DATA_REF) user_data;

  if( appl_data_ref -> cmd_pipe_ref != -1 ) {
    close( appl_data_ref -> cmd_pipe_ref );

    (void) unlink( appl_data_ref -> cmd_pipe_id );
  }

  exit( 0 );

} /* terminateCB */
