/*----------------------------------------------------------------------------
--
--  Module:           xtmEdAclAfs
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Edit the ACL list for a calendar in an AFS file system.
--
--  Filename:         xtmEdAclAfs.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-30
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmEdAclAfs.c, Version: 1.1, Date: 95/02/18 15:58:53";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <pwd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmAccBase.h"
#include "xtmAfsAclBs.h"
#include "xtmHelp.h"
#include "xitError.h"
#include "xitInfo.h"
#include "xitTools.h"
#include "xtmEdAclAfs.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* The number of ACLs we can handle. */
#define MAX_ACLS         200


/* Local widgets in the window. */
#define aclLi            dataLocalW[  0 ]
#define actionRc         dataLocalW[  1 ]
#define calLa            dataLocalW[  2 ]
#define flagMsgTb        dataLocalW[  3 ]
#define flagRc           dataLocalW[  4 ]
#define flagReadTb       dataLocalW[  5 ]
#define flagWriteTb      dataLocalW[  6 ]
#define idLa             dataLocalW[  7 ]
#define idTx             dataLocalW[  8 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record describing one instance of the ACL window. */
typedef struct {

  /* Has any fields changed? */
  Boolean  field_changed;

  /* Has any ACLs changed? */
  Boolean  acl_changed;

  /* Location of the calendar to edit. */
  char  cal_location[ XTM_GL_MAX_LOCATION + 1 ];

  /* Name of the calendar to edit. */
  char  cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* Default acl for the system administrators. */
  char  *system_admin_acl;

  /* ACL window. */
  Widget  aclW;

  /* Info reference. */
  XIT_IN_HANDLE  info_handle;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  /* Callback and user data to inform our creator of specific actions. */
  XTM_AA_ACTION_CB  actionCB;
  void              *user_data;

} ACL_REC, *ACL_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmEditAclAfs";

/* IDs for the help windows. */
static char  *acl_window_id = "EditAclAfs";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  accessListCB( Widget                widget, 
                ACL_REC_REF           acl_ref,
                XmListCallbackStruct  *call_data );

static void
  addCB( Widget       widget,
         ACL_REC_REF  acl_ref,
         XtPointer    call_data );

static void
  cancelCB( Widget       widget,
            ACL_REC_REF  acl_ref,
            XtPointer    call_data );

static void
  closeCB( Widget       widget,
           ACL_REC_REF  acl_ref,
           XtPointer    call_data );

static Widget
  createAclWindow( ACL_REC_REF  acl_ref,
                   Widget       parent );

static void
  deleteCB( Widget       widget,
            ACL_REC_REF  acl_ref,
            XtPointer    call_data );

static void 
  destroyCB( Widget       widget,
             ACL_REC_REF  acl_ref,
             XtPointer    call_data );

static void 
  fieldChangedCB( Widget       widget,
                  ACL_REC_REF  acl_ref,
                  XtPointer    call_data );

static void 
  helpCB( Widget       widget,
          ACL_REC_REF  acl_ref,
          XtPointer    call_data );

static void 
  okCB( Widget       widget,
        ACL_REC_REF  acl_ref,
        XtPointer    call_data );

static void
  saveAccessInfo( ACL_REC_REF  acl_ref );

static void 
  saveAndCloseCB( Widget       widget,
                  ACL_REC_REF  acl_ref,
                  XtPointer    call_data );

static void
  setAccessData( ACL_REC_REF  acl_ref );

static void
  setAccessItem( ACL_REC_REF  acl_ref,
                 char         *id,
                 int          flags );

static void
  setAction( ACL_REC_REF  acl_ref,
             Boolean      add_active,
             Boolean      delete_active );

static void
  setEmptyItem( ACL_REC_REF  acl_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_AA_HANDLE
  xtmAaInitialize( XTM_GL_CUSTOM_DATA_REF  custom_data_ref,
                   XIT_IN_HANDLE           info_handle,
                   Widget                  parent,
                   char                    *system_admin_acl,
                   XTM_AA_ACTION_CB        actionCB,
                   void                    *user_data )
{

  /* Variables. */
  ACL_REC_REF  acl_ref;


  /* Code. */

  /* Create and initialize our private data. */
  acl_ref = SysNew( ACL_REC );
  if( acl_ref == NULL )
    return( NULL );

  acl_ref -> custom_data_ref = custom_data_ref;
  acl_ref -> info_handle     = info_handle;
  acl_ref -> field_changed   = False;
  acl_ref -> actionCB        = actionCB;
  acl_ref -> user_data       = user_data;
  acl_ref -> aclW            = NULL;

  acl_ref -> system_admin_acl = SysNewString( system_admin_acl );


  /* Create the base window. */
  acl_ref -> aclW = createAclWindow( acl_ref, parent );
  if( acl_ref -> aclW == NULL ) {
    SysFree( acl_ref );

    return( NULL );
  }


  return( (XTM_AA_HANDLE) acl_ref );

} /* xtmAaInitialize */


/*----------------------------------------------------------------------*/

void
  xtmAaDestroy( XTM_AA_HANDLE  acl_handle )
{

  /* Variables. */
  ACL_REC_REF  acl_ref;


  /* Code. */

  if( acl_handle == NULL )
    return;

  /* Our private data. */
  acl_ref = (ACL_REC_REF) acl_handle;

  /* Get rid of all windows. */
  if( acl_ref -> aclW != NULL )
    XtDestroyWidget( acl_ref -> aclW );


  return;

} /* xtmAaDestroy */


/*----------------------------------------------------------------------*/

void
  xtmAaEditAcl( XTM_AA_HANDLE  acl_handle,
                char           *cal_name,
                char           *cal_location )
{

  /* Variables. */
  Arg          args[ 10 ];
  Cardinal     n;
  ACL_REC_REF  acl_ref;


  /* Code. */

  /* Our private data. */
  acl_ref = (ACL_REC_REF) acl_handle;


  /* Save the name and location of the calendar. */
  strncpy( acl_ref -> cal_name, cal_name, XTM_GL_MAX_CAL_NAME );
  acl_ref -> cal_name[ XTM_GL_MAX_CAL_NAME ] = '\0';

  strncpy( acl_ref -> cal_location, cal_location, XTM_GL_MAX_LOCATION );
  acl_ref -> cal_location[ XTM_GL_MAX_LOCATION ] = '\0';


  /* Make sure the window is visible. */
  XtManageChild( acl_ref -> aclW );


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( XtParent( acl_ref -> aclW ), args, n );


  /* Set the window data. */
  setAccessData( acl_ref );

  acl_ref -> acl_changed = False;


  return;

} /* xtmAaEditAcl */


/*----------------------------------------------------------------------*/

static Widget
  createAclWindow( ACL_REC_REF  acl_ref,
                   Widget       parent )
{

  /* Variables. */
  int       index;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    actionBu[ 2 ];
  Widget    aclFd;
  Widget    dataLocalW[ 9 ];
  Widget    workFo;

  static XIT_PUSH_STRUCT access_action_def[] = {
    { "AddPb",     "", "", True, NULL },
    { "DeletePb",  "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "IdTx",  NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   okCB,     NULL },
    { NULL, NULL,     NULL },
    { "",   cancelCB, NULL },
    { NULL, NULL,     NULL },
    { "",   helpCB,   NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = acl_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 2 ].data  = acl_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 4 ].data  = acl_ref;

  access_action_def[ 0 ].title = msgGetText( MXDI_ADD_CHANGE_ITEM );
  access_action_def[ 1 ].title = msgGetText( MXDI_DELETE_ITEM );


  /* Create a form dialog with buttons. */
  aclFd = xitCreateFormDialog( parent, "AclFd",
                               1, 0,
                               action_buttons,
                               XtNumber( action_buttons ) );

  XtAddCallback( aclFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) acl_ref );

  /* We want to know when this window is closed. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( aclFd ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( XtParent( aclFd ), &wm_delete_window, 1 );
    XmAddWMProtocolCallback( XtParent( aclFd ), wm_delete_window, 
                             (XtCallbackProc) cancelCB, (XtPointer) acl_ref );
  } /* block */

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_AFS_ACCESS_TITLE ) ); n++;
  XtSetValues( XtParent( aclFd ), args, n );


  /* Form to hold the information. */
  workFo = XtNameToWidget( aclFd, "AclFdFo" );


  /* The calendar we are editing. */
  calLa = xitCreateLabel( workFo, "CalLa", "", -1 );


  /* List with access information. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  aclLi = XmCreateScrolledList( workFo, "AclLi", args, n );

  XtAddCallback( aclLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) accessListCB, (XtPointer) acl_ref );
  XtAddCallback( aclLi, XmNdefaultActionCallback,
                 (XtCallbackProc) accessListCB, (XtPointer) acl_ref );


  /* AFS ID. */
  idLa = xitCreateLabel( workFo, "IdLa", 
                         msgGetText( MXDI_USER_LABEL ), -1 );

  idTx = xitCreateText( workFo, &text_buffer[ 0 ] );

  XtAddCallback( idTx, XmNvalueChangedCallback, 
                 (XtCallbackProc) fieldChangedCB, (XtPointer) acl_ref );


  /* Access flags. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  flagRc = XmCreateRowColumn( workFo, "FlagRc", args, n );

  flagReadTb  = xitCreateToggleButton( 
                  flagRc, "FlagReadTb", 
                  msgGetText( MXDI_READ_LABEL ), False );

  flagWriteTb = xitCreateToggleButton( 
                  flagRc, "FlagWriteTb", 
                  msgGetText( MXDI_WRITE_LABEL ), False );

  flagMsgTb   = xitCreateToggleButton( 
                  flagRc, "FlagMsgTb", 
                  msgGetText( MXDI_MESSAGE_PRIV_LABEL ), False );


  /* Action (add, delete and change). */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  actionRc = XmCreateRowColumn( workFo, "ActionRc", args, n );

  for( index = 0; index < XtNumber( access_action_def ); index++ )
    actionBu[ index ] = xitCreatePushButton( actionRc,
                                             &access_action_def[ index ] );

  XtAddCallback( actionBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) addCB, (XtPointer) acl_ref );
  XtAddCallback( actionBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) deleteCB, (XtPointer) acl_ref );


  /* Place the various elements together. */
  xitAttachWidget( calLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( aclLi ),
                   XmATTACH_WIDGET, calLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( idLa,
                   XmATTACH_WIDGET, XtParent( aclLi ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,              XmATTACH_NONE, NULL );
  xitAttachWidget( idTx,
                   XmATTACH_WIDGET, idLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( flagRc,
                   XmATTACH_WIDGET, idTx, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( actionRc,
                   XmATTACH_WIDGET, flagRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( calLa,             args, n );
  XtSetValues( XtParent( aclLi ), args, n );
  XtSetValues( idLa,              args, n );
  XtSetValues( idTx,              args, n );
  XtSetValues( flagRc,            args, n );
  XtSetValues( actionRc,          args, n );


  /* Manage the widgets. */
  XtManageChildren( actionBu, XtNumber( actionBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( aclFd, True );


  /* Make the final attachments. */
  xitAttachWidget( actionRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( flagRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, actionRc );
  xitAttachWidget( idTx,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, flagRc );
  xitAttachWidget( idLa,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, idTx );
  xitAttachWidget( XtParent( aclLi ),
                   XmATTACH_WIDGET, calLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,  XmATTACH_WIDGET, idLa );


  return( aclFd );

} /* createAclWindow */


/*----------------------------------------------------------------------*/

static void
  saveAccessInfo( ACL_REC_REF  acl_ref )
{

  /* Variables. */
  int                  index;
  int                  item_count;
  UINT32               diary_rights;
  UINT32               message_rights;
  char                 flags[ 30 ];
  char                 *char_ref;
  Arg                  args[ 3 ];
  Cardinal             n;
  Widget               mainW;
  Widget               tempW;
  XmString             *items;
  LST_DESC_TYPE        access_info;
  LST_DESC_TYPE        access_diary;
  LST_DESC_TYPE        access_message;
  LST_STATUS           lst_status;
  XTM_AB_AFS_ACL_INFO  acl_id;
  XTM_AB_STATUS        ab_status;
  XTM_AF_ACL_INFO      afs_acl_entry;
  XTM_AF_STATUS        af_status;


  /* Code. */

  mainW = XtNameToWidget( acl_ref -> aclW, "AclFdFo" );


  /* Create an new access list. */
  access_info    = LstLinkNew( sizeof( XTM_AB_AFS_ACL_INFO ), NULL );
  access_diary   = LstLinkNew( sizeof( XTM_AF_ACL_INFO ),     NULL );
  access_message = LstLinkNew( sizeof( XTM_AF_ACL_INFO ),     NULL );


  /* Fetch information from the access list. */
  tempW = XtNameToWidget( mainW, "AclLiSW.AclLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtSetArg( args[ n ], XmNitems,     &items ); n++;
  XtGetValues( tempW, args, n );

  for( index = 0; index < item_count; index++ ) {

    char_ref = xitStringGetString( *(items + index), CS );

    sscanf( char_ref, "%s %s", acl_id.id, flags );
    SysFree( char_ref );


    /* Check the flags. */
    acl_id.flags   = 0;
    diary_rights   = (XTM_AF_FLAG_LOOKUP | XTM_AF_FLAG_LOCK);
    message_rights = XTM_AF_FLAG_LOOKUP;

    if( strchr( flags, 'r' ) != NULL ) {
      flagSet( acl_id.flags, XTM_AB_FLAG_READ );
      flagSet( diary_rights, XTM_AF_FLAG_READ );
    }

    if( strchr( flags, 'w' ) != NULL ) {
      flagSet( acl_id.flags, XTM_AB_FLAG_WRITE );
      flagSet( acl_id.flags, XTM_AB_FLAG_READ );
      flagSet( diary_rights, XTM_AF_FLAG_WRITE );
      flagSet( diary_rights, XTM_AF_FLAG_READ );
      flagSet( diary_rights, XTM_AF_FLAG_DELETE );
      flagSet( diary_rights, XTM_AF_FLAG_INSERT );
    }

    if( strchr( flags, 'm' ) != NULL ) {
      flagSet( acl_id.flags, XTM_AB_FLAG_MSG );
      flagSet( message_rights, (XTM_AF_FLAG_INSERT | XTM_AF_FLAG_LOCK) );
    }

    /* Save the entry. */
    lst_status = LstLinkInsertLast( access_info, &acl_id );

    /* AFS ACL definitions. */
    afs_acl_entry.positive = True;
    strcpy( afs_acl_entry.id, acl_id.id );

    /* Diary rights. */
    afs_acl_entry.rights = diary_rights;
    lst_status = LstLinkInsertLast( access_diary, &afs_acl_entry );

    /* Message rights. */
    afs_acl_entry.rights = message_rights;
    lst_status = LstLinkInsertLast( access_message, &afs_acl_entry );

  } /* loop */


  /* The owner has all privileges. */
  {
    struct passwd  *password_ref;

    afs_acl_entry.rights = XTM_AF_FLAG_ALL;

    password_ref = getpwuid( getuid() );
    strcpy( afs_acl_entry.id, password_ref -> pw_name );

    lst_status = LstLinkInsertFirst( access_diary,   &afs_acl_entry );
    lst_status = LstLinkInsertFirst( access_message, &afs_acl_entry );
  }


  /* system:administrators privileges? */
  if( strlen( acl_ref -> system_admin_acl ) > 0 ) {
    strcpy( afs_acl_entry.id, "system:administrators" );

    afs_acl_entry.rights = xtmAfAclStringToFlag( acl_ref ->
                                                 system_admin_acl ); 

    lst_status = LstLinkInsertFirst( access_diary,   &afs_acl_entry );
    lst_status = LstLinkInsertFirst( access_message, &afs_acl_entry );
  }


  /* Save the access informatuon. */
  ab_status = xtmAbSaveAfsAccessInfo( acl_ref -> cal_location, access_info );

  /* Set ACL for Diary directory. */
  af_status = xtmAfSetAclForFile( acl_ref -> cal_location, access_diary );

  /* Set ACL for Message directory. */
  char_ref = (char *) SysMalloc( strlen( acl_ref -> cal_location ) + 50 );
  sprintf( char_ref, "%s/Message", acl_ref -> cal_location );

  af_status = xtmAfSetAclForFile( char_ref, access_message );
  SysFree( char_ref );


  /* Free access list. */
  LstLinkClear( access_info );
  LstLinkClear( access_diary );
  LstLinkClear( access_message );


  return;

} /* saveAccessInfo */


/*----------------------------------------------------------------------*/

static void
  setAccessData( ACL_REC_REF  acl_ref )
{

  /* Variables. */
  int                  index;
  int                  index1;
  char                 flags_buffer[ 20 ];
  char                 *char_ref;
  Arg                  args[ 5 ];
  Cardinal             n;
  Widget               mainW;
  Widget               tempW;
  XmString             list_items[ MAX_ACLS ];
  LST_DESC_TYPE        access_info;
  LST_STATUS           lst_status;
  XTM_AB_AFS_ACL_INFO  acl_info;


  /* Code. */

  mainW = XtNameToWidget( acl_ref -> aclW, "AclFdFo" );


  /* Name of the calendar we are using. */
  char_ref = (char *) SysMalloc( strlen( acl_ref -> cal_name ) +
                                 strlen( acl_ref -> cal_location ) + 10 );

  sprintf( char_ref, "%s (%s)", 
           acl_ref -> cal_name, acl_ref -> cal_location );

  tempW = XtNameToWidget( mainW, "CalLa" );
  xitStringSetLabel( tempW, char_ref );

  SysFree( char_ref );


  /* All is empty. */
  setEmptyItem( acl_ref );


  /* Fetch the access information for this database. */
  xtmAbGetAfsAccessInfo( acl_ref -> cal_location, &access_info );

  /* Set access control info. */
  if( LstLinkElements( access_info ) <= 0 )
    return;


  index = 0;

  lst_status = LstLinkCurrentFirst( access_info );

  while( lst_status == LST_OK ) {
    lst_status = LstLinkGetCurrent( access_info, &acl_info );

    /* Fetch the flags values. */
    flags_buffer[ 0 ] = '\0';

    if( flagIsSet( acl_info.flags, XTM_AB_FLAG_READ ) )
      strcat( flags_buffer, "r" );
    else
      strcat( flags_buffer, "-" );

    if( flagIsSet( acl_info.flags, XTM_AB_FLAG_WRITE ) )
      strcat( flags_buffer, "w" );
    else
      strcat( flags_buffer, "-" );

    if( flagIsSet( acl_info.flags, XTM_AB_FLAG_MSG ) )
      strcat( flags_buffer, "m" );
    else
      strcat( flags_buffer, "-" );

    char_ref = (char *) SysMalloc( strlen( acl_info.id ) + 50 );
    sprintf( char_ref, "%s  (%s)", acl_info.id, flags_buffer );

    list_items[ index ] = XmStringCreate( char_ref, CS );
    SysFree( char_ref );

    index++;

    if( index >= MAX_ACLS )
      break;

    /* Next entry. */
    lst_status = LstLinkCurrentNext( access_info );

  } /* while */


  tempW = XtNameToWidget( mainW, "AclLiSW.AclLi" );

  /* Assign the database locations to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );


  return;

} /* setAccessData */


/*----------------------------------------------------------------------*/

static void
  setAccessItem( ACL_REC_REF  acl_ref,
                 char         *id,
                 int          flags )
{

  /* Variables. */
  int       found_position;
  int       index;
  int       item_count;
  char      flags_buffer[ 20 ];
  char      read_id[ XTM_AB_MAX_AFS_ID_LENGTH + 1 ];
  char      *char_ref;
  Arg       args[ 3 ];
  Cardinal  n;
  Widget    listW;
  Widget    mainW;
  XmString  *items;
  XmString  xstr;


  /* Code. */

  mainW = XtNameToWidget( acl_ref -> aclW, "AclFdFo" );


  /* Fetch the flags values. */
  flags_buffer[ 0 ] = '\0';

  if( flagIsSet( flags, XTM_AB_FLAG_READ ) )
    strcat( flags_buffer, "r" );
  else
    strcat( flags_buffer, "-" );

  if( flagIsSet( flags, XTM_AB_FLAG_WRITE ) )
    strcat( flags_buffer, "w" );
  else
    strcat( flags_buffer, "-" );

  if( flagIsSet( flags, XTM_AB_FLAG_MSG ) )
    strcat( flags_buffer, "m" );
  else
    strcat( flags_buffer, "-" );

  if( flagIsSet( flags, XTM_AB_FLAG_PRIV ) )
    strcat( flags_buffer, "p" );
  else
    strcat( flags_buffer, "-" );

  /* Line to add to list. */
  char_ref = (char *) SysMalloc( strlen( id ) + 50 );
  sprintf( char_ref, "%s  %s", id, flags_buffer );

  xstr = XmStringCreate( char_ref, CS );
  SysFree( char_ref );


  /* Check if we have this id before? */
  listW = XtNameToWidget( mainW, "AclLiSW.AclLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtSetArg( args[ n ], XmNitems,     &items ); n++;
  XtGetValues( listW, args, n );

  found_position = 0;

  for( index = 0; index < item_count; index++ ) {

    char_ref = xitStringGetString( *(items + index), CS );
    sscanf( char_ref, "%s", read_id );
    SysFree( char_ref );

    if( strcmp( read_id, id ) == 0 )
      found_position = index + 1;

  } /* loop */

  /* Does the entry already exist? */
  if( found_position > 0 )
    XmListDeletePos( listW, found_position );


  /* Store the new location. */
  XmListAddItem( listW, xstr, 0 );
  XmListSetBottomPos( listW, 0 );
  XmListDeselectAllItems( listW );

  XmStringFree( xstr );  


  return;

} /* setAccessItem */


/*----------------------------------------------------------------------*/

static void
  setAction( ACL_REC_REF  acl_ref,
             Boolean      add_active,
             Boolean      delete_active )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( acl_ref -> aclW, "AclFdFo" );

  tempW = XtNameToWidget( mainW, "ActionRc.AddPb" );
  XtSetSensitive( tempW, add_active );

  tempW = XtNameToWidget( mainW, "ActionRc.DeletePb" );
  XtSetSensitive( tempW, delete_active );


  return;

} /* setAction */


/*----------------------------------------------------------------------*/

static void
  setEmptyItem( ACL_REC_REF  acl_ref )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( acl_ref -> aclW, "AclFdFo" );

  /* Empty the user, group and position field. */
  tempW = XtNameToWidget( mainW, "IdTx" );
  XmTextSetString( tempW, "" );


  /* Set the flag values. */
  tempW = XtNameToWidget( mainW, "FlagRc.FlagReadTb" );
  XmToggleButtonSetState( tempW, False );

  tempW = XtNameToWidget( mainW, "FlagRc.FlagWriteTb" );
  XmToggleButtonSetState( tempW, False );

  tempW = XtNameToWidget( mainW, "FlagRc.FlagMsgTb" );
  XmToggleButtonSetState( tempW, False );


  /* No items selected. */
  setAction( acl_ref, True, False );

  tempW = XtNameToWidget( mainW, "AclLiSW.AclLi" );
  XmListDeselectAllItems( tempW );

  acl_ref -> field_changed = False;


  return;

} /* setEmptyItem */


/*----------------------------------------------------------------------*/

static void 
  accessListCB( Widget                widget, 
                ACL_REC_REF           acl_ref,
                XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char    flags[ 20 ];
  char    id[ XTM_AB_MAX_AFS_ID_LENGTH + 1 ];
  char    *char_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( acl_ref -> aclW, "AclFdFo" );


  /* Fetch the selected name/directory line. */
  char_ref = xitStringGetString( call_data -> item, CS );

  flags[ 0 ] = '\0';

  sscanf( char_ref, "%s %s", id, flags );
  SysFree( char_ref );


  /* Set AFS id. */
  tempW = XtNameToWidget( mainW, "IdTx" );
  XmTextSetString( tempW, id );


  /* Set flags. */
  tempW = XtNameToWidget( mainW, "FlagRc.FlagReadTb" );
  if( strchr( flags, 'r' ) != NULL )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  tempW = XtNameToWidget( mainW, "FlagRc.FlagWriteTb" );
  if( strchr( flags, 'w' ) != NULL )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  tempW = XtNameToWidget( mainW, "FlagRc.FlagMsgTb" );
  if( strchr( flags, 'm' ) != NULL )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  setAction( acl_ref, True, True );

  acl_ref -> field_changed = False;


  return;

} /* accessListCB */


/*----------------------------------------------------------------------*/

static void
  addCB( Widget       widget,
         ACL_REC_REF  acl_ref,
         XtPointer    call_data )
{

  /* Variables. */
  int     flags;
  int     items;
  int     num_id;
  char    id[ XTM_AB_MAX_AFS_ID_LENGTH + 1 ];
  char    *char_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( acl_ref -> aclW, "AclFdFo" );


  /* Fetch the AFS ID. */
  tempW = XtNameToWidget( mainW, "IdTx" );

  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%s", id );
  SysFree( char_ref );

  if( items != 1 )
    return;


  /* Is this a valid id? */
  num_id = xtmAfFetchId( id );
  if( num_id == XTM_AF_NOID ) {
      xitErMessage( acl_ref -> aclW, XIT_ER_ERROR,
                    module_name, "addCB",
                    msgGetText( MXDI_NO_VALID_USERNAME ) );
    return;
  }


  /* Fetch the access flags. */
  flags = 0;

  tempW = XtNameToWidget( mainW, "FlagRc.FlagReadTb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( flags, XTM_AB_FLAG_READ );

  tempW = XtNameToWidget( mainW, "FlagRc.FlagWriteTb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( flags, XTM_AB_FLAG_WRITE | XTM_AB_FLAG_READ );

  tempW = XtNameToWidget( mainW, "FlagRc.FlagMsgTb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( flags, XTM_AB_FLAG_MSG );

  /* Any flags given? */
  if( flags == 0 ) {
    xitErMessage( acl_ref -> aclW, XIT_ER_ERROR,
                  module_name, "addCB",
                  msgGetText( MXDI_NO_ACCESS_FLAGS ) );
    return;
  }


  /* Set the new database. */
  setAccessItem( acl_ref, id, flags );

  /* All is empty. */
  setEmptyItem( acl_ref );

  acl_ref -> acl_changed = True;


  return;

} /* addCB */


/*----------------------------------------------------------------------*/

static void
  cancelCB( Widget       widget,
            ACL_REC_REF  acl_ref,
            XtPointer    call_data )
{

  /* Variables. */
  Widget  tempW;


  /* Code. */

  /* Any ACL changes? */
  if( acl_ref -> acl_changed ) {
    tempW = xitCreateQuestionDialog( 
              acl_ref -> aclW, "InformationDialog", 
	      msgGetText( MXDI_QUESTION_MESSAGE_LABEL ),
              msgGetText( MXDI_ALL_CHANGES_LOST ),
              closeCB, acl_ref,
              NULL, NULL );

    return;
  }

  closeCB( widget, acl_ref, NULL );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void
  deleteCB( Widget       widget,
            ACL_REC_REF  acl_ref,
            XtPointer    call_data )
{

  /* Variables. */
  int       item_count;
  Arg       args[ 3 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;
  XmString  *items;


  /* Code. */

  mainW = XtNameToWidget( acl_ref -> aclW, "AclFdFo" );


  tempW = XtNameToWidget( mainW, "AclLiSW.AclLi" );

  /* Fetch selected item from the list. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems, &items ); n++;
  XtGetValues( tempW, args, n );

  if( item_count != 1 )
    return;  

  /* Delete the selected short name/directory. */
  XmListDeleteItem( tempW, *items );

  /* All is empty. */
  setEmptyItem( acl_ref );

  acl_ref -> acl_changed = True;


  return;

} /* deleteCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget       widget,
             ACL_REC_REF  acl_ref,
             XtPointer    call_data )
{

  /* Code. */

  /* Call callback function? */
  if( acl_ref -> actionCB != NULL )
    (* acl_ref -> actionCB )( XTM_AA_REASON_DESTROY, acl_ref -> user_data );

  /* Release the user data. */
  if( acl_ref -> system_admin_acl != NULL )
    SysFree( acl_ref -> system_admin_acl );

  SysFree( acl_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  closeCB( Widget       widget,
           ACL_REC_REF  acl_ref,
           XtPointer    call_data )
{

  /* Code. */

  XtUnmanageChild( acl_ref -> aclW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  fieldChangedCB( Widget       widget,
                  ACL_REC_REF  acl_ref,
                  XtPointer    call_data )
{

  /* Code. */

  acl_ref -> field_changed = True;


  return;

} /* fieldChangedCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget       widget,
          ACL_REC_REF  acl_ref,
          XtPointer    call_data )
{

  /* Code. */

  xtmHlDisplayHelp( acl_ref -> info_handle, XTM_HL_WINDOW_INDEX,
                    acl_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget       widget,
        ACL_REC_REF  acl_ref,
        XtPointer    call_data )
{

  /* Variables. */
  Widget   tempW;


  /* Code. */

  /* Warn the user that he/she might forget to add sometinhg. */
  if( acl_ref -> field_changed ) {
    tempW = xitCreateQuestionDialog( 
              acl_ref -> aclW, "InformationDialog", 
	      msgGetText( MXDI_INFORMATION_LABEL ),
              msgGetText( MXDI_INFORMATION_NOT_SAVED ),
              saveAndCloseCB, acl_ref,
              NULL, NULL );

    return;
  }

  /* Save and close the window. */
  saveAndCloseCB( widget, acl_ref, call_data );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void 
  saveAndCloseCB( Widget       widget,
                  ACL_REC_REF  acl_ref,
                  XtPointer    call_data )
{

  /* Code. */

  /* Save the access information. */
  saveAccessInfo( acl_ref );


  /* Call callback function? */
  if( acl_ref -> actionCB != NULL )
    (* acl_ref -> actionCB )( XTM_AA_REASON_OK, acl_ref -> user_data );

  XtUnmanageChild( acl_ref -> aclW );


  return;

} /* saveAndCloseCB */
