/*----------------------------------------------------------------------------
--
--  Module:           System
--
--  Project:          Tools - General C objects.
--  System:           Sys - System utilities
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Various low-level system tools and utilities.
--
--  Filename:         System.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-01-30
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: System.c, Version: 1.1, Date: 95/02/18 14:32:31";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "Standard.h"
#include "System.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Print error messages on stderr? */
static Boolean  print_error_msg = True;

/* Debug on? */
static Boolean  use_debug  = False;

/* Memory allocation reference counter. */
static UINT32  memory_ref_count = 0;

static char  xdhome_buffer[ PATH_MAX + 10 ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  *SysBaseCalloc( size_t  elements,
                  size_t  element_size,
                  char    *file_name,
                  int     line_no )
{

  /* Variables. */
  void  *ref;


  /* Code. */

  ref = (void *) calloc( elements, element_size );

  if( ref == NULL && print_error_msg )
    fprintf( stderr, "SysBaseCalloc(), Cannot make a calloc()\n" );

  if( use_debug ) {
    if( line_no > 0 ) 
      printf( "%s: (%d), ", file_name, line_no );

    printf( "SysBaseCalloc(), ref: %d, ref_count: %d\n", 
            (int) ref, memory_ref_count );

    memory_ref_count++;
  }


  return( ref );

} /* SysBaseCalloc */


/*----------------------------------------------------------------------*/

void
  SysBaseFree( void  *pointer,
               char  *file_name,
               int   line_no )
{

  /* Code. */

  if( pointer == NULL )
    return;

  free( (void *) pointer );

  if( use_debug ) {
    if( line_no > 0 ) 
      printf( "%s: (%d), ", file_name, line_no );

    printf( "SysBaseFree(), ref: %d, ref_count: %d\n",
            (int) pointer, memory_ref_count );

    memory_ref_count++;
  }

} /* SysBaseFree */


/*----------------------------------------------------------------------*/

void
  *SysBaseMalloc( size_t  element_size,
                  char    *file_name,
                  int     line_no )
{

  /* Variables. */
  void  *ref;


  /* Code. */

  ref = (void *) malloc( element_size );

  if( ref == NULL && print_error_msg )
    fprintf( stderr, "SysBaseMalloc(), Cannot make a malloc(), size %d\n",
             element_size );

  if( use_debug ) {
    if( line_no > 0 ) 
      printf( "%s: (%d), ", file_name, line_no );

    printf( "SysBaseMalloc(), ref: %d, ref_count: %d\n", 
            (int) ref, memory_ref_count );

    memory_ref_count++;
  }


  return( ref );

} /* SysBaseMalloc */


/*----------------------------------------------------------------------*/

extern void
  *SysBaseRealloc( void    *pointer,
                   size_t  new_size,
                   char    *file_name,
                   int     line_no )
{

  /* Variables. */
  void  *ref;


  /* Code. */

  if( pointer == NULL ) {
    ref = SysBaseMalloc( new_size, file_name, line_no );

    return( ref );
  }

  ref = (void *) realloc( pointer, new_size );

  if( ref == NULL && print_error_msg )
    fprintf( stderr, "SysBaseRealloc(), Cannot make a realloc()\n" );

  if( use_debug ) {
    if( line_no > 0 ) 
      printf( "%s: (%d), ", file_name, line_no );

    printf( "SysBaseRealloc(), ref: %d, ref_count: %d\n", 
            (int) ref, memory_ref_count );

    memory_ref_count++;
  }


  return( ref );

} /* SysBaseRealloc */


/*----------------------------------------------------------------------*/

void
  SysDebugOn( Boolean  state )
{

  /* Code. */

  use_debug        = state;
  memory_ref_count = 0;


  return;

} /* SysDebugOn */


/*----------------------------------------------------------------------*/

char
  *SysExpandFilename( char  *filename )
{

  /* Variables. */
  char  buffer[ PATH_MAX + 1 ];
  char  *filename_copy;
  char  *new_filename;
  char  *part_ref;


  /* Code. */

  /* Make a copy of the filename. */
  filename_copy = SysNewString( filename );

  buffer[ 0 ] = '\0';
  part_ref = strtok( filename_copy, "/" );

  while( part_ref != NULL ) {

    /* Is the part an environment variable? */
    if( *part_ref == '$' ) {
      part_ref = getenv( part_ref + 1 );

      if( part_ref == NULL )
        part_ref = ".";
    }
    
    if( strlen( buffer ) > 0 || *filename == '/' )
      strcat( buffer, "/" );

    strcat( buffer, part_ref );

    part_ref = strtok( NULL, "/" );

  } /* while */

  SysFree( filename_copy );

  new_filename = SysNewString( buffer );


  return( new_filename );

} /* SysExpandFilename */


/*----------------------------------------------------------------------*/

void
  SysInitializeEnvironment()
{

#if defined( XDHOME ) && ! defined( XD_HAS_NO_PUTENV )
  /* If we don't have the XDHOME environment, try to set it. */
  if( getenv( "XDHOME" ) == NULL ) {
    sprintf( xdhome_buffer, "XDHOME=%s", XDHOME );
    putenv( xdhome_buffer );
  }
#endif

  /* We must have the XDHOME environment variable. */
  if( getenv( "XDHOME" ) == NULL ) {
    fprintf( stderr, "XDiary:\n"
             "The environment variable XDHOME is not defined.\n"
             "This environment variable must be defined for XDiary to "
             "function properly." );

    exit( 1 );
  }


  return;
  
}


/*----------------------------------------------------------------------*/

void
  SysPrintErrMsg( Boolean  state )
{

  /* Code. */

  print_error_msg = state;


  return;

} /* SysPrintErrMsg */
