/*----------------------------------------------------------------------------
--
--  Module:           SigHandler
--
--  Project:          Tools - General C objects.
--  System:           Sig - Signal handling riutines.
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Centralized signal management. Signal are cought and the appripriate
--    callbacks are called.
--
--  Filename:         SigHandler.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-17
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: SigHandler.c, Version: 1.1, Date: 95/02/18 14:32:30";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>

#include "System.h"

#include "SigHandler.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define  MAX_SIGNALS_REG  200


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {

  /* The id of the callback. */
  UINT32  id;

  /* The signal to catch. */
  int  signal;

  /* Watch out for signals for this process ID (0 - no PID). */
  int  pid;

  /* Function to call. */
  SIG_SIGNAL_CB  signalCB;

  /* User data. */
  void  *user_data;

} SIGNAL_REC, *SIGNAL_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "SigHandler";

/* Unique ID counter. */
static  int  unique_id = 0;

/* Our signal DB. */
static  SIGNAL_REC  signalDb[ MAX_SIGNALS_REG ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  sigCatchCB( int  this_signal );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  SigInitialize()
{

  /* Variables. */
  int  index;


  /* Code. */

  /* No callbacks registered. */
  for( index = 0; index < MAX_SIGNALS_REG; index++ )
    signalDb[ index ].id = 0;


  return;

} /* SigInitialize */


/*----------------------------------------------------------------------*/

UINT32
  SigRegister( int            for_signal,
               int            pid,
               SIG_SIGNAL_CB  signalCB,
               void           *user_data )
{

  /* Variables. */
  int index;

#if defined(_POSIX_SOURCE) || defined(_INCLUDE_POSIX_SOURCE)
  struct sigaction  sigact;
#endif


  /* Code. */

  /* Any free slots? */
  for( index = 0; index < MAX_SIGNALS_REG; index++ ) {
    if( signalDb[ index ].id == 0 )
      break;
  }

  if( index == MAX_SIGNALS_REG )
    return( 0 );


  /* Save the signal data. */
  unique_id++;

  signalDb[ index ].id        = unique_id;
  signalDb[ index ].signal    = for_signal;
  signalDb[ index ].pid       = pid;
  signalDb[ index ].signalCB  = signalCB;
  signalDb[ index ].user_data = user_data;

  /* Initialize the signal handler. */
#if defined(_POSIX_SOURCE) || defined(_INCLUDE_POSIX_SOURCE)
  sigact.sa_handler = sigCatchCB;
  sigemptyset( &sigact.sa_mask );
  sigact.sa_flags = 0;

  sigaction( for_signal, &sigact, NULL );
#else
  signal( for_signal, sigCatchCB );
#endif


  return( unique_id );

} /* SigRegister */


/*----------------------------------------------------------------------*/

void
  SigRemove( UINT32  id )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Find the registered id. */
  for( index = 0; index < MAX_SIGNALS_REG; index++ ) {

    if( signalDb[ index ].id == id ) {
      signalDb[ index ].id = 0;

      return;
    }

  } /* loop */


  return;

} /* SigRemove */


/*----------------------------------------------------------------------*/

static void
  sigCatchCB( int  this_signal )
{

  /* Variables. */
  int  index;
  int  pid;

#if defined(_POSIX_SOURCE) || defined(_INCLUDE_POSIX_SOURCE)
  int  status;
#else
  union wait  status;
#endif


  /* Code. */

  /* Signal SIGCHLD? */
  if( this_signal == SIGCHLD ) {

#ifdef XD_HAS_NO_WAITPID
    pid = wait3( &status, WNOHANG, (struct rusage *) NULL );

    while( pid > 0 ) {

      /* Check if we have a match signal/pid? */
      for( index = 0; index < MAX_SIGNALS_REG; index++ ) {
        if( signalDb[ index ].id     != 0 &&
            signalDb[ index ].signal == SIGCHLD &&
            signalDb[ index ].pid    == pid ) {

          /* The callback. */
          if( signalDb[ index ].signalCB != NULL )
            (* signalDb[ index ].signalCB)( SIGCHLD, 
                                            signalDb[ index ].user_data );
        }
      }

      pid = wait3( &status, WNOHANG, (struct rusage *) NULL );

    } /* while */
#else
    /* Check if we have a match signal/pid? */
    for( index = 0; index < MAX_SIGNALS_REG; index++ ) {
      if( signalDb[ index ].id != 0 &&
          signalDb[ index ].signal == SIGCHLD ) {

        pid = waitpid( signalDb[ index ].pid, &status, WNOHANG );
        if( pid == signalDb[ index ].pid &&
            signalDb[ index ].signalCB != NULL )
          (* signalDb[ index ].signalCB)( SIGCHLD, 
                                          signalDb[ index ].user_data );
      } /* if */
    }
#endif

  } /* if */


  /* Other signal. */
  if( this_signal != SIGCHLD ) {

    /* Check if we have a match signal/pid? */
    for( index = 0; index < MAX_SIGNALS_REG; index++ ) {
      if( signalDb[ index ].id != 0 &&
          signalDb[ index ].signal == this_signal ) {

        /* The callback. */
        if( signalDb[ index ].signalCB != NULL )
          (* signalDb[ index ].signalCB)( this_signal, 
                                          signalDb[ index ].user_data );
      }
    }

  } /* if */

  /* Reset the trap. */
#if ! defined(_POSIX_SOURCE) && ! defined(_INCLUDE_POSIX_SOURCE)
  signal( this_signal, sigCatchCB );
#endif

  return;

} /* sigCatchCB */


