/*----------------------------------------------------------------------------
--
--  Module:           xtmViewDb
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Routines for the view database.
--
--  Filename:         xtmViewDb.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-07-18
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmViewDb.c, Version: 1.1, Date: 95/02/18 15:52:55";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "xtmGlobal.h"
#include "xtmViewDb.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* How many calendars can we have in a view? */
#define  MAX_CAL_NO     40

/* Prefix to use for shadow entries. */
#define  SHADOW_PREFIX  "@_"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* One calendar in the view. */
typedef struct {

  /* Flags. */
  UINT32  flags;

  /* Name of calendar. */
  char  name[ XTM_GL_MAX_CAL_NAME + 1 ];

} CAL_REC;


/* Record descrining a single view. */
typedef struct {

  /* Flags for the view. */
  UINT32  flags;

  /* Name of the view. */
  char  name[ XTM_GL_MAX_VIEW_NAME + 1 ];

  /* The included calendars. */
  int      no_of_cals;
  CAL_REC  cals[ MAX_CAL_NO ];

  /* Color indexes for entries in this view. */
  int  bg_color;
  int  fg_color;
  int  imp_color;
  int  ro_color;

} VIEW_REC, *VIEW_REC_REF;


/* Calendar database description. */
typedef struct {

  /* ID to use for shadow entries. */
  int  shadow_id;

  /* Views from the customize file. */
  LST_DESC_TYPE  view_list;

  /* Shadow views. */
  LST_DESC_TYPE  shadow_view_list;

} VIEW_DB, *VIEW_DB_REF;


/* Shadow record. */
typedef struct {

  /* Shadow name. */
  char  shadow_name[ 10 ];

  /* View record. */
  VIEW_REC  view_info;

} SHADOW_REC, *SHADOW_REC_REF;



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmViewDb";



/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static int
  findCalIndex( VIEW_REC_REF  view_ref,
                char          *name );

static VIEW_REC_REF
  findViewRef( XTM_VD_HANDLE  view_handle,
               char           *name );

static LST_COMPARE
  searchViewName( VIEW_REC_REF  view_ref,
                  char          *name );

static LST_COMPARE
  searchShadowViewName( SHADOW_REC_REF  shadow_ref,
                        char            *name );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Boolean
  xtmVdAddCalendar( XTM_VD_HANDLE  view_handle,
                    char           *view_name,
                    char           *cal_name )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, view_name );
  if( view_ref == NULL )
    return( False );


  /* Add the calendar. */
  strcpy( view_ref -> cals[ view_ref -> no_of_cals ].name, cal_name );
  view_ref -> cals[ view_ref -> no_of_cals ].flags = 0;

  view_ref -> no_of_cals++;


  return( True );

} /* xtmVdAddCalendar */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdAddView( XTM_VD_HANDLE  view_handle,
                char           *name )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;
  VIEW_DB_REF   view_db_ref;
  LST_STATUS    lst_status;


  /* Code. */

  if( view_handle == NULL )
    return( False );

  view_db_ref = (XTM_VD_HANDLE) view_handle;


  /* Allocate a new view and fill in with default values. */
  view_ref = SysNew( VIEW_REC );
  view_ref -> flags      = 0;
  view_ref -> no_of_cals = 0;
  view_ref -> bg_color   = 0;
  view_ref -> fg_color   = 0;
  view_ref -> ro_color   = 0;
  view_ref -> imp_color  = 0;

  strcpy( view_ref -> name, name );


  /* Add the entry to the end of the list. */
  lst_status = LstLinkInsertLast( view_db_ref -> view_list, view_ref );
  if( lst_status != LST_OK )
    return( False );


  return( True );

} /* xtmVdAddView */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdAddCalFlags( XTM_VD_HANDLE  view_handle,
                    char           *view_name,
                    char           *cal_name,
                    UINT32         flags )
{

  /* Variables. */
  int           cal_index;
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, view_name );
  if( view_ref == NULL )
    return( False );

  cal_index = findCalIndex( view_ref, cal_name );
  if( cal_index < 0 )
    return( False );

  view_ref -> cals[ cal_index ].flags = flags;


  return( True );

} /* xtmVdAddCalFlags */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdAddViewBgColor( XTM_VD_HANDLE  view_handle,
                       char           *name,
                       int            color )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  view_ref -> bg_color = color;


  return( True );

} /* xtmVdFetchViewBgColor */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdAddViewFgColor( XTM_VD_HANDLE  view_handle,
                       char           *name,
                       int            color )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  view_ref -> fg_color = color;


  return( True );

} /* xtmVdAddViewFgColor */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdAddViewImpColor( XTM_VD_HANDLE  view_handle,
                        char           *name,
                        int            color )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  view_ref -> imp_color = color;


  return( True );

} /* xtmVdAddViewImpColor */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdAddViewRoColor( XTM_VD_HANDLE  view_handle,
                       char           *name,
                       int            color )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  view_ref -> ro_color = color;


  return( True );

} /* xtmVdAddViewRoColor */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdAddViewFlags( XTM_VD_HANDLE  view_handle,
                     char           *name,
                     UINT32         flags )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref != NULL )
    return( False );

  view_ref -> flags = flags;


  return( True );

} /* xtmVdAddViewFlags */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdCreateShadowView( XTM_VD_HANDLE  view_handle,
                         char           *name,
                         char           *shadow_name )
{

  /* Variables. */
  LST_STATUS    lst_status;
  SHADOW_REC    shadow_rec;
  VIEW_DB_REF   view_db_ref;
  VIEW_REC_REF  view_ref;


  /* Code. */

  *shadow_name = '\0';


  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  view_db_ref = (XTM_VD_HANDLE) view_handle;


  /* A new unique name for the shadow entry. */
  sprintf( shadow_rec.shadow_name, "%s%d",
           SHADOW_PREFIX, view_db_ref -> shadow_id );

  view_db_ref -> shadow_id++;
  if( view_db_ref -> shadow_id > 9999 )
    view_db_ref -> shadow_id = 0;


  memcpy( (void *) &shadow_rec.view_info,
          (void *) view_ref,
          sizeof( VIEW_REC ) );


  /* Add the entry to the end of the list. */
  lst_status = LstLinkInsertLast( view_db_ref -> shadow_view_list,
                                  &shadow_rec );
  if( lst_status != LST_OK )
    return( False );


  strcpy( shadow_name, shadow_rec.shadow_name );


  return( True );

} /* xtmVdCreateShadowView */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdDeleteCalendar( XTM_VD_HANDLE  view_handle,
                       char           *view_name,
                       char           *cal_name )
{

  /* Variables. */
  int           cal_index;
  int           index;
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view and the calendar. */
  view_ref = findViewRef( view_handle, view_name );
  if( view_ref == NULL )
    return( False );

  cal_index = findCalIndex( view_ref, cal_name );
  if( cal_index < 0 )
    return( False );


  /* Remove the calendar (compress entries). */
  for( index = cal_index + 1; index < view_ref -> no_of_cals; index++ ) {

    memcpy( (void *) &view_ref -> cals[ index - 1 ],
            (void *) &view_ref -> cals[ index ],
            sizeof( CAL_REC ) );

  } /* loop */

  view_ref -> no_of_cals--;


  return( True );

} /* xtmVdDeleteCalendar */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdDeleteView( XTM_VD_HANDLE  view_handle,
                   char           *name )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;
  VIEW_DB_REF   view_db_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  view_db_ref = (XTM_VD_HANDLE) view_handle;


  /* Remove the view (can be normal or shadow). */
  (void) LstLinkDeleteCurrent( view_db_ref -> view_list );


  return( True );

} /* xtmVdDeleteView */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFetchCalFlags( XTM_VD_HANDLE  view_handle,
                      char           *view_name,
                      char           *cal_name,
                      UINT32         *flags )
{

  /* Variables. */
  int           cal_index;
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, view_name );
  if( view_ref == NULL )
    return( False );

  cal_index = findCalIndex( view_ref, cal_name );
  if( cal_index < 0 )
    return( False );

  *flags = view_ref -> cals[ cal_index ].flags;


  return( True );

} /* xtmVdFetchCalFlags */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFetchCalNames( XTM_VD_HANDLE  view_handle,
                      char           *view_name,
                      char           **cal_names )
{

  /* Variables. */
  int           index;
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, view_name );
  if( view_ref == NULL )
    return( False );


  /* We know the length of string to return. */
  *cal_names  = (char *) SysMalloc( view_ref -> no_of_cals *
                                    (XTM_GL_MAX_CAL_NAME + 3) );
  **cal_names = '\0';


  /* Process all calendars. */
  for( index = 0; index < view_ref -> no_of_cals; index++ ) {
    strcat( *cal_names, view_ref -> cals[ index ].name );
    strcat( *cal_names, " " );
  }


  return( True );

} /* xtmVdFetchCalNames */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFetchViewBgColor( XTM_VD_HANDLE  view_handle,
                         char           *name,
                         int            *color )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  *color = view_ref -> bg_color;


  return( True );

} /* xtmVdFetchViewBgColor */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFetchViewFgColor( XTM_VD_HANDLE  view_handle,
                         char           *name,
                         int            *color )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  *color = view_ref -> fg_color;


  return( True );

} /* xtmVdFetchViewFgColor */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFetchViewImpColor( XTM_VD_HANDLE  view_handle,
                          char           *name,
                          int            *color )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  *color = view_ref -> imp_color;


  return( True );

} /* xtmVdFetchViewImpColor */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFetchViewRoColor( XTM_VD_HANDLE  view_handle,
                         char           *name,
                         int            *color )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref == NULL )
    return( False );

  *color = view_ref -> ro_color;


  return( True );

} /* xtmVdFetchViewRoColor */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFetchViewFlags( XTM_VD_HANDLE  view_handle,
                       char           *name,
                       UINT32         *flags )
{

  /* Variables. */
  VIEW_REC_REF  view_ref;


  /* Code. */

  /* Search the view. */
  view_ref = findViewRef( view_handle, name );
  if( view_ref != NULL )
    return( False );

  *flags = view_ref -> flags;


  return( True );

} /* xtmVdFetchViewFlags */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFetchViewNames( XTM_VD_HANDLE  view_handle,
                       char           **view_names )
{

  /* Variables. */
  int           view_items;
  LST_STATUS    lst_status;
  VIEW_DB_REF   view_db_ref;
  VIEW_REC_REF  view_ref;


  /* Code. */

  if( view_handle == NULL )
    return( False );

  view_db_ref = (XTM_VD_HANDLE) view_handle;


  view_items = LstLinkElements( view_db_ref -> view_list );

  if( view_items <= 0 )
    return( False );


  /* We know the length of string to return. */
  *view_names  = (char *) SysMalloc( view_items * (XTM_GL_MAX_VIEW_NAME + 3) );
  **view_names = '\0';


  /* Process all views. */
  lst_status = LstLinkCurrentFirst( view_db_ref -> view_list );

  while( lst_status == LST_OK ) {

    view_ref = (VIEW_REC_REF) LstLinkGetCurrentRef( view_db_ref -> view_list );

    strcat( *view_names, view_ref -> name );
    strcat( *view_names, " " );

    lst_status = LstLinkCurrentNext( view_db_ref -> view_list );

  } /* while */


  return( True );

} /* xtmVdFetchViewNames */


/*----------------------------------------------------------------------*/

void
  xtmVdFree( XTM_VD_HANDLE  view_handle )
{

  /* Variables. */
  VIEW_DB_REF  view_db_ref;


  /* Code. */

  if( view_handle == NULL )
    return;

  view_db_ref = (XTM_VD_HANDLE) view_handle;


  /* Free the shadow list. */
  if( view_db_ref -> shadow_view_list != NULL )
    LstLinkClear( view_db_ref -> shadow_view_list );

  /* Free the view list. */
  if( view_db_ref -> view_list != NULL )
    LstLinkClear( view_db_ref -> view_list );


  /* Free the record itself. */
  SysFree( view_db_ref );


  return;

} /* xtmVdFree */


/*----------------------------------------------------------------------*/

Boolean
  xtmVdFreeShadowView( XTM_VD_HANDLE  view_handle,
                       char           *name )
{

  /* Variables. */
  Boolean  status;


  /* Code. */

  /* The standard delete can do this. */
  status = xtmVdDeleteView( view_handle, name );


  return( status );

} /* xtmVdFreeShadowView */


/*----------------------------------------------------------------------*/

XTM_VD_HANDLE
  xtmVdInitialize()
{

  /* Variables. */
  VIEW_DB_REF  view_db_ref;


  /* Code. */

  /* Allocate space for a new calendar database. */
  view_db_ref = SysNew( VIEW_DB );

  view_db_ref -> view_list = 
    LstLinkNew( sizeof( VIEW_REC ), NULL );
  view_db_ref -> shadow_view_list = 
    LstLinkNew( sizeof( SHADOW_REC ), NULL );

  view_db_ref -> shadow_id = 0;
  

  return( (XTM_VD_HANDLE) view_db_ref );

} /* xtmVdInitialize */


/*----------------------------------------------------------------------*/

static int
  findCalIndex( VIEW_REC_REF  view_ref,
                char          *name )
{

  /* Variables. */
  int  index;


  /* Code. */

  if( view_ref -> no_of_cals < 1 )
    return( -1 );

  /* Find the calendar. */
  for( index = 0; index < view_ref -> no_of_cals; index++ ) {
    if( strcpy( view_ref -> cals[ index ].name, name ) == 0 )
      return( index );
  }


  return( -1 );

} /* findCalIndex */


/*----------------------------------------------------------------------*/

static VIEW_REC_REF
  findViewRef( XTM_VD_HANDLE  view_handle,
               char           *name )
{

  /* Variables. */
  LST_STATUS      lst_status;
  SHADOW_REC_REF  shadow_view_ref;
  VIEW_DB_REF     view_db_ref;
  VIEW_REC_REF    view_ref = NULL;


  /* Code. */

  if( view_handle == NULL )
    return( NULL );

  view_db_ref = (XTM_VD_HANDLE) view_handle;


  /* Search the view. */
  lst_status = LstLinkSearchFirst( view_db_ref -> view_list,
                                   (void *) name,
                                   (EQUALS_FUNC_TYPE) searchViewName );

  if( lst_status == LST_OK ) {
    view_ref = (VIEW_REC_REF) LstLinkGetCurrentRef( view_db_ref -> view_list );

  /* Search the entry in the shadow list. */
  } else {
    lst_status = LstLinkSearchFirst( view_db_ref -> shadow_view_list,
                                     (void *) name,
                                     (EQUALS_FUNC_TYPE) searchShadowViewName );
    if( lst_status == LST_OK ) {
      shadow_view_ref = (SHADOW_REC_REF) LstLinkGetCurrentRef( 
                                           view_db_ref -> shadow_view_list );

      view_ref = &shadow_view_ref -> view_info;
    }

  } /* if */


  return( view_ref );

} /* findViewRef */


/*----------------------------------------------------------------------*/

static LST_COMPARE
  searchViewName( VIEW_REC_REF  view_ref,
                  char          *name )
{

  /* Code. */

  if( strcmp( view_ref -> name, name ) == 0 )
    return( LST_EQUAL );


  return( LST_NOT_EQUAL );

} /* searchViewName */


/*----------------------------------------------------------------------*/

static LST_COMPARE
  searchShadowViewName( SHADOW_REC_REF  shadow_ref,
                        char            *name )
{

  /* Code. */

  if( strcmp( shadow_ref -> shadow_name, name ) == 0 )
    return( LST_EQUAL );


  return( LST_NOT_EQUAL );

} /* searchShadowViewName */
