/*----------------------------------------------------------------------------
--
--  Module:           xtmReminder
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    A reminder tool with 'light-weight' alarms, i.e. alarms to remind
--    you of events during the day.
--
--  Filename:         xtmReminder.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-02-16
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmReminder.c, Version: 1.1, Date: 95/02/18 15:52:40";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/Scale.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xitError.h"
#include "xitTools.h"
#include "XmUbTimeSl.h"
#include "xtmReminder.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the reminder window. */
#define alarmActiveTb      dataLocalW[  0 ]
#define alarmTs            dataLocalW[  1 ]
#define alarmTx            dataLocalW[  2 ]
#define alarmTxLa          dataLocalW[  3 ]
#define hoSp               dataLocalW[  4 ]
#define menuBr             dataLocalW[  5 ]
#define remindFr           dataLocalW[  6 ]
#define remindLa           dataLocalW[  7 ]
#define remindRc           dataLocalW[  8 ]
#define remindSelLa        dataLocalW[  9 ]
#define todayLa            dataLocalW[ 10 ]


/* Local widgets in the alarm window. */
#define markerPx           dataLocalW[  0 ]
#define messageLa          dataLocalW[  1 ]



/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmReminder";

/* Keyboard translations for newline in multi-line editor. */
static XtTranslations  newline_trans = NULL;

static char newlineTrans[] = 
  "<Key>Return:    newline()";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  alarmTimeChangedCB( Widget                        widget,
                      XTM_RD_REMINDER_REF           reminder_ref,
                      XmUbTimeSliderCallbackStruct  *call_ref );

static void
  alarmActiveCB( Widget                        widget,
                 XTM_RD_REMINDER_REF           reminder_ref,
                 XmToggleButtonCallbackStruct  *call_data );

static void
  alarmTimer( XTM_RD_REMINDER_REF  reminder_ref );

static void
  closeAlarmCB( Widget     widget,
                Widget     alarmW,
                XtPointer  call_data );

static void
  closeCB( Widget               widget,
           XTM_RD_REMINDER_REF  reminder_ref,
           XtPointer            call_data );

static Widget
  createReminderWindow( XTM_RD_REMINDER_REF  reminder_ref,
                        Widget               parent );

static void
  giveAlarm( XTM_RD_REMINDER_REF  reminder_ref,
             Widget               parent,
             char                 *message );

static void
  helpCB( Widget                     widget,
          XTM_RD_REMINDER_REF        reminder_ref,
          XmRowColumnCallbackStruct  *call_data );

static void
  reminderSelectedCB( Widget                        widget,
                      XTM_RD_REMINDER_REF           reminder_ref,
                      XmToggleButtonCallbackStruct  *call_data );

static void
  resetCB( Widget               widget,
           XTM_RD_REMINDER_REF  reminder_ref,
           XtPointer            call_data );

static void
  setAlarmIndicators( XTM_RD_REMINDER_REF  reminder_ref,
                      int                  reminder_no );

static void
  setWorkReminder( XTM_RD_REMINDER_REF  reminder_ref,
                   int                  reminder_no );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_RD_REMINDER_REF
  xtmRdInitializeReminder( Widget           parent,
                           XtAppContext     context,
                           XTM_RD_CLOSE_CB  closeCB,
                           XTM_RD_HELP_CB   helpCB,
                           void             *user_data )
{

  /* Variables. */
  int                  index;
  XTM_RD_REMINDER_REF  reminder_ref;


  /* Code. */

  /* Initialize translations? */
  if( newline_trans == NULL )
    newline_trans = XtParseTranslationTable( newlineTrans );


  reminder_ref = SysNew( XTM_RD_REMINDER );

  /* Default values. */
  reminder_ref -> sliderCB_active   = True;
  reminder_ref -> context           = context;
  reminder_ref -> reminderW         = NULL;
  reminder_ref -> curr_work_alarm   = 0;
  reminder_ref -> closeCB           = closeCB;
  reminder_ref -> helpCB            = helpCB;
  reminder_ref -> user_data         = user_data;

  for( index = 0; index < XTM_RD_MAX_ALARMS; index++ ) {
    reminder_ref -> alarm[ index ].active     = False;
    reminder_ref -> alarm[ index ].alarm_time = 0;
    reminder_ref -> alarm[ index ].text[ 0 ]  = '\0';
  }


  /* Create the reminder window. */
  reminder_ref -> reminderW = createReminderWindow( reminder_ref, 
                                                    parent );


  /* Start the alarm timer. */
  alarmTimer( reminder_ref );


  return( reminder_ref );

} /* xtmRdInitializeReminder */


/*----------------------------------------------------------------------*/

void
  xtmRdDestroyReminder( XTM_RD_REMINDER_REF  reminder_ref )
{

  /* Code. */


  return;

} /* xtmRdDestroyReminder */


/*----------------------------------------------------------------------*/

void 
  xtmRdDisplayReminder( XTM_RD_REMINDER_REF  reminder_ref )
{

  /* Variables. */
  char    buffer[ 50 ];
  Widget  tempW;


  /* Code. */

  /* Is the reminder initialized? */
  if( reminder_ref == NULL )
    return;


  /* Display the reminder being edited. */
  sprintf( buffer, "RemindFr.RemindRc.Select%dTb", 
           reminder_ref -> curr_work_alarm + 1 );

  tempW = XtNameToWidget( reminder_ref -> reminderW, 
                          "RemindTlBase.RemindTlFo" );
  tempW = XtNameToWidget( tempW, buffer );

  XmToggleButtonSetState( tempW, True, True );


  /* Start at the text field. */
  tempW = XtNameToWidget( reminder_ref -> reminderW,
                          "RemindTlBase.RemindTlFo.AlarmTxSW.AlarmTx" );
  xitGrabFocus( tempW );


  /* Make the window visible (if not already). */
  XtPopup( reminder_ref -> reminderW, XtGrabNone );


  /* Make sure the window is visible. */
  XRaiseWindow( XtDisplay( reminder_ref -> reminderW ), 
                XtWindow(  reminder_ref -> reminderW ) );

  XtMapWidget( reminder_ref -> reminderW );


  return;

} /* xtmRdDisplayReminder */


/*----------------------------------------------------------------------*/

void
  xtmRdHideReminder( XTM_RD_REMINDER_REF  reminder_ref )
{

  /* Code. */

  if( reminder_ref == NULL || reminder_ref -> reminderW == NULL )
    return;

  XtPopdown( reminder_ref -> reminderW );


  return;

} /* xtmRdHideReminder */


/*----------------------------------------------------------------------*/

Widget
  xtmRdToplevelWindow( XTM_RD_REMINDER_REF  reminder_ref )
{

  /* Code. */

  if( reminder_ref == NULL || reminder_ref -> reminderW == NULL )
    return( NULL );


  return( reminder_ref -> reminderW );

} /* xtmRdToplevelWindow */


/*----------------------------------------------------------------------*/

static void
  alarmTimer( XTM_RD_REMINDER_REF  reminder_ref )
{

  /* Variables. */
  int           index;
  int           next_minute;
  char          buffer[ 50 ];
  char          *char_ref;
  Widget        mainFo;
  Widget        tempW;
  TIM_TIME_REF  alarm_time;
  TIM_TIME_REF  now;


  /* Code. */

  mainFo = XtNameToWidget( reminder_ref -> reminderW, 
                           "RemindTlBase.RemindTlFo" );


  /* The time is.. */
  now = TimLocalTime( TimMakeTimeNow() );


  /* Update current date and time. */
  xtmFoFormatFullTime( now, buffer, sizeof( buffer ) );

  tempW = XtNameToWidget( mainFo, "TodayLa" );
  xitStringSetLabel( tempW, buffer );


  /* If we have any alarms, show them. */
  alarm_time = TimMakeTime( 1970, 1, 1,
                            TimHour(   now ),
                            TimMinute( now ), 0 );

  for( index = 0; index < XTM_RD_MAX_ALARMS; index++ ) {

    if( reminder_ref -> alarm[ index ].active &&
        alarm_time == reminder_ref -> alarm[ index ].alarm_time ) {

      /* If reminder is current, take text from window. */
      if( index == reminder_ref -> curr_work_alarm ) {
        tempW = XtNameToWidget( mainFo, "AlarmTxSW.AlarmTx" );

        char_ref = xitStringGetText( tempW );

        strcpy( reminder_ref -> alarm[ index ].text, char_ref );

        SysFree( char_ref );
      } /* if */

      giveAlarm( reminder_ref,
                 reminder_ref -> reminderW,
                 reminder_ref -> alarm[ index ].text );

      /* The alarm is not active anymore. */
      reminder_ref -> alarm[ index ].active = False;

      setAlarmIndicators( reminder_ref, index );

      if( index == reminder_ref -> curr_work_alarm )
        setWorkReminder( reminder_ref, index );

    } /* if */

  } /* loop */


  /* Schedule the next wakeup call. */
  next_minute = (60 - now % 60) * 1000;

  XtAppAddTimeOut( reminder_ref -> context, 
                   next_minute,
                   (XtTimerCallbackProc) alarmTimer,
                   (XtPointer) reminder_ref );


  return;

} /* alarmTimer */


/*----------------------------------------------------------------------*/

static Widget
  createReminderWindow( XTM_RD_REMINDER_REF  reminder_ref,
                        Widget               parent )
{

  /* Variables. */
  int       index;
  char      buffer[ 100 ];
  char      name[ 50 ];
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 11 ];
  Widget    menuCasc[ 2 ];
  Widget    menuFileBu[ 1 ];
  Widget    menuHelpBu[ 7 ];
  Widget    pulldownMenu[ 2 ];
  Widget    remindTl;
  Widget    selectLa[ XTM_RD_MAX_ALARMS  ];
  Widget    selectTb[ XTM_RD_MAX_ALARMS  ];
  Widget    tempW;
  Widget    workFo;

  static char  *pull_downs[] = { "pdown1", "pdown2" };

  static XIT_TEXT_STRUCT text_field_def[] = {
    { "AlarmTx", NULL, 5, True },
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "HelpPane" }
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "", "", NULL, "CloseBu", True, False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
    { "", "", NULL, "AboutBu",   True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   closeCB, NULL },
    { NULL, NULL,    NULL },
    { NULL, NULL,    NULL },
    { NULL, NULL,    NULL },
    { "",   resetCB, NULL },
  };


  /* Code. */

  /* Get text for menues and buttons. */
  action_buttons[ 0 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 0 ].data  = reminder_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_RESET_BUTTON );
  action_buttons[ 4 ].data  = reminder_ref;

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );
  help_casc[ 6 ].title    = msgGetText( MXDI_HELP_ABOUT );
  help_casc[ 6 ].mnemonic = msgGetText( MXDI_HELP_ABOUT_ACC );


  /* Create a separate window. */
  tempW = xitGetToplevelWidget( parent );

  remindTl = xitCreateToplevelDialog( tempW, "RemindTl", 
                                      1, 0,
                                      action_buttons, 
                                      XtNumber( action_buttons ) );

  sprintf( buffer, "%s", msgGetText( MXDI_REMINDER_TITLE ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetValues( remindTl, args, n );

  sprintf( buffer, "%s", msgGetText( MXDI_REMINDER_IC_TITLE ) );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, buffer ); n++;
  XtSetValues( remindTl, args, n );

  /* Close the window if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( remindTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( remindTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( remindTl, wm_delete_window, 
                             (XtCallbackProc) closeCB,
                             (XtPointer) reminder_ref );
  } /* block */


  /* Form to enclose the show window. */
  workFo = XtNameToWidget( remindTl, "RemindTlBase.RemindTlFo" );


  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ], 
                                                   &file_casc[ index ] );

  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) closeCB, (XtPointer) reminder_ref );

  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) helpCB, (XtPointer) reminder_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) helpCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Current date and time label. */
  todayLa = xitCreateLabel( workFo, "TodayLa", " \n ", XmALIGNMENT_CENTER );


  /* Reminder label. */
  remindLa = xitCreateLabel( workFo, "RemindLa",
                             msgGetText( MXDI_REMINDERS_LABEL ), -1 );


  /* The different reminders we have. */
  n = 0;
  remindFr = XmCreateFrame( workFo, "RemindFr", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns, 2 ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  XtSetArg( args[ n ], XmNisHomogeneous, False ); n++;
  remindRc = XmCreateRowColumn( remindFr, "RemindRc", args, n );


  /* Reminders selectors and alarm times. */
  for( index = 0; index < XTM_RD_MAX_ALARMS; index++ ) {
    sprintf( name,   "Select%dTb", index + 1 );
    sprintf( buffer, "%d",         index + 1 );

    selectTb[ index ] = xitCreateToggleButton( remindRc, name,
                                               buffer, False );

    XtAddCallback( selectTb[ index ],  XmNvalueChangedCallback,
                   (XtCallbackProc) reminderSelectedCB,
                   (XtPointer) reminder_ref );
  }

  n = 0;
  XtSetArg( args[ n ], XmNisHomogeneous, False ); n++;
  XtSetValues( remindRc, args, n );

  for( index = 0; index < XTM_RD_MAX_ALARMS; index++ ) {
    sprintf( name,   "Select%dLa", index + 1 );
    sprintf( buffer, "%-8.8s", " " );

    selectLa[ index ] = xitCreateLabel( remindRc, name, buffer, -1 );
  }


  /* Separator. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
  hoSp = XmCreateSeparatorGadget( workFo, "HoSp", args, n );


  /* The 'reminder definition' part. */

  /* Which reminder are we using? */
  remindSelLa = xitCreateLabel( workFo, "RemindSelLa", " ",
                                XmALIGNMENT_BEGINNING );


  /* Scale to select alarm time. */
  n = 0;
  alarmTs = XmUbCreateTimeSlider( workFo, "AlarmTs", args, n );
  XtAddCallback( alarmTs, XmNvalueChangedCallback, 
                 (XtCallbackProc) alarmTimeChangedCB,
                 (XtPointer) reminder_ref );



  /* Button to select if reminder is active or not. */
  alarmActiveTb = xitCreateToggleButton( 
                    workFo, "AlarmActiveTb", 
                    msgGetText( MXDI_REMINDERS_ACTIVE_LABEL ), False );

  XtAddCallback( alarmActiveTb, XmNvalueChangedCallback,
                 (XtCallbackProc) alarmActiveCB, (XtPointer) reminder_ref );


  /* Alarm text label. */
  alarmTxLa = xitCreateLabel( workFo, "AlarmTxLa", 
                              msgGetText( MXDI_TEXT_LABEL ), -1 );

  /* Alarm text to display. */
  alarmTx = xitCreateTextScrolled( workFo, &text_field_def[ 0 ] );

  XtOverrideTranslations( alarmTx, newline_trans );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, XTM_RD_MAX_ALARM_TEXT ); n++;
  XtSetValues( alarmTx, args, n );


  /* Put the parts together. */
  xitAttachWidget( menuBr, 
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( todayLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( remindLa, 
                   XmATTACH_WIDGET, todayLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( remindFr, 
                   XmATTACH_WIDGET, remindLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( hoSp,
                   XmATTACH_WIDGET, remindFr, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( remindSelLa,
                   XmATTACH_WIDGET, hoSp, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( alarmTs,
                   XmATTACH_WIDGET, remindSelLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,        XmATTACH_NONE, NULL );
  xitAttachWidget( alarmActiveTb,
                   XmATTACH_WIDGET, alarmTs, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( alarmTxLa,
                   XmATTACH_WIDGET, alarmActiveTb, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,          XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( alarmTx ),
                   XmATTACH_WIDGET, alarmTxLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( todayLa,             args, n );
  XtSetValues( remindLa,            args, n );
  XtSetValues( remindFr,            args, n );
  XtSetValues( remindSelLa,         args, n );
  XtSetValues( alarmTs,             args, n );
  XtSetValues( alarmActiveTb,       args, n );
  XtSetValues( alarmTxLa,           args, n );
  XtSetValues( XtParent( alarmTx ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset, 5 ); n++;
  XtSetValues( hoSp, args, n );

  /* Manage all the children. */
  XtManageChildren( menuCasc,     XtNumber( menuCasc ) );
  XtManageChildren( menuFileBu,   XtNumber( menuFileBu ) );
  XtManageChildren( menuHelpBu,   XtNumber( menuHelpBu ) );
  XtManageChildren( pulldownMenu, XtNumber( pulldownMenu ) );
  XtManageChildren( selectTb,     XtNumber( selectTb ) );
  XtManageChildren( selectLa,     XtNumber( selectLa ) );

  xitManageChildren( dataLocalW,  XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( remindTl, workFo, XTM_IC_ICON_REMINDER );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( remindTl, True );


  /* Make the final attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( XtParent( alarmTx ), args, n );


  return( remindTl );

} /* createReminderWindow */


/*----------------------------------------------------------------------*/

static void
  giveAlarm( XTM_RD_REMINDER_REF  reminder_ref,
             Widget               parent,
             char                 *message )
{

  /* Variables. */
  char      buffer[ 100 ];
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 2 ];
  Widget    remindAlarmTl;
  Widget    workFo;
  Widget    tempW;

  static Pixmap  normal_pixmap = None;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { NULL, NULL, NULL },
    { "",   NULL, NULL },
    { NULL, NULL, NULL },
  };


  /* Code. */

  action_buttons[ 1 ].label = msgGetText( MXDI_ALARM_CONFIRM );

  if( message == NULL || *message == '\0' )
    message = "---";

  /* Create a separate window. */
  tempW = xitGetToplevelWidget( parent );

  remindAlarmTl = xitCreateToplevelDialog( tempW, "RemindAlarmTl",
                                           2, 0,
                                           action_buttons,
                                           XtNumber( action_buttons ) );

  sprintf( buffer, "%s", msgGetText( MXDI_REMINDER_ALARM_TITLE ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle,    buffer ); n++;
  XtSetArg( args[ n ], XmNiconName, buffer ); n++;
  XtSetValues( remindAlarmTl, args, n );


  /* Catch when the user seslects Close from the system menu. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( remindAlarmTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( remindAlarmTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( remindAlarmTl, wm_delete_window, 
                             (XtCallbackProc) closeAlarmCB,
                             (XtPointer) remindAlarmTl );
  } /* block */



  /* Form to hold the alarm message. */
  workFo = XtNameToWidget( remindAlarmTl, 
                           "RemindAlarmTlBase.RemindAlarmTlFo" );


  /* Pixelmap label. */
  markerPx = xitCreateLabel( workFo, "MarkerPx", "", -1 );


  /* Fetch the alarm pixmap. */
  if( normal_pixmap == None )
    normal_pixmap = xtmIcFetchSimplePixmap( workFo, 
                                            XTM_IC_ICON_ALARM_CLOCK5, False );

  n = 0;
  XtSetArg( args[ n ], XmNlabelPixmap, normal_pixmap ); n++;
  XtSetArg( args[ n ], XmNlabelType,   XmPIXMAP      ); n++;
  XtSetValues( markerPx, args, n );


  /* Message label. */
  messageLa = xitCreateLabel( workFo, "MessageLa", 
                              message, XmALIGNMENT_BEGINNING );


  /* Put the Parts together. */
  xitAttachWidget( markerPx,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( messageLa,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, markerPx,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( markerPx,  args, n );
  XtSetValues( messageLa, args, n );


  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( remindAlarmTl, workFo, XTM_IC_ICON_DEFAULT );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( remindAlarmTl, True );


  XtPopup( remindAlarmTl, XtGrabNone );


  /* Give an alarm, just a boring bell. */
  XBell( XtDisplay( remindAlarmTl ), 100 );


  return;

} /* giveAlarm */


/*----------------------------------------------------------------------*/

static void
  setAlarmIndicators( XTM_RD_REMINDER_REF  reminder_ref,
                      int                  reminder_no )
{

  /* Variables. */
  int     index;
  char    buffer[ 50 ];
  Widget  mainFo;
  Widget  tempW;


  /* Code. */

  mainFo = XtNameToWidget( reminder_ref -> reminderW, 
                           "RemindTlBase.RemindTlFo" );

  /* Display alarm times. */
  for( index = 0; index < XTM_RD_MAX_ALARMS; index++ ) {

    sprintf( buffer, "RemindFr.RemindRc.Select%dLa", index + 1 );
    tempW = XtNameToWidget( mainFo, buffer );

    if( reminder_ref -> alarm[ index ].active )
      xtmFoFormatTime( reminder_ref -> alarm[ index ].alarm_time, 
                       buffer, sizeof( buffer ) );
    else
      sprintf( buffer, "%-8.8s", " " );

    if( reminder_no < 0 || reminder_no == index )
      xitStringSetLabel( tempW, buffer );

  } /* loop */


  return;

} /* setAlarmIndicators */


/*----------------------------------------------------------------------*/

static void
  setWorkReminder( XTM_RD_REMINDER_REF  reminder_ref,
                   int                  reminder_no )
{

  /* Variables. */
  Boolean       sliderCB_active;
  char          buffer[ 50 ];
  Widget        mainFo;
  Widget        tempW;
  TIM_TIME_REF  alarm_time;


  /* Code. */

  mainFo = XtNameToWidget( reminder_ref -> reminderW, 
                           "RemindTlBase.RemindTlFo" );


  /* Set the label telling wich reminder we are editing. */
  sprintf( buffer, msgGetText( MXDI_REMINDER_WORK_LABEL ), 
           reminder_no + 1 );

  tempW = XtNameToWidget( mainFo, "RemindSelLa" );
  xitStringSetLabel( tempW, buffer );


  /* Set the alarm time. */
  if( ! reminder_ref -> alarm[ reminder_no ].active &&
        reminder_ref -> alarm[ reminder_no ].alarm_time == 0 )
    reminder_ref -> alarm[ reminder_no ].alarm_time =
      TimLocalTime( TimMakeTimeNow() );

  alarm_time = reminder_ref -> alarm[ reminder_no ].alarm_time;


  /* Time scale. */
  tempW = XtNameToWidget( mainFo, "AlarmTs" );

  sliderCB_active = reminder_ref -> sliderCB_active;
  reminder_ref -> sliderCB_active = False;

  XmUbTimeSliderSetTime( tempW, alarm_time );

  reminder_ref -> sliderCB_active = sliderCB_active;


  /* Reminder is active? */
  tempW = XtNameToWidget( mainFo, "AlarmActiveTb" );
  XmToggleButtonSetState( tempW, 
                          reminder_ref -> alarm[ reminder_no ].active, 
                          False );


  /* Alarm text. */
  tempW = XtNameToWidget( mainFo, "AlarmTxSW.AlarmTx" );

  if( reminder_ref -> alarm[ reminder_no ].active ||
      reminder_ref -> alarm[ reminder_no ].text[ 0 ] != '\0' )
    XmTextSetString( tempW, reminder_ref -> alarm[ reminder_no ].text );


  /* Set the alarm indicators. */
  setAlarmIndicators( reminder_ref, reminder_no );


  return;

} /* setWorkReminder */


/*----------------------------------------------------------------------*/

static void
  alarmActiveCB( Widget                        widget,
                 XTM_RD_REMINDER_REF           reminder_ref,
                 XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */


  /* Code. */

  /* Alarm indicators. */
  if( call_data -> set )
    reminder_ref -> alarm[ reminder_ref -> curr_work_alarm ].active = True;
  else
    reminder_ref -> alarm[ reminder_ref -> curr_work_alarm ].active = False;


  /* Set the alarm indicators. */
  setAlarmIndicators( reminder_ref, reminder_ref -> curr_work_alarm );


  return;

} /* alarmActiveCB */


/*----------------------------------------------------------------------*/

static void
  closeCB( Widget               widget,
           XTM_RD_REMINDER_REF  reminder_ref,
           XtPointer            call_data )
{

  /* Code. */

  if( reminder_ref -> closeCB == NULL ) {
    XtPopdown( reminder_ref -> reminderW );

    return;
  }


  /* Call the user callback. */
  (* reminder_ref -> closeCB)( reminder_ref -> reminderW,
                               reminder_ref -> user_data,
                               reminder_ref );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void
  closeAlarmCB( Widget     widget,
                Widget     alarmW,
                XtPointer  call_data )
{

  /* Code. */

  XtDestroyWidget( alarmW );


  return;

} /* closeAlarmCB */


/*----------------------------------------------------------------------*/

static void
  helpCB( Widget                     widget,
          XTM_RD_REMINDER_REF        reminder_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  /* About window? */
  if( (int) call_data -> data == 6 ) {
    xtmHlDisplayAboutWindow( reminder_ref -> reminderW );

    return;
  }

  if( reminder_ref -> helpCB == NULL )
    return;

  /* Call the user callback. */
  (* reminder_ref -> helpCB)( reminder_ref -> reminderW, 
                              reminder_ref -> user_data,
                              (int) call_data -> data );


  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  reminderSelectedCB( Widget                        widget,
                      XTM_RD_REMINDER_REF           reminder_ref,
                      XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  int     index;
  char    buffer[ 50 ];
  char    *char_ref;
  Widget  mainFo;
  Widget  tempW;


  /* Code. */

  mainFo = XtNameToWidget( reminder_ref -> reminderW, 
                           "RemindTlBase.RemindTlFo" );

  if( ! call_data -> set ) {
    tempW = XtNameToWidget( mainFo, "AlarmTxSW.AlarmTx" );

    char_ref = xitStringGetText( tempW );
    strcpy( reminder_ref -> alarm[ reminder_ref -> curr_work_alarm ].text,
            char_ref );

    SysFree( char_ref );

    return;
  }


  /* Find the index of the reminder selected. */
  for( index = 0; index < XTM_RD_MAX_ALARMS; index++ ) {
    sprintf( buffer, "RemindFr.RemindRc.Select%dTb", index + 1 );

    tempW = XtNameToWidget( mainFo, buffer );
    if( tempW == widget )
      break;
  }

  reminder_ref -> curr_work_alarm = index;


  /* Display the selected reminder. */
  setWorkReminder( reminder_ref, reminder_ref -> curr_work_alarm );


  return;

} /* reminderSelectedCB */


/*----------------------------------------------------------------------*/

static void
  resetCB( Widget               widget,
           XTM_RD_REMINDER_REF  reminder_ref,
           XtPointer            call_data )
{

  /* Variables. */
  int     index;
  Widget  mainFo;
  Widget  tempW;


  /* Code. */

  mainFo = XtNameToWidget( reminder_ref -> reminderW, 
                           "RemindTlBase.RemindTlFo" );

  index = reminder_ref -> curr_work_alarm;

  /* The alarm is not active. */
  reminder_ref -> alarm[ index ].active     = False;
  reminder_ref -> alarm[ index ].alarm_time = 0;
  reminder_ref -> alarm[ index ].text[ 0 ]  = '\0';

  setWorkReminder( reminder_ref, index );

  tempW = XtNameToWidget( mainFo, "AlarmTxSW.AlarmTx" );
  XmTextSetString( tempW, reminder_ref -> alarm[ index ].text );


  return;

} /* resetCB */


/*----------------------------------------------------------------------*/

static void
  alarmTimeChangedCB( Widget                        widget,
                      XTM_RD_REMINDER_REF           reminder_ref,
                      XmUbTimeSliderCallbackStruct  *call_ref )
{

  /* Variables. */
  Widget             mainFo;
  Widget             tempW;
  TIM_TIME_REF       alarm_time;
  XTM_RD_ALARM_INFO  *this_alarm;


  /* Code. */

  if( ! reminder_ref -> sliderCB_active )
    return;

  this_alarm = &reminder_ref -> alarm[ reminder_ref -> curr_work_alarm ];

  mainFo = XtNameToWidget( reminder_ref -> reminderW, 
                           "RemindTlBase.RemindTlFo" );

  /* Fetch the alarm time. */
  alarm_time = XmUbTimeSliderGetTime( widget );

  /* Do we already have this alarm time? */
  if( this_alarm -> alarm_time == alarm_time )
    return;

  /* Save the alarm time. */
  this_alarm -> alarm_time = alarm_time;


  /* If an alarm was not active, make it active. */
  if( ! this_alarm -> active ) {
    tempW = XtNameToWidget( mainFo, "AlarmActiveTb" );

    XmToggleButtonSetState( tempW, True, True );
  }


  /* Update the alarm indicator). */
  setAlarmIndicators( reminder_ref, reminder_ref -> curr_work_alarm );


  return;

} /* alarmTimeChangedCB */
