/*----------------------------------------------------------------------------
--
--  Module:           xtmHoliday
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Read and check holiday dayes.
--
--  Filename:         xtmHoliday.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-01-14
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmHoliday.c, Version: 1.1, Date: 95/02/18 15:52:24";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmHoliday.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for holidays during one year. */
typedef struct {
  int  day_no;
} HOLIDAY_INFO_REC, *HOLIDAY_INFO_REF;

/* Global record to hold the holidays information. */
typedef struct {
  int               year;
  int               holidays_no;
  HOLIDAY_INFO_REF  *holidays;
} HOLIDAYS_REC, *HOLIDAYS_REF;

/* Our holidays database. */
typedef struct {
  Boolean       workdays[ 7 ];
  char          *holidays_dir;
  int           years_no;
  int           last_year;
  HOLIDAYS_REF  last_year_ref;
  HOLIDAYS_REF  *years;
} HOLIDAYS_DB, *HOLIDAYS_DB_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmHoliday";

/* Our holidays database. */
static HOLIDAYS_DB_REF  holidays_db = NULL;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static HOLIDAYS_REF
  loadYear( HOLIDAYS_DB_REF  holidays_db_ref,
            int              for_year );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmHoInitialize( char     *holidays_dir,
                   Boolean  workdays[] )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Already initialized? */
  if( holidays_db != NULL )
    return;

  holidays_db = (HOLIDAYS_DB_REF) SysMalloc( sizeof( HOLIDAYS_DB ) );
  if( holidays_db == NULL )
    return;

  holidays_db -> holidays_dir  = SysNewString( holidays_dir );
  holidays_db -> years_no      = 0;
  holidays_db -> years         = NULL;
  holidays_db -> last_year     = 0;
  holidays_db -> last_year_ref = NULL;

  for( index = 0; index < 7; index++ )
    holidays_db -> workdays[ index ] = workdays[ index ];


  return;

} /* xtmHoInitialize */


/*----------------------------------------------------------------------*/

Boolean
  xtmHoIsHoliday( TIM_TIME_REF  for_date )
{

  /* Variables. */
  int           day_no;
  int           index;
  int           year;
  HOLIDAYS_REF  year_ref;


  /* Code. */

  if( holidays_db == NULL )
    return( False );

  day_no = TimIndexOfDayInYear( for_date );
  year   = TimIndexOfYear( for_date );

  year_ref = NULL;

  /* Search the year. */
  if( holidays_db -> last_year == year ) {
    year_ref = holidays_db -> last_year_ref;

  } else {
    for( index = 0; index < holidays_db -> years_no; index++ ) {
      if( holidays_db -> years[ index ] -> year == year ) {
        year_ref = holidays_db -> years[ index ];
        break;
      }
    }
  } /* if */


  /* If the year was not found, load the information. */
  if( year_ref == NULL )
    year_ref = loadYear( holidays_db, year );

  holidays_db -> last_year     = year;
  holidays_db -> last_year_ref = year_ref;


  /* Is the day holiday? */
  for( index = 0; index < year_ref -> holidays_no; index++ ) {
    if( year_ref -> holidays[ index ] -> day_no == day_no )
      return( True );
  }


  return( False );

} /* xtmHoIsHoliday */


/*----------------------------------------------------------------------*/

Boolean
  xtmHoIsWorkday( TIM_TIME_REF  for_date )
{

  /* Variables. */
  Boolean  is_holiday;
  int      day_in_week;


  /* Code. */

  if( holidays_db == NULL )
    return( True );

  day_in_week = TimIndexOfDayInIsoWeek( for_date );
  if( ! holidays_db -> workdays[ day_in_week - 1 ] )
    return( False );
  
  is_holiday = xtmHoIsHoliday( for_date );
  if( is_holiday )
    return( False );


  return( True );

} /* xtmHoIsWorkday */


/*----------------------------------------------------------------------*/

static HOLIDAYS_REF
  loadYear( HOLIDAYS_DB_REF  holidays_db_ref,
            int              for_year )
{

  /* Variables. */
  int               holidays_no = 0;
  int               index;
  char              buffer[ PATH_MAX ];
  char              *dir_ref;
  char              *holidays_dir;
  HOLIDAY_INFO_REC  holiday_info[ 366 ];
  HOLIDAYS_REF      holidays_ref;


  /* Code. */

  /* Initialize. */
  for( index = 0; index < 366; index++ )
    holiday_info[ index ].day_no = 0;


  /* The holidays dirs are separated by a colon (:). */
  holidays_dir = SysNewString( holidays_db_ref -> holidays_dir );

  dir_ref = strtok( holidays_dir, ":" );

  while( dir_ref != NULL ) {

    FILE  *file_ref;

    /* Build the holiday filename and open the file. */
    sprintf( buffer, "%s/holidays%d", dir_ref, for_year );

    file_ref = fopen( buffer, "r" );

    if( file_ref == NULL ) {
      dir_ref = strtok( NULL, ":" );
      continue;
    }

    /* Read all the lines in the file and save the holidays. */
    while( fgets( buffer, sizeof( buffer ), file_ref ) != NULL ) {

      int              day_no;
      int              items;
      char             date_string[ 256 ];
      TIM_STATUS_TYPE  status;
      TIM_TIME_REF     holiday_date;

      /* Parse the line. */
      items = sscanf( buffer, "%s", date_string );
      if( items == 0 )
        continue;

      if( date_string[ 0 ] == '#' || date_string[ 0 ] == '!' )
        continue;

      /* Try to parse the date?. */
      if( items >= 1 ) {
        status = TimMakeDateFromIsoString( &holiday_date, date_string );

        if( status != TIM_OK ) {
          fprintf( stderr, "%s\n", msgGetText( MXDI_ERRMSG_HOLIDAY_DATE ) );
          continue;
        }
      }

      /* Save the holiday information. */
      day_no = TimIndexOfDayInYear( holiday_date );

      for( index = 0; index < holidays_no; index++ ) {
        if( holiday_info[ index ].day_no == day_no ||
            holiday_info[ index ].day_no == 0 )
          break;
      }

      holiday_info[ index ].day_no = day_no;

      if( index == holidays_no )
        holidays_no++;

    } /* while */


    fclose( file_ref );

    dir_ref = strtok( NULL, ":" );

  } /* while */

  SysFree( holidays_dir );


  /* Save the holidays. */
  holidays_ref = (HOLIDAYS_REF) SysMalloc( sizeof( HOLIDAYS_REC ) );
  holidays_ref -> year        = for_year;
  holidays_ref -> holidays_no = holidays_no;
  holidays_ref -> holidays    = NULL;

  if( holidays_no > 0 ) {
    holidays_ref -> holidays = (HOLIDAY_INFO_REF *) 
      SysMalloc( sizeof( HOLIDAY_INFO_REF ) * holidays_no );

    for( index = 0; index < holidays_no; index++ ) {
      holidays_ref -> holidays[ index ] = (HOLIDAY_INFO_REF) 
        SysMalloc( sizeof( HOLIDAY_INFO_REC ) );

      memcpy( (void *) holidays_ref -> holidays[ index ],
              (void *) &holiday_info[ index ],
              sizeof( HOLIDAY_INFO_REC ) );
    }
  }

  holidays_db_ref -> years_no++;
  index = holidays_db_ref -> years_no;

  holidays_db_ref -> years = (HOLIDAYS_REF *)
    SysRealloc( holidays_db_ref -> years, sizeof( HOLIDAYS_REF ) * index );

  holidays_db_ref -> years[ index - 1 ] = holidays_ref;


  return( holidays_ref );

} /* loadYear */
