/*----------------------------------------------------------------------------
--
--  Module:           xtmDbTools
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Interface file for module xtmDbTools
--
--  Filename:         xtmDbTools.h
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-05-28
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
/* SCCSID = @(#) Module: xtmDbTools.h, Version: 1.2, Date: 95/06/17 12:51:38 */


/* Only include this module once. */
#ifndef define_xtmDbTools_h
#  define define_xtmDbTools_h


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "xdndbm.h"
#include "Standard.h"
#include "LstLinked.h"
#include "TimDate.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Public file names. */
#define  XTM_DB_MESSAGE_FILE    "msg"

/* Max entries per day. */
#define XTM_DB_DATE_ID_SIZE     50

/* Max length for text in the entry record. */
#define XTM_DB_RECORD_TEXT_LEN  100

/* Max length for entry tag. */
#define XTM_DB_TAG_LEN          4

/* Max number of records in log file. */
#define XTM_DB_MAX_LOG_RECORDS  200

/* Version number for messages. */
#define XTM_DB_MESSAGE_VERSION  1

/* Flags when fetching entries. */
#define XTM_DB_FETCH_STANDING              (1<<0)
#define XTM_DB_FETCH_STICKY                (1<<1)
#define XTM_DB_FETCH_ALL_STANDING          (1<<2)
#define XTM_DB_FETCH_NO_NEW_LIST           (1<<3)
#define XTM_DB_FETCH_INCLUDE               (1<<4)
#define XTM_DB_FETCH_ALL_TEXT              (1<<5)
#define XTM_DB_FETCH_SKIP_INCLUDE          (1<<6)

/* Flags defined for an entry. */
#define XTM_DB_FLAG_ALARM                  (1<<0)
#define XTM_DB_FLAG_IMPORTANT              (1<<1)
#define XTM_DB_FLAG_NOTE_DONE              (1<<2)
#define XTM_DB_FLAG_EXT_FILE               (1<<3)
#define XTM_DB_FLAG_HIDE_IN_DAY_VIEW       (1<<4)
#define XTM_DB_FLAG_ARCHIVED               (1<<5)
#define XTM_DB_FLAG_PRIVATE                (1<<6)
#define XTM_DB_FLAG_HIDE_IN_CALENDAR       (1<<7)
#define XTM_DB_FLAG_HIDE_IN_SUMMARY        (1<<8)
#define XTM_DB_FLAG_HIDE_IN_PRINT          (1<<9)
#define XTM_DB_FLAG_INCLUDE                (1<<10)
#define XTM_DB_FLAG_IN_PRIV_DB             (1<<11)
#define XTM_DB_FLAG_PRIV_EXT_FILE          (1<<12)
#define XTM_DB_FLAG_HIDE_IN_DUMP           (1<<13)
#define XTM_DB_FLAG_HIDE_IN_ALARM          (1<<14)
#define XTM_DB_FLAG_ONLY_OWNER_DELETE      (1<<15)
#define XTM_DB_FLAG_ONLY_OWNER_CHANGE      (1<<16)
#define XTM_DB_FLAG_KEEP_IN_BG             (1<<17)
#define XTM_DB_FLAG_KEEP_IN_FG             (1<<18)

#define XTM_DB_FLAG_ACTION_SCRIPT          (1<<20)
#define XTM_DB_FLAG_ACTION_TEXT            (1<<21)
#define XTM_DB_FLAG_ACTION_ALARM1          (1<<22)
#define XTM_DB_FLAG_ACTION_ALARM2          (1<<23)
#define XTM_DB_FLAG_ACTION_ALARM3          (1<<24)
#define XTM_DB_FLAG_ACTION_ALARM4          (1<<25)
#define XTM_DB_FLAG_ACTION_ALARM5          (1<<26)

/* Flags defined for a standing entry. */
#define XTM_DB_FLAG_SE_STANDING            (1<<0)
#define XTM_DB_FLAG_SE_STICKY              (1<<1)
#define XTM_DB_FLAG_SE_HIDE_IN_CALENDAR    (1<<2)
#define XTM_DB_FLAG_SE_DAY_IN_MONTH        (1<<3)
#define XTM_DB_FLAG_SE_1ST                 (1<<4)
#define XTM_DB_FLAG_SE_2ND                 (1<<5)
#define XTM_DB_FLAG_SE_3RD                 (1<<6)
#define XTM_DB_FLAG_SE_1ST_LAST            (1<<7)
#define XTM_DB_FLAG_SE_2ND_LAST            (1<<8)
#define XTM_DB_FLAG_SE_3RD_LAST            (1<<9)
#define XTM_DB_FLAG_SE_NWDAY_PREV          (1<<10)
#define XTM_DB_FLAG_SE_NWDAY_NEXT          (1<<11)
#define XTM_DB_FLAG_SE_NWDAY_SKIP          (1<<12)

/* Flags for database operations. */
#define XTM_DB_FLAG_MODE_READ              (1<<0)
#define XTM_DB_FLAG_MODE_WRITE             (1<<1)
#define XTM_DB_FLAG_MODE_PRIV              (1<<2)
#define XTM_DB_FLAG_MODE_MSG               (1<<3)
#define XTM_DB_FLAG_MODE_SETID             (1<<8)

/* Flags for log records. */
#define XTM_DB_FLAG_LOG_SAVE               (1<<0)
#define XTM_DB_FLAG_LOG_DELETE             (1<<1)
#define XTM_DB_FLAG_LOG_COPY               (1<<2)
#define XTM_DB_FLAG_LOG_MOVE               (1<<3)

/* Flags for protection. */
#define XTM_DB_PROT_READ                   (1<<0)
#define XTM_DB_PROT_WRITE                  (1<<1)
#define XTM_DB_PROT_DELETE                 (1<<2)
#define XTM_DB_PROT_CHANGE                 (1<<3)

/* Name of key and data files. */
#define  ENTRY_DB_NAME             "idDB"
#define  ENTRY_DB_NAME_DIR         "idDB.dir"
#define  ENTRY_DB_NAME_PAG         "idDB.pag"

#define  DATE_DB_NAME              "dateDB"
#define  DATE_DB_NAME_DIR          "dateDB.dir"
#define  DATE_DB_NAME_PAG          "dateDB.pag"

#define  STAND_ENTRY_DB_NAME       "standDB"
#define  STAND_ENTRY_DB_NAME_DIR   "standDB.dir"
#define  STAND_ENTRY_DB_NAME_PAG   "standDB.pag"

#define  PRIV_ENTRY_DB_NAME        "Private/privDB"
#define  PRIV_ENTRY_DB_NAME_DIR    "Private/privDB.dir"
#define  PRIV_ENTRY_DB_NAME_PAG    "Private/privDB.pag"

#define  ID_FILE                   "entryId"
#define  LOG_FILE                  "dbLog"
#define  LOCK_FILE                 "dbLock"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Status from database calls. */
typedef enum {
  XTM_DB_OK, XTM_DB_ERROR, XTM_DB_UNKNOWN, XTM_DB_LOCKED
} XTM_DB_STATUS;


/* Identifiers for the databases managed. */
typedef enum {
  XTM_DB_ENTRY_DB, 
  XTM_DB_DATE_DB, 
  XTM_DB_STAND_ENTRY_DB, 
  XTM_DB_PRIV_ENTRY_DB, 
  XTM_DB_ALL_ENTRY_DB
} XTM_DB_DATABASES;


/* Standing is valid every: */
typedef enum {
  XTM_DB_VALID_NONE,
  XTM_DB_VALID_DAY,
  XTM_DB_VALID_WEEK,
  XTM_DB_VALID_MONTH,
  XTM_DB_VALID_YEAR,
  XTM_DB_VALID_MONTH_LAST
} XTM_DB_STAND_VALID;


/* Type of entry. */
typedef enum { 
  XTM_DB_DAY_NOTE, 
  XTM_DB_DAY_ENTRY, 
  XTM_DB_UNDEFINED, 
  XTM_DB_ALL_ENTRIES
} XTM_DB_ENTRY_TYPE;

/* Category for entries. */
typedef enum {
  XTM_DB_ENTRY_LIST, 
  XTM_DB_REP_ENTRY_LIST,
  XTM_DB_STICKY_LIST
} XTM_DB_ENTRY_CATEGORY;


/* The different tyoes of entries. */
typedef enum {
  XTM_DB_TYPE_APPOINTMENT,
  XTM_DB_TYPE_STAND_APPOINTMENT,
  XTM_DB_TYPE_NOTE,
  XTM_DB_TYPE_STAND_NOTE,
  XTM_DB_TYPE_STICKY_NOTE
} XTM_DB_ALL_ENTRY_TYPES;


/* Reference to database. */
typedef DBM  *XTM_DB_DATABASE;


/* Request to create a database in XDiary. */
typedef struct {
  char              *directory;
  XTM_DB_DATABASES  database;
} XTM_DB_CREATE_REQUEST;


/* Request to open a database in XDiary. */
typedef struct {
  char              *name;
  char              *directory;
  UINT32            operations;
  int               lock_timeout;
  int               uid_locking;
  XTM_DB_DATABASES  database;
} XTM_DB_OPEN_REQUEST;


/* Request for a entry id. */
typedef struct {
  char     *directory;
  UINT32   operations;
  Boolean  lock_file;
} XTM_DB_ID_REQUEST;


/* Header record in log file. */
typedef struct {
  UINT32  last_changed;
  UINT32  current_ref;
  UINT32  records;
  UINT32  max_log_records;
  char    unused[ 50 - 4 * 4 ];
} XTM_DB_LOG_HEADER;


/* Log record in log file. */
typedef struct {
  UINT32  time_stamp;
  UINT32  entry_id;
  UINT32  flags;
  UINT32  changed_by;
} XTM_DB_LOG_RECORD;


/* Database references for entries (all databases). */
typedef struct {
  char             database_dir[ 150 ];
  char             name[ 16 ];
  XTM_DB_DATABASE  entry_db;
  XTM_DB_DATABASE  date_db;
  XTM_DB_DATABASE  stand_entry_db;
  XTM_DB_DATABASE  private_db;
  UINT32           operations;
  int              lock_fd;
  Boolean          locked;
  Boolean          set_id;
} XTM_DB_ENTRY_DATABASES;


/* Database references for messages. */
typedef struct {
  char      database_dir[ 150 ];
  int       lock_fd;
  Boolean   locked;
  Boolean   set_id;
} XTM_DB_MESSAGE_DATABASE;


/* XDiary message record. */
typedef struct {
  UINT8   version;
  char    from[ 21 ];
  UINT32  flags;
  UINT32  date;
  UINT32  time;
  UINT16  duration;
  UINT16  message_length;
  UINT16  text_length;
} XTM_DB_MESSAGE_DEF, *XTM_DB_MESSAGE_REF;


/* Key and record containing data for an entry. */
typedef struct {
  UINT32  id;
} XTM_DB_ENTRY_KEY;

typedef struct {
  UINT32  id;
  UINT32  time_stamp;
  UINT32  date_stamp;
  UINT32  last_update;
  UINT32  owner;
  UINT8   entry_type;
  UINT8   entry_category;
  char    tag[ XTM_DB_TAG_LEN + 1 ];
  UINT16  duration;
  UINT32  flags;
  INT16   alarm_offset[ 5 ];
  UINT8   alarm_valid[ 5 ];
  UINT8   alarm_melody;
  UINT8   day_list_lines;
  UINT8   alarm_lines;
  char    text[ XTM_DB_RECORD_TEXT_LEN + 1 ];
  UINT8   fg_color_index;
  UINT8   bg_color_index;
  INT32   crc;

  char    unused[ 170 - 7*4 - 1*2 - 5*2 - 5*1 - 7*1 -
                  XTM_DB_TAG_LEN - 1 -
                  XTM_DB_RECORD_TEXT_LEN - 1 ];
} XTM_DB_ENTRY_DEF, *XTM_DB_ENTRY_REF;


/* Key and record containing entries for a day. */
typedef struct {
  UINT32  date;
} XTM_DB_DATE_KEY;

typedef struct {
  UINT32  date;
  UINT32  id[ XTM_DB_DATE_ID_SIZE ];

  char   unused[ 256 - 1*4 - XTM_DB_DATE_ID_SIZE*4 ];
} XTM_DB_DATE_DEF, *XTM_DB_DATE_REF;


/* Key and record for standing entries. */
typedef struct {
  UINT32  id;
} XTM_DB_STAND_ENTRY_KEY;

typedef struct {
  UINT32  id;
  UINT32  from;
  UINT32  to;
  UINT32  flags;
  UINT8   valid_days[ 7 ];
  UINT16  every_n;
  UINT8   valid_every;
  UINT32  skip_week[ 2 ];
  
  char    unused[ 48 - 4*4 - 2*4 - 7*1 - 1*2 - 1*1 ];
} XTM_DB_STAND_ENTRY_DEF, *XTM_DB_STAND_ENTRY_REF;


/* Key and record for private data. */
typedef struct {
  UINT32  id;
} XTM_DB_PRIV_KEY;

typedef struct {
  UINT32  id;
  char    text[ XTM_DB_RECORD_TEXT_LEN + 1 ];

  char    unused[ 110 - 1 * 4 - XTM_DB_RECORD_TEXT_LEN - 1 ];
} XTM_DB_PRIV_DEF, *XTM_DB_PRIV_REF;


/* Record describng one entry (a combination of many records). */
typedef struct {
  XTM_DB_ENTRY_DEF        entry;
  XTM_DB_STAND_ENTRY_DEF  stand_entry;
  char                    db_name[ 16 ];
  char                    *all_text;
} XTM_DB_ALL_ENTRY_DEF, *XTM_DB_ALL_ENTRY_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

XTM_DB_STATUS
  xtmDbChangesInLog( char               *directory,
                     TIM_TIME_REF       since,
                     XTM_DB_LOG_RECORD  changes[],
                     int                max_changes,
                     int                *no_changes );

void
  xtmDbCheckDbOperations( char     *directory,
                          Boolean  force_check,
                          UINT32   *operations );

void
  xtmDbClearAccessCache();

XTM_DB_STATUS
  xtmDbCloseEntryDb( XTM_DB_ENTRY_DATABASES *entry_databases );

XTM_DB_STATUS
  xtmDbCreateDatabase( XTM_DB_CREATE_REQUEST  *create_request );

XTM_DB_STATUS
  xtmDbDeleteEntry( XTM_DB_ENTRY_DATABASES  *databases,
                    UINT32                  id );

XTM_DB_STATUS
  xtmDbDeleteMessage( char    *db_dir,
                      UINT32  msg_id );

Boolean
  xtmDbDoesStandingMatch( XTM_DB_STAND_ENTRY_REF  stand_ref,
                          UINT32                  flags,
                          Boolean                 check_non_wday,
                          TIM_TIME_REF            date );

XTM_DB_STATUS
  xtmDbFetchDates( XTM_DB_ENTRY_DATABASES  *databases,
                   LST_DESC_TYPE           *list_ref );

XTM_DB_STATUS
  xtmDbFetchEntry( XTM_DB_ENTRY_DATABASES  *databases,
                   UINT32                  id,
                   XTM_DB_ALL_ENTRY_REF    entry_ref,
                   char                    **text_ref );

XTM_DB_STATUS
  xtmDbFetchEntriesInDay( XTM_DB_ENTRY_DATABASES  *databases,
                          TIM_TIME_REF            date,
                          UINT32                  flags,
                          LST_DESC_TYPE           *entry_list_ref,
                          LST_DESC_TYPE           *note_list_ref );

XTM_DB_STATUS
  xtmDbFetchFileInfo( char         *filename,
                      struct stat  *file_info );

XTM_DB_STATUS
  xtmDbFetchMessage( char                *db_dir,
                     UINT32              msg_id,
                     XTM_DB_MESSAGE_REF  msg_info,
                     char                **message,
                     char                **text );

XTM_DB_STATUS
  xtmDbFetchStandEntries( XTM_DB_ENTRY_DATABASES  *databases,
                          LST_DESC_TYPE           *entry_list_ref,
                          LST_DESC_TYPE           *note_list_ref );

XTM_DB_STATUS
  xtmDbGenerateId( XTM_DB_ID_REQUEST  *request,
                   UINT32             *id );

void
  xtmDbGetEntryPermissions( UINT32  db_operations,
                            int     entry_owner,
                            UINT32  entry_flags,
                            UINT32  *can_do_flags );

XTM_DB_STATUS
  xtmDbInitializeAuxFiles( char  *directory );

void
  xtmDbInitializeProcessId();

XTM_DB_STATUS
  xtmDbInsertEntry( XTM_DB_ENTRY_DATABASES  *databases,
                    XTM_DB_ALL_ENTRY_REF    entry_ref,
                    char                    *text_ref );

XTM_DB_STATUS
  xtmDbInsertMessage( char                *db_dir,
                      XTM_DB_MESSAGE_REF  msg_info,
                      char                *message,
                      char                *text );

Boolean
  xtmDbIsEntryDefined( XTM_DB_ENTRY_DATABASES  *database,
                       LST_DESC_TYPE           *stand_entries,
                       TIM_TIME_REF            cal_date );

XTM_DB_STATUS
  xtmDbOpenEntryDb( XTM_DB_OPEN_REQUEST     *open_request,
                    XTM_DB_ENTRY_DATABASES  *entry_databases );

void
  xtmDbUseFileLock( Boolean  use_file_lock );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

#endif
