/*----------------------------------------------------------------------------
--
--  Module:           xtmCustIncl
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Include defined databases in this database.
--
--  Filename:         xtmCustIncl.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-10-15
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCustIncl.c, Version: 1.1, Date: 95/02/18 15:52:05";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmCustIncl.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the 'Customize Include' window. */
#define actionRc           dataLocalW[  0 ]
#define anchorSp           dataLocalW[  1 ]
#define dbLi               dataLocalW[  2 ]
#define doInclDbAr         dataLocalW[  3 ]
#define doInclDbPb         dataLocalW[  4 ]
#define doInclDbRc         dataLocalW[  5 ]
#define doRemDbAr          dataLocalW[  6 ]
#define doRemDbPb          dataLocalW[  7 ]
#define doRemDbRc          dataLocalW[  8 ]
#define explainLa          dataLocalW[  9 ]
#define inclDbLa           dataLocalW[ 10 ]
#define inclInDbLa         dataLocalW[ 11 ]
#define inclLi             dataLocalW[ 12 ]



/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for included calendars customization. */
typedef struct {

  /* Selected database. */
  char  db_selected[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* The include window. */
  Widget  inclW;

  /* Calendar database. */
  XTM_CD_HANDLE  cal_db_handle;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_CI_ACTION_CB  actionCB;

} INCL_REC, *INCL_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCustInclude";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  cancelCB( Widget        widget,
            INCL_REC_REF  incl_ref,
            XtPointer     call_data );

static Widget
  createIncludeWindow( INCL_REC_REF  incl_ref,
                       Widget        parent );

static void 
  dbListSelCB( Widget                widget, 
               INCL_REC_REF          incl_ref,
               XmListCallbackStruct  *call_data );

static void 
  destroyCB( Widget        widget,
             INCL_REC_REF  incl_ref,
             XtPointer     call_data );

static void 
  inclCB( Widget        widget, 
          INCL_REC_REF  incl_ref,
          XtPointer     call_data );

static void 
  inclListSelCB( Widget                widget, 
                 INCL_REC_REF          incl_ref,
                 XmListCallbackStruct  *call_data );

static void 
  okCB( Widget        widget,
        INCL_REC_REF  incl_ref,
        XtPointer     call_data );

static void 
  removeCB( Widget        widget, 
            INCL_REC_REF  incl_ref,
            XtPointer     call_data );

static void
  setIncludeData( INCL_REC_REF  incl_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_CI_HANDLE
  xtmCiInitialize( XTM_GL_CUSTOM_DATA_REF  custom_data_ref,
                   Widget                  parent,
                   XTM_CI_ACTION_CB        actionCB,
                   void                    *user_data )
{

  /* Variables. */
  INCL_REC_REF  incl_ref;


  /* Code. */

  /* Create and initialize our private data. */
  incl_ref = SysNew( INCL_REC );
  if( incl_ref == NULL )
    return( NULL );

  incl_ref -> custom_data_ref = custom_data_ref;
  incl_ref -> actionCB        = actionCB;
  incl_ref -> user_data       = user_data;


  /* Create the groups window. */
  incl_ref -> inclW = createIncludeWindow( incl_ref, parent );

  if( incl_ref -> inclW == NULL ) {
    SysFree( incl_ref );

    return( NULL );
  }


  return( (XTM_CI_HANDLE) incl_ref );

} /* xtmCiInitialize */


/*----------------------------------------------------------------------*/

void
  xtmCiDestroy( XTM_CI_HANDLE  incl_handle )
{

  /* Variables. */
  INCL_REC_REF  incl_ref;


  /* Code. */

  if( incl_handle == NULL )
    return;

  /* Our private data. */
  incl_ref = (INCL_REC_REF) incl_handle;


  /* Destroy the window. */
  XtDestroyWidget( incl_ref -> inclW );


  return;

} /* xtmCiDestroy */


/*----------------------------------------------------------------------*/

void 
  xtmCiEditCustInclude( XTM_CI_HANDLE  incl_handle,
                        XTM_CD_HANDLE  cal_db_handle,
                        char           *db_name )
{

  /* Variables. */
  INCL_REC_REF  incl_ref;


  /* Code. */

  if( incl_handle == NULL )
    return;

  /* Our private data. */
  incl_ref = (INCL_REC_REF) incl_handle;


  /* Save the database we are editing. */
  incl_ref -> cal_db_handle = cal_db_handle;
  strcpy( incl_ref -> db_selected, db_name );


  /* Fill in the window fields. */
  setIncludeData( incl_ref );


  /* Make sure the window is visible. */
  XtManageChild( incl_ref -> inclW );


  return;

} /* xtmCiEditCustInclude */


/*----------------------------------------------------------------------*/

static Widget
  createIncludeWindow( INCL_REC_REF  incl_ref,
                       Widget        parent )
{

  /* Variables. */
  char       *char_ref;
  Arg        args[ 10 ];
  Cardinal   n;
  Dimension  offset;
  Dimension  width;
  Widget     cuInclFd;
  Widget     workFo;
  Widget     dataLocalW[ 13 ];

  static XIT_ARROW_STRUCT arrow_action[] = {
    { "", True, XmARROW_LEFT,  NULL },
    { "", True, XmARROW_RIGHT, NULL },
  };

  static XIT_PUSH_STRUCT move_action[] = {
    { "", "", "", True, NULL },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   okCB,     NULL },
    { NULL, NULL,     NULL },
    { NULL, NULL,     NULL },
    { "",   cancelCB, NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = incl_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 3 ].data  = incl_ref;


  /* Create a form dialog with buttons. */
  cuInclFd = xitCreateFormDialog( parent, "CuInclFd",
                                  1, 0,
                                  action_buttons,
                                  XtNumber( action_buttons ) );

  XtAddCallback( cuInclFd,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) incl_ref );

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( cuInclFd, args, n );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( cuInclFd, "CuInclFdFo" );


  /* Explain what this is. */
  explainLa = xitCreateLabel( workFo, "", 
                              msgGetText( MXDI_EXPLAIN_INCLUDE_DB ), -1 );

  /* Labels telling what we are doing. */
  inclDbLa   = xitCreateLabel( workFo, "", 
                               msgGetText( MXDI_INCLUDE_DB ), -1 );

  inclInDbLa = xitCreateLabel( workFo, "", 
                               msgGetText( MXDI_INCLUDE_IN_THE_DB ), -1 );


  /* Lists with defined databases. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmEXTENDED_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  dbLi = XmCreateScrolledList( workFo, "DbLi", args, n );

  XtAddCallback( dbLi, XmNextendedSelectionCallback,
                 (XtCallbackProc) dbListSelCB, (XtPointer) incl_ref );


  /* Lists with included databases. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmEXTENDED_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  inclLi = XmCreateScrolledList( workFo, "InclLi", args, n );

  XtAddCallback( inclLi, XmNextendedSelectionCallback,
                 (XtCallbackProc) inclListSelCB, (XtPointer) incl_ref );


  /* Form to hold actions. */
  n = 0;
  actionRc = XmCreateRowColumn( workFo, "ActionRc", args, n );


  /* Include database (display). */
  char_ref = msgGetText( MXDI_INCLUDE_ACTION_LABEL );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  doInclDbRc = XmCreateRowColumn( actionRc, "", args, n );

  doInclDbAr = xitCreateArrowPushButton( doInclDbRc, &arrow_action[ 0 ] );

  move_action[ 0 ].title = char_ref;
  doInclDbPb = xitCreatePushButton( doInclDbRc, &move_action[ 0 ] );

  XtAddCallback( doInclDbAr, XmNactivateCallback,
                 (XtCallbackProc) inclCB, (XtPointer) incl_ref );
  XtAddCallback( doInclDbPb, XmNactivateCallback,
                 (XtCallbackProc) inclCB, (XtPointer) incl_ref );


  /* Remove database (display). */
  char_ref = msgGetText( MXDI_INCLUDE_REMOVE_ACTION_LABEL );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  doRemDbRc = XmCreateRowColumn( actionRc, "", args, n );

  doRemDbAr = xitCreateArrowPushButton( doRemDbRc, &arrow_action[ 1 ] );

  move_action[ 0 ].title = char_ref;
  doRemDbPb = xitCreatePushButton( doRemDbRc, &move_action[ 0 ] );

  XtAddCallback( doRemDbAr, XmNactivateCallback,
                 (XtCallbackProc) removeCB, (XtPointer) incl_ref );
  XtAddCallback( doRemDbPb, XmNactivateCallback,
                 (XtCallbackProc) removeCB, (XtPointer) incl_ref );

  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
  XtSetValues( doInclDbPb, args, n );
  XtSetValues( doRemDbPb,  args, n );


  /* Create anchors. */
  n = 0;
  XtSetArg( args[ n ], XmNseparatorType,  XmNO_LINE );  n++;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_POSITION );  n++;
  XtSetArg( args[ n ], XmNleftPosition,   50 );  n++;
  anchorSp = XmCreateSeparatorGadget( workFo, "AnchorSp", args, n );


  /* Place the main elements together. */
  xitAttachWidget( explainLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( inclInDbLa,
                   XmATTACH_WIDGET, explainLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( inclDbLa,
                   XmATTACH_WIDGET, explainLa, XmATTACH_WIDGET, anchorSp,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE,   NULL );
  xitAttachWidget( XtParent( inclLi ),
                   XmATTACH_WIDGET, inclDbLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( actionRc,
                   XmATTACH_WIDGET, inclDbLa, 
                   XmATTACH_WIDGET, XtParent( inclLi ),
                   XmATTACH_NONE,   NULL,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( XtParent( dbLi ),
                   XmATTACH_WIDGET, inclInDbLa, XmATTACH_WIDGET, actionRc,
                   XmATTACH_NONE,   NULL,       XmATTACH_NONE,   NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( explainLa,          args, n );
  XtSetValues( inclDbLa,           args, n );
  XtSetValues( inclInDbLa,         args, n );
  XtSetValues( XtParent( dbLi ),   args, n );
  XtSetValues( XtParent( inclLi ), args, n );
  XtSetValues( actionRc,           args, n );


  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( cuInclFd, True );


  /* Make the final attachments. */
  xitAttachWidget( XtParent( dbLi ),
                   XmATTACH_WIDGET, inclInDbLa, XmATTACH_WIDGET, anchorSp,
                   XmATTACH_FORM,   NULL,       XmATTACH_FORM,   NULL );
  xitAttachWidget( actionRc,
                   XmATTACH_WIDGET, inclDbLa, XmATTACH_NONE, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( inclLi ),
                   XmATTACH_WIDGET, inclDbLa, XmATTACH_FORM, NULL,
                   XmATTACH_WIDGET, anchorSp, XmATTACH_FORM, NULL );


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( XtParent( cuInclFd ), args, n );


  /* Attach the action form in the middle of the window. */
  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtGetValues( actionRc, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_WIDGET ); n++;
  XtSetArg( args[ n ], XmNleftWidget,     anchorSp ); n++;
  XtSetArg( args[ n ], XmNleftOffset,     (-1) * width / 2 ); n++;
  XtSetValues( actionRc, args, n );

  offset = width / 2;

  n = 0;
  XtSetArg( args[ n ], XmNrightOffset, offset ); n++;
  XtSetValues( XtParent( inclLi ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftOffset, offset ); n++;
  XtSetValues( XtParent( dbLi ), args, n );
  XtSetValues( inclDbLa,         args, n );


  return( cuInclFd );

} /* createIncludeWindow */


/*----------------------------------------------------------------------*/

static void
  setIncludeData( INCL_REC_REF  incl_ref )
{

  /* Variables. */
  Boolean           ok;
  int               index;
  int               index1;
  char              *char_ref;
  char              *db_names;
  char              *hide_label;
  char              *separator;
  char              buffer[ 200 ];
  Arg               args[ 10 ];
  Cardinal          n;
  Widget            mainW;
  Widget            tempW;
  XmString          list_items[ 100 ];
  XmString          xstr;
  XTM_CD_CAL_INFO   sel_db_info;
  XTM_CD_INCL_CALS  sel_db_incl;


  /* Code. */

  mainW      = XtNameToWidget( incl_ref -> inclW, "CuInclFdFo" );
  hide_label = msgGetText( MXDI_INCLUDE_HIDE_LABEL );
  separator  = msgGetText( MXDI_INCLUDE_NO_HIDE_LABEL );


  /* Window title. */
  sprintf( buffer, msgGetText( MXDI_INCLUDE_TITLE ), incl_ref -> db_selected );
  xstr = XmStringCreate( buffer, CS );

  n = 0;
  XtSetArg( args[ n ], XmNdialogTitle, xstr ); n++;
  XtSetValues( incl_ref -> inclW, args, n );

  XmStringFree( xstr );



  /* This is the selected database. */
  ok = xtmCdFetchNamedDb( incl_ref -> cal_db_handle,
                          incl_ref -> db_selected, 
                          &sel_db_info, &sel_db_incl );
  if( ! ok )
    return;


  /* Set 'known' databases. */
  ok = xtmCdFetchDbNames( incl_ref -> cal_db_handle, &db_names );
  char_ref = db_names;
  index    = 0;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( incl_ref -> cal_db_handle, db_name,
                              &db_info, NULL );

    /* Is this database already in the include list? */
    if( ! xtmCdSearchIncludeDb( &sel_db_incl, db_info.short_name ) &&
          strcmp( db_info.short_name, sel_db_info.short_name ) != 0 ) {

      sprintf( buffer, "%-15.15s %s", 
               db_info.short_name, db_info.directory );

      list_items[ index ] = XmStringCreate( buffer, CS );
      index++;

    } /* if */

  } while( True );

  SysFree( db_names );


  tempW = XtNameToWidget( mainW, "DbLiSW.DbLi" );

  /* Assign the database locations to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );



  /* Set the 'included' databases. */
  index = 0;

  for( index1 = 0; index1 < sel_db_incl.no; index1++ ) {

    XTM_CD_CAL_INFO  db_info;

    (void) xtmCdFetchNamedDb( incl_ref -> cal_db_handle, 
                              sel_db_incl.db[ index1 ].name, 
                              &db_info, NULL );

    sprintf( buffer, "%-15.15s  %s  %s", 
             db_info.short_name, separator, db_info.directory );

    list_items[ index ] = XmStringCreate( buffer, CS );
    index++;

  } /* loop */


  tempW = XtNameToWidget( mainW, "InclLiSW.InclLi" );

  /* Assign the database info to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );


  return;

} /* setIncludeData */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget        widget,
            INCL_REC_REF  incl_ref,
            XtPointer     call_data )
{

  /* Code. */

  XtUnmanageChild( incl_ref -> inclW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  dbListSelCB( Widget                widget, 
               INCL_REC_REF          incl_ref,
               XmListCallbackStruct  *call_data )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( incl_ref -> inclW, "CuInclFdFo" );

  /* No selections in the include list. */
  tempW = XtNameToWidget( mainW, "InclLiSW.InclLi" );

  XmListDeselectAllItems( tempW );


  return;

} /* dbListSelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             INCL_REC_REF  incl_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( incl_ref -> actionCB != NULL )
    (* incl_ref -> actionCB)( XTM_CI_REASON_DESTROY,
                              NULL, NULL, incl_ref -> user_data );


  /* Release the user data. */
  SysFree( incl_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  inclCB( Widget        widget, 
          INCL_REC_REF  incl_ref,
          XtPointer     call_data )
{

  /* Variables. */
  int              index;
  int              selected_item_count;
  char             buffer[ 200 ];
  char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  char             *buffer_ref;
  char             *separator;
  Arg              args[ 5 ];
  Cardinal         n;
  Widget           mainW;
  Widget           dbListW;
  Widget           inclListW;
  XmString         *selected_items;
  XmString         xstr;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  mainW = XtNameToWidget( incl_ref -> inclW, "CuInclFdFo" );

  separator = msgGetText( MXDI_INCLUDE_NO_HIDE_LABEL );
  dbListW   = XtNameToWidget( mainW, "DbLiSW.DbLi" );
  inclListW = XtNameToWidget( mainW, "InclLiSW.InclLi" );

  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems,     &selected_items ); n++;
  XtGetValues( dbListW, args, n );


  /* Move all selected databases to the include list. */
  for( index = 0; index < selected_item_count; index++ ) {

    buffer_ref = xitStringGetString( *(selected_items + index), CS );

    sscanf( buffer_ref, "%s", db_name );
    SysFree( buffer_ref );

    (void) xtmCdFetchNamedDb( incl_ref -> cal_db_handle, db_name,
                              &db_info, NULL );

    sprintf( buffer, "%-15.15s  %s  %s", 
             db_info.short_name, separator, db_info.directory );


    /* Add database to the include list. */
    xstr = XmStringCreate( buffer, CS );

    XmListAddItem( inclListW, xstr, 0 );
    XmStringFree( xstr );

  } /* loop */


  /* Remove all selected items. */
  {
    XmString  list_items[ 100 ];

    for( index = 0; index < selected_item_count; index++ )
      list_items[ index ] = XmStringCopy( *(selected_items + index) );

    for( index = 0; index < selected_item_count; index++ ) {
      XmListDeleteItem( dbListW, list_items[ index ] );
      XmStringFree( list_items[ index ] );
    }
  }

  return;

} /* inclCB */


/*----------------------------------------------------------------------*/

static void 
  inclListSelCB( Widget                widget, 
                 INCL_REC_REF          incl_ref,
                 XmListCallbackStruct  *call_data )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( incl_ref -> inclW, "CuInclFdFo" );

  /* No selections in the database list. */
  tempW = XtNameToWidget( mainW, "DbLiSW.DbLi" );

  XmListDeselectAllItems( tempW );


  return;

} /* inclListSelCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget        widget,
        INCL_REC_REF  incl_ref,
        XtPointer     call_data )
{

  /* Variables. */
  int               index;
  int               item_count;
  char              *buffer_ref;
  Arg               args[ 5 ];
  Cardinal          n;
  Widget            mainW;
  Widget            dbListW;
  Widget            inclListW;
  XmString          *items;
  XTM_CD_CAL_INFO   db_info;
  XTM_CD_INCL_CALS  db_incl;


  /* Code. */

  mainW     = XtNameToWidget( incl_ref -> inclW, "CuInclFdFo" );
  dbListW   = XtNameToWidget( mainW, "DbLiSW.DbLi" );
  inclListW = XtNameToWidget( mainW, "InclLiSW.InclLi" );


  /* Fetch the database we are changing. */
  (void) xtmCdFetchNamedDb( incl_ref -> cal_db_handle, 
                            incl_ref -> db_selected,
                            &db_info, &db_incl );


  /* Fetch all items in the 'include' window. */
  n = 0;
  XtSetArg( args[ n ], XmNitems,     &items ); n++;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtGetValues( inclListW, args, n );

  if( item_count > XTM_CD_MAX_CAL_INCL )
    item_count = XTM_CD_MAX_CAL_INCL;

  for( index = 0; index < item_count; index++ ) {

    buffer_ref = xitStringGetString( *(items + index), CS );

    sscanf( buffer_ref, "%s", db_incl.db[ index ].name );

    SysFree( buffer_ref );

  } /* loop */

  db_incl.no = item_count;


  /* Do we have a user action callback registered? */
  if( incl_ref -> actionCB != NULL )
    (* incl_ref -> actionCB)( XTM_CI_REASON_OK,
                              incl_ref -> db_selected, &db_incl,
                              incl_ref -> user_data );


  XtUnmanageChild( incl_ref -> inclW );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void 
  removeCB( Widget        widget, 
            INCL_REC_REF  incl_ref,
            XtPointer     call_data )
{

  /* Variables. */
  int              index;
  int              selected_item_count;
  char             buffer[ 200 ];
  char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  char             *buffer_ref;
  Arg              args[ 5 ];
  Cardinal         n;
  Widget           mainW;
  Widget           dbListW;
  Widget           inclListW;
  XmString         *selected_items;
  XmString         xstr;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  mainW     = XtNameToWidget( incl_ref -> inclW, "CuInclFdFo" );
  dbListW   = XtNameToWidget( mainW, "DbLiSW.DbLi" );
  inclListW = XtNameToWidget( mainW, "InclLiSW.InclLi" );


  /* No selections in the include list. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems, &selected_items ); n++;
  XtGetValues( inclListW, args, n );


  /* Move all selected databases to the database list. */
  for( index = 0; index < selected_item_count; index++ ) {

    buffer_ref = xitStringGetString( *(selected_items + index), CS );

    sscanf( buffer_ref, "%s", db_name );
    SysFree( buffer_ref );


    (void) xtmCdFetchNamedDb( incl_ref -> cal_db_handle, db_name,
                              &db_info, NULL );

    sprintf( buffer, "%-15.15s %s", db_name, db_info.directory );


    /* Add database to the include list. */
    xstr = XmStringCreate( buffer, CS );

    XmListAddItem( dbListW, xstr, 0 );
    XmStringFree( xstr );

  } /* loop */


  /* Remove all selected items. */
  {
    XmString  list_items[ 100 ];

    for( index = 0; index < selected_item_count; index++ )
      list_items[ index ] = XmStringCopy( *(selected_items + index) );

    for( index = 0; index < selected_item_count; index++ ) {
      XmListDeleteItem( inclListW, list_items[ index ] );
      XmStringFree( list_items[ index ] );
    }
  }


  return;

} /* removeCB */
