/*----------------------------------------------------------------------------
--
--  Module:           xtmCpMvEntry
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Move or copy an entry in the day list to another time or date.
--
--  Filename:         xtmCpMvEntry.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-01-23
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCpMvEntry.c, Version: 1.1, Date: 95/02/18 15:52:00";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xtmDbMisc.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "XmUbTimeB.h"
#include "xtmCpMvEntry.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the copy/move window. */
#define databaseFs    dataLocalW[  0 ]
#define databaseLa    dataLocalW[  1 ]
#define dbRc          dataLocalW[  2 ]
#define messageLa     dataLocalW[  3 ]
#define toTimeTb      dataLocalW[  4 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Information about the entry to copy/move. */
typedef struct {

  /* Move/Copy to this database. */
  char  to_cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* The main window. */
  Widget  copyMoveW;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Entry being moved/copied. */
  XTM_DB_ALL_ENTRY_REF  entry_ref;

  /* Do we copy or move? */
  XTM_CM_OPERATION  operation;

  /* Move/Copy to this date. */
  TIM_TIME_REF  to_date;
  TIM_TIME_REF  to_time;

} CM_REC, *CM_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCopyMoveEntry";

/* IDs for the help windows. */
static char  *copy_move_window_id = "CopyMove";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  copyCB( Widget      widget,
          CM_REC_REF  cm_ref,
          XtPointer   call_data );

static Widget
  createCopyMoveWindow( Widget            parent,
                        CM_REC_REF        cm_ref,
                        XTM_CM_OPERATION  operation );

static void 
  destroyCB( Widget      widget,
             CM_REC_REF  cm_ref,
             XtPointer   call_data );

static void
  doMoveCopyCB( Widget      widget,
                CM_REC_REF  cm_ref,
                XtPointer   call_data );

static void 
  helpCB( Widget      widget,
          CM_REC_REF  cm_ref,
          XtPointer   call_data );

static void
  moveCB( Widget      widget,
          CM_REC_REF  cm_ref,
          XtPointer   call_data );

static Boolean
  moveOrCopyEntry( CM_REC_REF  cm_ref );

static Boolean
  readEntry( Widget      parent,
             CM_REC_REF  cm_ref,
             char        *cal_name,
             UINT32      entry_id );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  xtmCmCopyMoveEntry( XTM_GL_BASE_DATA_REF  appl_data_ref,
                      Widget                parent,
                      XTM_CM_OPERATION      operation,
                      char                  *cal_name,
                      UINT32                entry_id )
{

  /* Variables. */
  Boolean          ok;
  Widget           mainW;
  Widget           tempW;
  CM_REC_REF       cm_ref;
  XTM_CD_CAL_INFO  cal_info;


  /* Code. */

  /* Create and initialize our private data. */
  cm_ref = SysNew( CM_REC );
  if( cm_ref == NULL )
    return;

  cm_ref -> appl_data_ref = appl_data_ref;
  cm_ref -> operation     = operation;

  /* Fetch the entry and calendr information. */
  ok = readEntry( parent, cm_ref, cal_name, entry_id );
  if( ! ok )
    return;

  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                          cal_name, &cal_info, NULL );
  if( ! ok ) {
    SysFree( cm_ref );
    return;
  }

  /* Create the copy/move window. */
  cm_ref -> copyMoveW = createCopyMoveWindow( parent, cm_ref, operation );

  mainW = XtNameToWidget( cm_ref -> copyMoveW, "CopyMoveFdFo" );


  /* Depending on the type of the entry, take different actions. */
  tempW = XtNameToWidget( mainW, "ToTimeTb" );

  if( cm_ref -> entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE ) {
    (void) XmUbTimeBoxSetStartDate( tempW, 
                                    cm_ref -> entry_ref -> entry.date_stamp );
  } else {
    (void) XmUbTimeBoxSetStartDate( tempW,
                                    cm_ref -> entry_ref -> entry.date_stamp );
    (void) XmUbTimeBoxSetStartTime( tempW,
                                    cm_ref -> entry_ref -> entry.time_stamp );
  }


  /* Database name. */
  tempW = XtNameToWidget( mainW, "DbRc.DatabaseFs" );
  if( flagIsSet( cal_info.operations, XTM_DB_FLAG_MODE_WRITE ) )
    xitFieldSelectSetCurrent( tempW, cm_ref -> entry_ref -> db_name, False );
  else
    xitFieldSelectSetCurrent( tempW, "", False );


  /* Make the form visible. */
  XtManageChild( cm_ref -> copyMoveW );


  return;

} /* xtmCmCopyMoveEntry */


/*----------------------------------------------------------------------*/

void
  xtmCmCopyMoveEntryTo( XTM_GL_BASE_DATA_REF  appl_data_ref,
                        Widget                parent,
                        XTM_CM_OPERATION      operation,
                        char                  *from_cal_name,
                        UINT32                from_entry_id,
                        char                  *to_cal_name,
                        TIM_TIME_REF          to_date,
                        TIM_TIME_REF          to_time )
{

  /* Variables. */
  Boolean     ok;
  char        buffer[ 100 ];
  char        date_buffer[ 20 ];
  char        time_buffer[ 20 ];
  char        *to_do;
  Widget      tempW;
  CM_REC_REF  cm_ref;


  /* Code. */

  /* Create and initialize our private data. */
  cm_ref = SysNew( CM_REC );
  if( cm_ref == NULL )
    return;

  cm_ref -> appl_data_ref = appl_data_ref;
  cm_ref -> operation     = operation;


  /* Fetch the entry. */
  ok = readEntry( parent, cm_ref, from_cal_name, from_entry_id );
  if( ! ok )
    return;


  /* This is what we want to do. */
  cm_ref -> to_date = to_date;
  cm_ref -> to_time = to_time;

  strcpy( cm_ref -> to_cal_name, to_cal_name );


  /* Ask for confirmation. */
  if( operation == XTM_CM_MOVE )
    to_do = msgGetText( MXDI_MOVE_BUTTON );
  else
    to_do = msgGetText( MXDI_COPY_BUTTON );

  xtmFoFormatDate( to_date, date_buffer, sizeof( date_buffer ) );
  xtmFoFormatTime( to_time, time_buffer, sizeof( time_buffer ) );


  /* Message to display. */
  if( cm_ref -> entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE )
    sprintf( buffer, msgGetText( MXDI_COPY_MOVE_TO ),
             to_do, date_buffer, "" );
  else
    sprintf( buffer, msgGetText( MXDI_COPY_MOVE_TO ),
             to_do, date_buffer, time_buffer );

  if( cm_ref -> appl_data_ref -> custom_data -> confirm_actions ) {
    tempW = xitCreateQuestionDialog( 
              parent, "QuestionDialog",
              msgGetText( MXDI_MOVE_COPY_TITLE ),
              buffer,
              doMoveCopyCB, cm_ref, 
              NULL, NULL );

    XtAddCallback( tempW, XmNdestroyCallback, 
                   (XtCallbackProc) destroyCB, (XtPointer) cm_ref );

    return;
  }


  /* Copy the entry. */
  doMoveCopyCB( NULL, cm_ref, NULL );

  /* Release the user data. */
  if( cm_ref -> entry_ref != NULL ) {
    SysFree( cm_ref -> entry_ref -> all_text );
    SysFree( cm_ref -> entry_ref );
  }

  SysFree( cm_ref );


  return;

} /* xtmCmCopyMoveEntryTo */


/*----------------------------------------------------------------------*/

static Widget
  createCopyMoveWindow( Widget            parent,
                        CM_REC_REF        cm_ref,
                        XTM_CM_OPERATION  operation )
{

  /* Variables. */
  Arg                     args[ 5 ];
  Cardinal                n;
  Widget                  copyMoveFd;
  Widget                  dataLocalW[ 5 ];
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", copyCB, NULL },
    { "", moveCB, NULL },
    { "", NULL,   NULL },
    { "", helpCB, NULL },
  };


  /* Code. */

  appl_data_ref   = cm_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Text items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_COPY_BUTTON );
  action_buttons[ 0 ].data  = cm_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_MOVE_BUTTON );
  action_buttons[ 1 ].data  = cm_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 2 ].data  = cm_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 3 ].data  = cm_ref;


  /* Create a form dialog with buttons. */
  if( operation == XTM_CM_COPY )
    copyMoveFd = xitCreateFormDialog( parent, "CopyMoveFd",
                                      1, 0,
                                      action_buttons,
                                      XtNumber( action_buttons ) );
  else
    copyMoveFd = xitCreateFormDialog( parent, "CopyMoveFd",
                                      2, 0,
                                      action_buttons,
                                      XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_MOVE_COPY_TITLE ) ); n++;
  XtSetValues( XtParent( copyMoveFd ), args, n );

  XtAddCallback( copyMoveFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) cm_ref );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( copyMoveFd, "CopyMoveFdFo" );


  /* Label indication what is moved. */
  messageLa = xitCreateLabel( workFo, "MessageLa",
                              msgGetText( MXDI_MOVE_COPY_APPOINTMENT ), -1 );


  /* Move/Copy to date and time. */
  if( cm_ref -> entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE )
    toTimeTb = xtmFlCreateDateField( workFo, "ToTimeTb" );
  else
    toTimeTb = xtmFlCreateDateTimeField( workFo, "ToTimeTb" );


  /* Move/Copy to database. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  dbRc = XmCreateRowColumn( workFo, "DbRc", args, n );

  databaseLa = xitCreateLabel( dbRc, "DatabaseLa", 
                               msgGetText( MXDI_DATABASE_LABEL ), -1 );

  /* Database selections. */
  databaseFs = xtmFlCreateDbField( dbRc, "DatabaseFs",
                                   custom_data_ref -> cal_db_handle,
                                   True,
                                   NULL, NULL );


  /* Place the elements together. */
  xitAttachWidget( messageLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( toTimeTb,
                   XmATTACH_WIDGET, messageLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( dbRc,
                   XmATTACH_WIDGET, toTimeTb, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( messageLa, args, n );
  XtSetValues( toTimeTb,  args, n );
  XtSetValues( dbRc,      args, n );


  /* Manage all the children except the date and time field. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( copyMoveFd, True );


  return( copyMoveFd );

} /* createCopyMoveWindow */


/*----------------------------------------------------------------------*/

static Boolean
  moveOrCopyEntry( CM_REC_REF  cm_ref )
{

  /* Variables. */
  Boolean                 ok;
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = cm_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( cm_ref -> copyMoveW, "CopyMoveFdFo" );


  /* Fetch the date and time. */
  tempW = XtNameToWidget( mainW, "ToTimeTb"  );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False,
                       &cm_ref -> to_date );
  if( ! ok )
    return( False );

  if( cm_ref -> entry_ref -> entry.entry_type != XTM_DB_DAY_NOTE ) {
    ok = xtmFoFetchTime( mainW, tempW, XTM_FO_START_TIME, False,
                         &cm_ref -> to_time );
    if( ! ok )
      return( False );
  }


  /* Fetch selected database. */
  tempW = XtNameToWidget( mainW, "DbRc.DatabaseFs" );

  xitFieldSelectGetCurrent( tempW, &char_ref );
  if( char_ref == NULL )
    return( False );

  if( *char_ref == '\0' ) {
    xitErMessage( mainW, XIT_ER_ERROR,
                  module_name, "moveOrCopyEntry",
                  msgGetText( MXDI_NO_DB_SELECTED ) );

    SysFree( char_ref );
    return( False );
  }

  strcpy( cm_ref -> to_cal_name, char_ref );
  SysFree( char_ref );


  /* Do the copy operation. */
  doMoveCopyCB( NULL, cm_ref, NULL );


  return( True );

} /* moveOrCopyEntry */


/*----------------------------------------------------------------------*/

static Boolean
  readEntry( Widget      parent,
             CM_REC_REF  cm_ref,
             char        *cal_name,
             UINT32      entry_id )
{

  /* Variables. */
  Boolean                 ok;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;


  /* Code. */

  appl_data_ref = cm_ref -> appl_data_ref;


  /* Data to hold the entry. */
  cm_ref -> entry_ref = SysNew( XTM_DB_ALL_ENTRY_DEF );
  cm_ref -> entry_ref -> all_text = NULL;


  /* Open the database and fetch the entry. */
  ok = xtmDmOpenDatabase( appl_data_ref, cal_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    raise exception;


  /* Fetch the entry. */
  db_status = xtmDbFetchEntry( &database, entry_id, 
                               cm_ref -> entry_ref, 
                               &cm_ref -> entry_ref -> all_text );

  xtmDbCloseEntryDb( &database );


  if( db_status != XTM_DB_OK ) {
    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "readEntry",
                  msgGetText( MXDI_NO_ENTRY ) );

    raise exception;
  }


  return( True );


  /* Exception handler. */
  exception:
    if( cm_ref -> entry_ref -> all_text != NULL )
      SysFree( cm_ref -> entry_ref -> all_text );

    SysFree( cm_ref -> entry_ref );

    return( False );

} /* readEntry */


/*----------------------------------------------------------------------*/

static void
  copyCB( Widget      widget,
          CM_REC_REF  cm_ref,
          XtPointer   call_data )
{

  /* Variables. */
  Boolean  ok;


  /* Code. */

  /* Do the copy operation. */
  cm_ref -> operation = XTM_CM_COPY;

  ok = moveOrCopyEntry( cm_ref );
  if( ! ok )
    return;

  XtDestroyWidget( cm_ref -> copyMoveW );


  return;

} /* copyCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget      widget,
             CM_REC_REF  cm_ref,
             XtPointer   call_data )
{

  /* Code. */

  /* Release the user data. */
  if( cm_ref -> entry_ref != NULL ) {
    SysFree( cm_ref -> entry_ref -> all_text );
    SysFree( cm_ref -> entry_ref );
  }

  SysFree( cm_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  doMoveCopyCB( Widget      widget,
                CM_REC_REF  cm_ref,
                XtPointer   call_data )
{

  /* Variables. */
  Boolean                 ok;
  UINT32                  id;
  XTM_DB_ALL_ENTRY_DEF    new_entry;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_ID_REQUEST       id_request;
  XTM_DB_STATUS           db_status;


  /* Code. */

  /* Open the database to save the entry. */
  ok = xtmDmOpenDatabase( cm_ref -> appl_data_ref,
                          cm_ref -> to_cal_name, XTM_DB_FLAG_MODE_WRITE,
                          &database );
  if( ! ok )
    return;


  /* Generate new entry id. */
  id_request.directory  = database.database_dir;
  id_request.operations = 0;
  id_request.lock_file  = False;

  db_status = xtmDbGenerateId( &id_request, &id );
  if( db_status != XTM_DB_OK )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "doMoveCopyEntry",
                  msgGetText( MXDI_ERRMSG_GENERATE_ID ) );


  /* The new entry is a copy of the old (including all_text). */
  memcpy( &new_entry, cm_ref -> entry_ref, sizeof( XTM_DB_ALL_ENTRY_DEF ) );

  new_entry.entry.id       = id;
  new_entry.stand_entry.id = id;

  new_entry.entry.date_stamp = cm_ref -> to_date;
  new_entry.entry.time_stamp = cm_ref -> to_time;

  /* New entries are not archived. */
  if( cm_ref -> operation == XTM_CM_COPY )
    flagClear( new_entry.entry.flags, XTM_DB_FLAG_ARCHIVED );


  /* Modify starting day for repeated entries. */
  if( new_entry.entry.entry_category == XTM_DB_REP_ENTRY_LIST ||
      new_entry.entry.entry_category == XTM_DB_STICKY_LIST ) {
    new_entry.stand_entry.from = cm_ref -> to_date;

    if( new_entry.stand_entry.to > 0 &&
        new_entry.stand_entry.to < new_entry.stand_entry.from )
      new_entry.stand_entry.to = 0;
  }


  /* Save the entry. */
  db_status = xtmDbInsertEntry( &database, &new_entry, new_entry.all_text );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "doMoveCopyEntry",
                  msgGetText( MXDI_ERRMSG_INSERT_ENTRY ) );

    xtmDbCloseEntryDb( &database );

    return;
  }


  /* If this a move operation, delete the old entry. */
  if( cm_ref -> operation == XTM_CM_MOVE ) {

    /* Different database? */
    if( strcmp( cm_ref -> to_cal_name,
                cm_ref -> entry_ref -> db_name ) != 0 ) {
      xtmDbCloseEntryDb( &database );

      ok = xtmDmOpenDatabase( cm_ref -> appl_data_ref,
                              cm_ref -> entry_ref -> db_name, 
                              XTM_DB_FLAG_MODE_WRITE,
                              &database );
      if( ! ok )
        return;
    }

    /* Remove the old entry. */
    (void) xtmDbDeleteEntry( &database, cm_ref -> entry_ref -> entry.id );

  } /* if */


  xtmDbCloseEntryDb( &database );


  /* Update the day list/calendar. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;

} /* doMoveCopyCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget      widget,
          CM_REC_REF  cm_ref,
          XtPointer   call_data )
{

  /* Code. */

  xtmHlDisplayHelp( cm_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    copy_move_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  moveCB( Widget      widget,
          CM_REC_REF  cm_ref,
          XtPointer   call_data )
{

  /* Variables. */
  Boolean  ok;


  /* Code. */

  /* Do the move. */
  cm_ref -> operation = XTM_CM_MOVE;

  ok = moveOrCopyEntry( cm_ref );
  if( ! ok )
    return;

  XtDestroyWidget( cm_ref -> copyMoveW );


  return;

} /* moveCB */
