/*-
 * @(#)ttfont.c -- Functions for handling FreeType library
 */

#include <stdio.h>
#include <stdlib.h>

#include "ttfont.h"
#include "chconv.h"

/* Show debug messages */
#define dprintf(args)	 fprintf args

static int freetype_initcount = 0;
static TT_Engine engine;
static TT_Error error;

int
ttfont_init()
{
    if (freetype_initcount == 0) {
	if (error = TT_Init_FreeType(&engine)) {
	    dprintf((stderr, "ttfont_init(): %s\n", TT_ErrToString18(error)));
	    return 0;
	}
	dprintf((stderr, "ttfont_init(): FreeType initialized.\n"));
    }
    freetype_initcount++;

    return 1;
}

void
ttfont_deinit()
{
    if (freetype_initcount <= 0)
	return;
    if (--freetype_initcount == 0) {
	TT_Done_FreeType(engine);
	dprintf((stderr, "ttfont_deinit(): FreeType done.\n"));
    }
}

FontFaceHints *
ttfont_load(char *fontfile, double point)
{
    FontFaceHints *hints;
    TT_Face face;
    TT_Face_Properties properties;
    TT_UShort num_faces;

    hints = (FontFaceHints *)calloc(sizeof(FontFaceHints), 1);
    if (!hints) {
	dprintf((stderr,
		 "ttfont_load(): Could not allocate FontFaceHints.\n"));
	return NULL;
    }

    hints->fontfile   = fontfile;
    hints->name	      = NULL;		/* Filled with full font's name */
    hints->resolution = 96;		/* Get from anywhare */
    hints->point      = point;
    hints->pixel      = hints->point * hints->resolution / 72;

    if (error = TT_Open_Face(engine, hints->fontfile, &face))
	goto abort;

    if (error = TT_Get_Face_Properties(face, &properties)) {
	TT_Close_Face(face);
	goto abort;
    }

    hints->num_faces  = properties.num_Faces;
    hints->ttcno      = 0;

    dprintf((stderr, "ttfont_load(): Font loaded successfully.\n"));
    TT_Close_Face(face);

    return hints;

abort:
    dprintf((stderr, "ttfont_load(): %s\n", TT_ErrToString18(error)));
    free(hints);

    return NULL;
}

void
ttfont_unload(FontFaceHints *hints)
{
    free(hints);
    dprintf((stderr, "ttfont_unload(): Font unloaded.\n"));
}

FontFaceInfo *
ttfont_open_face(FontFaceHints *hints)
{
    FontFaceInfo *info;
    TT_UShort num_face, num_charmaps;
    TT_UShort platform_id, encoding_id;
    int	i;

    num_face = hints->ttcno;
    if (num_face > hints->num_faces)
	num_face = 0;

    info = (FontFaceInfo *)calloc(sizeof(FontFaceInfo), 1);
    if (!info) {
	dprintf((stderr,
		 "ttfont_open_face(): Could not allocate FontFaceInfo.\n"));
	return NULL;
    }

    /* function as code converter */
    info->chconv_func = NULL;
    info->chconv_rev_func = NULL;
    for (i=0; i<256; i++) info->advance[i] = -1;
    
    if (error = TT_Open_Collection(engine,
				   hints->fontfile, /* FontFaceHints */
				   num_face,
				   &info->face)) {  /* FontFaceInfo */
	dprintf((stderr, "ttfont_open_face(): %s\n", TT_ErrToString18(error)));
	return NULL;
    }

    dprintf((stderr,
	     "ttfont_open_face(): Face %d (of %d) opened.\n",
	     num_face + 1, hints->num_faces));

    if (error = TT_Get_Face_Properties(info->face, &info->properties)) {
	dprintf((stderr, "ttfont_open_face(): %s\n", TT_ErrToString18(error)));
	goto close_face;
    }

    if (error = TT_New_Instance(info->face, &info->instance)) {
	dprintf((stderr, "ttfont_open_face(): %s\n", TT_ErrToString18(error)));
	goto close_face;
    }

    if (error = TT_Set_Instance_Resolutions(info->instance,
					    hints->resolution,
					    hints->resolution)) {
	dprintf((stderr, "ttfont_open_face(): %s\n", TT_ErrToString18(error)));
	goto done_instance;
    }
    if (error = TT_Set_Instance_CharSize(info->instance,
					 hints->point * 64)) {
	dprintf((stderr, "ttfont_open_face(): %s\n", TT_ErrToString18(error)));
	goto done_instance;
    }

    TT_Get_Instance_Metrics(info->instance, &info->imetrics);

    if (error = TT_New_Glyph(info->face, &info->glyph)) {
	dprintf((stderr, "ttfont_open_face(): %s\n", TT_ErrToString18(error)));
	goto done_glyph;
    }

    num_charmaps = TT_Get_CharMap_Count(info->face);
    for (i = 0; i < num_charmaps; i++) {
	TT_Get_CharMap_ID(info->face, i, &platform_id, &encoding_id);
	dprintf((stderr,
		 "ttfont_open_face(): Found charmap platform: %d, encoding: %d.\n",
		 platform_id, encoding_id));

	/* In case of platform is Microsoft */
	if (platform_id == 3) {
	    info->chconv_func =
		(encoding_id == 1) ? chconv_euc_to_unicode :
		(encoding_id == 2) ? chconv_euc_to_sjis : NULL;
	    info->chconv_rev_func =
		(encoding_id == 1) ? chconv_unicode_to_euc :
		(encoding_id == 2) ? chconv_sjis_to_euc : NULL;
	}

	/* Found handlable charmap */
	if (info->chconv_func && info->chconv_rev_func) {
	    TT_Get_CharMap(info->face, i, &info->charmap);
	    break;	
	}
    }

    if (i == num_charmaps) {
	dprintf((stderr, "ttfont_open_face(): Not found handlable charmap.\n"));
	goto done_glyph;
    }

    return info;

done_glyph:
    TT_Done_Glyph(info->glyph);
done_instance:
    TT_Done_Instance(info->instance);
close_face:
    TT_Close_Face(info->face);
abort:
    return NULL;
}

void
ttfont_close_face(FontFaceInfo *info)
{
    TT_Done_Glyph(info->glyph);
    TT_Done_Instance(info->instance);
    TT_Close_Face(info->face);
    dprintf((stderr, "ttfont_close_face(): Face closed.\n"));
}

TT_Raster_Map *
ttfont_create_raster(int width, int height, int is_pixmap)
{
    TT_Raster_Map *rmap;

    rmap = (TT_Raster_Map *)malloc(sizeof(TT_Raster_Map));
    if (!rmap)
	return NULL;

    rmap->rows = height;
    rmap->width = width;
    rmap->flow = TT_Flow_Down;
    if (is_pixmap)
	rmap->cols = (rmap->width + 3) & -4;
    else
	rmap->cols = (rmap->width + 7) >> 3;
    rmap->size = rmap->rows * rmap->cols;
    if (rmap->size > 0)
	rmap->bitmap = (char *)calloc(rmap->size, 1);
    else
	rmap->bitmap = NULL;

    return rmap;    
}

void
ttfont_delete_raster(TT_Raster_Map *rmap)
{
    if (!rmap)
	return;
    free(rmap->bitmap);
    free(rmap);
}

TT_Raster_Map *
ttfont_get_font(FontFaceInfo *info, int code, int smoothing)
{
    TT_Raster_Map *rmap;
    TT_UShort internal_code;
    TT_Pos tx, ty;
    int	depth, upm;

    depth = info->properties.header->yMin;
    upm = info->properties.header->Units_Per_EM;

    if (code >= 0x80) code = info->chconv_func(code);
    internal_code = TT_Char_Index(info->charmap, code);
    if (!internal_code)
	internal_code = TT_Char_Index(info->charmap, ' ');
	
    rmap = ttfont_create_raster(info->imetrics.y_ppem,
				info->imetrics.x_ppem,
				smoothing);
    if (!rmap)
	return NULL;
    
    TT_Load_Glyph(info->instance, info->glyph, internal_code, TTLOAD_DEFAULT);

    tx = info->imetrics.x_ppem / 100;
    ty = -depth * info->imetrics.y_ppem / upm;

    if (smoothing)
	TT_Get_Glyph_Pixmap(info->glyph, rmap, tx * 64, ty * 64);
    else
	TT_Get_Glyph_Bitmap(info->glyph, rmap, tx * 64, ty * 64);

    return rmap;
}

int ttfont_advance(FontFaceInfo* info, int code) {
	int internal_code; TT_Glyph_Metrics metrics;
	if (code > 0x80) return info->imetrics.y_ppem;
	if (info->advance[code] != -1) return info->advance[code];
	internal_code = TT_Char_Index(info->charmap, code);
	TT_Load_Glyph(info->instance, info->glyph, internal_code, TTLOAD_DEFAULT);
	TT_Get_Glyph_Metrics(info->glyph, &metrics);
	info->advance[code] = metrics.advance / 64;
	return info->advance[code];
}

char *
ttfont_render_font(FontFaceInfo *info,
		   int *width, int *height, int code, int smoothing)
{
    TT_Raster_Map *rmap, *rtmp;
    char *bitmap;
    char *src, *dest;
    int	x, y, b;

    rmap = ttfont_get_font(info, code, smoothing);
    if (!rmap)
	return NULL;

    bitmap = (char *)malloc(rmap->width * rmap->rows);
    if (!bitmap) {
	ttfont_delete_raster(rmap);
	return NULL;
    }

    for (y = 0; y < rmap->rows; y++) {
	src = rmap->bitmap + y * rmap->cols;
	dest = bitmap + y * rmap->width;
	for (x = 0; x < rmap->cols; x++) {
	    if (smoothing)
	        *dest++ = *src++;
	    else {
	        for (b = 0x80; b > 0; b >>= 1)
		    *dest++ = (*src & b) ? 4 : 0;
		src++;
	    }
	}
    }
    *width = rmap->width;
    *height = rmap->rows;

    ttfont_delete_raster(rmap);

    return bitmap;
}

/*
 * ttfont_is_usable_japanese() returns 1 if the font file contains
 * Japanese fonts, otherwise (maybe file has roman fonts) returns 0.
 * -- Kazunori Ueno <jagarl@creater.club.ne.jp>
 */
int
ttfont_is_usable_japanese(FontFaceInfo *info)
{
    TT_CharMap charmap = info->charmap;

    return (TT_Char_Index(charmap, info->chconv_func(0xa3c1)) > 0) &&
	   (TT_Char_Index(charmap, info->chconv_func(0xa4a2)) > 0) &&
	   (TT_Char_Index(charmap, info->chconv_func(0xa5c2)) > 0) &&
	   (TT_Char_Index(charmap, info->chconv_func(0xb0a1)) > 0); 
}

/*
 * ttfont_read_name_string() gets full font's name. 
 * -- Kazunori Ueno <jagarl@creater.club.ne.jp>
 */
int
ttfont_read_name_string(FontFaceInfo *info, char *dest, int dest_len)
{
    TT_UShort num_names;
    int	i, j;
    TT_UShort platform_id, encoding_id, language_id, name_id;
    char *name;
    TT_UShort name_len;

    num_names = TT_Get_Name_Count(info->face);
    for (i = 0; i < num_names; i++) {
	if (error = TT_Get_Name_ID(info->face, i,
				   &platform_id, &encoding_id,
				   &language_id, &name_id)) {
	    dprintf((stderr,
		     "ttfont_read_name_string(): %s\n", TT_ErrToString18(error)));
	    continue;
	}
	if (name_id != 4)
	    continue;

	if (error = TT_Get_Name_String(info->face, i, &name, &name_len)) {
	    dprintf((stderr,
		     "ttfont_read_name_string(): %s\n", TT_ErrToString18(error)));
	    continue;
	}
	if (platform_id == 3 &&
	    (encoding_id == 1 || encoding_id == 2)) {
	    char *p = dest;
	    int len = (dest_len > name_len) ? name_len : dest_len;
	    for (j = 0; j < len; j += 2) {
		unsigned int c1 = name[j] & 0xff;
		unsigned int c2 = name[j + 1] & 0xff;
		if (c1 == 0)
		    *p++ = c2;
		else {
		    c1 = info->chconv_rev_func((c1 << 8) | c2);
		    *p++ = c1 >> 8;
		    *p++ = c1 & 0xff;
		}
	    }
	    dest[j] = '\0';
	    if (language_id == 0x411)
		break;
	}
    }

    return 1;
}

/* ttfont.c ends here */
