// -*- c++ -*-

/*
 *  Author: Arvin Schnell
 */


#ifndef pcm_h
#define pcm_h


#include <stdint.h>


class PCM
{

public:

    enum pcm_type_t { DEVICE, FILE };
    enum pcm_mode_t { READ, WRITE };
    enum pcm_format_t { S8, S16, S32 };

    /**
     * Create a PCM object.  The device driver accepts the string "default" as
     * name for a reasonable default.  buffer_size is in frames.
     */
    static PCM* create (const char* name, pcm_type_t type, pcm_mode_t mode,
			pcm_format_t format, int channels, int rate,
			int buffer_size);

    /**
     * Create a PCM object.
     */
    static PCM* create (const char* name, pcm_type_t type, pcm_mode_t mode,
			pcm_format_t format, int channels, int rate);

    /**
     * Destructor.
     */
    virtual ~PCM ();

    /**
     * Displays some information on cerr.
     */
    virtual void info () const = 0;

    /**
     * Only useful for files.  Returns size of file in frames.  On error -1 is
     * returned.
     */
    virtual size_t size () = 0;

    /**
     * Only useful for files.  Reposition file offset.  On error -1 is
     * returned.
     */
    virtual off_t seek (off_t offset, int whence) = 0;

    /**
     * Returns number of frames read (may be 0).  On error -1 is returned.
     */
    virtual size_t read (void* buffer, size_t frames) = 0;

    /**
     * Returns number of frames written.  On error -1 is returned.
     */
    virtual size_t write (void* buffer, size_t frames) = 0;

    /**
     * Returns bytes per frame.
     */
    int get_bytes_per_frame () const { return channels * (1 << format); }

    /**
     * Returns buffer size in frames.
     */
    int get_buffer_size () const { return buffer_size; }

protected:

    PCM (const char* name, pcm_type_t type, pcm_mode_t mode,
	 pcm_format_t format, int channels, int rate, int buffer_size);

    bool ok;

    char* name;
    pcm_type_t type;
    pcm_mode_t mode;
    pcm_format_t format;
    int channels;
    int rate;

    int buffer_size;

    void swap_buffer (void* buffer, size_t frames) const;

    uint16_t uint16_to_le (uint16_t i) const;
    uint32_t uint32_to_le (uint32_t i) const;

};


#endif
