

/*
 *  Author: Arvin Schnell
 */


#include <string.h>
#include <stdlib.h>
#ifdef __NetBSD__
#  include <machine/byte_swap.h>
#elif __FreeBSD__
#  include <sys/endian.h>
#else
#  include <byteswap.h>
#endif
#include <iostream>

using std::cerr;

#include "pcm.h"
#include "pcm-cdr.h"
#include "pcm-wav.h"
#include "pcm-oss.h"
#ifdef HAVE_ALSA
#  include "pcm-alsa.h"
#endif


#ifndef __BYTE_ORDER
#  error "undefined endianness"
#elif __BYTE_ORDER != __BIG_ENDIAN && __BYTE_ORDER != __LITTLE_ENDIAN
#  error "unsupported endianness"
#endif


PCM*
PCM::create (const char* name, pcm_type_t type, pcm_mode_t mode,
	     pcm_format_t format, int channels, int rate, int buffer_size)
{
    switch (type)
    {
	case DEVICE: {

	    const char* tmp = name;

#ifdef HAVE_ALSA
	    if (strcmp (tmp, "default") == 0)
		tmp = "plughw:0,0";
#else
#  if defined(__NetBSD__) || defined(__FreeBSD__)
	    if (strcmp (tmp, "default") == 0)
		tmp = "/dev/dsp";
#  else
	    if (strcmp (tmp, "default") == 0)
		tmp = "/dev/audio";
#  endif
#endif

	    if (strncmp (tmp, "/dev/", 5) == 0) {
		PCM* pcm = new PCMOSS (tmp, type, mode, format, channels, rate, buffer_size);
		if (!pcm->ok) {
		    delete pcm;
		    pcm = 0;
		}
		return pcm;
	    }

#ifdef HAVE_ALSA
	    if (strncmp (tmp, "plughw:", 7) == 0) {
		PCM* pcm = new PCMALSA (tmp, type, mode, format, channels, rate, buffer_size);
		if (!pcm->ok) {
		    delete pcm;
		    pcm = 0;
		}
		return pcm;
	    }
#endif

	    cerr << "error: Unknown name for pcm interface\n";
	    return 0;

	} break;

	case FILE: {

	    int len = strlen (name);

	    if (len > 4 && strcmp (&name[len - 4], ".cdr") == 0) {
		PCM* pcm = new PCMCDR (name, type, mode, format, channels, rate, buffer_size);
		if (!pcm->ok) {
		    delete pcm;
		    pcm = 0;
		}
		return pcm;
	    }

	    if (len > 4 && strcmp (&name[len - 4], ".wav") == 0) {
		PCM* pcm = new PCMWAV (name, type, mode, format, channels, rate, buffer_size);
		if (!pcm->ok) {
		    delete pcm;
		    pcm = 0;
		}
		return pcm;
	    }

	    cerr << "error: Unknown name for pcm interface\n";
	    return 0;

	} break;

	default:
	    cerr << "error: Illegal type for pcm interface\n";
	    return 0;
    }
}


PCM*
PCM::create (const char* name, pcm_type_t type, pcm_mode_t mode,
	     pcm_format_t format, int channels, int rate)
{
    return create (name, type, mode, format, channels, rate, 1);
}


PCM::PCM (const char* name, pcm_type_t type, pcm_mode_t mode,
	  pcm_format_t format, int channels, int rate, int buffer_size)
    : name (strdup (name)),
      type (type),
      mode (mode),
      format (format),
      channels (channels),
      rate (rate),
      buffer_size (buffer_size)
{
    ok = false;
}


PCM::~PCM ()
{
    free (name);
}


void
PCM::info () const
{
    cerr << "name: " << name << '\n'
	 << "type: " << (type == DEVICE ? "device" : "file") << '\n'
	 << "mode: " << (mode == READ ? "read" : "write") << '\n'
	 << "format: signed " << (8 << format) << " bits\n"
	 << "channels: " << channels << '\n'
	 << "rate: " << rate << " Hz\n"
	 << "buffer size: " << buffer_size << " frames, "
	 << buffer_size * get_bytes_per_frame () << " bytes, "
	 << 1000.0 * buffer_size / rate << " ms\n";
}


void
PCM::swap_buffer (void* buffer, size_t frames) const
{
    size_t n = frames * channels;

    switch (format)
    {
	case S8: {
	} break;

	case S16: {
	    int16_t* tmp_buffer = (int16_t*) buffer;
	    for (size_t i = n; i != 0; i--) {
#if defined(__NetBSD__) || defined(__FreeBSD__)
		*tmp_buffer = bswap16 (*tmp_buffer);
#else
		*tmp_buffer = bswap_16 (*tmp_buffer);
#endif
		tmp_buffer++;
	    }
	} break;

	case S32: {
	    int32_t* tmp_buffer = (int32_t*) buffer;
	    for (size_t i = n; i != 0; i--) {
#if defined(__NetBSD__) || defined(__FreeBSD__)
		*tmp_buffer = bswap32 (*tmp_buffer);
#else
		*tmp_buffer = bswap_32 (*tmp_buffer);
#endif
		tmp_buffer++;
	    }
	} break;
    }
}


uint16_t
PCM::uint16_to_le (uint16_t i) const
{
#if __BYTE_ORDER == __BIG_ENDIAN
#  if defined(__NetBSD__) || defined(__FreeBSD__)
    return bswap16 (i);
#  else
    return bswap_16 (i);
#  endif
#else
    return i;
#endif
}


uint32_t
PCM::uint32_to_le (uint32_t i) const
{
#if __BYTE_ORDER == __BIG_ENDIAN
#  if defined(__NetBSD__) || defined(__FreeBSD__)
    return bswap32 (i);
#  else
    return bswap_32 (i);
#  endif
#else
    return i;
#endif
}
