#!/usr/bin/ruby
#
# Author:: Oliver M. Bolzer <oliver@fakeroot.net>
# Copyright:: Copyright (c) 2002 Oliver M. Bolzer. All rights reserved.
# Licence:: Ruby licence.

require 'test/unit'
require 'dbi'
require 'vapor/tuplemgr'
require 'vapor/persistable'

# dummy class for testing
class Person
  include Vapor::Persistable
end
class City
  include Vapor::Persistable
end
class Student
  include Vapor::Persistable
end

## test class for TulpeManager
class Test_TupleManager < Test::Unit::TestCase

  include Vapor
  include Vapor::Exceptions

  # the db mus be already set up
  DBNAME = 'bolzer_test'
  DBHOST = 'db'
  DBPORT = '5432'
  DBUSER = 'bolzer'
  DBPASS = 'foo02bar'

  Testdata = 'testdata.sql'

  ## preparation, whip stuff into RDBMS
  def setup

    ## connect to database
    driver_url = [ 'DBI', 'pg', DBNAME, DBHOST ].join(':')
    @dbh = DBI.connect( driver_url, DBUSER, DBPASS )
    
    ## read in test data 
    testdata = File.open( File.dirname(__FILE__) + '/' + Testdata ){|file| file.readlines } 
    testdata.each{|line| line.chomp!; line.sub!( /--.*$/, '' )}  ## remove comments
    testdata.delete('')                                          ## remove comment-only lines
    testdata = testdata.join(' ')

    begin
       @dbh.execute( 'SET CLIENT_MIN_MESSAGES TO ERROR;' )
    rescue # ignore error happening on PostgreSQL <= 7.3
    end
    
    ## put data into database
    testdata.split(';').each{ |statement| @dbh.execute( statement + ";" ) } 
    
    ## instanciate TupleManager
    @mgr = TupleManager.new( DBNAME, DBUSER, DBPASS, DBHOST, DBPORT )

    ## set default OID of transaction log object, to make all tests run smoothly
    @mgr.transaction_log = 0
  end # setup()

  def teardown

    ## delete all talbes
    leaftables = Array.new
    parenttables = Array.new
    @dbh.execute( "SELECT relname, relhassubclass FROM pg_class WHERE relkind = 'r' AND relname !~ '^pg_';").each{|row|
      if row[1] then
        parenttables << row[0]
      else 
        leaftables << row[0]
      end
    }
    leaftables.each{|table| @dbh.execute( 'DROP TABLE "' + table + '";' ) }
    parenttables.each{|table| @dbh.execute( 'DROP TABLE "' + table + '";' ) }

    ## delete all sequences
    sequences = Array.new
    @dbh.execute( "SELECT relname FROM pg_class WHERE relkind = 'S' AND relname !~ '^pg_';").each{|row|
      sequences << row[0] 
    }
    sequences.each{|seq|
      @dbh.execute( 'DROP SEQUENCE "' + seq + '";' )
    }

  end # teardown()

  ## test constructor
  def test_initialize

    ## test correct number of arguments
    assert_raises( ArgumentError ){ TupleManager.new }
    assert_raises( ArgumentError ){ TupleManager.new( 'db' ) }
    assert_raises( ArgumentError ){ TupleManager.new( 'db','user') }
    # 4 or 5 arguments is correct
    assert_raises( ArgumentError ){ TupleManager.new( 'db','user','pass','host', 1234, 'toomuch') }

    # test refusal to connect to Repository with incorrent schema-version
    @dbh.execute( %!UPDATE ":Vapor::RepositoryInfo" SET value = '99' WHERE name = 'schema-version'! )
    assert_raises( RepositoryOfflineError ){ 
      @mgr = TupleManager.new( DBNAME, DBUSER, DBPASS, DBHOST, DBPORT )
    }

  end # test_initialize()

  ## test retrieval of objects
  def test_get_tuple
  
    ## check method invocation
    assert_respond_to( @mgr, "get_tuple" )
    assert_raises( ArgumentError ){ @mgr.get_tuple }
    assert_raises( ArgumentError ){ @mgr.get_tuple( 1234, 1234 ) }
    assert_raises( TypeError ){ @mgr.get_tuple( 'foo' ) }

    ## try retrieving an non-existent tuple
    assert_nil( @mgr.get_tuple( 0 ) ) 

    object = nil
    assert_nothing_raised{
      ## retrieve a Person ( 51987, 'Big', 'Seal' ) 
      object = @mgr.get_tuple( 51987 )
    }      
      assert_kind_of( Hash, object )
      assert_equal( object['_type'], ::Person )
      assert_equal( object['_oid'], 51987 )
      assert_equal( object['first_name'], 'Big' )
      assert_equal( object['last_name'], 'Seal' )
      assert_equal( Time.utc( 1979, 2, 28, 0, 0, 0 ), object['birthday'] )
    

  end # test_get_tuple()

  # test retrieval of class metadata, in this case the list
  # of a class' attributes of the reference type
  def test_get_class_attributes
    # correct way to call is one String argument
    assert_respond_to( @mgr, 'get_class_attributes' )
    assert_raises( ArgumentError ){ @mgr.get_class_attributes() }
    assert_raises( ArgumentError ){ @mgr.get_class_attributes( "foo", "bar" ) }
    
    # return nil on non-existing class
    assert_nil( @mgr.get_class_attributes( "ASDF" ) )
   
    # test base classes Person City
    result = nil
    result = @mgr.get_class_attributes( "Person" )
    assert_kind_of( Array, result )
    assert_equal( 6, result.size )
    result.each{|r| assert_kind_of( Vapor::ClassAttribute, r ) }

    result = nil
    result = @mgr.get_class_attributes( "City" )
    assert_kind_of( Array, result )
    assert_equal( 2, result.size )
    result.each{|r| assert_kind_of( Vapor::ClassAttribute, r ) }

    # test class with inheritance
    result = nil
    assert_nothing_raised{ result = @mgr.get_class_attributes( "Student" ) }
    assert_kind_of( Array, result )
    assert_equal( 7, result.size )
    result.each{|r| assert_kind_of( Vapor::ClassAttribute, r ) }
    
  end # test_get_class_attributes()

  # test retrieval of all objects from a class
  def test_get_all_instances
    
    # correct way to call is one Class or fully qualified class name
    # and a flag for inclusion of subclasses
    assert_respond_to( @mgr, :get_all_instances )
    assert_raises( ArgumentError ){ @mgr.get_all_instances() }
    assert_raises( ArgumentError ){ @mgr.get_all_instances( nil ) }
    assert_raises( ArgumentError ){ @mgr.get_all_instances( nil, nil, nil ) }
    assert_raises( ArgumentError ){ @mgr.get_all_instances( "" ) }

    # search without subclasses
    result = nil
    assert_nothing_raised{ result = @mgr.get_all_instances( 'Person', false ) }
    assert_kind_of( Array, result )
    assert_equal( 2, result.size )
    assert( result.include?( 51987 ) )
    assert( result.include?( 40125 ) )

    # search with subclasses
    result = nil
    assert_nothing_raised{ result = @mgr.get_all_instances( 'Person', true ) }
    assert_kind_of( Array, result )
    assert_equal( 3, result.size )
    assert( result.include?( 51987 ) )
    assert( result.include?( 40125 ) )
    assert( result.include?( 228299 ) )

  end # test_get_all_instances()

  # test retrieval of array's as part of tuples 
  def test_get_tuple_with_arrays
    assert_respond_to( @mgr, "get_tuple" )

    ## retrieve a Persen with nicknames ( 51987, 'Big', 'Seal' ) => [biggi,Godzilla]
    object = nil
    assert_nothing_raised{ object = @mgr.get_tuple( 51987 ) }
  
    assert_equal( ["biggi", "Godzilla" ], object['nickNames'] )

  end # test_get_tuple_with_arrays()

  # test insertation of new object
  def test_new_tuple
    assert_respond_to( @mgr, :new_tuple )
    assert_raises( ArgumentError ){ @mgr.new_tuple }
    assert_raises( ArgumentError ){ @mgr.new_tuple( Class ) }
    assert_raises( ArgumentError ){ @mgr.new_tuple( Class, 1233, nil ) }
    assert_raises( TypeError ){ @mgr.new_tuple( String, 1233 ) }

    assert_nothing_raised{ @mgr.new_tuple( Person, 123 ) }

    ## check if data is correctly saved

    # Object in object list with correct type
    klass = @dbh.execute( 'SELECT _oid, _class FROM ":Vapor::ObjectList" WHERE _oid = 123;' )
    klass = klass.fetch
    assert_not_nil( klass )
    assert_equal( 123, klass[0] )
    assert_equal( 'Person', klass[1] )

    # object in class' table, everything NULL except oid
    obj = @dbh.execute( 'SELECT * FROM "Person" WHERE _oid = 123;' )
    obj = obj.fetch_hash
    assert_not_nil( obj )
    assert_equal( 123, obj['_oid'] )
    assert_equal( 0, obj['_revision'] )
    
    obj.each{ |key, value|
      unless key == '_oid' or key == '_revision' then
        assert_nil( value )
      end
    }

    # object in unknwon class
    klass = Class.new
    klass.__send__ :include, Persistable
    assert_raises( ClassNotKnownError ){ @mgr.new_tuple( klass, 456 ) }
  end # test_new_tuple()

  # test updating of tuple
  def test_update_tuple
    ## correct calling convention: TupleManager#update_tuple( Class, Integer, Hash)
    
    # test number of arguments
    assert_raises( ArgumentError ){ @mgr.update_tuple }
    assert_raises( ArgumentError ){ @mgr.update_tuple( Class ) }
    assert_raises( ArgumentError ){ @mgr.update_tuple( Class, 123 ) }
    assert_raises( ArgumentError ){ @mgr.update_tuple( Class, 123, {}, nil ) }
    
    # test type of arguments
    assert_raises( TypeError ){ @mgr.update_tuple( Object, 1169, {} ) }
    assert_raises( TypeError ){ @mgr.update_tuple( City, "foo", {} ) }
    assert_raises( TypeError ){ @mgr.update_tuple( City, 1169, nil ) }

    # set OID of transaction log belonging to current transaction
    log_oid = rand( 1000 )
    assert_nothing_raised{ @mgr.transaction_log = log_oid }

    # the real thing
    assert_nothing_raised{ @mgr.update_tuple( City, 1169, { :name => 'Antarktikus', :altitude => 100, '_revision' => 1 } ) }

    # test if correctly recorded
    obj = @dbh.execute( 'SELECT * FROM "City" WHERE _oid = 1169;' )
    obj = obj.fetch_hash
    assert_not_nil( obj )
    assert_equal( 1169, obj['_oid'] )
    assert_equal( 2, obj['_revision' ] )
    assert_equal( log_oid, obj['_last_change'] )
    assert_equal( 'Antarktikus' , obj['name'] )
    assert_equal( 100, obj['altitude'] )

    # give error when tuple is not fount
    assert_raises( DeletedObjectError ){ @mgr.update_tuple( City, 1170, { :name => 'Foo', :altitude => 23, 'revision' => 1 } ) }
    
    # give error when tuple to be updated has already been updated
    # by somebody else
    assert_raises( StaleObjectError ){ @mgr.update_tuple( City, 64212, { :name => 'Arktiva', :altitude => 2, 'revision' => 4 } ) }

    # object in unknwon class
    klass = Class.new
    klass.__send__ :include, Persistable
    assert_raises( ClassNotKnownError ){ @mgr.update_tuple( klass, 456, { :foo => ''} ) }
  end # test_update_tuple()

  # test updating with array-type
  def test_update_tuple_array
    assert_respond_to( @mgr, :update_tuple )

    # save two array-attributes
    person_attrs = { :friends => [], :nickNames => [ 'foo', 'bar' ], '_revision' => 1 }
    assert_nothing_raised{ @mgr.update_tuple( Person, 51987, person_attrs ) }
  
    obj = @dbh.execute( 'SELECT * FROM "Person" WHERE _oid = 51987' )
    obj = obj.fetch_hash
    assert_not_nil( obj )
    assert_equal( 51987, obj['_oid'] )
    assert_equal( 2, obj['_revision'] )
    assert_equal( [], obj['friends'] )
    assert_equal( [ 'foo', 'bar' ], obj['nickNames'] )

  end # test_update_tuple_array()

  # test updating with uniqness violation
  def test_update_tuple_unique
    assert_respond_to( @mgr, :update_tuple )
    assert_respond_to( @mgr, :new_tuple )
    
    # two students with same ID, Student.student_id is UNIQUE
    student1 = { :student_id => 1, :first_name => "Peter", '_revision' => 0 }
    student2 = { :student_id => 1, :first_name => "John", '_revision' => 0 }

    # create two empty students and update the first
    assert_nothing_raised{
      @mgr.new_tuple( Student, 10617 )
      @mgr.new_tuple( Student, 10618 )
      @mgr.update_tuple( Student, 10617, student1 )
    }

    # try the second one, violating uniqueness constraints 
    assert_raises( UniquenessError ){
      @mgr.update_tuple( Student, 10618, student2 )
    }  
  end # test_update_tuple_unique()

  # test correct saving of old tuple into historic table
  def test_update_tuple_keep_historic
    assert_respond_to( @mgr, :update_tuple )
    assert_respond_to( @mgr, :get_tuple )
   
    # retrieve tuple before modification
    old_tuple = @mgr.get_tuple( 1169 )
    old_tuple.delete( '_type' )
    new_tuple = old_tuple.dup
    new_tuple['altitude'] = 99
    new_tuple.delete( '_oid' )

    # update tuple thing
    assert_nothing_raised{ @mgr.update_tuple( City, 1169, new_tuple ) }

    # cheack if old tuple is correctly saved in historic table 
    obj = @dbh.execute( %!SELECT * FROM "_City" WHERE _oid = 1169 and _revision = #{old_tuple['_revision']};! )
    obj = obj.fetch_hash
    assert_not_nil( obj )
    assert_equal( old_tuple, obj )

  end # test_update_tuple_keep_historic()

  # test next_oid_high()
  def test_next_oid_high

    assert_respond_to( @mgr, :next_oid_high )
    assert_raises( ArgumentError ){ @mgr.next_oid_high( nil ) }

    # get current next_oid_high value
    prev_value = @dbh.execute( %!SELECT currval('":Vapor::oid_high"')! )
    prev_value = prev_value.fetch[0]
    assert_not_nil( prev_value )
    assert_kind_of( Integer, prev_value)

    # now retrieve the next_oid_high
    oid_high = nil
    assert_nothing_raised{ oid_high = @mgr.next_oid_high }
    assert_kind_of( Integer, oid_high )
    
    # this value should have been increased by one
    assert_equal( prev_value + 1, oid_high )

  end #test_next_oid_high()

  # test deletion of objects
  def test_delete_tuple
    assert_respond_to( @mgr, :delete_tuple )
    assert_raises( ArgumentError ){ @mgr.delete_tuple }
    assert_raises( ArgumentError ){ @mgr.delete_tuple( Class ) }

    assert_nothing_raised{ @mgr.delete_tuple( Person, 51987, 1 ) }

    # check deletion from ObjectList
    obj = @dbh.execute( 'SELECT _oid FROM ":Vapor::ObjectList" WHERE _oid = 51987' )
    obj = obj.fetch
    assert_nil( obj )

    # check deletion from class' table
    obj = @dbh.execute( 'SELECT _oid FROM "Person" WHERE _oid = 51987' )
    obj = obj.fetch
    assert_nil( obj )

    # stale object case
    assert_raises( StaleObjectError ){ @mgr.delete_tuple( City, 64212, 4 ) }
    
    # object in unknwon class
    klass = Class.new
    klass.__send__ :include, Persistable
    assert_raises( ClassNotKnownError ){ @mgr.delete_tuple( klass, 456, 4 ) }
      
  end # test_delete_tuple()

  # test that historic copy is kept when tuple is deleted
  def test_delete_tuple_kepp_historic
    assert_respond_to( @mgr, :delete_tuple )
    assert_respond_to( @mgr, :get_tuple )
   
    # retrieve tuple before modification
    old_tuple = @mgr.get_tuple( 1169 )
    old_tuple.delete( '_type' )
    new_tuple = old_tuple.dup
    new_tuple.delete( '_oid' )

    # update tuple thing
    assert_nothing_raised{ @mgr.delete_tuple( City, 1169, old_tuple['_revision']) }

    # cheack if old tuple is correctly saved in historic table 
    obj = @dbh.execute( %!SELECT * FROM "_City" WHERE _oid = 1169 and _revision = #{old_tuple['_revision']};! )
    obj = obj.fetch_hash
    assert_not_nil( obj )
    assert_equal( old_tuple, obj )
  end # test_delete_tuple_kepp_historic()

  # test searching for tuples
  def test_search_tuples
    assert_respond_to( @mgr, :search_tuples )
    assert_raises( ArgumentError ){ @mgr.search_tuples }
    assert_raises( ArgumentError ){ @mgr.search_tuples( Class ) }
    assert_raises( ArgumentError ){ @mgr.search_tuples( Class , nil, nil,  nil ) }
    assert_raises( TypeError ){ @mgr.search_tuples( "foo", Vapor::BasicQueryStatement.new( :foo, '=', 1 ) ) }
    assert_raises( TypeError ){ @mgr.search_tuples(Class, "foo" ) }

    # search for a single object with a BasicQuery
    objects = nil
    statement = Vapor::BasicQueryStatement.new( :name, '=', 'Antarktikus' )
    assert_nothing_raised{ objects = @mgr.search_tuples( City, statement, false ) }
    assert_kind_of( Array, objects )
    assert_equal( 1, objects.size )
    assert_equal( [ 1169 ], objects )

    # two objects with a simple query
    objects = nil
    statement = Vapor::BasicQueryStatement.new( :altitude, '=', 1 )
    assert_nothing_raised{ objects = @mgr.search_tuples( City, statement, false ) }
    assert_kind_of( Array, objects )
    assert_equal( 2, objects.size )
    assert( objects.include?( 64212 ) )
    assert( objects.include?( 64213 ) )

    # a single object with a complex query
    st1 = Vapor::BasicQueryStatement.new( :altitude, '=', 1 ) 
    st2 = Vapor::BasicQueryStatement.new( :name, '=', 'Tokyo' )
    statement = Vapor::ComplexQueryStatement.new( st1, 'AND', st2 )
    assert_nothing_raised{ objects = @mgr.search_tuples( City, statement, false ) }
    assert_kind_of( Array, objects )
    assert_equal( 1, objects.size )
    assert_equal( [ 64213 ], objects )

    # no-result case
    statement = Vapor::BasicQueryStatement.new( :name, '=', 'Berlin' )
    assert_nothing_raised{ objects = @mgr.search_tuples( City, statement, false ) }
    assert_nil( objects )

    # search with and without subclasses
    statement = Vapor::BasicQueryStatement.new( :inhabits, '=', 64212 )
    
    assert_nothing_raised{ objects = @mgr.search_tuples( Person, statement, false ) }
    assert_kind_of( Array, objects )
    assert_equal( 2, objects.size )

    assert_nothing_raised{ objects = @mgr.search_tuples( Person, statement, true ) }
    assert_kind_of( Array, objects )
    assert_equal( 3, objects.size )

    # search for non-existent klass
    klass = Class.new
    klass.__send__ :include, Vapor::Persistable
    assert_raises( ClassNotKnownError){@mgr.search_tuples( klass, statement, false )}

  end # test_search_tuples()

  # test QueryStatement -> SQL conversion
  def test_query_to_sql
    assert_respond_to( @mgr, :query_to_sql )

    # simple statement with String
    statement = Vapor::BasicQueryStatement.new( :name, '=', 'Antarktis' )
    expected_string = %!"name" = 'Antarktis'!
    assert_equal( expected_string, @mgr.query_to_sql( statement ) )

    # simple statement with Integer
    statement = Vapor::BasicQueryStatement.new( :oid, '=', 1 )
    expected_string = "_oid = 1"
    assert_equal( expected_string, @mgr.query_to_sql( statement ) )

    # simple statement with Float
    statement = Vapor::BasicQueryStatement.new( :foo, '=', 3.14 )
    expected_string = %!"foo" = 3.14!
    assert_equal( expected_string, @mgr.query_to_sql( statement ) )

    # simple statement with Boolean
    statement = Vapor::BasicQueryStatement.new( :foo, '=', true )
    expected_string = %!"foo" = 't'!
    assert_equal( expected_string, @mgr.query_to_sql( statement ) )
    
    # simple statement with Date
    date = Time.now
    statement = Vapor::BasicQueryStatement.new( :foo, '=', date )
    expected_string = %!"foo" = '#{date.to_s}'!
    assert_equal( expected_string, @mgr.query_to_sql( statement ) )

    # simple statement with SIMILAR_TO operator
    statement = Vapor::BasicQueryStatement.new( :foo, '~', 'b?r*b%_az\?\*' )
    expected_string = %q!"foo" LIKE 'b_r%b\\\\%\\\\_az?*'!
    assert_equal( expected_string, @mgr.query_to_sql( statement ) )

    # complex statement
    st1 = Vapor::BasicQueryStatement.new( :name, '=', 'Tokyo' )
    st2 = Vapor::BasicQueryStatement.new( :altitude, '=', 1 ) 
    statement = Vapor::ComplexQueryStatement.new( st1, 'AND', st2 )
    expected_string = %!("name" = 'Tokyo') AND ("altitude" = 1)! 
    assert_equal( expected_string, @mgr.query_to_sql( statement ) )

    # slightly more complex statement
    st1 = Vapor::BasicQueryStatement.new( :first_name, '=', 'Big' )
    st2 = Vapor::BasicQueryStatement.new( :last_name, '=', 'Seal' )
    st3 = Vapor::ComplexQueryStatement.new( st1, 'AND', st2 )
    st4 = Vapor::BasicQueryStatement.new( :inhabits, '=', 64212 )
    statement = Vapor::ComplexQueryStatement.new( st3, 'AND', st4 )
    expected_string = %!(("first_name" = 'Big') AND ("last_name" = 'Seal')) AND ("inhabits" = 64212)!
    assert_equal( expected_string, @mgr.query_to_sql( statement ) )

  end # test_query_to_sql()

  # test transaction
  def test_transaction
    assert_respond_to( @mgr, :begin_transaction )
    assert_respond_to( @mgr, :rollback_transaction )
    assert_respond_to( @mgr, :commit_transaction )
    assert_respond_to( @mgr, :in_transaction? )

    # start a transaction
    assert_equal( false, @mgr.in_transaction? )
    assert_nothing_raised{ @mgr.begin_transaction }
    assert_equal( true, @mgr.in_transaction? )

    # can't start second withcout finishing first
    assert_raises( NestedTransactionError ){ @mgr.begin_transaction }
    
    # let's abort it
    assert_equal( true, @mgr.in_transaction? )
    assert_nothing_raised{ @mgr.rollback_transaction }
    assert_equal( false, @mgr.in_transaction? )
   
    # can't rollback if nothing going on
    assert_raises( StaleTransactionError ){ @mgr.rollback_transaction }
    assert_equal( false, @mgr.in_transaction? )

    # can't commit non-running transaction
    assert_raises(  StaleTransactionError ){ @mgr.commit_transaction }

    # start a new transaction we can commit
    assert_nothing_raised{ @mgr.begin_transaction }
    assert_equal( true, @mgr.in_transaction? )
    assert_nothing_raised{ @mgr.commit_transaction }
    assert_equal( false, @mgr.in_transaction? )
    
  end # test_transaction

  # test retrieval of archived tuple
  def test_get_archived_tuple

    ## check method invocation  get_archived_tuple( klass, oid, rev )
    assert_respond_to( @mgr, "get_archived_tuple" )
    assert_raises( ArgumentError ){ @mgr.get_archived_tuple }
    assert_raises( ArgumentError ){ @mgr.get_archived_tuple( Class) }
    assert_raises( ArgumentError ){ @mgr.get_archived_tuple( Class, 1234 ) }
    assert_raises( ArgumentError ){ @mgr.get_archived_tuple( Class, 1234, 2, 3 ) }
    assert_raises( TypeError ){ @mgr.get_archived_tuple( Class, 'foo', 1 ) }

    ## try retrieving an non-existent tuple
    assert_nil( @mgr.get_archived_tuple( City, 0, 1 ) ) 

    ## try retrieving an non-existing revision
    assert_nil( @mgr.get_archived_tuple( City, 64212, 20 ) )

    object = nil
    assert_nothing_raised{
      ## retrieve a archived tuple 
      object = @mgr.get_archived_tuple( City, 64212, 4 )
    }
      assert_kind_of( Hash, object )
      assert_equal( object['_type'], ::City )
      assert_equal( object['_oid'], 64212 )
      assert_equal( object['name'], 'Arktika' )
      assert_equal( object['altitude'], 2 )
  end # test_get_archived_tuple()

  # test recording of transaction log
  def test_get_set_transaction_log
    assert_respond_to( @mgr, :transaction_log )
    assert_respond_to( @mgr, :transaction_log= )

    assert_raises( ArgumentError ){ @mgr.transaction_log( nil ) }
    assert_raises( TypeError ){ @mgr.transaction_log = "Foo" }

    # set ro zero per default in test constructor, set and verify readability
    assert_equal( 0, @mgr.transaction_log )
    log_oid = rand( 1000 )
    assert_nothing_raised{ @mgr.transaction_log  = log_oid }
    assert_equal( log_oid, @mgr.transaction_log )

  end # test_get_set_transaction_log()

end # class Test_TupleManager

