#!/usr/bin/ruby
#
# Author:: Oliver M. Bolzer <oliver@fakeroot.net>
# Copyright:: Copyright (c) 2002 Oliver M. Bolzer. All rights reserved.
# Licence:: Ruby licence.

require 'test/unit'
require 'mock'

require 'vapor/persistencemgr'
require 'vapor/persistable'
require 'vapor/extent'
require 'vapor/utils'

# dummy class to be instanciated from persistent storage
class Foo
  include Vapor::Persistable
  attr_accessor :bar
  attr_reader :baz
  attr_reader :dirty_marker
  def dirty_marker=(val)
    @dirty_marker=val
    self.mark_dirty
  end
end # class Foo

# test class for PersistenceManager
class Test_PManager < Test::Unit::TestCase
  include Vapor
  include Vapor::Exceptions

  # pre-test preparation method
  def setup
    @mock_tmgr = Mock.new( "TupleManager" )
    @mock_tmgr.__next( :new ){|a,b,c,d,e| @mock_tmgr }

    prop = Hash.new
    prop[ 'Vapor.Datastore.BackendClass'  ] = @mock_tmgr 
    prop[ 'Vapor.Autocommit'              ] = true 

    @mgr = PersistenceManager.new( prop )

  end # setup()

  # post-test clean up method
  def teardown
    Foo.metadata = nil
  end # teardown()

  ## check constructor 
  def test_initialize
    ## check number of arguments, 1 is correct
    assert_raises( ArgumentError ){ PersistenceManager.new( ) }
    assert_raises( ArgumentError ){ PersistenceManager.new( nil, nil ) }
   
  end # def test_initialize

  # test object retrieval 
  def test_get_object
    assert_respond_to( @mgr, 'get_object' )
    assert_raises( TypeError ){ @mgr.get_object( "foo" ) }
    @mock_tmgr.__next(:get_tuple){|oid| 
      assert_kind_of( Integer, oid )
      nil
    }
    assert_nil( @mgr.get_object( 0 ) )
   
    # test if PM correctly calls backend to get info on an object
    @mock_tmgr.__next(:get_tuple){ |oid|
      assert_kind_of( Integer, oid )
      assert_equal( oid, 12345 )
      { '_oid' => 12345, 'bar' => "Bar", 'baz' => 123, '_type' => Foo }
    }
    @mock_tmgr.__next(:get_class_attributes){|klass| Array.new}

    # test object retrieval and correctness of values
    foo = nil
    assert_nothing_raised{ foo = @mgr.get_object( 12345 ) }
    assert_not_nil( foo )
    assert_kind_of( Vapor::Persistable, foo )
    assert_kind_of( Foo, foo )
    assert_respond_to( foo, 'oid' )
    assert_equal( foo.oid, 12345 )
    assert_respond_to( foo, 'bar' )
    assert_equal( foo.bar, "Bar" )
    assert_respond_to( foo, 'bar' )
    assert_equal( foo.baz, 123 )

    # freshly retrieved object's state is PERSISTENT and was not TRANSIENT
    assert_equal( Persistable::PERSISTENT, foo.state )
    assert_equal( false, foo.vapor_was_transient? )

    # test object-equality by ID
    assert_same( foo, @mgr.get_object( 12345 ) )
    @mock_tmgr.__verify
    
  end # test_get()

  # test retrieval of an object that references another Persistable
  def test_get_object_reference
    assert_respond_to( @mgr, 'get_object' )
    @mock_tmgr.__next(:get_tuple){|oid| 
      { '_oid' => 12345, '_last_change' => 45,
        'bar' => "Bar", 'baz' => 123, '_type' => Foo }
    }
    @mock_tmgr.__next(:get_class_attributes){|klass| 
      [ ClassAttribute.new( "bar", ClassAttribute::String, false ),
        ClassAttribute.new( "baz", ClassAttribute::Reference, false ) ]
    }
    @mock_tmgr.__next(:get_tuple){|oid| # referenced object
      assert_equal( 123, oid )
      { '_oid' => 123, 'bar' => "Foo", 'baz' => nil, '_type' => Foo }
    }
    @mock_tmgr.__next(:get_tuple){|oid| # changelog object
      assert_equal( 45, oid )
      { '_oid' => 45, '_type' => Foo } 
    }
    
    # test object retrieval and correctness of values
    foo = nil
    assert_nothing_raised{ foo = @mgr.get_object( 12345 ) }
    assert_not_nil( foo )
    assert_kind_of( Foo, foo )
    assert_kind_of( Foo, foo.baz )
    assert_equal( 123, foo.baz.oid )
    assert_equal( nil, foo.baz.baz )
    assert_same( 45, foo.instance_eval{ @vapor_changelog }.oid )

  end # test_get_object_reference()

  # test retrieval all all-instances-of-class Extent
  def test_get_extent
    # correct way to call: one Class
    assert_respond_to( @mgr, :get_extent )
    assert_raises( ArgumentError ){ @mgr.get_extent() }
    assert_raises( ArgumentError ){ @mgr.get_extent( nil, nil ) }
    assert_raises( ArgumentError ){ @mgr.get_extent( nil ) }

    ## retrieve unknown class
    @mock_tmgr.__next(:get_all_instances){|klass, subclasses| 
      []
    }
    assert_nothing_raised{ @mgr.get_extent( Class.new ) }

    ## retrieve an class without subclasses
    extent = nil
    klass = Class.new
    @mock_tmgr.__next(:get_all_instances){|k, subclasses|
      assert_equal( false, subclasses )
      [1,2]
    }
    assert_nothing_raised{ extent = @mgr.get_extent( klass, false ) }
    assert_instance_of( Extent, extent )
    assert_equal( @mgr, extent.persistence_manager )
    assert_same( klass , extent.candidate_class  )
    assert_equal( 2, extent.size )
    assert_same( false, extent.has_subclasses? )

    # retrieve an class with subclasses
    extent = nil
    @mock_tmgr.__next(:get_all_instances){|k, subclasses|
      assert_equal( true, subclasses )
      [1,2,3]
    }
    assert_nothing_raised{ extent = @mgr.get_extent( klass, true ) }
    assert_instance_of( Extent, extent )
    assert_same( klass , extent.candidate_class  )
    assert_equal( 3, extent.size )
    assert_same( true, extent.has_subclasses? )

    @mock_tmgr.__verify

  end # test_get_extent

  # retrieve an unused OID
  def test_next_oid
    @mock_tmgr.__next( :next_oid_high ){ 123 }
    assert_respond_to( @mgr, :next_oid )

    i = nil
    j = nil

    assert_nothing_raised{ i = @mgr.next_oid }
    assert_nothing_raised{ j = @mgr.next_oid } 
    assert_kind_of( Integer, i )
    assert_kind_of( Integer, j )

    assert( j > i )

  end # test_next_oid()

  # insert a new object into the cache
  def test_new_object

    assert_respond_to( @mgr, :new_object ) 
    @mock_tmgr.__next( :begin_transaction ){}
    @mock_tmgr.__next( :next_oid_high ){ 123 }
    @mock_tmgr.__next( :get_class_attributes ){ [] }
    @mock_tmgr.__next( :new_tuple ){}
    @mock_tmgr.__next( :update_tuple ){}

    f = nil
    @mgr.transaction.begin
    assert_nothing_raised{
      f = Foo.new
      f.make_persistent
    }

    assert_same( f, @mgr.get_object( f.oid ) )

    ## non-Persistable
    assert_raises( TypeError ){ @mgr.new_object( 123 ) }
    ## TRANSIENT Persistable
    assert_raises( ArgumentError ){ @mgr.new_object( Foo.new ) }

  end # test_new_object()

  # test flushing of object_cache
  def test_flush_new
    assert_respond_to( @mgr, :flush_all )

    @mock_tmgr.__next( :begin_transaction ){}
    @mgr.transaction.begin
    @mock_tmgr.__verify

    @mock_tmgr.__next( :next_oid_high ){ 123 }
    f = Foo.new
    Foo.metadata = [Vapor::ClassAttribute.new( :bar, Vapor::ClassAttribute::String, false )]
    
    f.make_persistent

    i = 0
    @mgr.each_loaded_object{ |obj|
      i += 1 if obj.state != Persistable::PERSISTENT }
    assert( i > 0 , "At least one object should not be PERSISTENT" )

    objectid = nil 

    @mock_tmgr.__next( :new_tuple ){|klass, oid|
     assert_instance_of( Class, klass )
     assert_kind_of( Integer, oid )
     objectid = oid
    }

    @mock_tmgr.__next( :update_tuple ){ |klass, oid, attributes|
      assert_instance_of( Class, klass )
      assert_kind_of( Integer, oid )
      assert_instance_of( Hash, attributes )

      assert_equal( objectid, oid )
    }

    assert_nothing_raised{ @mgr.flush_all }

    i = 0
    @mgr.each_loaded_object{ |obj|
      i += 1 if  obj.state != Persistable::PERSISTENT }
    assert_equal( 0 , i,  "All objects should be PERSISTENT, #{i} weren't" )
   
    assert_nothing_raised{ @mock_tmgr.__verify }
  end # test_flush_new()

  # test deletion of objects
  def test_flush_delete
    assert_respond_to( @mgr, :flush_all )
    f = Foo.new
   
    @mock_tmgr.__next( :begin_transaction ){}
    @mgr.transaction.begin

    @mock_tmgr.__next( :next_oid_high ){1}
    @mock_tmgr.__next( :get_class_attributes ){ [ Vapor::ClassAttribute.new('bar', Vapor::ClassAttribute::String, false ) ] }
    assert_nothing_raised{ f.make_persistent }
    
    assert_nothing_raised{ f.delete_persistent }
    assert_same( Vapor::Persistable::DELETED, f.state )

    @mock_tmgr.__next( :delete_tuple ){ |klass, oid, revision |
      assert_equal( f.class.to_s, klass.to_s )
      assert_equal( f.oid, oid )
    }
    assert_nothing_raised{ @mgr.flush_all }

    assert_equal( Vapor::Persistable::TRANSIENT, f.state )
    assert_nil( f.oid )

    # object not in cache anymore
    @mgr.each_loaded_object{ |obj| assert( obj != f ) }

    @mock_tmgr.__verify
  end # test_flush_delete()

  # test flushing of DIRTY objects
  def test_flush_diry
    assert_respond_to( @mgr, :flush_all )
   
    @mock_tmgr.__next( :begin_transaction ){}
    @mgr.transaction.begin

    # prepare an PERSISTENT object
    f = Foo.new
    @mock_tmgr.__next( :next_oid_high ){1}
    @mock_tmgr.__next( :get_class_attributes ){[] }
    @mock_tmgr.__next( :new_tuple ){}
    @mock_tmgr.__next( :update_tuple ){}
    assert_nothing_raised{
      f.make_persistent
      @mgr.flush_all
    }
    assert_same( Vapor::Persistable::PERSISTENT, f.state )
    
    # make it DIRTY
    assert_nothing_raised{ f.mark_dirty }
    assert_same( Vapor::Persistable::DIRTY, f.state )

    # now flush and check that it got flushed
    @mock_tmgr.__next( :update_tuple ){ |klass, oid, attributes|
      assert_same( Foo, klass )
      assert_instance_of( Hash, attributes )
      assert_equal( f.oid, oid )
    }
    assert_nothing_raised{ @mgr.flush_all }
    assert_same( Vapor::Persistable::PERSISTENT, f.state )
   
  end # test_flush_diry()
  
  #check flushing of an DIRTY->TRANSIENT reference
  def test_flush_dirty_reference_to_transient
    assert_respond_to( @mgr, :flush_all )
   
    @mock_tmgr.__next( :begin_transaction ){}
    @mgr.transaction.begin

    # prepare a PERSISTENT object
    f = Foo.new
    @mock_tmgr.__next( :next_oid_high ){1}
    @mock_tmgr.__next( :get_class_attributes ){
      [ClassAttribute.new( "bar",  Vapor::ClassAttribute::Reference, false )] }
    @mock_tmgr.__next( :new_tuple ){}
    @mock_tmgr.__next( :update_tuple ){}
    assert_nothing_raised{
      f.make_persistent
      @mgr.flush_all
    }
    assert_same( Vapor::Persistable::PERSISTENT, f.state )
    
    # create a TRANSIENT object and set a reference from PERSISTENT to it,
    # making it DIRTY
    b = Foo.new
    f.bar = b
    assert_nothing_raised{ f.mark_dirty }
    assert_equal( Vapor::Persistable::DIRTY, f.state )
    assert_equal( Vapor::Persistable::TRANSIENT, b.state )

    # now flush and check that b is made PERSISTENT and
    # both are flushed
    bar_id = nil
    @mock_tmgr.__next( :new_tuple ){ |klass, oid|
      assert_same( Foo, klass)
      bar_id = oid
    }
     @mock_tmgr.__next( :update_tuple ){ |klass, oid, attributes|
       assert_same( Foo,klass)
       assert_same( bar_id, oid )
    }
    @mock_tmgr.__next( :update_tuple ){ |klass, oid, attributes|
      assert_same( Foo, klass )
      assert_instance_of( Hash, attributes )
      assert_equal( f.oid, oid )
    }
    assert_nothing_raised{ @mgr.flush_all }
    assert_equal(  Vapor::Persistable::PERSISTENT, f.state )
    assert_equal(  Vapor::Persistable::PERSISTENT, b.state )
    
  end #test_flush_dirty_reference_to_transient()

  
  # test searching for objects
  def test_query
    assert_respond_to( @mgr, :query )

    # case with result, no subclasses
    result_oids = [ 1, 2, 3 ]
    @mock_tmgr.__next( :search_tuples ){|klass, statement, subclasses|
      assert_kind_of( Class, klass )
      assert_kind_of( Vapor::QueryStatement, statement )
      assert_equal( false, subclasses )
      result_oids 
    }

    extent = nil
    assert_nothing_raised{ extent = @mgr.query( Foo, "bar = ? AND baz = ?", [1,2], false ) }
    assert_instance_of( Extent, extent )
    assert_equal( 3, extent.size )
    assert_equal( result_oids, extent.oid_array )
    assert_equal( false, extent.has_subclasses? )

    # case with result, with subclasses
    @mock_tmgr.__next( :search_tuples ){|klass, statement, subclasses|
      assert_equal( true, subclasses )
      result_oids
    }
    extent = nil
    assert_nothing_raised{ extent = @mgr.query( Foo, "bar = ? AND baz = ?", [1,2], true ) }
    assert_equal( true, extent.has_subclasses? )

    # case with no result
    @mock_tmgr.__next( :search_tuples ){ nil }
    extent = nil
    assert_nothing_raised{ extent = @mgr.query( Foo, "bar = ?", [666] ) }
    assert_instance_of( Extent, extent )
    assert_equal( 0, extent.size )

    # query with similar-to operator
    @mock_tmgr.__next( :search_tuples ){|klass, statement, subclasses|
      assert_kind_of( Vapor::QueryStatement, statement )
      assert_equal( '~', statement.operator )
      assert_equal( 'b?r*b%_az\?\*', statement.value ) 
      result_oids 
    }
    assert_nothing_raised{ extent = @mgr.query( Foo, "bar ~ ?", ['b?r*b%_az\?\*']) }
    
    ## invalid querys
    
    # non-matching brackets
    assert_raises( InvalidQueryError ){ @mgr.query( Foo, "(bar = ?", [666] ) }
    # junk
    assert_raises( InvalidQueryError ){ @mgr.query( Foo, "%&;?,", [666] ) }
    # not enough arguments
    assert_raises( InvalidQueryError ){ @mgr.query( Foo, "bar = ? AND baz = ?", [1] ) }
    
    
  end # test_query()

  # test on/off of Autocommit-Mode
  def test_autocommit
    assert_respond_to( @mgr, :autocommit? )
    assert_respond_to( @mgr, "autocommit=" )

    @mock_tmgr.__next( :new ){|a,b,c,d,e| @mock_tmgr }
    prop = Hash.new
    prop[ 'Vapor.Datastore.BackendClass'  ] = @mock_tmgr 
    pmgr = PersistenceManager.new( prop )

    # autocommit mode on by default
    assert_equal( true, pmgr.autocommit? )

    # trun it off
    assert_nothing_raised{ pmgr.autocommit = false }
    assert_equal( false, pmgr.autocommit? )

    # turn it on again, changes should be flushed
    assert_nothing_raised{ pmgr.autocommit = true }
    assert_equal( true, pmgr.autocommit? )

    # turning it on while already on, should not flush
    assert_nothing_raised{ pmgr.autocommit = true }

    # trun off at initialization with property
    @mock_tmgr.__next( :new ){|a,b,c,d,e| @mock_tmgr }
    prop[ 'Vapor.Autocommit' ] = false
    pmgr2 = PersistenceManager.new( prop )
    assert_equal( false, pmgr2.autocommit? )
    
    
    @mock_tmgr.__verify
   
  end # test_autocommit()
    
  # test wheter objects are properly flushed in Autocommit-Mode
  def test_try_change_state
    assert_respond_to( @mgr, :try_change_state )

    assert_raises( ArgumentError ){ @mgr.try_change_state(){} }
    assert_raises( ArgumentError ){ @mgr.try_change_state( nil, nil ){} }
    assert_raises( TypeError ){ @mgr.try_change_state( "" ){} }
    assert_raises( ArgumentError ){ @mgr.try_change_state( Foo.new ) }

    # hi-jack Transaction's begin/commit/rollback methods to call to
    # @mock_tmgr
    $mock_tmgr = @mock_tmgr  # neccersary evil
    @mgr.transaction.instance_eval %Q{
      alias :orig_begin :begin
      alias :orig_commit :commit
      alias :orig_rollback :rollback

      def begin
        $mock_tmgr.begin
        @active = true
      end
      def commit
        $mock_tmgr.commit
        @active = false
      end
      def rollback
        $mock_tmgr.rollback
        @active = false
      end
    }

    # prepare Persistable we are going to use everywere
    foo = Foo.new

    # autocommit = true and no transaction flush_object should be called
    @mock_tmgr.__next( :begin ){}
    @mock_tmgr.__next( :commit ){}
    assert_nothing_raised{ @mgr.try_change_state( foo ){} }
    @mock_tmgr.__verify
 
    # autocommit = false and transaction not started
    assert_nothing_raised{ @mgr.autocommit = false }
    assert_raises( StaleTransactionError ){ @mgr.try_change_state( foo ){} }
    @mock_tmgr.__verify

    # autocommit = false and transaction started, nothing raised and
    # flush_object not called, object added to transaction's modified list
    @mock_tmgr.__next( :begin ){}
    assert_nothing_raised{ 
      @mgr.transaction.begin
      @mgr.try_change_state( foo ){} 
    }
    assert_equal( true, @mgr.transaction.log_entry.modified_objects.include?( foo ) )
    @mock_tmgr.__verify
   
    ## restore original flush_object method
    $mock_tmgr = nil      # globals are evil
    @mgr.transaction.instance_eval %Q{
      alias :begin orig_begin 
      alias :commit :orig_commit
      alias :rollback :orig_rollback
    }

  end # test_try_change_state()

  # test try_change_state for readonly persistables
  def test_try_change_state_readonly
    assert_respond_to( @mgr, :try_change_state )
    
    foo = Foo.new
    Foo.metadata = []
    foo_content = { '_type' => Foo, '_oid' => 1, '_revision' => 2, 'dirty_marker' => 0  }
    
    ## raise execption on readonly persistable
    @mock_tmgr.__next( :get_archived_tuple ){|klass,oid,revision|foo_content }
   
    assert_nothing_raised{ foo.load_attributes_readonly( foo_content ) }
    assert_raises( PersistableReadOnlyError ){ @mgr.try_change_state( foo ){} }
    
    # attribute preservation on readonly dirty marking
    @mock_tmgr.__next( :get_archived_tuple ){|klass,oid,revision|foo_content }
    assert_equal( 0, foo.dirty_marker )
    assert_raises( PersistableReadOnlyError ){ foo.dirty_marker = 1 }
    assert_equal( Persistable::READONLY, foo.state )
    assert_equal( 0, foo.dirty_marker )

  end #test_try_change_state()
  
  # test Transaction reader
  def test_transaction
    assert_respond_to( @mgr, :transaction )
    assert_kind_of( Transaction, @mgr.transaction )

    # starting an transaction resets Autocommit-Mode
    assert_equal( true, @mgr.autocommit? )

    @mock_tmgr.__next( :begin_transaction ){}
    assert_nothing_raised{ @mgr.transaction.begin } 
    @mock_tmgr.__verify
    assert_equal( false, @mgr.autocommit? )
    @mock_tmgr.__next( :next_oid_high ){1}
    @mock_tmgr.__next( :transaction_log= ){|log|}
    @mock_tmgr.__next( :new_tuple ){}
    @mock_tmgr.__next( :update_tuple ){}
    @mock_tmgr.__next( :commit_transaction ){}
    @mock_tmgr.__next( :transaction_log= ){|log|}
    assert_nothing_raised{ @mgr.transaction.commit }
    assert_equal( false, @mgr.autocommit? )

    @mock_tmgr.__verify
  end # test_transaction()

  # test refreshing of object
  def test_refresh_object
    assert_respond_to( @mgr, :refresh_object )
    assert_raises( ArgumentError ){ @mgr.refresh_object() }
    assert_raises( ArgumentError ){ @mgr.refresh_object( nil, nil ) }
    assert_raises( TypeError ){ @mgr.refresh_object( "foo" ) }
  
    # create PERSISTENT object
    foo_attrs = { '_oid' => 123, 'bar' => "bar", 'baz' => "baz", '_type' => Foo }
    foo = Foo.new
    foo.load_attributes( foo_attrs )
    assert_equal( Persistable::PERSISTENT, foo.state )
    assert_equal( "bar", foo.bar )

    # object has changed values
    foo_attrs[ 'bar' ] = "foo"
    @mock_tmgr.__next( :get_tuple ){|oid|
      assert_equal( 123, oid )
      foo_attrs
    }
    @mock_tmgr.__next( :get_class_attributes ){ [] }
    # update it
    assert_nothing_raised{ @mgr.refresh_object( foo ) }
    assert_equal( Persistable::PERSISTENT, foo.state )
    assert_equal( "foo", foo.bar )
 
    # make sure it doesn't appear in the transaction log's modified object
    # list
    assert_equal( false, @mgr.transaction.log_entry.modified_objects.include?( foo ) )
  end # test_refresh_object()

  # test rolling back a single object
  def test_rollback_obj
    assert_respond_to( @mgr, :rollback_object )
    assert_raises( ArgumentError ){ @mgr.rollback_object() }
    assert_raises( ArgumentError ){ @mgr.rollback_object( nil, nil ) }

    # test subject: Persistable
    foo = Foo.new
    foo.bar = "bar"
    
    # start transaction
    @mock_tmgr.__next( :begin_transaction ){}
    @mgr.transaction.begin

    # rollback( TRANSIENT ) -> TRANSIENT. no value change
    assert_equal( Persistable::TRANSIENT, foo.state )
    assert_equal( true, foo.vapor_was_transient? )
    assert_equal( "bar", foo.bar )
    assert_nothing_raised{ @mgr.rollback_object( foo ) }
    assert_equal( Persistable::TRANSIENT, foo.state )
    assert_equal( "bar", foo.bar )

    # rollback( NEW ) -> TRANSIENT, no value change
    @mock_tmgr.__next( :next_oid_high ){1}
    @mock_tmgr.__next( :get_class_attributes ){ [] }
    assert_nothing_raised{ foo.make_persistent } 
    assert_equal( Persistable::NEW, foo.state )
    assert_equal( true, foo.vapor_was_transient? )
    assert_not_nil( foo.oid )
    assert_nothing_raised{ @mgr.rollback_object( foo ) }
    assert_equal( Persistable::TRANSIENT, foo.state )
    assert_equal( "bar", foo.bar )
    assert_nil( foo.oid )
    @mock_tmgr.__verify

    # rollback( NEW->DELETED ) -> TRANSIENT, no value change
    assert_nothing_raised{ foo.make_persistent; foo.delete_persistent }
    assert_equal( Persistable::DELETED, foo.state )
    assert_equal( true, foo.vapor_was_transient? )
    assert_not_nil( foo.oid )
    assert_nothing_raised{ @mgr.rollback_object( foo ) }
    assert_equal( Persistable::TRANSIENT, foo.state )
    assert_equal( true, foo.vapor_was_transient? )
    assert_equal( "bar", foo.bar )
    assert_nil( foo.oid )

    # make it PERSISTENT
    foo_attrs_orig = { '_oid' => 123, 'bar' => 'bar', 'baz' => 'baz', '_type' => Foo }
    foo_attrs_new = { '_oid' => 123, 'bar' => 'BAR', 'baz' => 'baz', '_type' => Foo }
    foo.load_attributes( foo_attrs_orig )
    foo.vapor_post_commit

    # rollback( PERSISTENT ) -> PERSISTENT, new values
    @mock_tmgr.__next( :get_tuple ){ foo_attrs_new }
    assert_equal( Persistable::PERSISTENT, foo.state )
    assert_equal( false, foo.vapor_was_transient? )
    assert_equal( "bar", foo.bar )
    assert_equal( 123, foo.oid )
    assert_nothing_raised{ @mgr.rollback_object( foo ) }
    assert_equal( Persistable::PERSISTENT, foo.state )
    assert_equal( 'BAR', foo.bar )

    # rollback( DIRTY ) -> PERSISTENT, new values
    @mock_tmgr.__next( :get_tuple ){ foo_attrs_new }
    foo.bar = "FOO"
    foo.mark_dirty
    assert_equal( Persistable::DIRTY, foo.state )
    assert_equal( false, foo.vapor_was_transient? )
    assert_nothing_raised{ @mgr.rollback_object( foo ) }
    assert_equal( Persistable::PERSISTENT, foo.state )
    assert_equal( 'BAR', foo.bar )

    # rollback( DELETED ) -> PERSISTENT, new values
    @mock_tmgr.__next( :get_tuple ){ foo_attrs_new }
    foo.bar = "FOO"
    foo.delete_persistent
    assert_equal( Persistable::DELETED, foo.state )
    assert_equal( false, foo.vapor_was_transient? )
    assert_nothing_raised{ @mgr.rollback_object( foo ) }
    assert_equal( Persistable::PERSISTENT, foo.state )
    assert_equal( 'BAR', foo.bar )
   
    @mock_tmgr.__verify
  end # test_rollback_obj()
 
  # test optimistic locking, error cases
  def test_optimistic

    Foo.metadata = []

    # object stale
    @mock_tmgr.__next( :begin_transaction ){}
    assert_nothing_raised{ @mgr.transaction.begin }
      @mock_tmgr.__next( :get_tuple ){ |oid|
        { '_oid' => oid, '_revision' => 2, 'bar' => 'bar', '_type' => Foo }
      }
      foo = @mgr.get_object( 666 )
      foo.mark_dirty
    @mock_tmgr.__next( :next_oid_high ){1}
    @mock_tmgr.__next( :transaction_log= ){|val|}
    @mock_tmgr.__next( :update_tuple ){ |klass, oid, attributes|
        raise StaleObjectError
      }
    @mock_tmgr.__next( :rollback_transaction ){}
    @mock_tmgr.__next( :get_tuple ){ |oid| { '_oid' => oid, '_revision' => 3, 'bar' => 'bar', '_type' => Foo } }
    begin
      @mgr.transaction.commit
    rescue StaleObjectError => e
      assert_kind_of( StaleObjectError, e )
      assert_equal( foo, e.causing_object )
      raised = true
    end
    assert_equal( true, raised, "StaleObjectError not raised" )
   
    @mock_tmgr.__verify
    
  end # test_optimistic()

  # test retrieval of archived object
  def test_get_object_version
    assert_respond_to( @mgr, :get_object_version )

    # correct calling convention  get_object_version( klass, oid, revision )
    assert_raises( ArgumentError ){ @mgr.get_object_version() }
    assert_raises( ArgumentError ){ @mgr.get_object_version( Class ) }
    assert_raises( ArgumentError ){ @mgr.get_object_version( Class, 123 ) }
    assert_raises( ArgumentError ){ @mgr.get_object_version( Class, 123, 123, nil ) }
    assert_raises( TypeError ){ @mgr.get_object_version( Class, 'foo', 123 ) }
    assert_raises( TypeError ){ @mgr.get_object_version( Class, 123, 'foo' ) }

    # call forwarding 
    @mock_tmgr.__next( :get_archived_tuple){ |klass, oid, revision|
      assert_kind_of( Integer, oid )    
      assert_kind_of( Integer, revision )
      assert_equal( Foo.to_s, klass.to_s )
      nil
    
    }
    assert_nil( @mgr.get_object_version( Foo, 0, 0 ) )
    @mock_tmgr.__verify
    
    # test object retrieval and correctness of values
    @mock_tmgr.__next( :get_archived_tuple ){ |klass, oid, revision|
      { '_oid' => oid, '_revision' => revision, '_type' => klass, 
        'bar' => "Bar", 'baz' => 123
      } 
    }
    @mock_tmgr.__next( :get_class_attributes ){ [] } 
    foo = nil
    assert_nothing_raised{ foo = @mgr.get_object_version( Foo, 123, 1 ) }
    assert_not_nil( foo )
    assert_kind_of( Vapor::Persistable, foo )
    assert_kind_of( Foo, foo )
    assert_equal( Vapor::Persistable::READONLY, foo.state )

    # unequality on repeated calls
    @mock_tmgr.__next( :get_archived_tuple ){ |klass, oid, revision|
      { '_oid' => oid, '_revision' => revision, '_type' => klass, 
        'bar' => "Bar", 'baz' => 123
      } 
    }
    assert_not_same( foo, @mgr.get_object_version( foo.class, foo.oid, foo.revision ) ) 

    @mock_tmgr.__verify

  end # test_get_object_version
end # class Test_PManager
