#!/usr/bin/ruby
#
# Author:: Oliver M. Bolzer <oliver@fakeroot.net>
# Copyright:: Copyright (c) 2002 Oliver M. Bolzer. All rights reserved.
# Licence:: Ruby licence.
# $Id: Integrated.rb 269 2003-11-07 15:54:17Z bolzer $
#
# Test overall functionality of VAPOR, letting all the components interact.
# Considered Usability-test, not unit-test


require 'test/unit'
require 'vapor'

# fake class we've saved persistently
class Person
  include Vapor::Persistable 
  
  attr_accessor :first_name   # String
  attr_accessor :last_name    # String
  attr_accessor :birthday     # Date
  attr_accessor :nickNames    # String[]
  attr_accessor :inhabits     # City
  attr_accessor :friends      # Person[]
end

class City
  include Vapor::Persistable

  attr_reader :name         # String
  attr_reader :altitude     # Integer

  def initialize( name = 'Tokyo', altitude = 0 )
    @name = name
    @altitude = altitude 
  end

end

class Student < Person
  attr_accessor  :student_id
end

# test class for overall funtioncality of Vapor
class Test_Vapor < Test::Unit::TestCase

  Properties = Hash.new
  Properties[ 'Vapor.Datastore.Username'] = 'bolzer'
  Properties[ 'Vapor.Datastore.Password' ] = 'foo02bar'
  Properties[ 'Vapor.Datastore.Name' ] = 'bolzer_vapor_test'
  Properties[ 'Vapor.Datastore.Host' ] = 'db'
  Properties[ 'Vapor.Datastore.Port' ] = 5432
  Properties[ 'Vapor.Autocommit' ]     = false

  # setup a valid VAPOR-environment
  def setup
    @pmgr = Vapor::PersistenceManager.new( Properties )
    @pmgr.transaction.committer = $0 
  end # setup()

  # test reading single stored object
  def test_remarshal_single_object
    bear = nil
   
    # remarshal the Object ( we just know it's OID )
    assert_nothing_raised{ bear = @pmgr.get_object( 40125 ) }

    # check it's type
    assert_instance_of( Person, bear )
    assert_kind_of( Vapor::Persistable, bear )

    # check various attributes   
    attrvalue = nil

    assert_respond_to( bear, :oid )
    assert_equal( 40125, bear.oid )
    assert_respond_to( bear, :revision )
    assert_equal( 1, bear.revision )

    assert_respond_to( bear, :first_name ) # String
    assert_nothing_raised{ attrvalue = bear.first_name }
    assert_instance_of( String, attrvalue )
    assert_equal( 'Bear', attrvalue )
    
    assert_respond_to( bear, :last_name )  # String
    assert_nothing_raised{ attrvalue = bear.last_name }
    assert_instance_of( String, attrvalue ) 
    assert_equal( 'Polaris', attrvalue )

    assert_respond_to( bear, :birthday )   # Date
    assert_nothing_raised{ attrvalue = bear.birthday }
    assert_instance_of( Time, attrvalue )
    assert_equal( Time.utc( 2002, 6, 15, 0, 0, 0 ), attrvalue ) 
    
    assert_respond_to( bear, :nickNames )  # String[]
    assert_nothing_raised{ attrvalue = bear.nickNames }
    assert_instance_of( Array, attrvalue )
    assert_equal( ["Pooh"], attrvalue )

    #Person.reference_attributes = [ [ 'inhabits', false ], ['friends', true] ]
    assert_respond_to( bear, :inhabits )   # City (Reference)
    assert_nothing_raised{ attrvalue = bear.inhabits }
    assert_instance_of( City, attrvalue )
    assert_same( @pmgr.get_object( 64212 ), attrvalue )
    
    assert_respond_to( bear, :friends )    # Person[] (Reference[])
    assert_nothing_raised{ attrvalue = bear.friends }
    assert_instance_of( Array, attrvalue )
    assert_equal( [ @pmgr.get_object( 51987 ) ], attrvalue )

  end # test_remarshal_single_object()

  # instanciate a new object and make it persistent
  def test_make_new_object_persistent

    tokyo = City.new( 'Tokyo', 0 )
    assert_kind_of( Vapor::Persistable, tokyo )
    assert_respond_to( tokyo, :oid )
    assert_respond_to( tokyo, :revision )
    assert_nil( tokyo.oid )
    assert_nil( tokyo.revision )
    assert_equal( Vapor::Persistable::TRANSIENT, tokyo.state )
    
    cyril = Person.new
    cyril.first_name = "Cyril"
    cyril.last_name = "Bitteric"
    cyril.nickNames = [ "cebit" ]
    cyril.inhabits = tokyo
    cyril.friends = [cyril]

    assert_nothing_raised{ @pmgr.transaction.begin }
    assert_nothing_raised{ cyril.make_persistent }
    
    assert_kind_of( Integer, cyril.oid )
    assert( cyril.oid > 0 )
    assert_equal( 0, cyril.revision )
    assert_equal( Vapor::Persistable::NEW, tokyo.state )
    assert_same( cyril, @pmgr.get_object( cyril.oid ) )

    # should have made tokyo persistent, too
    assert_kind_of( Integer, tokyo.oid )
    assert( tokyo.oid > 0 )
    assert_equal( 0, tokyo.revision )
    assert_equal( Vapor::Persistable::NEW, tokyo.state )
    assert_same( tokyo, @pmgr.get_object( tokyo.oid ) )

    assert_nothing_raised{ 
      @pmgr.transaction.message = "test_make_new_object_persistent()"  
      @pmgr.transaction.commit
    }

    assert_equal( Vapor::Persistable::PERSISTENT, cyril.state ) 
    assert_equal( Vapor::Persistable::PERSISTENT, tokyo.state )
    assert_equal( 1, cyril.revision )
    assert_equal( 1, tokyo.revision )
  end # test_make_new_object_persistent()

  # delete a persistent object
  def test_delete_persistent_object
  
    @pmgr.autocommit = true
    # first, create a persistent object
    tokyo = City.new( 'Tokyo', 0 )
    assert_nothing_raised{ tokyo.make_persistent }
    assert_equal( Vapor::Persistable::PERSISTENT, tokyo.state )
    assert_not_nil( tokyo.revision )
    
    # now get rid of it
    assert_nothing_raised{ tokyo.delete_persistent }
    assert_equal( Vapor::Persistable::TRANSIENT, tokyo.state )
    assert_nil( tokyo.oid )
    assert_nil( tokyo.revision )
  end # test_delete_persistent_object()

  # update an object
  def test_object_update

    assert_nothing_raised{ @pmgr.transaction.begin }

    # first load an object
    seal = nil
    assert_nothing_raised{ seal = @pmgr.get_object( 51987 ) }
    seal.first_name = "Gigantic"
    seal.nickNames = ["asdf"]
    revision = seal.revision
    assert_nothing_raised{ seal.mark_dirty }
    assert_equal( Vapor::Persistable::DIRTY, seal.state )

    assert_nothing_raised{
      @pmgr.transaction.message = "test_object_update()"
      @pmgr.transaction.commit
    }
    assert_equal( Vapor::Persistable::PERSISTENT, seal.state )
    assert_equal( revision + 1, seal.revision )
  end # test_object_update

  # test retrieving of all objects of a class
  def test_get_extent

    # without subclasses
    extent = nil
    assert_nothing_raised{ extent = @pmgr.get_extent( Person, false ) }
    assert_kind_of( Vapor::Extent, extent )
    assert( extent.size > 0 )
    assert_equal( true, extent.reject{|obj| obj.class == Person }.empty? )

    # with subclasses
    extent = nil
    assert_nothing_raised{ extent = @pmgr.get_extent( Person, true ) }
    assert_equal( false, extent.reject{|obj| obj.== Person }.empty? )
  
  end # test_get_extent()

  # test searching for objects
  def test_query

    extent = nil
    assert_nothing_raised{ extent = @pmgr.query( Person, "first_name = ?", ["Bear"] ) }
    assert_equal( 1, extent.size )
    assert_equal( extent[0], @pmgr.get_object( 40125 ) )

    extent = nil
    assert_nothing_raised{ extent = @pmgr.query( Person, "first_name ~ ?", ["?ea*"] ) }
    assert_equal( 1, extent.size )
    assert_equal( extent[0], @pmgr.get_object( 40125 ) )

  end # test_query()

  # test creation of objects that violate uniqueness constraints
  def test_violate_uniqueness

    assert_nothing_raised{ @pmgr.autocommit = true }
    # create two students with identical ID
    student1 = Student.new
    student1.first_name = "Peter"
    student1.student_id = 23
    student2 = Student.new
    student2.first_name ="John"
    student2.student_id = 23
 
    # make them both unique
    assert_raises( Vapor::UniquenessError ){
      student1.make_persistent
      student2.make_persistent
    }

  end # test_violate_uniqueness

  # test uniqueness violation in block
  def test_violate_uniqueness_in_block
    assert_nothing_raised{ @pmgr.autocommit = true }
    assert_raises( Vapor::UniquenessError ){
      student1 = student2 = nil 
      @pmgr.transaction{|t|
        student1 = Student.new
        student1.first_name = "Peter"
        student1.student_id = 23
        student1.make_persistent
        student2 = Student.new
        student2.first_name = 'John'
        student2.student_id = 23
        student2.make_persistent
      }
    }
  end #test_violate_uniqueness_in_block

  # test behaviour when in Autocommit-Mode
  def test_autocommit

    assert_nothing_raised{ @pmgr.autocommit = true }
    assert_equal( true, @pmgr.autocommit? ) 

    tokyo = City.new( 'Tokyo', 0 )
    assert_kind_of( Vapor::Persistable, tokyo )
    assert_equal( Vapor::Persistable::TRANSIENT, tokyo.state )
    assert_nothing_raised{ tokyo.make_persistent }
    assert_equal( Vapor::Persistable::PERSISTENT, tokyo.state )
    assert_nothing_raised{ tokyo.mark_dirty }
    assert_equal( Vapor::Persistable::PERSISTENT, tokyo.state )
    assert_nothing_raised{ tokyo.delete_persistent }
    assert_equal( Vapor::Persistable::TRANSIENT, tokyo.state )

  end # test_autocommit()

  # test rollback
  def test_rollback

    tx = nil
    assert_nothing_raised{
      @pmgr.autocommit = false
      tx = @pmgr.transaction
      tx.begin
    }

    oli = nil
    assert_nothing_raised{
      oli = Person.new
      oli.first_name = "Oliver"
      oli.last_name = "B."
      oli.nickNames = ["OliB"]
      oli.make_persistent
    }
    assert_equal( Vapor::Persistable::NEW, oli.state ) 
    assert_equal( ["OliB"], oli.nickNames )

    assert_nothing_raised{
      tx.message = "test_rollback()"
      tx.commit
    }
    assert_equal( Vapor::Persistable::PERSISTENT, oli.state )

    assert_equal( false, tx.active? )
    assert_nothing_raised{
      tx.begin 
      oli.nickNames.push( "BigO" )
      assert_equal( ["OliB", "BigO"], oli.nickNames )
      oli.mark_dirty
    }
    assert_equal( Vapor::Persistable::DIRTY, oli.state )

    assert_nothing_raised{ tx.rollback }

    assert_equal( Vapor::Persistable::PERSISTENT, oli.state )
    assert_equal( ["OliB"], oli.nickNames )

  end # test_rollback

  # test capability of recording changelogs
  def test_versioning
    tx = nil
    assert_nothing_raised{
      @pmgr.autocommit = false
      tx = @pmgr.transaction
    }

    oli = nil
    oli = Person.new
    oli.first_name = "Oliver"
    oli.last_name = "1"
    oli.nickNames = ["OliB"]
  
    assert_nothing_raised{
      tx.begin
        tx.message = "1"
        oli.make_persistent
      tx.commit
    }
    assert_equal( 1, oli.revision )

    assert_nothing_raised{
      tx.begin
        tx.message = "2"
        oli.last_name = "2"
        oli.mark_dirty
      tx.commit
    }
    assert_equal( 2, oli.revision )

    # keep oid for re-retrieval and forget running PersistenceManager
    oli_oid = oli.oid
    @pmgr = nil    
    
    # create new PersistenceManager, so instances will be loaded again
    @pmgr = Vapor::PersistenceManager.new( Properties )

    oli2 = @pmgr.get_object( oli_oid )

    # check basics
    assert_not_same( oli, oli2 )
    assert_equal( oli.oid, oli2.oid )
    assert_equal( oli.revision, oli2.revision )
    assert_equal( oli.last_name, oli2.last_name )

    # check change_log
    assert_kind_of( Vapor::TransactionLog, oli2.vapor_changelog )
    assert_equal( $0, oli2.vapor_changelog.committer )
    assert_equal( "2", oli2.vapor_changelog.message )
    assert_equal( 1, oli2.vapor_changelog.modified_objects.size )
    assert_same( oli2, oli2.vapor_changelog.modified_objects.first )

    # retrieve older resvition
    oli2_old = nil
    assert_nothing_raised{ oli2_old = oli2.get_version( 1 ) }
    
    # check it
    assert_not_nil( oli2_old )
    assert_equal( oli2.oid, oli2_old.oid )
    assert_not_same( oli2.revision, oli2_old.revision )
    assert_equal( 1, oli2_old.revision )
    assert_equal( "1", oli2_old.last_name )
    assert_not_nil( oli2_old.vapor_changelog )
    assert_equal( $0, oli2_old.vapor_changelog.vapor_changelog.committer )
    assert_equal( "1", oli2_old.vapor_changelog.vapor_changelog.message )

  end # test_versioning() 

end # class Test_Vapor
