/*
Copyright 2020 The Tekton Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

		http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package resources

var (
	// DefaultImage is the image used by default.
	DefaultImage = "override-with-el:latest"
	// DefaultPort is the port used by default.
	DefaultPort = 8080
	// DefaultSetSecurityContext is the SetSecurityContext value used by default.
	DefaultSetSecurityContext = true
	// DefaultEventListenerEvent is the EventListenerEvent value used by default.
	DefaultEventListenerEvent = "disable"
	// DefaultReadTimeout is the ReadTimeout used by default.
	DefaultReadTimeout = int64(5)
	// DefaultWriteTimeout is the WriteTimeout used by default.
	DefaultWriteTimeout = int64(40)
	// DefaultIdleTimeout is the IdleTimeout used by default.
	DefaultIdleTimeout = int64(30)
	// DefaultTimeOutHandler is the TimeOutHandler timeout used by default.
	DefaultTimeOutHandler = int64(5)
	// DefaultPeriodSeconds is the PeriodSeconds used by default.
	DefaultPeriodSeconds = 10
	// DefaultFailureThreshold is the FailureThreshold used by default.
	DefaultFailureThreshold = 3
	// DefaultHTTPClientReadTimeOut is the HTTPClient ReadTimeOut used by default.
	DefaultHTTPClientReadTimeOut = int64(30)
	// DefaultHTTPClientKeepAlive is the HTTPClient KeepAlive used by default
	DefaultHTTPClientKeepAlive = int64(30)
	// DefaultHTTPClientTLSHandshakeTimeout is the HTTPClient TLS Handshake timeout used by default
	DefaultHTTPClientTLSHandshakeTimeout = int64(10)
	// DefaultHTTPClientResponseHeaderTimeout is the HTTPClient Response Header Timeout
	DefaultHTTPClientResponseHeaderTimeout = int64(10)
	// DefaultHTTPClientExpectContinueTimeout is the HTTPClient Expect Continue Timeout
	DefaultHTTPClientExpectContinueTimeout = int64(1)
	// DefaultStaticResourceLabels are the StaticResourceLabels used by default.
	DefaultStaticResourceLabels = map[string]string{
		"app.kubernetes.io/managed-by": "EventListener",
		"app.kubernetes.io/part-of":    "Triggers",
	}
	// DefaultSystemNamespace is the default system namespace used.
	DefaultSystemNamespace = "tekton-pipelines"
)

type Config struct {
	// Image defines the container that we use to run in the EventListener Pods.
	Image *string
	// Port defines the port for the EventListener to listen on.
	Port *int
	// SetSecurityContext defines if the security context is set.
	SetSecurityContext *bool
	// SetEventListenerEvent defines to enable or disable of emitting events for EventListener.
	SetEventListenerEvent *string
	// ReadTimeOut defines the read timeout for EventListener Server.
	ReadTimeOut *int64
	// WriteTimeOut defines the write timeout for EventListener Server.
	WriteTimeOut *int64
	// IdleTimeOut defines the read timeout for EventListener Server.
	IdleTimeOut *int64
	// TimeOutHandler defines the timeout for Timeout Handler of EventListener Server.
	TimeOutHandler *int64
	// HTTPClientReadTimeOut defines the Read timeout for HTTP Client
	HTTPClientReadTimeOut *int64
	// HTTPClientKeepAlive defines the Keep Alive for HTTP Client
	HTTPClientKeepAlive *int64
	// HTTPClientTLSHandshakeTimeout defines the Handshake timeout for HTTP Client
	HTTPClientTLSHandshakeTimeout *int64
	// HTTPClientResponseHeaderTimeout defines the Response Header timeout for HTTP Client
	HTTPClientResponseHeaderTimeout *int64
	// HTTPClientExpectContinueTimeout defines the Expect timeout for HTTP Client
	HTTPClientExpectContinueTimeout *int64
	// PeriodSeconds defines Period Seconds for the EventListener Liveness and Readiness Probes.
	PeriodSeconds *int
	// FailureThreshold defines the Failure Threshold for the EventListener Liveness and Readiness Probes.
	FailureThreshold *int
	// StaticResourceLabels is a map with all the labels that should be on all resources generated by the EventListener.
	StaticResourceLabels map[string]string
	// SystemNamespace is the namespace where the reconciler is deployed.
	SystemNamespace string
}

type ConfigOption func(d *Config)

// MakeConfig is a helper to build a config that is consumed by an EventListener.
// It generates a default Config for the EventListener without any flags set and accepts functions for modification.
func MakeConfig(ops ...ConfigOption) *Config {
	c := &Config{
		Image:                 &DefaultImage,
		Port:                  &DefaultPort,
		SetSecurityContext:    &DefaultSetSecurityContext,
		SetEventListenerEvent: &DefaultEventListenerEvent,

		ReadTimeOut:                     &DefaultReadTimeout,
		WriteTimeOut:                    &DefaultWriteTimeout,
		IdleTimeOut:                     &DefaultIdleTimeout,
		TimeOutHandler:                  &DefaultTimeOutHandler,
		HTTPClientReadTimeOut:           &DefaultHTTPClientReadTimeOut,
		HTTPClientKeepAlive:             &DefaultHTTPClientKeepAlive,
		HTTPClientTLSHandshakeTimeout:   &DefaultHTTPClientTLSHandshakeTimeout,
		HTTPClientResponseHeaderTimeout: &DefaultHTTPClientResponseHeaderTimeout,
		HTTPClientExpectContinueTimeout: &DefaultHTTPClientExpectContinueTimeout,
		PeriodSeconds:                   &DefaultPeriodSeconds,
		FailureThreshold:                &DefaultFailureThreshold,

		StaticResourceLabels: DefaultStaticResourceLabels,
		SystemNamespace:      DefaultSystemNamespace,
	}

	for _, op := range ops {
		op(c)
	}
	return c
}
