/* This file is part of statist
**
** It is distributed under the GNU General Public License.
** See the file COPYING for details.
** 
** (c) 1997 Dirk Melcher
**  Doerper Damm 4
**  49134 Wallenhorst
**  GERMANY
**  Tel. 05407/7636
**  email: Dirk.Melcher@usf.Uni-Osnabrueck.DE
**
**  small Changes by Bernhard Reiter  http://www.usf.Uni-Osnabrueck.DE/~breiter
**  $Id: plot.c,v 1.8 2001/02/17 19:26:09 breiter Exp $
***************************************************************/
/* plot.c fuer STATIST */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>

#include "statist.h"
#include "funcs.h"
#include "data.h"
#include "plot.h"
#include "gettext.h"
 
#define SETDEFAULT   fprintf(pipef, "set nokey\n"); \
  fprintf(pipef, "set noparametric\n");             \
  fprintf(pipef, "set nolog x\n");                  \
  fprintf(pipef, "set nogrid\n");                   \
  fprintf(pipef, "set nolabel\n");                  \
  fprintf(pipef, "set xtic\n");                     \
  fprintf(pipef, "set ytics\n");                    \
  fprintf(pipef, "set ylabel\n");		    \
  fprintf(pipef, "linetype=1\n");

#ifndef MSDOS
#define PLOT_MSG   out_d(_("Creating gnuplot-graphic ...\n\n",	\
	"Erstelle gnuplot-Graphik ...\n\n") )
#define PLOT_CLOSE fflush(pipef)
#else
#define PLOT_MSG   out_d(_("Creating gnuplot command file '%s'\n\n",	\
	"Erstelle Kommandodatei '%s' fuer gnuplot\n\n"), GPL_COM)
#define PLOT_CLOSE FCLOSE(pipef)
#endif

char GPL_DAT[300];
char GPL_COM[16];

BOOLEAN init_gnuplot() {
#ifndef MSDOS
  switch (gnupl_open) {
    case CRASH: return FALSE;
      break;
    case TRUE: return TRUE;
      break;
    case FALSE: {
      if ((pipef = popen("gnuplot -geometry 450x300", "w"))==NULL) {
        out_err(ERR, ERR_FILE, ERR_LINE, 
	  	_("gnuplot-initialization failed!",
	  	"gnuplot-Initialisierung gescheitert!") );
	gnupl_open = CRASH;
        return FALSE;
      } 
      else {
        gnupl_open = TRUE;
	sprintf(GPL_DAT, "%.256s/stat_gpl.dat", getenv("HOME"));
/*	printf("GPL_DAT: %s\n", GPL_DAT);	*/
        return TRUE;
      }
    }
    default: return FALSE;
  }

#else
  strcpy(GPL_COM, "stat_gpl.com");
  FOPEN(GPL_COM, "w", pipef);
  gnupl_open = TRUE;
  sprintf(GPL_DAT, "stat_gpl.dat");
  return TRUE;
#endif
}



BOOLEAN plot_pair(REAL x[], REAL y[], int n, REAL a0, REAL a1,
		  char *xlab, char *ylab) {
  FILE *tempf;
  int i;
  char aline[160];

  if (!init_gnuplot()) {
    return FALSE;
  }

  FOPEN(GPL_DAT, "wt", tempf);
  for (i=0; i<n; i++) {
    fprintf(tempf, "%g %g\n", x[i], y[i]);
  }
  FCLOSE(tempf);
  sprintf(aline, "f(x)=%g + %g*x\n", a0, a1);
  fprintf(pipef, "%s", aline);
  SETDEFAULT;
  fprintf(pipef, "set xlabel '%s'\n", xlab);
  fprintf(pipef, "set ylabel '%s'\n", ylab);
  fprintf(pipef, _("set title 'STATIST: Linear Regression'\n",
			 "set title 'STATIST: Lineare Regression'\n") );
  fprintf(pipef, "plot '%s', f(x)\n", GPL_DAT);
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}


BOOLEAN plot_tripl(REAL x[], REAL y[], REAL z[], int n,
                   REAL a0, REAL a1, REAL a2,
		   char *xlab, char *ylab, char *zlab) {
  FILE *tempf;
  int i;
  char aline[160];

  if (!init_gnuplot()) {
    return FALSE;
  }

  FOPEN(GPL_DAT, "wt", tempf);
  for (i=0; i<n; i++) {
    fprintf(tempf, "%g %g %g\n", x[i], y[i], z[i]);
  }
  FCLOSE(tempf);
  sprintf(aline,  "f(u,v)=%g + %g*u + %g*v\n", a0, a1, a2);
  SETDEFAULT;
  fprintf(pipef, "set parametric\n");
  fprintf(pipef, "%s", aline);
  fprintf(pipef, "set xlabel '%s'\n", xlab);
  fprintf(pipef, "set ylabel '%s'\n", ylab);
  fprintf(pipef, "set zlabel '%s'\n", zlab);
  fprintf(pipef, _("set title 'STATIST: Multiple Linear Regression'\n",
			 "set title 'STATIST: Multiple Lineare Regression'\n") );
  fprintf(pipef, "splot [%g:%g][%g:%g][] ",
          get_min(x,n), get_max(x,n),get_min(y,n), get_max(y,n)) ;
  fprintf(pipef, "'%s', '%s' with impulses, u,v,f(u,v)\n", GPL_DAT, GPL_DAT);
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}


BOOLEAN plot_cdf_ks(REAL z[], int n, REAL z0, REAL fn1, REAL fn2, 
		 REAL x[], char *xlab) {
  const REAL zmin=(-3.0), zmax=3.0;
  FILE *tempf;
  REAL cdf;
  int i;
  
  if (!init_gnuplot()) {
    return FALSE;
  }

  FOPEN(GPL_DAT, "wt", tempf);
  fprintf(tempf, "%g %g\n", zmin, 0.0);    
  fprintf(tempf, "%g %g\n", z[0], 0.0);  
  for (i=0; i<(n-1); i++) {
    cdf = (REAL)(i+1)/(REAL)n;
    fprintf(tempf, "%g %g\n", z[i], cdf);
    fprintf(tempf, "%g %g\n", z[i+1], cdf);
  }
  fprintf(tempf, "%g %g\n", z[n-1], 1.0); 
  fprintf(tempf, "%g %g\n", zmax, 1.0);
/*  
  fprintf(tempf, "\n%g %g\n%g %g\n", zmin, fn1, zmax, fn1);
  fprintf(tempf, "\n%g %g\n%g %g\n", zmin, fn2, zmax, fn2);
*/  
  FCLOSE(tempf);
  
  SETDEFAULT;
  fprintf(pipef, "set xlabel 'N(%s)'\n", xlab);
  fprintf(pipef, "set ylabel 'S(%s)'\n", xlab);
  fprintf(pipef, _("set title 'STATIST: KS-Lilliefors-Test'\n",
  			 "set title 'STATIST: KS-Lilliefors-Test'\n") );
  fprintf(pipef, "u(x)=%g\n", fn1);
  fprintf(pipef, "l(x)=%g\n", fn2);
  fprintf(pipef, "plot [%g:%g] [0:1.02] '%s' with lines, norm(x), u(x), ",
	  zmin, zmax, GPL_DAT);
  fprintf(pipef, "l(x) with lines 3\n");
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}

BOOLEAN plot_cdf(REAL z[], int n, char *xlab) {
  const REAL zmin= z[0], zmax = z[n-1]; 
  FILE *tempf;
  REAL cdf;
  int i;
  
  if (!init_gnuplot()) {
    return FALSE;
  }

  FOPEN(GPL_DAT, "wt", tempf);
/*  fprintf(tempf, "%g %g\n", z[0], 0.0);  */
  for (i=0; i<(n-1); i++) {
    cdf = (REAL)(i+1)/(REAL)n;
    fprintf(tempf, "%g %g\n", z[i], cdf);
    fprintf(tempf, "%g %g\n", z[i+1], cdf);
  }
  fprintf(tempf, "%g %g\n", zmax, 1.0);
  FCLOSE(tempf);
  
  SETDEFAULT;
  fprintf(pipef, "set xlabel '%s'\n", xlab);
  fprintf(pipef, "set ylabel '%%(%s)'\n", xlab);
  fprintf(pipef, _("set title 'STATIST: Cummulative distribtion'\n",
  			 "set title 'STATIST: Kummulative Verteilung'\n") );
  fprintf(pipef, "plot [%g:%g] [0:1.02] '%s' with lines\n", zmin, zmax, GPL_DAT);
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}


BOOLEAN plot_histo(REAL x[], int y[], int n, REAL step, REAL data[],
		   char *datalab) {
  FILE *tempf;
  unsigned int i, miny, maxy;
  REAL minx, maxx;

  if (!init_gnuplot()) {
    return FALSE;
  }

  FOPEN(GPL_DAT, "wt", tempf);
  for (i=0; i<n; i++) {
    fprintf(tempf, "%g %i\n", x[i], y[i]);
  }
  miny = 0;
  maxy = (int)(get_maxint(y, n)*1.05) + 1;
  minx = get_min(x, n);
  maxx = get_max(x, n);
  FCLOSE(tempf);
  SETDEFAULT;
  fprintf(pipef, _("set title 'STATIST: Histogram'\n",
  			 "set title 'STATIST: Histogramm'\n") );
  fprintf(pipef, _("set ylabel 'Frequency'\n",
  			 "set ylabel 'Hufigkeit'\n") );
  fprintf(pipef, "set xlabel '%s'\n", datalab);
  if(((maxx+step/2.)-(minx-step/2.)) /step >10)
  {
  	fprintf(pipef, "set xtics\n");
  } else
  	fprintf(pipef, "set xtics %g, %g, %g\n", 
		(minx-step/2.), step, (maxx+step/2.));
  fprintf(pipef, "plot [%g:%g][%i:%i] '%s' with boxes linetype linetype\n",
          minx-step/2., maxx+step/2., miny, maxy, GPL_DAT);
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}


BOOLEAN plot_histo_discrete(REAL x[], int y[], int n, REAL step, REAL data[],
		   char *datalab) {
  FILE *tempf;
  unsigned int i, miny, maxy;
  REAL minx, maxx;

  if (!init_gnuplot()) {
    return FALSE;
  }

  FOPEN(GPL_DAT, "wt", tempf);
  for (i=0; i<n; i++) {
    fprintf(tempf, "%g %i\n", x[i], y[i]);
  }
  miny = 0;
  maxy = get_maxint(y, n) + 1;
  minx = get_min(x, n) - (step/2.);
  maxx = get_max(x, n) + (step/2.);
  FCLOSE(tempf);
  SETDEFAULT;
  fprintf(pipef, _("set title 'STATIST: Histogram'\n",
  			 "set title 'STATIST: Histogramm'\n") );
  fprintf(pipef, _("set ylabel 'Frequency'\n",
  			 "set ylabel 'Hufigkeit'\n") );
  fprintf(pipef, "set xlabel '%s'\n", datalab);
  fprintf(pipef, "set xtics %g, %g, %g\n", (minx+step/2.), step, (maxx-step/2.));
/*  fprintf(pipef, "set boxwidth %g\n", step/4. ); */
/*  fprintf(pipef, "plot [%g:%g][%i:%i] '%s' with boxes\n", */
  fprintf(pipef, "plot [%g:%g][%i:%i] '%s' with impulses\n",
          minx, maxx, miny, maxy, GPL_DAT);
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}

BOOLEAN plot_probit(REAL dose[], REAL num[], REAL effect[], int n,
                    REAL a0, REAL a1, REAL dose0, REAL dose1, 
		    char *doselab, char *effectlab) {
  FILE *tempf;
  int i;
#ifndef STATIST_X
  out_r("plot_probit: doselab=|%s|, effectlab=|%s|\n", 
	 doselab, effectlab);
#endif
  if (!init_gnuplot()) {
    return FALSE;
  }

  FOPEN(GPL_DAT, "wt", tempf);
  for (i=0; i<n; i++) {
    fprintf(tempf, "%g %g\n", dose[i], (effect[i]/num[i])*100.);
  }
  FCLOSE(tempf);
  SETDEFAULT;
  fprintf(pipef, _("set xlabel 'Dose %s'\n",
  			 "set xlabel 'Dosis %s'\n"), doselab);
  fprintf(pipef, _("set ylabel 'Effect %s [%%]'\n",
  			 "set ylabel 'Wirkung %s [%%]'\n"), effectlab);
  fprintf(pipef, "set grid\n");
  fprintf(pipef, "set log x\n");
  fprintf(pipef, "a0=%g\n", a0);
  fprintf(pipef, "a1=%g\n", a1);
  fprintf(pipef, "f(x)=norm((log10(x)*a1+a0)-5)*100\n");
  fprintf(pipef, _("set title 'STATIST: Probitanalysis'\n",
  			 "set title 'STATIST: Probitanalyse'\n") );
  fprintf(pipef, "plot [%g:%g] '%s', f(x)\n",
          dose0, dose1, GPL_DAT);
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}



BOOLEAN plot_box(REAL x[], int n, REAL median, REAL mean, REAL q_l,
                 REAL q_u, REAL w_l, REAL w_u, REAL no_l, 
		 REAL no_u, char *xlab) {
  FILE *tempf;
  int i;
  REAL min, max, off;

  if (!init_gnuplot()) {
    return FALSE;
  }

  min = get_min(x, n);
  max = get_max(x, n);
  off = (max-min)/20.;

  FOPEN(GPL_DAT, "wt", tempf);
  fprintf(tempf, "%g 10\n", q_l);
  fprintf(tempf, "%g 12\n", q_l);
  fprintf(tempf, "%g 12\n", q_u);
  fprintf(tempf, "%g 10\n", q_u);
  fprintf(tempf, "%g 10\n\n", q_l);
  fprintf(tempf, "%g 10\n", median);
  fprintf(tempf, "%g 12\n\n", median);
  fprintf(tempf, "%g 11\n", w_l);
  fprintf(tempf, "%g 11\n\n", q_l);
  fprintf(tempf, "%g 10.7\n", w_l);
  fprintf(tempf, "%g 11.3\n\n", w_l);
  fprintf(tempf, "%g 11\n", q_u);
  fprintf(tempf, "%g 11\n\n", w_u);
  fprintf(tempf, "%g 10.7\n", w_u);
  fprintf(tempf, "%g 11.3\n\n", w_u);
  fprintf(tempf, "%g 11.1\n", no_l+off*0.2);
  fprintf(tempf, "%g 11.1\n", no_l);
  fprintf(tempf, "%g 10.9\n", no_l);
  fprintf(tempf, "%g 10.9\n\n", no_l+off*0.2);
  fprintf(tempf, "%g 11.1\n", no_u-off*0.2);
  fprintf(tempf, "%g 11.1\n", no_u);
  fprintf(tempf, "%g 10.9\n", no_u);
  fprintf(tempf, "%g 10.9\n\n", no_u-off*0.2);
  FCLOSE(tempf);

  SETDEFAULT;

  for (i=0; i<n; i++) {
    if ((x[i]>w_u) || (x[i]<w_l)) {
      fprintf(pipef, "set label '*' at %g, 11 center\n", x[i]);
    }
  }

  fprintf(pipef, "set noytics\n");
  fprintf(pipef, "set label\n");
  fprintf(pipef, "set label 'o' at %g, 11 center\n", mean);
  fprintf(pipef, "set label 'n=%i' at %g, 12.7 right\n", n, max);
  fprintf(pipef, _("set title 'STATIST: Box-and-Whisker Plot'\n",
  			 "set title 'STATIST: Box-and-Whisker Plot'\n") );
  fprintf(pipef, "set xlabel '%s'\n", xlab);
  fprintf(pipef, "plot [%g:%g][9:13] '%s' with lines\n",
          min-off, max+off, GPL_DAT);
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}




BOOLEAN  plot_poly(REAL x[], REAL y[], int n, REAL a[], int npoly,
		   char *xlab, char *ylab) {
  FILE *tempf;
  int i;
  char term[255], aline[255];

  if (!init_gnuplot()) {
    return FALSE;
  }

  FOPEN(GPL_DAT, "wt", tempf);
  for (i=0; i<n; i++) {
    fprintf(tempf, "%g %g\n", x[i], y[i]);
  }
  FCLOSE(tempf);
  strcpy(aline, "f(x) = ");
  for (i=0; i<=npoly; i++) {
    fprintf(pipef, "a%i=%g\n", i, a[i]);
    sprintf(term, "a%i*x**%i+", i, i);
    strncat(aline, term, 255-strlen(aline));
  }
  strcat(aline, "0\n");
  fprintf(pipef, "%s\n", aline);
  SETDEFAULT;
  fprintf(pipef, "set xlabel '%s'\n", xlab);
  fprintf(pipef, "set ylabel '%s'\n", ylab);
  fprintf(pipef, _("set title 'STATIST: Polynomial Regression of Order %i'\n",
  			 "set title 'STATIST: Polynomregression %i. Grades'\n"), 
		npoly);
  fprintf(pipef, "plot '%s', f(x)\n", GPL_DAT);
  PLOT_CLOSE;
  PLOT_MSG;
  return TRUE;
}


#ifndef MSDOS
BOOLEAN plot_command() {
  char aline[80];

  if (!init_gnuplot()) {
    return FALSE;
  }

  do {
    out_d("gnuplot> ");
    fgets(aline, 79, stdin);
    if ((aline[0]!='.') && (strcmp("quit", aline)!=0)) {
      fprintf(pipef, "%s\n", aline);
      fflush(pipef);
    }
  }
  while ((aline[0]!='.') && (strcmp("quit", aline)!=0));
  if (strcmp(aline, "quit")==0) {
    pclose(pipef);
    gnupl_open = FALSE;
  }
  return TRUE;
}
#endif
