/*--------------------------------------------------------
   util.c - several routines for exiting, logging and signals.

--------------------------------------------------------*/


/*--------------------------------------------------------

REALNETWORKS LICENSE AGREEMENT AND WARRANTY 
             DISCLAIMER

_____________________________________________

Free Real-Time Streaming Protocol (RTSP) Firewall 
             Proxy License

IMPORTANT -- READ CAREFULLY: This RealNetworks 
License Agreement ("License Agreement") is a legal 
agreement between you (either an individual or an 
entity) and RealNetworks, Inc.  and its suppliers and 
licensors collectively ("RN") for the software product 
listed above, which includes computer software and 
associated media and printed material, whether provided 
in a physical form or received on-line form ("Software").  
By clicking on the "Accept" button or opening the 
package, you are consenting to be bound by this Agreement.  
If you do not agree to all of the terms of this agreement, 
click the "Do Not Accept" button and, if you received the 
Software by package, return the product to the place of 
purchase.

__________________________________________________________

1. GRANT OF LICENSE.

Subject to the provisions contained in this License Agreement, 
RN hereby grants you a non-exclusive, non-transferable, 
perpetual, worldwide license to use, modify or redistribute the 
Software subject to the following terms and conditions:

(a) The copyright notice (" 1998 RealNetworks, 
Inc.") and this copy of  this License Agreement shall 
appear on all copies and/or any derivative versions 
of the Software you create or distribute.

(b)	You acknowledge and agree that RN is and shall be 
the exclusive owner of all right, title and interest, 
including copyright, in the Software.

All rights not expressly granted to you are reserved to RN.

2.  SOFTWARE MAINTENANCE AND UPGRADES. 

RN is not obligated to provide maintenance or updates to you 
for the Software. However, any maintenance or updates 
provided by RN shall be covered by this Agreement.

3.  DISCLAIMER OF WARRANTY.

The Software is deemed accepted by you.  Because RN is 
providing you the Software for free, the Software is provided 
to you AS IS, WITHOUT WARRANTY OF ANY KIND. TO 
THE MAXIMUM EXTENT PERMITTED BY 
APPLICABLE LAW, REALNETWORKS FURTHER 
DISCLAIMS ALL
WARRANTIES, INCLUDING WITHOUT LIMITATION 
ANY IMPLIED WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE, AND NONINFRINGEMENT. THE ENTIRE 
RISK ARISING OUT OF THE USE OR PERFORMANCE 
OF THE SOFTWARE REMAINS WITH YOU. TO THE 
MAXIMUM EXTENT PERMITTED BY APPLICABLE 
LAW, IN NO
EVENT SHALL REALNETWORKS OR ITS SUPPLIERS 
BE LIABLE FOR ANY CONSEQUENTIAL, INCIDENTAL, 
DIRECT, INDIRECT, SPECIAL, PUNITIVE, OR OTHER 
DAMAGES WHATSOEVER (INCLUDING, WITHOUT 
LIMITATION, DAMAGES FOR LOSS OF BUSINESS 
PROFITS, BUSINESS INTERRUPTION, LOSS OF 
BUSINESS INFORMATION, OR OTHER PECUNIARY 
LOSS) ARISING OUT OF THIS AGREEMENT OR THE 
USE OF OR INABILITY TO USE THE SOFTWARE, EVEN 
IF REALNETWORKS HAS BEEN ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGES. BECAUSE SOME 
STATES/JURISDICTIONS DO NOT ALLOW THE 
EXCLUSION OR LIMITATION OF LIABILITY FOR 
CONSEQUENTIAL OR INCIDENTAL DAMAGES, THE 
ABOVE LIMITATION MAY NOT APPLY TO YOU.

4. INDEMNIFICATION. 

You hereby agree to defend, indemnify, and hold RN, its 
directors, officers, employees and agents, harmless from any 
and all claims, damages, and expenses (including attorneys 
fees and costs) of any nature arising out of the use, 
modification, or redistribution the Software or any derivative 
versions thereof.

5. U.S. GOVERNMENT RESTRICTED RIGHTS AND 
EXPORT RESTRICTIONS. 

The Software is provided with RESTRICTED RIGHTS. Use, 
duplication, or disclosure by the Government is subject to 
restrictions as set forth in subparagraph (a) through (d) of the 
of the Commercial Computer Software-Restricted Rights at 
FAR 52.227-19, as applicable, or subparagraph (c)(1)(ii) of 
The Rights in Technical Data and Computer Software clause 
of DFARS 252.227-7013, and in similar clauses in the NASA 
FAR supplement, as applicable.  Manufacturer is 
RealNetworks, Inc., 1111 Third Avenue, Suite 500, Seattle, 
Washington 98101.  You acknowledge that none of the 
Software or underlying information or technology may be 
downloaded or otherwise exported or re-exported (i) into (or 
to a national or resident of) Cuba, Iraq, Libya, Yugoslavia, 
North Korea, Iran, Syria, Sudan or Angola or any other 
country to which the U.S. has embargoed goods; or (ii) to 
anyone on the U.S. Treasury Department's list of Specially 
Designated Nationals or the U.S. Commerce Department's 
Table of Denial Orders.  By using the Software, you are 
agreeing to the foregoing and you are representing and 
warranting that you are not located in, under the control of, or 
a national or resident or resident of any such country or on any 
such list.

6. GOVERNING LAW; ATTORNEYS FEES. 

This agreement shall be governed by the laws of the State of 
Washington and you further consent to jurisdiction by the state 
and federal courts sitting in the State of Washington. If either 
RN or you employs
attorneys to enforce any rights arising out of or relating to this 
Agreement, the prevailing party shall be entitled to recover 
reasonable attorneys' fees.

8.  ENTIRE AGREEMENT. 

This agreement constitutes the complete and exclusive 
agreement between RN and you with respect to the subject 
matter hereof, and supersedes all prior oral or written 
understandings, communications or agreements not 
specifically incorporated herein.  This agreement may not be 
modified except in a writing duly signed by an authorized 
representative of RN and you.    

Copyright  1997-1998 RealNetworks, Inc. and or its 
suppliers.  1111 Third Avenue, Suite 2900, Seattle, 
Washington 98101 U.S.A.  All rights are reserved.



RTSP Proxy License Agreement 8-98

--------------------------------------------------------*/




#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>
#include <errno.h>
#ifdef USE_SYS_SYSLOG_H
#include <sys/syslog.h>
#else
#include <syslog.h>
#endif /* USE_SYS_SYSLOG_H */
#include <strings.h>
#ifdef NEED_STRING_H
#include <string.h>
#endif /* NEED_STRING_H */
#include <stdarg.h>
#include <signal.h>

#include "util.h"
#include "setup.h"


/* a buffer for message to print at exit */
#ifndef MAX_EXIT_MESSAGE
#define MAX_EXIT_MESSAGE 1024
#endif /* MAX_EXIT_MESSAGE */



/*----------------------------------------------------------------------
   exit_proxy() -- Exit, potentially logging a message to stderr and/or syslog.

   Paramaters:
     n - An exit code, from the ExitValue enum.

   Return:  nothing (exits before returning)
----------------------------------------------------------------------*/
void 
exit_proxy (ExitValue n)
{
	switch (n)
	{
		case EXIT_NORMAL:
		case EXIT_CLIENT_READ_ERROR:
		case EXIT_SERVER_READ_ERROR:
			/* normal exit conditions, so exit quietly with 0 */
	  		log_message (INFO_LOG, "proxy exiting");
			exit (0);
			break;

		case EXIT_USAGE:
		case EXIT_VERSION:
		case EXIT_QUIET:
			/* even more terse for "-v" and "-h" exit, and for
			   exiting processes which are only used to properly
			   start as a daemon.
			 */
			exit (0);
			break;

		default:
			log_message (INFO_LOG, exit_string (n));
	  		log_message (INFO_LOG, "proxy exiting");
			exit (n);
			break;
	}


}



/*----------------------------------------------------------------------
   exit_string() -- Given an exit value, return a descriptive string.

   Paramaters:
     n - An exit code, from the ExitValue enum.

   Return:
     A character string describing the exit code.

   Re-uses a static message buffer, so copy it if you want to keep it around.
----------------------------------------------------------------------*/
char *
exit_string (ExitValue n)
{
	static char error_message[MAX_EXIT_MESSAGE+1];

	switch (n)
	{
		case EXIT_NORMAL:
			return "Normal Exit";
			break;

		case EXIT_GENERIC_ERROR:
			return "Unknown Error";
			break;

		case EXIT_ACCEPT_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "accept() returned a bad value: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_SELECT_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "select() returned a bad value: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_SOCKET_CREATE_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "socket creation error: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_SOCKET_CONNECT_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "socket connect error: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_BIND_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "bind() error: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_LISTEN_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				"listen() error: errno=%d (%s)",
				errno, strerror(errno));
			return error_message;
			break;

		case EXIT_FORK_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "fork() error: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_CHROOT_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "chroot() error: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_SETUID_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "setuid() error: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_UNKNOWN_USER_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "setuid() error: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_BAD_USER_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "setuid() error: errno=%d (%s)",
				  errno, strerror(errno));
			return error_message;
			break;

		case EXIT_PORT_ALLOCATION_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "port allocation error");
			return error_message;
			break;

		case EXIT_MALLOC_ERROR:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "malloc() error");
			return error_message;
			break;

		case EXIT_INFINITE_LOOP:
			snprintf (error_message, MAX_EXIT_MESSAGE,
				  "exiting to prevent infinite loop");
			return error_message;
			break;


		default:
			return "";
			break;

	}

	return "";
}



/*======================================================================*/

/* This sets up the values for the level paramater to syslog(3). */
#if defined DEBUG && defined DEBUG_TO_SYSLOG
/* so everything goes to one log for ease of debugging */
#define DEBUG_LEVEL LOG_DEBUG
#define ERROR_LEVEL LOG_DEBUG
#define INFO_LEVEL  LOG_DEBUG
#else
#define ERROR_LEVEL LOG_ERR
#define INFO_LEVEL  LOG_INFO
#endif /* DEBUG && DEBUG_TO_SYSLOG */



/*----------------------------------------------------------------------
   log_message() -- A wrapper routine for using syslog().

   Paramaters:
     flag - An identifier indicating which log to send the message to.
            This may be one of DEBUG_LOG, ERROR_LOG or INFO_LOG.
     format - A printf-style format string.
     ... - additional arguments, like printf.

   Return: nothing
----------------------------------------------------------------------*/
void
log_message (int flag, char *format, ...)
{ 
	char buff[LOG_MESSAGE_MAX+1]="";
	va_list args;

	va_start (args, format);
	vsnprintf (buff, LOG_MESSAGE_MAX, format, args);

	switch (flag)
	{
		/* may not want to also send debug output to syslog */
#if defined DEBUG && defined DEBUG_TO_SYSLOG
		case DEBUG_LOG:
			openlog (SYSLOG_IDENT, LOG_PID, SYSLOG_FACILITY);
			syslog (SYSLOG_FACILITY|DEBUG_LEVEL, buff);
			closelog ();
			break;
#endif /* DEBUG && DEBUG_TO_SYSLOG */

		case ERROR_LOG:
			openlog (SYSLOG_IDENT, LOG_PID, SYSLOG_FACILITY);
			syslog (SYSLOG_FACILITY|ERROR_LEVEL, buff);
			closelog ();
			break;

		case INFO_LOG:
			openlog (SYSLOG_IDENT, LOG_PID, SYSLOG_FACILITY);
			syslog (SYSLOG_FACILITY|INFO_LEVEL, buff);
			closelog ();
			break;

		default:
			break;
	}

/* For DEBUG we always fprintf() it, even if we also syslog() it. */
#ifdef DEBUG
	fprintf (stderr, "%s[%d]: %s\n", SYSLOG_IDENT, (int)getpid(), buff);
#endif /* DEBUG */

	va_end(args);
}





/*======================================================================*/



/*----------------------------------------------------------------------
   sig_func() -- Mimic signal() using sigaction() for robustness.

   Paramaters:
     sig - The signal to set the signal handler for
     sig_func - The signal handler function to associate with the signal.

   Return:
     A pointer to the previous signal handler for this signal.

   On some platforms, such as libc5-based Linux, signal() is less relable
   than sigaction().  You can define USE_SIGNAL if necessary to just call
   signal().
----------------------------------------------------------------------*/
sig_func *
set_signal_handler (int sig, sig_func *func)
{
# ifdef USE_SIGNAL
  return (signal (sig, func));
# else

  struct sigaction action, prev_action;
  sigemptyset (&action.sa_mask);
  action.sa_flags = 0;
  action.sa_handler = func;

  if (sig == SIGALRM)
  {
#   ifdef SA_INTERRUPT
    action.sa_flags |= SA_INTERRUPT;
#   endif /* SA_INTERRUPT */
  }
  else
  {
#   ifdef SA_RESTART
    action.sa_flags |= SA_RESTART;
#   endif /* SA_RESTART */
  }

  if (sigaction (sig, &action, &prev_action) < 0)
    return (SIG_ERR);

  return (prev_action.sa_handler);

#endif /* USE_SIGNAL */
}


/*======================================================================*/



#ifdef NEED_SNPRINTF
/*----------------------------------------------------------------------
   snprintf() -- Simplistic replacement for snprintf().

   Paramaters:
     str    - A preallocaded string buffer in which to place the results.
     n      - Maximum length of str.
     format - A printf-style format string.
     ...    - Additional arguments, like printf.

   Return:
       The return value is the number of characters stored, not
       including the terminating null.

   **NO** string length check occurs in this implementation, which
   defeats the entire purpose of using snprintf().  This implementation
   should not be used if at all possible.
----------------------------------------------------------------------*/
int
snprintf (char *str, size_t n, const char *format, ...)
{
        va_list args;
        int ret=0;

        va_start (args, format);
        ret = vsprintf (str, format, args); /* yikes! don't do this! */
	va_end(args);
	return ret;
}
#endif



#ifdef NEED_VSNPRINTF
/*----------------------------------------------------------------------
   vsnprintf() -- Simplistic replacement for vsnprintf().

   Paramaters:
     str    - A preallocaded string buffer in which to place the results.
     n      - Maximum length of str.
     format - A printf-style format string.
     ap     - Additional arguments, like vprintf.

   Return:
       The return value is the number of characters stored, not
       including the terminating null.

   **NO** string length check occurs in this implementation, which
   defeats the entire purpose of using snprintf().  This implementation
   should not be used if at all possible.
----------------------------------------------------------------------*/
int
vsnprintf (char *str, size_t n, const char *format, va_list ap)
{
	return vsprintf (str, format, ap); /* yikes! don't do this! */
}
#endif
