/*  QuadruplebackEngine.cpp - Game engine.

    quadrupleback - A video game where intruders must be circled.
    Copyright (C) 2012-2024 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#include "QuadruplebackEngine.h"

#include <flatzebra/PixmapLoadError.h>

#include "Enemy.h"

// Some XPM images are compiled as C, in addition to be loaded by the
// QuadruplebackEngine constructor, so that the C structures can be used
// for collision detection.
//
#include "images/cherry.xpm"
#include "images/apple.xpm"
#include "images/pear.xpm"
#include "images/horseshoe.xpm"
#include "images/skull.xpm"
#include "images/skate0.xpm"
#include "images/spider0.xpm"
#include "images/yoyo.xpm"

#include <fstream>

using namespace std;
using namespace flatzebra;


// Normally defined by Makefile.am:
#ifndef PKGDATADIR
#define PKGDATADIR "."
#endif

#ifndef PKGSOUNDDIR
#define PKGSOUNDDIR "sounds"
#endif


///////////////////////////////////////////////////////////////////////////////


QuadruplebackEngine::QuadruplebackEngine(const string &windowManagerCaption,
                                         bool _useSound,
                                         bool _useAcceleratedRendering)
  : GameEngine(Couple(900, 675), windowManagerCaption, false, true, _useAcceleratedRendering),
                  // may throw string exception
    paused(false),
    tickCount(0),
    errorMessage(),

    quitKS(SDLK_ESCAPE),
    startKS(SDLK_SPACE),
    fullScreenKS(SDLK_F11),
    pauseResumeKS(SDLK_p),
    upKS(SDLK_UP),
    downKS(SDLK_DOWN),
    leftKS(SDLK_LEFT),
    rightKS(SDLK_RIGHT),
    throttleKS(SDLK_LCTRL),

    livesArea(MARGIN,
              MARGIN,
              theScreenSizeInPixels.x - 2 * MARGIN,
              CHAR_HEIGHT,
              mapRGBA(  0, 128, 255),
              mapRGBA(255, 255, 255)),
    scoreArea(MARGIN,
              theScreenSizeInPixels.y - MARGIN - CHAR_HEIGHT,  // at bottom
              theScreenSizeInPixels.x - 2 * MARGIN,
              CHAR_HEIGHT,
              mapRGBA(  0, 128, 255),
              mapRGBA(255, 255, 255)),
    gameArea( MARGIN,
              MARGIN + livesArea.size.y,
              theScreenSizeInPixels.x - 2 * MARGIN,
              theScreenSizeInPixels.y - 2 * MARGIN - livesArea.size.y - scoreArea.size.y,
              mapRGBA(  0,   0,   0),  // unused
              mapRGBA(  0,   0,   0)),
    presentationColor(  mapRGBA(255, 255, 255)),
    errorMessageColor(  mapRGBA(255,   0,   0)),
    pauseBGColor(       mapRGBA( 64,  64,  64)),
    snakeColor(         mapRGBA(  0, 128, 255)),
    loopColor(          mapRGBA(255,   0, 255)),
    intersectionColor(  mapRGBA(255, 255,   0)),
    currentLifeColor(   mapRGBA(255, 128,   0)),
    otherLivesColor(    mapRGBA(0,   128, 255)),
    floatingNumberColor(mapRGBA(255, 128,   0)),
    yoyoWireColor(      mapRGBA(255, 128,   0)),
    deathColor(         mapRGBA(  0, 255,   0, 128)),
    joystick(),
    snake(),
    enemies(),
    maxNumEnemies(0),
    minTimeBeforeNextEnemy(0),
    maxTimeBeforeNextEnemy(0),
    nextEnemyTime(size_t(-1)),
    numEnemiesKilled(0),
    gamePhase(WAITING_FOR_PLAYER),
    preLifeEndTime(size_t(-1)),
    deathEndTime(size_t(-1)),
    numLives(0),
    score(0),
    floatingNumbers(),
    cherryPA(),
    applePA(),
    pearPA(),
    horseshoePA(),
    skullPA(),
    skatePA(),
    spiderPA(),
    yoyoPA(),
    fontDefBuffer(),
    useSound(_useSound),
    theSoundMixer(NULL),
    enemyAppearsSound(),
    enemyCapturedSound(),
    enemyTouchedSound()
{
    loadPixmap("cherry.xpm", cherryPA, 0);
    loadPixmap("apple.xpm", applePA, 0);
    loadPixmap("pear.xpm", pearPA, 0);
    loadPixmap("horseshoe.xpm", horseshoePA, 0);
    loadPixmap("skull.xpm", skullPA, 0);
    loadPixmap("skate0.xpm", skatePA, 0);
    loadPixmap("skate1.xpm", skatePA, 1);
    loadPixmap("spider0.xpm", spiderPA, 0);
    loadPixmap("spider1.xpm", spiderPA, 1);
    loadPixmap("yoyo.xpm", yoyoPA, 0);

    const char *pkgDataDir = getenv("PKGDATADIR");
    if (pkgDataDir == NULL)
        pkgDataDir = PKGDATADIR;
    const char *pkgSoundDir = getenv("PKGSOUNDDIR");
    if (pkgSoundDir == NULL)
        pkgSoundDir = PKGSOUNDDIR;

    // Load font definition file.
    // Must be consistent with CHAR_WIDTH and CHAR_HEIGHT.
    {
        enum { FILE_LEN = 9216 };
        fontDefBuffer.resize(FILE_LEN);
        assert(fontDefBuffer.length() == FILE_LEN);

        string filename = string(pkgDataDir) + "/fonts/9x18B.fnt";

        ifstream file(filename.c_str());
        if (!file)
            throw string("failed to open font definition file ") + filename;
        file.read(const_cast<char *>(fontDefBuffer.data()), FILE_LEN);
        if (!file)
            throw string("failed to read font definition file ") + filename;
        if (file.tellg() != FILE_LEN)
            throw string("failed to read entire font definition file ") + filename;

        // Set the font to be used by gfxWriteString(), i.e., SDL2_gfx's stringRGBA().
        gfxPrimitivesSetFont(fontDefBuffer.data(), 9, 18);
    }

    if (useSound)
    {
        try
        {
            if (theSoundMixer == NULL)
                theSoundMixer = new SoundMixer();
        }
        catch (const SoundMixer::Error &e)
        {
            assert(theSoundMixer == NULL);
            cerr << PACKAGE << ": failed to create sound mixer: " << e.what() << endl;
            useSound = false;
            errorMessage = "failed to initialize sound device";
        }

        if (useSound)
        {
            try
            {
                string d = string(pkgSoundDir) + "/";

                enemyAppearsSound.init(d  + "enemyAppears.wav");
                enemyCapturedSound.init(d + "enemyCaptured.wav");
                enemyTouchedSound.init(d  + "enemyTouched.wav");
            }
            catch (const SoundMixer::Error &e)
            {
                // Not supposed to happen if package correctly installed.
                // Print trace for debugging purposes.
                cerr << PACKAGE << ": failed to load sounds: " << e.what() << endl;
                errorMessage = "failed to load sounds";
                useSound = false;
                delete theSoundMixer;
                theSoundMixer = NULL;
            }
        }
    }

    resetGame();
}


/*virtual*/
QuadruplebackEngine::~QuadruplebackEngine()
{
    deleteVectorElements(enemies);
    deleteVectorElements(floatingNumbers);
    delete theSoundMixer;
}


void
QuadruplebackEngine::loadPixmap(const char *filePath,
                                PixmapArray &pa,
                                size_t index)
{
    static const string dir = getDirPathFromEnv(PKGPIXMAPDIR, "PKGPIXMAPDIR");
    GameEngine::loadPixmap(dir + filePath, pa, index);
}


void
QuadruplebackEngine::resetGame()
{
    deleteVectorElements(enemies);

    maxNumEnemies = 10;
    minTimeBeforeNextEnemy = FPS * 1;
    maxTimeBeforeNextEnemy = FPS * 3;
    nextEnemyTime = size_t(-1);
    numEnemiesKilled = 0;

    gamePhase = WAITING_FOR_PLAYER;
    numLives = 0;

    initNewLife();
}


void
QuadruplebackEngine::initNewLife()
{
    snake.reset(M_PI / 2, gameArea.size / 2, 80, 14, RCouple(0, -16384));

    deleteVectorElements(enemies);

    preLifeEndTime = -1;
    deathEndTime = size_t(-1);
    nextEnemyTime = size_t(-1);
}


//virtual
void
QuadruplebackEngine::processKey(SDL_Keycode keysym, bool pressed)
{
    quitKS.check(keysym, pressed);
    startKS.check(keysym, pressed);
    fullScreenKS.check(keysym, pressed);
    pauseResumeKS.check(keysym, pressed);
    upKS.check(keysym, pressed);
    downKS.check(keysym, pressed);
    leftKS.check(keysym, pressed);
    rightKS.check(keysym, pressed);
    throttleKS.check(keysym, pressed);
}


//virtual
void
QuadruplebackEngine::processActivation(bool appActive)
{
    if (!appActive)  // if app has just been made inactive
    {
        if (numLives > 0)  // if game is being played
        {
            paused = true;  // pause it
            drawScene();  // redraw scene to show pause message
        }
    }

    // If app has just been made active again, do nothing:
    // if a game was being played, it is now paused and
    // the pause message will remain displayed. The user can
    // thus choose the right moment to resume playing.
}


RCouple
QuadruplebackEngine::addPointToSnakeAndFindIntersections(double directionInRadians,
                                                      double speed)
{
    RCouple displacement = RCouple(cos(directionInRadians), -sin(directionInRadians))
                           * (speed * snake.maxSpeedInPixels);
    const RCouple &curheadPos = snake.points.back();
    RCouple newHeadPos = curheadPos + displacement;
    clamp(newHeadPos.x, 0, gameArea.size.x - 1);
    clamp(newHeadPos.y, 0, gameArea.size.y - 1);

    addPointToSnakeAndFindIntersections(newHeadPos);

    return newHeadPos;
}


void
QuadruplebackEngine::addPointToSnakeAndFindIntersections(const RCouple &newSegEnd)
{
    RCouple newSegStart = snake.points.back();

    snake.points.addPoint(newSegEnd);

    size_t numEnemiesKilled = 0;

    if (snake.points.size() >= 3)
    {
        #if 0
        cout << "findSnakeLoop(" << newSegStart << ", " << newSegEnd << "): ";
        for (PointSequence::const_iterator it = snake.points.begin(); it != snake.points.end(); ++it)
            cout << *it << " ";
        cout << "." << endl;
        #endif

        size_t numPointsToCheck = snake.points.size() - 2;  // do not check last segment
        RCouple prevPoint = snake.points.getPoint(0);
        for (size_t i = 1; i < numPointsToCheck; ++i)
        {
            const RCouple &curPoint = snake.points.getPoint(i);

            RCouple pt;
            if (segmentIntersection(prevPoint, curPoint, newSegStart, newSegEnd, pt))
            {
                //cout << "INTERSECTION: " << pt << endl;

                Intersection &inter = snake.points.addIntersection(pt, i - 1, numPointsToCheck);

                vector<RCouple> *newestLoop = inter.addLoop();
                snake.points.findNewestLoop(*newestLoop);

                numEnemiesKilled += killCircledEnemies(*newestLoop);
            }

            prevPoint = curPoint;
        }
    }

    if (numEnemiesKilled > 0)
        playSoundEffect(enemyCapturedSound);
}


size_t
QuadruplebackEngine::killCircledEnemies(const vector<RCouple> &loop)
{
    long scoredPoints = 0;
    size_t numKilledNow = 0;
    RCouple avgPos(0, 0);

    for (std::vector<Enemy *>::iterator it = enemies.begin();
                                       it != enemies.end(); )
    {
        Enemy *e = *it;
        if (e->isCapturedByLoop(loop))
        {
            scoredPoints += e->getPointValue();
            avgPos += e->getCapturePos();

            delete e;
            removeVectorElementQuick(enemies, it);

            ++numKilledNow;
            ++numEnemiesKilled;  // global counter
        }
        else
            ++it;
    }

    if (numKilledNow > 0)
    {
        scoredPoints *= numKilledNow;

        // Start displaying a number that slowly floats up for about a second.
        // The number's position is the average of the capture positions
        // of the captured enemies.
        //
        floatingNumbers.push_back(new FloatingNumber(scoredPoints, avgPos / numKilledNow, 1, FPS));

        score += scoredPoints;
    }

    return numKilledNow;
}


bool
QuadruplebackEngine::isPointOnEnemy(const RCouple &p) const
{
    for (std::vector<Enemy *>::const_iterator it = enemies.begin();
                                             it != enemies.end(); ++it)
    {
        const Enemy &e = **it;
        if (e.touchesPosition(p))
            return true;

        if (e.getType() == Enemy::YOYO)  // check yoyo's wire
        {
            RCouple wireTopLeft, wireSize;
            e.getWireRect(wireTopLeft, wireSize);

            // Increase width of rectangle to help catch snake head.
            wireTopLeft.x -= 3;
            wireSize.x += 6;

            if (RCouple::rectangleCollision(p, RCouple(1, 1), wireTopLeft, wireSize))
                return true;
        }
    }
    return false;
}


/*virtual*/
bool
QuadruplebackEngine::tick()
{
    tickCount++;

    joystick.update();

    if (quitKS.justPressed())
        return false;

    // Check for full screen toggle.
    //
    if (fullScreenKS.justPressed())
        setFullScreenMode(!inFullScreenMode());  // ignore failure

    // Check for pause/resume.
    //
    if (numLives > 0
        && !paused
        && (joystick.buttonJustPressed(Joystick::BACK_BTN) || pauseResumeKS.justPressed()))
    {
        paused = true;
    }
    else if (paused
        && (joystick.buttonJustPressed(Joystick::BACK_BTN) || pauseResumeKS.justPressed()))
    {
        paused = false;
    }

    if (!paused)
    {
        switch (gamePhase)
        {
        case WAITING_FOR_PLAYER:
        {
            assert(numLives == 0);

            moveSnakeRandomly();

            bool startBtnJustP = joystick.buttonJustPressed(Joystick::START_BTN);
            bool startKSJustP = startKS.justPressed();
            if (startBtnJustP
                || startKSJustP)
            {
                resetGame();
                numLives = 3;
                score = 0;
                gamePhase = DELAY_BEFORE_LIFE_START;
                preLifeEndTime = tickCount + FPS * 1;
            }
            break;
        }
        case DELAY_BEFORE_LIFE_START:
            assert(numLives > 0);
            if (tickCount >= preLifeEndTime)
            {
                gamePhase = PLAYING;
                preLifeEndTime = -1;
            }
            break;
        case PLAYING:
            assert(numLives > 0);
            play();
            break;
        case PLAYER_DYING:
            assert(numLives > 0);
            if (tickCount >= deathEndTime)  // if finished dying:
            {
                --numLives;
                if (numLives > 0)  // if another life to play:
                {
                    initNewLife();
                    gamePhase = DELAY_BEFORE_LIFE_START;
                    preLifeEndTime = tickCount + FPS * 1;
                }
                else
                    resetGame();
            }
            break;
        default:
            assert(false);
            return false;
        }
    }

    drawScene();

    fullScreenKS.remember();
    pauseResumeKS.remember();

    return true;
}


void
QuadruplebackEngine::moveSnakeRandomly()
{
    double speed = snake.maxSpeedInPixels / 20;

    if (tickCount % 3 == 0)
    {
        RCouple toCenter = gameArea.size / 2 - snake.getHeadPos();
        double toCenterLen = toCenter.length();

        if (toCenterLen < 300 && (rand() % 100 < 75 || toCenterLen < 100))
            snake.dirInRadians += (double(randSize(0, 50)) - 25) * M_PI / 180;
        else
            snake.dirInRadians = atan2(- toCenter.y, toCenter.x);
    }

    addPointToSnakeAndFindIntersections(snake.dirInRadians, speed);
}


void
QuadruplebackEngine::play()
{
    // Get joystick position (overridden by keyboard).
    //
    RCouple joystickPos(joystick.getXAxisValue(), joystick.getYAxisValue());

    const double minSqRadius = 10000 * 10000;

    RCouple keyboardPos;
    if (leftKS.isPressed())
        keyboardPos.x -= 32767;
    if (rightKS.isPressed())
        keyboardPos.x += 32767;
    if (upKS.isPressed())
        keyboardPos.y -= 32767;
    if (downKS.isPressed())
        keyboardPos.y += 32767;

    if (keyboardPos.isNonZero())  // if keyboard provided direction, use it instead of joystick
        joystickPos = keyboardPos;

    double directionInRadians = 0;
    double sqRadius = square(joystickPos.x) + square(joystickPos.y);
    if (sqRadius < minSqRadius)  // if joystick is not clearly oriented, keep previous direction
        joystickPos = snake.lastJoystickPos;
    else
        snake.lastJoystickPos = joystickPos;  // remember for next frame

    if (joystick.getButton(Joystick::A_BTN) || throttleKS.isPressed())
        joystickPos *= 0.5;

    directionInRadians = atan2(- joystickPos.y, joystickPos.x);
    double speed = joystickPos.length() / 36600.0;

    RCouple oldHeadPos = snake.getHeadPos();

    RCouple newHeadPos = addPointToSnakeAndFindIntersections(directionInRadians, speed);

    // Check if any part of the added segment touches an enemy.
    //
    RCouple delta = newHeadPos - oldHeadPos;
    enum { STEPS = 4 };
    for (size_t j = 0; j <= STEPS; ++j)
    {
        RCouple p = oldHeadPos + double(j) / STEPS * delta;
        if (isPointOnEnemy(p))
        {
            gamePhase = PLAYER_DYING;
            deathEndTime = tickCount + FPS * 2;

            playSoundEffect(enemyTouchedSound);
            return;
        }
    }


    if (enemies.size() < maxNumEnemies)  // if room for one more enemy:
    {
        if (nextEnemyTime == size_t(-1))  // if no enemy scheduled:
        {
            // The delay before the next enemy is in a range that gets closer to zero
            // as the game progresses.
            //
            minTimeBeforeNextEnemy = size_t(FPS * std::max(1 - numEnemiesKilled / 30.0, 0.0));
            maxTimeBeforeNextEnemy = size_t(FPS * std::max(3 - numEnemiesKilled / 15.0, 1.0));

            nextEnemyTime = tickCount + randSize(minTimeBeforeNextEnemy, maxTimeBeforeNextEnemy);
        }
        else if (tickCount >= nextEnemyTime)  // if time of scheduled enemy arrived:
        {
            const RCouple enemySize(40, 40);

            // Choose a random position that is not close to the snake's head.
            //
            RCouple pos;
            do
            {
                pos.x = randSize(4, gameArea.size.x - enemySize.x - 4);
                pos.y = randSize(4, gameArea.size.y - enemySize.y - 4);
            } while (isPositionNearSnakeHead(pos, enemySize.x * 4));

            // Choose enemy type.
            // Only introduce skulls after some time.
            // Limit total number of skulls, because they are not killable.
            // The yoyo requires special treatment in a few places.
            //
            Enemy::Type type;
            do
            {
                type = Enemy::chooseRandomly();
            } while (type == Enemy::SKULL && (numEnemiesKilled < 10 || getNumSkulls() >= 4));

            if (type == Enemy::YOYO)
                pos.y *= 0.45;  // restrict creation of yoyo to top half of game area, because it extends downward

            PixmapArray *pa = NULL;  // array of images to use in animation
            const char **xpm = NULL;  // pixmap to use for collision detection
            double speed = 0;  // in pixels per frame
            size_t tickOfDeath = size_t(-1);
            switch (type)
            {
            case Enemy::CHERRY:    pa = &cherryPA;      xpm = cherry_xpm;               break;
            case Enemy::APPLE:     pa = &applePA;       xpm = apple_xpm;                break;
            case Enemy::PEAR:      pa = &pearPA;        xpm = pear_xpm;                 break;
            case Enemy::HORSESHOE: pa = &horseshoePA;   xpm = horseshoe_xpm; speed = 3; break;
            case Enemy::SKULL:     pa = &skullPA;       xpm = skull_xpm;
                                   tickOfDeath = tickCount + randSize(10, 60) * FPS;    break;
            case Enemy::SKATE:     pa = &skatePA;       xpm = skate0_xpm;    speed = 3; break;
            case Enemy::SPIDER:    pa = &spiderPA;      xpm = spider0_xpm;   speed = 6; break;
            case Enemy::YOYO:      pa = &yoyoPA;        xpm = yoyo_xpm;      speed = 4; break;
            default: assert(!"invalid enemy type chosen");
            }

            RSprite *sprite = new RSprite(*pa, pos, RCouple(), RCouple(),
                                          RCouple(0, 0), enemySize);
            sprite->currentPixmapIndex = 0;

            Enemy *e = new Enemy(type, sprite, xpm, speed, tickOfDeath);
            enemies.push_back(e);

            if (type == Enemy::YOYO)
                e->setYoyoMaxLength(randSize(theScreenSizeInPixels.y / 4, theScreenSizeInPixels.y / 2));

            nextEnemyTime = size_t(-1);

            playSoundEffect(enemyAppearsSound);
        }
    }

    // Animate and move the enemies.
    //
    for (std::vector<Enemy *>::iterator it = enemies.begin();
                                       it != enemies.end(); )
    {
        Enemy *e = *it;

        // If this enemy dies automatically after a certain time, destroy it.
        //
        if (e->getTickOfDeath() != size_t(-1) && tickCount >= e->getTickOfDeath())
        {
            delete e;
            removeVectorElementQuick(enemies, it);
            continue;
        }

        ++it;

        RSprite &s = *e->sprite;

        // Advance the pixmap index in case this enemy has more than one image.
        //
        if (tickCount % e->getAnimationFreq() == 0)
        {
            e->updatePixmapIndex();
        }

        if (e->getType() == Enemy::YOYO)
        {
            e->animateYoyo();
        }
        else
        {
            if (e->isTimeToChangeTarget(tickCount))
            {
                RCouple targetPos;
                size_t numTicks = 0;
                if (rand() % 3 == 0)
                {
                    // Stupid mode.
                    double x = randReal(0.1, 0.9);
                    double y = randReal(0.1, 0.9);
                    targetPos = RCouple(gameArea.size.x * x, gameArea.size.y * y);
                    numTicks = randSize(3 * FPS, 15 * FPS);
                }
                else  // smart mode:
                {
                    targetPos = snake.getHeadPos();
                    numTicks = FPS;
                }
                e->setTargetPos(targetPos, tickCount + numTicks);
            }

            // Compute a vector that points from the enemy to the snake's head.
            // Adjust the vector's length to the enemy's speed.
            //
            RCouple delta = e->getTargetPos() - s.getCenterPos();
            double deltaLen = delta.length();
            if (deltaLen >= 1)
                delta *= e->getSpeed() / deltaLen;

            if (e->movesHorizOnly())
                delta.y = 0;

            s.setSpeed(delta);
            s.addSpeedToPos();
        }
    }
}


size_t
QuadruplebackEngine::getNumSkulls() const
{
    size_t num = 0;

    for (std::vector<Enemy *>::const_iterator it = enemies.begin();
                                             it != enemies.end(); ++it)
        if ((*it)->getType() == Enemy::SKULL)
            ++num;

    return num;
}


bool
QuadruplebackEngine::isPositionNearSnakeHead(const RCouple &pos, double maxDistance) const
{
    RCouple delta = snake.getHeadPos() - pos;
    return delta.length() <= maxDistance;
}


void
QuadruplebackEngine::drawFilledRect(const RCouple &topLeft,
                                    const RCouple &size,
                                    SDL_Color color)
{
    fillRect(int(topLeft.x), int(topLeft.y), int(size.x), int(size.y), color);
}


void
QuadruplebackEngine::drawAreaBG(const Area &area)
{
    drawFilledRect(area.topLeft, area.size, area.bgColor);
}


void
QuadruplebackEngine::drawScene()
{
    // Fill the whole screen in white.
    // Most of the screen will be filled in black by gameArea.
    // A white frame will be left around gameArea.
    //
    drawFilledRect(RCouple(0, 0), theScreenSizeInPixels, livesArea.bgColor);
    drawAreaBG(livesArea);
    drawAreaBG(gameArea);
    drawAreaBG(scoreArea);


    // Draw enemies.
    //
    for (std::vector<Enemy *>::const_iterator it = enemies.begin();
                                             it != enemies.end(); ++it)
    {
        const Enemy &e = **it;

        if (e.getType() == Enemy::YOYO)  // draw yoyo's wire first:
        {
            RCouple wireTopLeft, wireSize;  // dimensions of wire's rectangle
            e.getWireRect(wireTopLeft, wireSize);
            drawFilledRect(wireTopLeft + gameArea.topLeft, wireSize, yoyoWireColor);
        }

        copySpritePixmap(*e.sprite, e.sprite->currentPixmapIndex, e.sprite->getPos() + gameArea.topLeft);
    }

    // Draw snake.
    //
    if (snake.points.size() >= 2)
    {
        size_t numPoints = snake.points.size();
        RCouple prevPoint = snake.points.getPoint(0) + gameArea.topLeft;
        for (size_t i = 1; i < numPoints; ++i)
        {
            RCouple curPoint = snake.points.getPoint(i) + gameArea.topLeft;

            if (prevPoint != curPoint)
                drawLine(int(prevPoint.x), int(prevPoint.y), int(curPoint.x), int(curPoint.y), snakeColor);

            prevPoint = curPoint;
        }

        if (false)  // draw loops and intersection points (useful for debugging)
            for (vector<Intersection *>::const_iterator it = snake.points.getIntersections().begin();
                                                     it != snake.points.getIntersections().end(); ++it)
            {
                const Intersection *inter = *it;

                const vector<vector<flatzebra::RCouple> *> &loops = inter->getLoops();
                for (vector<vector<flatzebra::RCouple> *>::const_iterator jt = loops.begin();
                                                                         jt != loops.end(); ++jt)
                    drawLoop(**jt);
            }
    }

    // Snake's head (useful if snake is entirely in one position):
    RCouple head = snake.getHeadPos() + gameArea.topLeft;
    fillCircle(int(head.x), int(head.y), 3, snakeColor);


    // Show number of lives left as three horizontal bars in livesArea.
    // They are blue, except for the "current" life which is red.
    //
    if (numLives > 0)
    {
        const RCouple barSize(40, 4);
        const double gapWidth = 20;
        const RCouple totalSize(barSize.x * 3 + gapWidth * 2, barSize.y);

        RCouple topLeft = livesArea.topLeft + (livesArea.size - totalSize) / 2;

        for (size_t i = 0; i < 3; ++i)  // left to right
        {
            drawFilledRect(topLeft, barSize, 3 - numLives == i ? currentLifeColor : otherLivesColor);
            topLeft.x += barSize.x + gapWidth;
        }
    }


    // Draw (and animate) the floating numbers, if any.
    //
    for (vector<FloatingNumber *>::iterator it = floatingNumbers.begin();
                                           it != floatingNumbers.end(); )
    {
        FloatingNumber *fn = *it;
        if (fn->numTicksLeft == 0)
        {
            removeVectorElementQuick(floatingNumbers, it);
            delete fn;
        }
        else
        {
            char tmp[32];
            snprintf(tmp, sizeof(tmp), "%ld", fn->number);

            // Position string so that number is centered at fn->pos
            // in the game area.
            //
            size_t widthInPixels = strlen(tmp) * CHAR_WIDTH;
            RCouple pos = fn->pos + gameArea.topLeft - RCouple(widthInPixels, CHAR_HEIGHT) / 2;
            gfxWriteString(tmp, pos.round(), floatingNumberColor);

            fn->pos.y -= fn->speed;
            --fn->numTicksLeft;
            ++it;
        }
    }


    // Draw written info (score, etc).
    //
    if (gamePhase == WAITING_FOR_PLAYER)
    {
        static const char *lines[] =
        {
            "Quadrupleback " VERSION " - by Pierre Sarrazin",
            "",
            "Based on Doubleback, a Color Computer game by Dale Lear",
            "",
            "",
            "Object: score points by enclosing objects with the line",
            "",
            "Avoid touching objects with the head of the line",
            "",
            "You have 3 lives",
            "",
            "",
            "To move: keyboard arrows or joystick  ",
            "To slow down: [Control] or gamepad [A]",
            "",
            "",
            "Press [Space] or [Start] to begin game",
            NULL
        };


        Sint16 y = 10 * CHAR_HEIGHT;
        for (int i = 0; lines[i] != NULL; ++i, y += CHAR_HEIGHT)
            writeCenteredString(y, lines[i], presentationColor);

        if (!errorMessage.empty())
        {
            y += CHAR_HEIGHT;
            writeCenteredString(y, ("ERROR: " + errorMessage).c_str(), errorMessageColor);
        }
    }
    else
    {
        if (paused)
        {
            const char *msg = "  PAUSED -- to resume, press [P] or gamepad [Back] button  ";
            RCouple bgRectSize(strlen(msg) * CHAR_WIDTH, 3 * CHAR_HEIGHT);
            RCouple bgRectPos = (theScreenSizeInPixels - bgRectSize) / 2;
            drawFilledRect(bgRectPos, bgRectSize, pauseBGColor);
            writeCenteredString(bgRectPos.y + CHAR_HEIGHT, msg, presentationColor);
        }
    }

    // Always display score, so that score of finished game remains visible.
    //
    char tmp[32];
    snprintf(tmp, sizeof(tmp), "%ld", score);
    writeCenteredString(scoreArea.topLeft.y, tmp, scoreArea.fgColor);

    // Draw a transparent rectangle over the whole screen.
    //
    if (gamePhase == PLAYER_DYING)
        drawFilledRect(RCouple(0, 0), theScreenSizeInPixels, deathColor);
}


void
QuadruplebackEngine::writeCenteredString(Sint16 y, const char *s, SDL_Color color)
{
    int x = (theScreenSizeInPixels.x - (int) strlen(s) * CHAR_WIDTH) / 2;
    gfxWriteString(s, Couple(x, y), color);
}


void
QuadruplebackEngine::drawLoop(const vector<RCouple> &loop)
{
    vector<RCouple>::const_iterator it = loop.begin();
    if (it == loop.end())
        return;
    RCouple prevPoint = *it++ + gameArea.topLeft;
    for ( ; it != loop.end(); ++it)
    {
        const RCouple &curPoint = *it + gameArea.topLeft;

        if (prevPoint != curPoint)
            drawLine(int(prevPoint.x), int(prevPoint.y), int(curPoint.x), int(curPoint.y), loopColor);

        prevPoint = curPoint;
    }
}


void
QuadruplebackEngine::playSoundEffect(SoundMixer::Chunk &wb)
{
    if (useSound && theSoundMixer != NULL)
    {
        try
        {
            theSoundMixer->playChunk(wb);
        }
        catch (const SoundMixer::Error &e)
        {
            cerr << PACKAGE << ": playSoundEffect: " << e.what() << endl;
        }
    }
}
